//=============================================================================
// NrWindowSizeManager.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc ウィンドウサイズ変更プラグイン v1.0.0
 * @author NJ
 *
 * @param EnabledSizes
 * @text 有効サイズリスト
 * @type number[]
 * @default ["0", "1", "2", "3", "4"]
 * @desc オプション画面で選択可能な画面サイズラベルの配列
 *
 * @help
 * スクリプトコマンド:
 *   changeWindowSize(n)   // サイズを対応した番号に変更
 *   toggleFullScreen()    // フルスクリーン状態をトグルで切り替える
 *
 * サイズ番号一覧:
 *   0 : 816  * 624
 *   1 : 1024 * 576
 *   2 : 1280 * 720
 *   3 : 1600 * 900
 *   4 : 1920 * 1080
 *
 * このプラグインは、画面サイズとフルスクリーン状態を保存してリアルタイムに変更し、
 * 次回起動時にはそのサイズを読み込んで起動します。
 *
 * バージョン
 *   1.0.0 - 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(() => {
    const parameters = PluginManager.parameters("NrWindowSizeManager");
    const EnabledSizes = JSON.parse(parameters.EnabledSizes || "[]").map(Number);

    const NrWindowSizeManager = {
        SIZES: {
            0: { width: 816, height: 624 },
            1: { width: 1024, height: 576 },
            2: { width: 1280, height: 720 },
            3: { width: 1600, height: 900 },
            4: { width: 1920, height: 1080 }
        },
        FILE_NAME: 'windowsize',

        resizeWindow(width, height) {
            const frameWidth = window.outerWidth - window.innerWidth;
            const frameHeight = window.outerHeight - window.innerHeight;
            window.resizeTo(width + frameWidth, height + frameHeight);
        },

        applySize(label) {
            ConfigManager.screenSize = label;
            this.saveSettings(label);
            if (!Graphics._isFullScreen()) {
                this.applyRecordedSize();
            }
        },

        applyRecordedSize() {
            const label = ConfigManager.screenSize;
            const size = this.SIZES[label];
            if (size) {
                this.resizeWindow(size.width, size.height);
            }
        },

        saveSettings(currentLabel) {
            const data = {
                sizeLabel: currentLabel ?? ConfigManager.screenSize,
                isFullScreen: Graphics._isFullScreen()
            };
            StorageManager.saveToLocalFile(this.FILE_NAME, JSON.stringify(data)).catch(() => {});
        },

        async loadSettings() {
            try {
                const json = await StorageManager.loadFromLocalFile(this.FILE_NAME);
                return JSON.parse(json);
            } catch {
                const initialLabel = this.getClosestLabelToCurrentSize();
                ConfigManager.screenSize = initialLabel;
                this.saveSettings(initialLabel);
                return {
                    sizeLabel: initialLabel,
                    isFullScreen: false
                };
            }
        },

        getClosestLabelToCurrentSize() {
            const w = window.innerWidth;
            const h = window.innerHeight;
            let closest = null;
            let minDiff = Infinity;
            for (const [label, size] of Object.entries(this.SIZES)) {
                const diff = Math.abs(size.width - w) + Math.abs(size.height - h);
                if (diff < minDiff) {
                    closest = Number(label);
                    minDiff = diff;
                }
            }
            return closest ?? 0;
        },

        async applySavedSettings() {
            const data = await this.loadSettings();
            if (!data) return;
            if (data.sizeLabel !== undefined && this.SIZES[data.sizeLabel]) {
                ConfigManager.screenSize = data.sizeLabel;
                this.applyRecordedSize();
            }
            if (data.isFullScreen && !Graphics._isFullScreen()) {
                Graphics._requestFullScreen();
            }
        },

        getAvailableLabels() {
            return EnabledSizes.filter(id => this.SIZES.hasOwnProperty(id));
        },

        getSizeOptions() {
            return this.getAvailableLabels().map(id => {
                const size = this.SIZES[id];
                return `${size.width}×${size.height}`;
            });
        }
    };

    const _SceneManager_initGraphics = SceneManager.initGraphics;
    SceneManager.initGraphics = function() {
        const hasSavedSize = ConfigManager.screenSize !== undefined;
        if (hasSavedSize) {
            Graphics.initialize(this._screenWidth, this._screenHeight, this._boxWidth, this._boxHeight);
            // Graphics.setLoadingImage('img/system/Loading.png'); // RPGツクールMZには存在しないため無効化
        } else {
            _SceneManager_initGraphics.call(this);
        }
    };

    const _Scene_Boot_start = Scene_Boot.prototype.start;
    Scene_Boot.prototype.start = function() {
        _Scene_Boot_start.call(this);
        setTimeout(() => {
            NrWindowSizeManager.applySavedSettings();
        }, 500);
    };

    let _prevFullScreen = Graphics._isFullScreen();
    const _SceneManager_updateMain = SceneManager.updateMain;
    SceneManager.updateMain = function() {
        _SceneManager_updateMain.call(this);
        const nowFullScreen = Graphics._isFullScreen();
        if (_prevFullScreen && !nowFullScreen) {
            NrWindowSizeManager.applyRecordedSize();
        }
        _prevFullScreen = nowFullScreen;
    };

    const _SceneManager_terminate = SceneManager.terminate;
    SceneManager.terminate = function() {
        NrWindowSizeManager.saveSettings();
        _SceneManager_terminate.call(this);
    };

    window.addEventListener("beforeunload", () => {
        NrWindowSizeManager.saveSettings();
    });

    window.changeWindowSize = function(label) {
        NrWindowSizeManager.applySize(label);
    };

    window.toggleFullScreen = function() {
        Graphics._switchFullScreen();
    };

    const _Window_Options_addGeneralOptions = Window_Options.prototype.addGeneralOptions;
    Window_Options.prototype.addGeneralOptions = function() {
        _Window_Options_addGeneralOptions.call(this);
        this.addCommand("画面サイズ", "screenSize");
        this.addCommand("フルスクリーン", "fullScreen");
    };

    const _Window_Options_statusText = Window_Options.prototype.statusText;
    Window_Options.prototype.statusText = function(index) {
        const symbol = this.commandSymbol(index);
        let value = this.getConfigValue(symbol);
        if (symbol === "screenSize") {
            const labels = NrWindowSizeManager.getAvailableLabels();
            const sizeOptions = NrWindowSizeManager.getSizeOptions();
            const idx = labels.indexOf(value);
            return sizeOptions[idx] || "";
        } else if (symbol === "fullScreen") {
            return value ? "ON" : "OFF";
        } else {
            return _Window_Options_statusText.call(this, index);
        }
    };

    const _Window_Options_processOk = Window_Options.prototype.processOk;
    Window_Options.prototype.processOk = function() {
        const index = this.index();
        const symbol = this.commandSymbol(index);
        const value = this.getConfigValue(symbol);
        if (symbol === "screenSize") {
            const labels = NrWindowSizeManager.getAvailableLabels();
            const currentIndex = labels.indexOf(value);
            const newIndex = (currentIndex + 1) % labels.length;
            const newValue = labels[newIndex];
            this.changeValue(symbol, newValue);
            NrWindowSizeManager.applySize(newValue);
        } else if (symbol === "fullScreen") {
            this.changeValue(symbol, !value);
            const newValue = this.getConfigValue(symbol);
            if (newValue !== Graphics._isFullScreen()) {
                Graphics._switchFullScreen();
            }
        } else {
            _Window_Options_processOk.call(this);
        }
    };

    const _ConfigManager_makeData = ConfigManager.makeData;
    ConfigManager.makeData = function() {
        const config = _ConfigManager_makeData.call(this);
        config.screenSize = this.screenSize;
        config.fullScreen = this.fullScreen;
        return config;
    };

    const _ConfigManager_applyData = ConfigManager.applyData;
    ConfigManager.applyData = function(config) {
        _ConfigManager_applyData.call(this, config);
        this.screenSize = Number.isInteger(config.screenSize) ? config.screenSize : 0;
        this.fullScreen = typeof config.fullScreen === "boolean" ? config.fullScreen : false;
    };
    
})();
