//=============================================================================
// OnChat2.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc チャットログ機能
 * @author Onmoremind
 * 
 * @param ChatLogSettings
 * @text チャットログ設定
 * @type struct<ChatLogSettings>
 * @desc チャットログ表示に関する各種設定
 *
 * @command ChatLog
 * @text ログ
 * @desc チャットログ操作
 *
 * @arg action
 * @text 処理内容
 * @type select
 * @option 左にログを追加
 * @option 右にログを追加
 * @option ログをクリア
 * @default 左にログを追加
 *
 * @arg message
 * @text 表示メッセージ
 * @desc 表示するテキスト（ログ追加時のみ有効）
 * @type string
 * @default
 *
 * @command ChatStamp
 * @text スタンプ
 * @desc チャットログにスタンプを追加
 *
 * @arg side
 * @text 表示位置
 * @desc スタンプを表示する位置
 * @type select
 * @option 左
 * @option 右
 * @default 左
 *
 * @arg stampName
 * @text スタンプ画像名
 * @desc 使用するピクチャ
 * @type file
 * @dir img/pictures
 *
 * @arg width
 * @text 幅
 * @desc スタンプの表示幅
 * @type number
 * @default 0
 * @min 0
 *
 * @arg height
 * @text 高さ
 * @desc スタンプの表示高さ
 * @type number
 * @default 0
 * @min 0
 * 
 * @arg commonEventId
 * @text コモンイベントID
 * @desc クリックで呼び出すコモンイベント
 * @type common_event
 * @default 0
 *
 * @command ChatWindow
 * @text チャットウィンドウ設定
 * @desc チャットログウィンドウの位置やサイズ、不透明度を変更
 *
 * @arg x
 * @text X座標
 * @type number
 * @default 0
 *
 * @arg y
 * @text Y座標
 * @type number
 * @default 0
 *
 * @arg width
 * @text 幅
 * @type number
 * @default 480
 *
 * @arg height
 * @text 高さ
 * @type number
 * @default 216
 *
 * @arg opacity
 * @text 不透明度
 * @desc 0で完全透明、255で不透明
 * @type number
 * @default 255
 * @max 255
 * @min 0
 *
 * @arg paddingTop
 * @text 上余白
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg paddingBottom
 * @text 下余白
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg paddingLeft
 * @text 左余白
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg paddingRight
 * @text 右余白
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg lineSpacing
 * @text 行間
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg fontSize
 * @text フォントサイズ
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg iconSize
 * @text アイコンサイズ
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg scrollAmount
 * @text スクロール量
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg defaultStampWidth
 * @text デフォルトスタンプ幅
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg defaultStampHeight
 * @text デフォルトスタンプ高さ
 * @type number
 * @default -1
 * @desc -1で変更しない
 *
 * @arg customBackgroundImage
 * @text カスタム背景画像
 * @type file
 * @dir img/pictures
 * @desc 背景画像を指定。空欄で変更しない
 *
 * @arg fontName
 * @text 使用フォント名
 * @type string
 * @desc 空欄で変更しない
 */

/*:
 * @param LayoutPresets
 * @text レイアウトプリセット一覧
 * @type struct<ChatLogSettings>[]
 * @desc 複数のチャットログレイアウト設定。スイッチごとに適用。
 * @default []
 */

/*~struct~ChatLogSettings:
 *
 * @param SwitchID
 * @text 表示スイッチID
 * @type switch
 * @desc チャットログの表示・非表示を切り替えるスイッチ
 * @default 1
 *
 * @param DisableChatSwitchID
 * @text チャット無効スイッチID
 * @type switch
 * @desc ONのときチャットログ追加を無効化
 * @default 2
 *
 * @param ForceHideSwitchIds
 * @text 強制非表示スイッチID一覧
 * @type switch[]
 * @desc ONのときチャットログウィンドウを強制的に非表示にするスイッチ
 * @default []
 *
 * @param MaxLogEntries
 * @text 最大ログ件数
 * @type number
 * @desc 保存する最大件数（古い順に削除）
 * @default 100
 *
 * @param WindowX
 * @text ウィンドウX座標
 * @type number
 * @default 0
 *
 * @param WindowY
 * @text ウィンドウY座標
 * @type number
 * @default 0
 *
 * @param WindowWidth
 * @text ウィンドウ幅
 * @type number
 * @default 480
 *
 * @param WindowHeight
 * @text ウィンドウ高さ
 * @type number
 * @default 216
 *
 * @param SwitchMoves
 * @text スイッチ移動設定
 * @type struct<ChatLogSwitchMove>[]
 * @desc 指定スイッチON時にウィンドウを移動させる設定
 * @default []
 *
 * @param WindowOpacity
 * @text ウィンドウ透明度
 * @type number
 * @min 0
 * @max 255
 * @default 255
 *
 * @param PaddingTop
 * @text 上余白
 * @type number
 * @default 8
 *
 * @param PaddingBottom
 * @text 下余白
 * @type number
 * @default 8
 *
 * @param PaddingLeft
 * @text 左余白
 * @type number
 * @default 12
 *
 * @param PaddingRight
 * @text 右余白
 * @type number
 * @default 12
 *
 * @param LineSpacing
 * @text 行間
 * @type number
 * @default 4
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @default 28
 *
 * @param IconSize
 * @text アイコンサイズ
 * @type number
 * @default 32
 *
 * @param ScrollAmount
 * @text スクロール量
 * @type number
 * @default 40
 *
 * @param DefaultStampWidth
 * @text デフォルトスタンプ幅
 * @type number
 * @default 100
 *
 * @param DefaultStampHeight
 * @text デフォルトスタンプ高さ
 * @type number
 * @default 100
 *
 * @param CustomBackgroundImage
 * @text カスタム背景画像
 * @type file
 * @dir img/pictures
 * @desc 背景画像を指定。空欄なら無し
 *
 * @param FontName
 * @text 使用フォント名
 * @type string
 * @default GameFont
 */

/*~struct~ChatLogSwitchMove:
 *
 * @param Name
 * @text 設定名
 * @type string
 * @default ""
 * @desc 管理しやすいよう任意の名前を付けます（任意）
 *
 * @param SwitchId
 * @text トリガースイッチID
 * @type switch
 * @desc ONの間ウィンドウを移動させるスイッチ
 * @default 0
 *
 * @param TargetX
 * @text 移動先X座標
 * @type number
 * @default ""
 * @desc スイッチON時のウィンドウX座標（未入力で初期値を使用）
 *
 * @param TargetY
 * @text 移動先Y座標
 * @type number
 * @default ""
 * @desc スイッチON時のウィンドウY座標（未入力で初期値を使用）
 *
 * @param Duration
 * @text 移動時間(フレーム)
 * @type number
 * @min 0
 * @default 20
 * @desc ウィンドウが目標座標へ移動するまでのフレーム数
 *
 * @param Easing
 * @text イージング
 * @type select
 * @option リニア
 * @value linear
 * @option イーズイン(Quad)
 * @value easeInQuad
 * @option イーズアウト(Quad)
 * @value easeOutQuad
 * @option イーズインアウト(Quad)
 * @value easeInOutQuad
 * @default easeOutQuad
 * @desc イージングの種類
 */

(() => {
  "use strict";
  


  function getPluginParameter(pluginName, key, defaultValue) {
    const parameters = PluginManager.parameters(pluginName);
    return parameters[key] !== undefined ? parameters[key] : defaultValue;
  }

  function safeJsonParse(jsonString, defaultValue) {
    try {
      return JSON.parse(jsonString);
    } catch (e) {
      return defaultValue;
    }
  }

  function parseLayoutPresets(rawPresets, baseParameters) {
    if (!rawPresets || rawPresets === "[]") {
      const raw = getPluginParameter("OnChat2", "ChatLogSettings", "{}");
      const chatParams = safeJsonParse(raw, {});
      return [{
        ...baseParameters,
        ...chatParams,
      }];
    }

    try {
      const presetsArray = JSON.parse(rawPresets);
      return presetsArray.map(presetJson => {
        const preset = safeJsonParse(presetJson, {});
        return {
          ...baseParameters,
          ...preset,
        };
      });
    } catch (e) {
      console.warn("LayoutPresetsの解析に失敗しました:", e);
      const raw = getPluginParameter("OnChat2", "ChatLogSettings", "{}");
      const chatParams = safeJsonParse(raw, {});
      return [{
        ...baseParameters,
        ...chatParams,
      }];
    }
  }

  const baseParameters = PluginManager.parameters("OnChat2");
  const rawPresets = getPluginParameter("OnChat2", "LayoutPresets", "[]");
  const layoutPresets = parseLayoutPresets(rawPresets, baseParameters);
  

  
  function getCurrentLayoutSettings() {
    if (!$gameSwitches) {
      return layoutPresets[0] || baseParameters;
    }

    for (let i = 0; i < layoutPresets.length; i++) {
      const preset = layoutPresets[i];
      const switchId = Number(preset.SwitchID || 0);
      const switchState = switchId > 0 ? $gameSwitches.value(switchId) : false;

      if (switchId > 0 && switchState) {
        return preset;
      }
    }
    
    return layoutPresets[0] || baseParameters;
  }

  let parameters = layoutPresets[0] || baseParameters;

  const DEFAULT_MOVE_DURATION = 20;
  const DEFAULT_MOVE_EASING = "easeOutQuad";

  function parseSwitchMoves(raw) {
    if (!raw) return [];

    let rawArray;
    try {
      rawArray = JSON.parse(raw);
    } catch (error) {
      console.warn("ChatLogのSwitchMovesを解析できませんでした:", error);
      return [];
    }

    return rawArray
      .map((entry, index) => {
        const data = safeJsonParse(entry, null);
        if (!data) return null;

        const switchId = Number(data.SwitchId || 0);
        if (!switchId) return null;

        const hasX = data.TargetX !== undefined && data.TargetX !== "";
        const hasY = data.TargetY !== undefined && data.TargetY !== "";

        const durationRaw =
          data.Duration !== undefined && data.Duration !== ""
            ? Number(data.Duration)
            : DEFAULT_MOVE_DURATION;
        const duration = Math.max(0, durationRaw || 0);

        const easing = (data.Easing || DEFAULT_MOVE_EASING).toString();

        return {
          name: data.Name || "",
          switchId,
          targetX: hasX ? Number(data.TargetX) : null,
          targetY: hasY ? Number(data.TargetY) : null,
          duration,
          easing,
          key: `switch:${switchId}:${index}`,
        };
      })
      .filter(Boolean);
  }

  function clamp01(value) {
    return Math.max(0, Math.min(1, value));
  }

  function lerp(a, b, t) {
    return a + (b - a) * t;
  }

  function applyEasing(easing, t) {
    const clamped = clamp01(t);
    switch (easing) {
      case "easeInQuad":
        return clamped * clamped;
      case "easeOutQuad":
        return clamped * (2 - clamped);
      case "easeInOutQuad":
        return clamped < 0.5
          ? 2 * clamped * clamped
          : -1 + (4 - 2 * clamped) * clamped;
      case "linear":
      default:
        return clamped;
    }
  }

  let fontName = parameters["FontName"] || "GameFont";
  let customBackgroundImage = parameters["CustomBackgroundImage"] || "";
  let fontLoaded = false;

  if (fontName !== "GameFont") {
    const fontFileName = fontName.endsWith(".woff")
      ? fontName
      : `${fontName}.woff`;
    const baseFontName = fontName.replace(/\.woff$/i, "");

    const font = new FontFace(baseFontName, `url("fonts/${fontFileName}")`);
    font
      .load()
      .then((loadedFont) => {
        document.fonts.add(loadedFont);
        fontLoaded = true;
      })
      .catch((err) => {
        console.warn(
          `フォント "${baseFontName}" の読み込みに失敗しました:`,
          err
        );
        fontLoaded = true;
      });

    fontName = baseFontName;
  } else {
    fontLoaded = true;
  }

  const _Scene_Boot_isReady = Scene_Boot.prototype.isReady;
  Scene_Boot.prototype.isReady = function () {
    return _Scene_Boot_isReady.call(this) && fontLoaded;
  };

  function getCurrentSettings() {
    const currentParams = getCurrentLayoutSettings();
    return {
      disableSwitchId: Number(currentParams["DisableChatSwitchID"] || 2),
      visibleSwitchId: Number(currentParams["SwitchID"] || 1),
      forceHideSwitchIds: JSON.parse(
        currentParams["ForceHideSwitchIds"] || "[]"
      ).map(Number),
      maxEntries: Number(currentParams["MaxLogEntries"] || 100),
      padding: {
        top: Number(currentParams["PaddingTop"] || 8),
        bottom: Number(currentParams["PaddingBottom"] || 8),
        left: Number(currentParams["PaddingLeft"] || 12),
        right: Number(currentParams["PaddingRight"] || 12),
      },
      fontSize: Number(currentParams["FontSize"] || 28),
      iconSize: Number(currentParams["IconSize"] || 32),
      scrollAmount: Number(currentParams["ScrollAmount"] || 40),
      defaultStampWidth: Number(currentParams["DefaultStampWidth"] || 100),
      defaultStampHeight: Number(currentParams["DefaultStampHeight"] || 100),
      backgroundImage: currentParams["CustomBackgroundImage"] || "",
      voiceCompleteId: Number(currentParams["VoiceCompleteSwitchID"] || 0),
      fontName: fontName,
      window: {
        x: Number(currentParams["WindowX"] || 0),
        y: Number(currentParams["WindowY"] || 0),
        width: Number(currentParams["WindowWidth"] || 480),
        height: Number(currentParams["WindowHeight"] || 216),
        opacity: Number(currentParams["WindowOpacity"] || 255),
      },
      lineSpacing: Number(currentParams["LineSpacing"] || 4),
      switchMoves: parseSwitchMoves(currentParams["SwitchMoves"]),
    };
  }

  const settings = getCurrentSettings();

  let {
    x: currentWindowX,
    y: currentWindowY,
    width: currentWindowWidth,
    height: currentWindowHeight,
    opacity: currentWindowOpacity,
  } = settings.window;

  const baseWindowX = settings.window.x;
  const baseWindowY = settings.window.y;
  const switchMoveConfigs = settings.switchMoves || [];

  let windowMoveTween = null;
  let lastTargetKey = null;
  let manualPositionOverride = false;
  let lastSwitchStates = {};

  let {
    top: paddingTop,
    bottom: paddingBottom,
    left: paddingLeft,
    right: paddingRight,
  } = settings.padding;

  let scrollAmount = settings.scrollAmount;
  let maxLogEntries = settings.maxEntries;
  let disableChatSwitchId = settings.disableSwitchId;
  const DEBUG = true;
  let switchId = settings.visibleSwitchId;
  let fontSize = settings.fontSize;
  let iconSize = settings.iconSize;
  let lineSpacing = settings.lineSpacing;
  let fontNameFinal = settings.fontName;
  let voiceCompleteSwitchId = settings.voiceCompleteId;
  let chatEntries = [];
  let scrollY = 0;
  let totalLogHeight = 0;
  let maxScrollY = 0;
  let $chatLogData = {
    entries: [],
    scrollY: 0,
    wasVisible: false,
    windowX: currentWindowX,
    windowY: currentWindowY,
    windowWidth: currentWindowWidth,
    windowHeight: currentWindowHeight,
    windowOpacity: currentWindowOpacity,
  };

  function resolveActiveSwitchTarget() {
    for (const config of switchMoveConfigs) {
      if (config.switchId && $gameSwitches.value(config.switchId)) {
        return {
          x: config.targetX !== null ? config.targetX : baseWindowX,
          y: config.targetY !== null ? config.targetY : baseWindowY,
          duration: config.duration,
          easing: config.easing,
          key: config.key,
        };
      }
    }

    return {
      x: baseWindowX,
      y: baseWindowY,
      duration: DEFAULT_MOVE_DURATION,
      easing: DEFAULT_MOVE_EASING,
      key: "base",
    };
  }

  function startWindowTween(targetX, targetY, duration, easing, key) {
    windowMoveTween = {
      startX: currentWindowX,
      startY: currentWindowY,
      targetX,
      targetY,
      duration: Math.max(0, duration || 0),
      easing: easing || DEFAULT_MOVE_EASING,
      elapsed: 0,
      key,
    };
  }

  function updateWindowTween(windowInstance) {
    if (!windowMoveTween) return;

    if (windowMoveTween.duration === 0) {
      currentWindowX = windowMoveTween.targetX;
      currentWindowY = windowMoveTween.targetY;
      windowInstance.move(
        currentWindowX,
        currentWindowY,
        currentWindowWidth,
        currentWindowHeight
      );
      windowMoveTween = null;
      return;
    }

    windowMoveTween.elapsed = Math.min(
      windowMoveTween.elapsed + 1,
      windowMoveTween.duration
    );
    const progress = windowMoveTween.elapsed / windowMoveTween.duration;
    const eased = applyEasing(windowMoveTween.easing, progress);

    const newX = Math.round(
      lerp(windowMoveTween.startX, windowMoveTween.targetX, eased)
    );
    const newY = Math.round(
      lerp(windowMoveTween.startY, windowMoveTween.targetY, eased)
    );

    if (currentWindowX !== newX || currentWindowY !== newY) {
      currentWindowX = newX;
      currentWindowY = newY;
      windowInstance.move(
        currentWindowX,
        currentWindowY,
        currentWindowWidth,
        currentWindowHeight
      );
    }

    if (windowMoveTween.elapsed >= windowMoveTween.duration) {
      windowMoveTween = null;
    }
  }

  function Window_ChatLog() {
    this.initialize(...arguments);
  }

  Window_ChatLog.prototype = Object.create(Window_Base.prototype);
  Window_ChatLog.prototype.constructor = Window_ChatLog;

  Window_ChatLog.prototype.initialize = function () {
    if ($gameSwitches) {
      const currentSettings = getCurrentSettings();
      
      currentWindowX = currentSettings.window ? currentSettings.window.x : currentWindowX;
      currentWindowY = currentSettings.window ? currentSettings.window.y : currentWindowY;
      currentWindowWidth = currentSettings.window ? currentSettings.window.width : currentWindowWidth;
      currentWindowHeight = currentSettings.window ? currentSettings.window.height : currentWindowHeight;
      currentWindowOpacity = currentSettings.window ? currentSettings.window.opacity : currentWindowOpacity;
      
      switchId = currentSettings.visibleSwitchId || switchId;
      disableChatSwitchId = currentSettings.disableSwitchId || disableChatSwitchId;
      fontSize = currentSettings.fontSize || fontSize;
      iconSize = currentSettings.iconSize || iconSize;
      lineSpacing = currentSettings.lineSpacing || lineSpacing;
      
      if (currentSettings.padding) {
        paddingTop = currentSettings.padding.top;
        paddingBottom = currentSettings.padding.bottom;
        paddingLeft = currentSettings.padding.left;
        paddingRight = currentSettings.padding.right;
      }
    }
    
    const rect = new Rectangle(
      currentWindowX,
      currentWindowY,
      currentWindowWidth,
      currentWindowHeight
    );
    Window_Base.prototype.initialize.call(this, rect);

    if (typeof this.padding === "function") {
      this._paddingCustom = 0;
      this.padding = function () {
        return this._paddingCustom;
      };
    } else {
      this.padding = 0;
    }
    this.createContents();

    this.contents.fontSize = fontSize;

    if (customBackgroundImage) {
      this.windowskin = new Bitmap();
      this.opacity = 0;
      this.backOpacity = 0;
      this.contentsOpacity = 255;
      this._backgroundSprite = new Sprite();
      const bgBitmap = ImageManager.loadPicture(customBackgroundImage);
      this._backgroundSprite.bitmap = bgBitmap;
      this.addChildToBack(this._backgroundSprite);

      bgBitmap.addLoadListener(() => {
        const w = bgBitmap.width;
        const h = bgBitmap.height;

        this._backgroundSprite.x = 0;
        this._backgroundSprite.y = 0;
        this.setWindowSize(
          w,
          h,
          currentWindowX,
          currentWindowY,
          currentWindowOpacity
        );
      });
    } else {
      this.opacity = currentWindowOpacity;
      this.backOpacity = 192;
      this.contentsOpacity = 255;
    }
    this.hide();
    this.updateVisibility();
    this.initializeMouseScroll();
    this.redrawLog();
    this._draggingScrollbar = false;
    this._dragOffsetY = 0;

    this._refreshChatMask();
  };

  Window_ChatLog.prototype.updateBackgroundImage = function () {
    if (this._backgroundSprite) {
      this.removeChild(this._backgroundSprite);
      this._backgroundSprite = null;
    }
    
    if (customBackgroundImage && customBackgroundImage.trim() !== "") {
      this.windowskin = new Bitmap();
      this.opacity = 0;
      this.backOpacity = 0;
      this.contentsOpacity = 255;
      this._backgroundSprite = new Sprite();
      const bgBitmap = ImageManager.loadPicture(customBackgroundImage);
      this._backgroundSprite.bitmap = bgBitmap;
      this.addChildToBack(this._backgroundSprite);

      bgBitmap.addLoadListener(() => {
        const w = bgBitmap.width;
        const h = bgBitmap.height;
        this._backgroundSprite.x = 0;
        this._backgroundSprite.y = 0;
        this.setWindowSize(
          w,
          h,
          currentWindowX,
          currentWindowY,
          currentWindowOpacity
        );
      });
    } else {
      this.windowskin = ImageManager.loadSystem("Window");
      this.opacity = currentWindowOpacity;
      this.backOpacity = 192;
      this.contentsOpacity = 255;
    }
  };

  Window_ChatLog.prototype._refreshChatMask = function () {
    try {
      if (!this._chatMask) {
        this._chatMask = new PIXI.Graphics();
        this.addChild(this._chatMask);
        if (this._contentsSprite) {
          this._contentsSprite.mask = this._chatMask;
        }
      }
      const w = this.contents.width;
      const h = this.contents.height;
      const x = paddingLeft;
      const y = paddingTop;
      const rw = Math.max(0, w - paddingLeft - paddingRight);
      const rh = Math.max(0, h - paddingTop - paddingBottom);
      this._chatMask.clear();
      this._chatMask.beginFill(0xffffff, 1);
      this._chatMask.drawRect(x, y, rw, rh);
      this._chatMask.endFill();
    } catch (e) {
    }
  };

  Window_ChatLog.prototype.updateSwitchDrivenPosition = function () {
    let switchStateChanged = false;
    for (const config of switchMoveConfigs) {
      if (config.switchId) {
        const currentState = $gameSwitches.value(config.switchId);
        const lastState = lastSwitchStates[config.switchId];
        if (lastState !== undefined && lastState !== currentState) {
          switchStateChanged = true;
        }
        lastSwitchStates[config.switchId] = currentState;
      }
    }
    
    if (switchStateChanged) {
      manualPositionOverride = false;
    }
    
    if (manualPositionOverride) return;
    
    const target = resolveActiveSwitchTarget();

    if (!target) return;

    const upcomingKey = target.key;
    const needsNewTween =
      (windowMoveTween &&
        (windowMoveTween.targetX !== target.x ||
          windowMoveTween.targetY !== target.y ||
          lastTargetKey !== upcomingKey)) ||
      (!windowMoveTween &&
        (lastTargetKey !== upcomingKey ||
          currentWindowX !== target.x ||
          currentWindowY !== target.y));

    if (needsNewTween) {
      const alreadyAtTarget =
        currentWindowX === target.x && currentWindowY === target.y;

      if (alreadyAtTarget) {
        windowMoveTween = null;
        lastTargetKey = upcomingKey;
      } else {
        startWindowTween(
          target.x,
          target.y,
          target.duration,
          target.easing,
          upcomingKey
        );
        lastTargetKey = upcomingKey;
      }
    }

    updateWindowTween(this);
  };

  Window_ChatLog.prototype._refreshFrame = function () {
    if (customBackgroundImage) return;
    Window_Base.prototype._refreshFrame.call(this);
  };

  Window_ChatLog.prototype.updateSettings = function () {
    const newSettings = getCurrentSettings();
    
    const needsPositionUpdate = 
      currentWindowX !== newSettings.window.x || 
      currentWindowY !== newSettings.window.y ||
      currentWindowWidth !== newSettings.window.width ||
      currentWindowHeight !== newSettings.window.height ||
      currentWindowOpacity !== newSettings.window.opacity;
    
    if (
      switchId !== newSettings.visibleSwitchId ||
      disableChatSwitchId !== newSettings.disableChatSwitchId ||
      fontSize !== newSettings.fontSize ||
      needsPositionUpdate ||
      JSON.stringify(settings.forceHideSwitchIds) !== JSON.stringify(newSettings.forceHideSwitchIds)
    ) {
      switchId = newSettings.visibleSwitchId;
      disableChatSwitchId = newSettings.disableChatSwitchId;
      fontSize = newSettings.fontSize;
      iconSize = newSettings.iconSize;
      lineSpacing = newSettings.lineSpacing;
      scrollAmount = newSettings.scrollAmount;
      maxLogEntries = newSettings.maxEntries;
      
      paddingTop = newSettings.padding.top;
      paddingBottom = newSettings.padding.bottom;
      paddingLeft = newSettings.padding.left;
      paddingRight = newSettings.padding.right;
      
      this.setWindowSize(
        newSettings.window.width,
        newSettings.window.height,
        newSettings.window.x,
        newSettings.window.y,
        newSettings.window.opacity
      );
      
      if (switchId !== newSettings.visibleSwitchId) {
        manualPositionOverride = false;
      }
      
      if (customBackgroundImage !== newSettings.backgroundImage) {
        customBackgroundImage = newSettings.backgroundImage;
        this.updateBackgroundImage();
      }
      
      if (fontNameFinal !== newSettings.fontName) {
        fontNameFinal = newSettings.fontName;
        fontName = newSettings.fontName;
      }
      
      Object.assign(settings, newSettings);
      
      this.redrawLog();
    }
  };

  Window_ChatLog.prototype.updateVisibility = function () {
    this.updateSettings();
    
    const forceHidden = settings.forceHideSwitchIds.some((id) =>
      $gameSwitches.value(id)
    );
    if (forceHidden || !$gameSwitches.value(switchId)) {
      this.hideWindow();
    } else {
      this.showWindow();
    }
  };

  Window_ChatLog.prototype.textPadding = function () {
    return 0;
  };

  Window_ChatLog.prototype.lineHeight = function () {
    return this.contents.fontSize + 6;
  };

  Window_ChatLog.prototype.convertEscapeCharacters = function (text) {
    text = text.replace(/\\+n(?![\[\d])/g, "\n");
    text = text.replace(/\\/g, "\x1b");
    
    text = text.replace(/\x1bV\[(\d+)\]/gi, (_, n) => {
      return $gameVariables.value(Number(n));
    });

    text = text.replace(/\\SV\[([^\]]+)\]/gi, (_, arg) => {
      const expanded = this.convertEscapeCharacters(arg);
      this._lastVoiceName = expanded;
      return "";
    });

    text = text.replace(/\x1bSC\[(\d+)\]/gi, (_, n) => {
      const varValue = $gameVariables.value(Number(n));
      this._lastVoiceName = varValue;
      return "";
    });

    text = text.replace(/\x1bN\[(\d+)\]/gi, (_, n) => {
      const actor = $gameActors.actor(Number(n));
      return actor ? actor.name() : "";
    });

    text = text.replace(/\x1bITEM\[(\d+)\]/gi, (_, id) => {
      const item = $dataItems[parseInt(id, 10)];
      return item ? item.name : "";
    });

    text = text.replace(/\x1bWEAPON\[(\d+)\]/gi, (_, id) => {
      const wep = $dataWeapons[parseInt(id, 10)];
      return wep ? wep.name : "";
    });

    text = text.replace(/\x1bARMOR\[(\d+)\]/gi, (_, id) => {
      const arm = $dataArmors[parseInt(id, 10)];
      return arm ? arm.name : "";
    });

    text = text.replace(/\x1bG/gi, TextManager.currencyUnit);

    return text;
  };

  Window_ChatLog.prototype.initializeMouseScroll = function () {};
  
  Window_ChatLog.prototype.scrollYPlus = function () {
    if (this._scrollAnimation) {
      this._scrollAnimation.active = false;
    }
    scrollY = Math.min(scrollY + scrollAmount, maxScrollY);
    this.redrawLog();
  };

  Window_ChatLog.prototype.scrollYMinus = function () {
    if (this._scrollAnimation) {
      this._scrollAnimation.active = false;
    }
    scrollY = Math.max(scrollY - scrollAmount, 0);
    this.redrawLog();
  };

  Window_ChatLog.prototype.redrawLogWithNewEntryOffset = function (newEntryOffset) {
    this.contents.clear();
    this._playedVoices = [];
    this._stampHitAreas = [];
    this.updateTotalHeight();
    
    let currentY = this.contents.height - paddingBottom + scrollY;
    const visibleTop = paddingTop;
    const visibleBottom = this.contents.height - paddingBottom;
    
    for (let i = chatEntries.length - 1; i >= 0; i--) {
      const entry = chatEntries[i];
      let entryHeight;
      if (entry.height) {
        entryHeight = entry.height;
      } else if (entry.text && entry.text.includes("\n")) {
        const lineCnt = entry.text.split("\n").length;
        entryHeight = (fontSize + lineSpacing) * lineCnt;
      } else {
        entryHeight = fontSize + lineSpacing;
      }
      
      currentY -= entryHeight;
      
      let drawY = currentY;
      if (i === chatEntries.length - 1) {
        drawY = currentY + newEntryOffset;
      }
      
      if (drawY + entryHeight <= visibleTop) break;
      if (drawY >= visibleBottom) continue;
      
      if (entry.type === "text") {
        this.contents.fontSize = fontSize;
        const txt = this.convertEscapeCharacters(entry.text);
        let x;
        if (entry.align === "right") {
          const w = this.textWidth(txt);
          x = this.contents.width - paddingRight - w;
        } else {
          x = paddingLeft;
        }
        this.drawChatText(entry.text, x, drawY, entry.align);
      } else if (entry.type === "stamp") {
        const bitmap = ImageManager.loadPicture(entry.stampName);
        if (bitmap && bitmap.isReady()) {
          const w = entry.width;
          const h = entry.height;
          const x = entry.align === "right"
            ? this.contents.width - w - paddingRight
            : paddingLeft;

          this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, x, drawY, w, h);
          this._stampHitAreas.push({
            x, y: drawY, width: w, height: h,
            commonEventId: entry.commonEventId || 0,
          });
        } else {
          bitmap.addLoadListener(() => this.redrawLogWithNewEntryOffset(newEntryOffset));
        }
      }
    }

    this.drawScrollBar();
    this.contents.fontSize = fontSize;
  };

  Window_ChatLog.prototype.redrawLog = function () {
    this.contents.clear();
    this._playedVoices = [];
    this._stampHitAreas = [];
    this.updateTotalHeight();
    let currentY = this.contents.height - paddingBottom + scrollY;
    const visibleTop = paddingTop;
    const visibleBottom = this.contents.height - paddingBottom;
    
    for (let i = chatEntries.length - 1; i >= 0; i--) {
      const entry = chatEntries[i];
      let entryHeight;
      if (entry.height) {
        entryHeight = entry.height;
      } else if (entry.text && entry.text.includes("\n")) {
        const lineCnt = entry.text.split("\n").length;
        entryHeight = (fontSize + lineSpacing) * lineCnt;
      } else {
        entryHeight = fontSize + lineSpacing;
      }
      currentY -= entryHeight;
      
      let drawY = currentY;
      if (typeof this._temporaryNewLogOffset === 'number') {
        if (i === chatEntries.length - 1) {
          drawY = currentY + this._temporaryNewLogOffset;
        } else {
          drawY = currentY + this._temporaryNewLogOffset;
        }
      }
      
      if (drawY + entryHeight <= visibleTop) break; 
      if (drawY >= visibleBottom) continue;
      if (entry.type === "text") {
        this.contents.fontSize = fontSize;

        const txt = this.convertEscapeCharacters(entry.text);
        let x;
        if (entry.align === "right") {
          const w = this.textWidth(txt);
          x = this.contents.width - paddingRight - w;
        } else {
          x = paddingLeft;
        }
        this.drawChatText(entry.text, x, drawY, entry.align);
      } else if (entry.type === "stamp") {
        const bitmap = ImageManager.loadPicture(entry.stampName);
        if (bitmap && bitmap.isReady()) {
          const w = entry.width;
          const h = entry.height;
          const x =
            entry.align === "right"
              ? this.contents.width - w - paddingRight
              : paddingLeft;

          this.contents.blt(
            bitmap,
            0,
            0,
            bitmap.width,
            bitmap.height,
            x,
            drawY,
            w,
            h
          );
          this._stampHitAreas.push({
            x,
            y: drawY,
            width: w,
            height: h,
            commonEventId: entry.commonEventId || 0,
          });
        } else {
          bitmap.addLoadListener(() => this.redrawLog());
        }
      }
    }

    this.drawScrollBar();
    this.contents.fontSize = fontSize;
  };

  Window_ChatLog.prototype.drawScrollBar = function () {
    const barWidth = 8;
    const marginRight = 4;
    const visibleHeight = this.contents.height - paddingTop - paddingBottom;

    if (totalLogHeight <= visibleHeight) return;
    const trackX = this.contents.width - paddingRight - barWidth - marginRight;
    const trackY = paddingTop;
    const trackH = visibleHeight;
    this.contents.paintOpacity = 48;
    this.contents.fillRect(trackX, trackY, barWidth, trackH, "#000000");
    const barRatio = visibleHeight / totalLogHeight;
    const barHeight = Math.floor(visibleHeight * barRatio);
    const maxScroll = totalLogHeight - visibleHeight;
    const barY =
      paddingTop +
      Math.floor(
        ((maxScroll - scrollY) / maxScroll) * (visibleHeight - barHeight)
      );

    this.contents.paintOpacity = 128;
    this.contents.fillRect(trackX, barY, barWidth, barHeight, "#FFFFFF");
    this.contents.paintOpacity = 255;
  };

  Window_ChatLog.prototype.updateTotalHeight = function () {
    let total = 0;
    for (let i = chatEntries.length - 1; i >= 0; i--) {
      const entry = chatEntries[i];
      let entryHeight;
      if (entry.height) {
        entryHeight = entry.height;
      } else if (entry.text && entry.text.includes("\n")) {
        entryHeight = (fontSize + lineSpacing) * 2;
      } else {
        entryHeight = fontSize + lineSpacing;
      }
      if (entry.text && entry.text.includes("\n")) {
        const normalizedText = this.convertEscapeCharacters(entry.text);
        const lineCount = normalizedText.split("\n").length;
        entryHeight = (fontSize + lineSpacing) * lineCount;
      }
      total += entryHeight;
    }
    totalLogHeight = total;

    const visibleHeight = this.contents.height - paddingTop - paddingBottom;
    maxScrollY = Math.max(totalLogHeight - visibleHeight, 0);

    scrollY = Math.min(scrollY, maxScrollY);
  };

  Window_ChatLog.prototype.updateDragScroll = function () {
    const visibleHeight = this.contents.height - paddingTop - paddingBottom;
    if (totalLogHeight <= visibleHeight) return;

    const barWidth = 8;
    const marginRight = 4;
    const pad =
      typeof this.padding === "function"
        ? this.padding()
        : Number(this.padding || 0);
    const mx = TouchInput.x - (this.x + pad);
    const my = TouchInput.y - (this.y + pad);
    const trackX = this.contents.width - paddingRight - barWidth - marginRight;
    const trackY = paddingTop;
    const trackH = visibleHeight;
    const barH = Math.floor((visibleHeight * visibleHeight) / totalLogHeight);
    const maxScroll = totalLogHeight - visibleHeight;
    const barY =
      trackY + Math.floor((trackH - barH) * (1 - scrollY / maxScroll));

    if (!this._draggingScrollbar && TouchInput.isTriggered()) {
      if (
        mx >= trackX &&
        mx <= trackX + barWidth &&
        my >= barY &&
        my <= barY + barH
      ) {
        this._draggingScrollbar = true;
        this._dragOffsetY = my - barY;
        return;
      }
      if (
        mx >= trackX &&
        mx <= trackX + barWidth &&
        my >= trackY &&
        my <= trackY + trackH
      ) {
        let targetRelY = my - trackY - barH / 2;
        targetRelY = Math.max(0, Math.min(trackH - barH, targetRelY));
        const targetScrollY = Math.round(maxScroll * (1 - targetRelY / (trackH - barH)));
        scrollY = targetScrollY;
        this.redrawLog();
        return;
      }
    }

    if (this._draggingScrollbar) {
      if (TouchInput.isPressed()) {
        let relY = my - this._dragOffsetY - trackY;
        relY = Math.max(0, Math.min(trackH - barH, relY));
        scrollY = Math.round(maxScroll * (1 - relY / (trackH - barH)));
        this.redrawLog();
        return;
      }
      if (TouchInput.isReleased()) {
        this._draggingScrollbar = false;
      }
    }
  };

  Window_ChatLog.prototype.animateNewLogEntry = function (lineHeight) {
    if (this._newLogAnimation && this._newLogAnimation.active) {
      this.redrawLog();
      return;
    }
    
    const animationOffset = lineHeight;
    
    this._newLogAnimation = {
      offset: animationOffset,
      targetOffset: 0,
      duration: 60,
      elapsed: 0,
      active: true
    };
  };

  Window_ChatLog.prototype.scrollToPosition = function (targetScrollY, animated = false) {
    if (!animated) {
      scrollY = targetScrollY;
      this.redrawLog();
      return;
    }
    
    const startScrollY = scrollY;
    
    if (startScrollY === targetScrollY) {
      return;
    }
    
    if (this._scrollAnimation && this._scrollAnimation.active) {
      this._scrollAnimation.active = false;
    }
    
    this._scrollAnimation = {
      startScrollY: startScrollY,
      targetScrollY: targetScrollY,
      duration: 60,
      elapsed: 0,
      active: true
    };
  };

  Window_ChatLog.prototype.scrollToBottom = function (animated = false) {
    this.updateTotalHeight();
    if (!animated) {
      scrollY = 0;
      this.redrawLog();
      return;
    }
    
    this.scrollToPosition(0, true);
  };

  Window_ChatLog.prototype.clearLog = function () {
    chatEntries = [];
    scrollY = 0;
    totalLogHeight = 0;
    maxScrollY = 0;
    this.contents.clear();
  };

  Window_ChatLog.prototype.hideWindow = function () {
    this.visible = false;
    this.hide();
  };

  Window_ChatLog.prototype.showWindow = function () {
    this.visible = true;
    this.show();
    this.redrawLog();
  };

  Window_ChatLog.prototype.setWindowSize = function (
    width,
    height,
    x,
    y,
    opacity
  ) {
    if (width !== undefined) currentWindowWidth = width;
    if (height !== undefined) currentWindowHeight = height;
    if (x !== undefined) currentWindowX = x;
    if (y !== undefined) currentWindowY = y;
    if (opacity !== undefined) currentWindowOpacity = opacity;

    this.move(
      currentWindowX,
      currentWindowY,
      currentWindowWidth,
      currentWindowHeight
    );
    this.opacity = currentWindowOpacity;
    this.createContents();
    this.redrawLog();
    this._refreshChatMask();

    windowMoveTween = null;
    lastTargetKey = null;
  };

  Window_ChatLog.prototype.drawChatText = function (text, x, y, align) {
    const normalizedText = text.replace(/\\+n(?![\[\d])/g, "\n");
    const textState = {
      index: 0,
      x: x,
      y: y,
      left: x,
      text: this.convertEscapeCharacters(normalizedText),
      drawing: true,
    };

    this.contents.fontFace = fontName;
    this.contents.fontSize = fontSize;
    this.resetTextColor();
    this.contents.fontBold = false;
    this.contents.fontItalic = false;
    this.contents.outlineWidth = 4;

    const controlCodeMarker = "\x1b";
    while (textState.index < textState.text.length) {
      const c = textState.text.charAt(textState.index);
      if (c === controlCodeMarker) {
        textState.index++;
        const code = this.obtainEscapeCode(textState);
        this.processChatEscapeCharacter(code, textState);
      } else if (c === "\n") {
        textState.x = textState.left;
        textState.y += this.lineHeight();
        textState.index++;
      } else {
        const w = this.textWidth(c);
        this.contents.drawText(
          c,
          textState.x,
          textState.y,
          w,
          this.lineHeight(),
          "left"
        );
        textState.x += w;
        textState.index++;
      }
    }
  };

  Window_ChatLog.prototype.resetFontSettings = function () {
    this.contents.fontFace = fontName;
    this.contents.fontSize = fontSize;
    this.resetTextColor();
    this.contents.fontBold = false;
    this.contents.fontItalic = false;
  };

  Window_ChatLog.prototype.processChatEscapeCharacter = function (
    code,
    textState
  ) {
    switch (code) {
      case "C": {
        const colorIndex = this.obtainEscapeParam(textState);
        this.changeTextColor(ColorManager.textColor(colorIndex));
        break;
      }
      case "I": {
        const iconIndex = this.obtainEscapeParam(textState);
        this.drawIcon(iconIndex, textState.x, textState.y + 2);
        textState.x += this.iconWidth;
        break;
      }
      case "FS": {
        const size = this.obtainEscapeParam(textState);
        this.contents.fontSize = size;
        break;
      }
      case "V": {
        const variableId = this.obtainEscapeParam(textState);
        const value = $gameVariables.value(variableId).toString();
        for (let i = 0; i < value.length; i++) {
          const char = value.charAt(i);
          const w = this.textWidth(char);
          this.contents.drawText(
            char,
            textState.x,
            textState.y,
            w,
            this.lineHeight(),
            "left"
          );
          textState.x += w;
        }
        break;
      }
      case "N": {
        const actorId = this.obtainEscapeParam(textState);
        const actor = $gameActors.actor(actorId);
        const name = actor ? actor.name() : "";
        for (let i = 0; i < name.length; i++) {
          const char = name.charAt(i);
          const w = this.textWidth(char);
          this.contents.drawText(
            char,
            textState.x,
            textState.y,
            w,
            this.lineHeight(),
            "left"
          );
          textState.x += w;
        }
        break;
      }
      case "ITEM": {
        const itemId = this.obtainEscapeParam(textState);
        const item = $dataItems[itemId];
        const name = item ? item.name : "";
        for (let i = 0; i < name.length; i++) {
          const char = name.charAt(i);
          const w = this.textWidth(char);
          this.contents.drawText(
            char,
            textState.x,
            textState.y,
            w,
            this.lineHeight(),
            "left"
          );
          textState.x += w;
        }
        break;
      }
      case "WEAPON": {
        const weaponId = this.obtainEscapeParam(textState);
        const weapon = $dataWeapons[weaponId];
        const name = weapon ? weapon.name : "";
        for (let i = 0; i < name.length; i++) {
          const char = name.charAt(i);
          const w = this.textWidth(char);
          this.contents.drawText(
            char,
            textState.x,
            textState.y,
            w,
            this.lineHeight(),
            "left"
          );
          textState.x += w;
        }
        break;
      }
      case "ARMOR": {
        const armorId = this.obtainEscapeParam(textState);
        const armor = $dataArmors[armorId];
        const name = armor ? armor.name : "";
        for (let i = 0; i < name.length; i++) {
          const char = name.charAt(i);
          const w = this.textWidth(char);
          this.contents.drawText(
            char,
            textState.x,
            textState.y,
            w,
            this.lineHeight(),
            "left"
          );
          textState.x += w;
        }
        break;
      }
      case "G": {
        const unit = TextManager.currencyUnit;
        for (let i = 0; i < unit.length; i++) {
          const char = unit.charAt(i);
          const w = this.textWidth(char);
          this.contents.drawText(
            char,
            textState.x,
            textState.y,
            w,
            this.lineHeight(),
            "left"
          );
          textState.x += w;
        }
        break;
      }
      case "FB": {
        this.contents.fontBold = !this.contents.fontBold;
        break;
      }
      case "FI": {
        this.contents.fontItalic = !this.contents.fontItalic;
        break;
      }
      case "OW": {
        const width = this.obtainEscapeParam(textState);
        this.contents.outlineWidth = width;
        break;
      }
      case "SV": {
        const match = textState.text
          .substring(textState.index)
          .match(/^\[([^\]]+)\]/);
        if (match) {
          let fileName = match[1];
          textState.index += match[0].length;

          fileName = this.convertEscapeCharacters(fileName);

          if (this._playedVoices && this._playedVoices.includes(fileName)) {
          } else {
            this._lastVoiceName = fileName;
          }
        }
        break;
      }
      case "SC": {
        const variableId = this.obtainEscapeParam(textState);
        const fileName = $gameVariables.value(variableId);
        if (this._playedVoices && this._playedVoices.includes(fileName)) {
        } else {
          this._lastVoiceName = fileName;
        }
        break;
      }
    }
  };

  Window_ChatLog.prototype.obtainEscapeCode = function (textState) {
    const regExp = /^([A-Z]+)/i;
    const match = textState.text.substring(textState.index).match(regExp);
    if (match) {
      textState.index += match[1].length;
      return match[1].toUpperCase();
    }
    return "";
  };

  Window_ChatLog.prototype.obtainEscapeParam = function (textState) {
    const regExp = /^\[(\d+)\]/;
    const match = textState.text.substring(textState.index).match(regExp);
    if (match) {
      textState.index += match[0].length;
      return Number(match[1]);
    } else {
      return 0;
    }
  };

  Window_ChatLog.prototype._localX = function () {
    const pad =
      typeof this.padding === "function"
        ? this.padding()
        : Number(this.padding || 0);
    return TouchInput.x - this.x - pad;
  };
  
  Window_ChatLog.prototype._localY = function () {
    const pad =
      typeof this.padding === "function"
        ? this.padding()
        : Number(this.padding || 0);
    return TouchInput.y - this.y - pad;
  };

  const _Window_ChatLog_update = Window_ChatLog.prototype.update;
  Window_ChatLog.prototype.update = function () {
    if (_Window_ChatLog_update) {
      _Window_ChatLog_update.call(this);
    } else {
      Window_Base.prototype.update.call(this);
    }

    this.updateSwitchDrivenPosition();
    this.updateDragScroll();

    if (this._newLogAnimation && this._newLogAnimation.active) {
      this._newLogAnimation.elapsed++;
      const progress = this._newLogAnimation.elapsed / this._newLogAnimation.duration;
      
      if (progress >= 1) {
        this._newLogAnimation.active = false;
        this._temporaryNewLogOffset = null;
        this.redrawLog();
      } else {
        const eased = 1 - Math.pow(1 - progress, 2);
        const currentOffset = this._newLogAnimation.offset * (1 - eased);
        
        this._temporaryNewLogOffset = currentOffset;
        this.redrawLog();
      }
    }

    if (this._scrollAnimation && this._scrollAnimation.active) {
      this._scrollAnimation.elapsed++;
      const progress = this._scrollAnimation.elapsed / this._scrollAnimation.duration;
      
      if (progress >= 1) {
        this._scrollAnimation.active = false;
        scrollY = this._scrollAnimation.targetScrollY;
        this.redrawLog();
      } else {
        const eased = 1 - Math.pow(1 - progress, 2); 
        const currentScrollY = this._scrollAnimation.startScrollY + 
          (this._scrollAnimation.targetScrollY - this._scrollAnimation.startScrollY) * eased;
        scrollY = currentScrollY;
        this.redrawLog();
      }
    }

    if (this.visible) {
      const wheelY = TouchInput.wheelY;
      if (wheelY > 0) {
        this.scrollYMinus();
      } else if (wheelY < 0) {
        this.scrollYPlus();
      }
    }

    if (this.visible && TouchInput.isTriggered()) {
      const lx = this._localX();
      const ly = this._localY();
      const visibleLeft = paddingLeft;
      const visibleRight = this.contents.width - paddingRight;
      const visibleTop = paddingTop;
      const visibleBottom = this.contents.height - paddingBottom;
      if (
        lx < visibleLeft ||
        lx > visibleRight ||
        ly < visibleTop ||
        ly > visibleBottom
      ) {
        return;
      }
      for (const area of this._stampHitAreas) {
        if (
          lx >= area.x &&
          lx <= area.x + area.width &&
          ly >= area.y &&
          ly <= area.y + area.height
        ) {
          if (area.commonEventId > 0) {
            $gameTemp.reserveCommonEvent(area.commonEventId);
          }
          break;
        }
      }
    }
  };

  Window_ChatLog.prototype.addChatText = function (text, align) {
    const currentSettings = getCurrentSettings();
    if ($gameSwitches.value(currentSettings.disableSwitchId)) return;

    const rawText = text;
    const displayText = this.convertEscapeCharacters(text);

    if (
      displayText &&
      window.NrSimpleVoiceControlParams &&
      window.NrSimpleVoiceControlParams.enableVoicePlayback
    ) {
      if (this._lastVoiceName == undefined) {
        const svMatch = /\\SV\[(.+?)\]/i.exec(rawText);
        this._lastVoiceName = svMatch ? svMatch[1] : null;
      }

      if (this._lastVoiceName) {
          const voiceName = String(this._lastVoiceName).trim();
          if (voiceName && window.NrPlayVoice) {
              window.NrPlayVoice(voiceName);
              this._lastVoiceName = null;
          }
      }
    }

    const lineCount = displayText.split("\n").length;
    const entryHeight = (fontSize + lineSpacing) * lineCount;
    
    const entry = {
      type: "text",
      text: displayText,
      align: align,
      fontSize: fontSize,
      height: entryHeight,
    };

    chatEntries.push(entry);
    if (chatEntries.length > maxLogEntries) {
      chatEntries.shift();
    }

    this.updateTotalHeight();
    
    if (!this._newLogAnimation || !this._newLogAnimation.active) {
      this.animateNewLogEntry(entryHeight);
    }
    
    this.showWindow();
  };

  Window_ChatLog.prototype.addStamp = function (
    stampName,
    align,
    width,
    height,
    commonEventId = 0
  ) {
    const currentSettings = getCurrentSettings();
    if ($gameSwitches.value(currentSettings.disableSwitchId)) return;

    const defaultWidth = Number(parameters["DefaultStampWidth"]) || 100;
    const defaultHeight = Number(parameters["DefaultStampHeight"]) || 100;
    const targetWidth = width || defaultWidth;
    const targetHeight = height || defaultHeight;
    const bitmap = ImageManager.loadPicture(stampName);

    bitmap.addLoadListener(() => {
      let finalWidth = bitmap.width;
      let finalHeight = bitmap.height;

      if (finalWidth > targetWidth || finalHeight > targetHeight) {
        const scale = Math.min(
          targetWidth / finalWidth,
          targetHeight / finalHeight
        );
        finalWidth = Math.floor(finalWidth * scale);
        finalHeight = Math.floor(finalHeight * scale);
      }

      const entry = {
        type: "stamp",
        stampName,
        align,
        width: finalWidth,
        height: finalHeight,
        commonEventId,
      };

      chatEntries.push(entry);
      if (chatEntries.length > maxLogEntries) chatEntries.shift();

      this.updateTotalHeight();
      
      if (!this._newLogAnimation || !this._newLogAnimation.active) {
        this.animateNewLogEntry(finalHeight);
      }
      
      this.showWindow();
    });
  };

  const _Game_Interpreter_pluginCommand =
    Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function (command, args) {
    if (_Game_Interpreter_pluginCommand) {
      _Game_Interpreter_pluginCommand.call(this, command, args);
    }

    if (!SceneManager._scene || !SceneManager._scene._chatLogWindow) {
      return;
    }

    const chatWindow = SceneManager._scene._chatLogWindow;
    if (!args) args = [];

    switch (command) {
      case "chatleft":
      case "chatright": {
        const text = args.join(" ").replace(/_/g, " ");
        const align = command === "chatleft" ? "left" : "right";
        chatWindow.addChatText(text, align);
        break;
      }
      case "chatsize": {
        const w = Number(args[0] || currentWindowWidth);
        const h = Number(args[1] || currentWindowHeight);
        const x = Number(args[2] || currentWindowX);
        const y = Number(args[3] || currentWindowY);
        const op =
          args[4] !== undefined ? Number(args[4]) : currentWindowOpacity;
        chatWindow.setWindowSize(w, h, x, y, op);
        break;
      }
      case "chatscroll": {
        const val = Number(args[0] || defaultScrollAmount);
        scrollAmount = val;
        break;
      }
      case "chatclear": {
        chatWindow.clearLog();
        break;
      }
      case "chatstopvoice": {
        if (chatWindow._currentVoiceBuffer && chatWindow._currentVoiceBuffer.isPlaying()) {
          chatWindow._currentVoiceBuffer.stop();
          chatWindow._currentVoiceBuffer = null;
        }
        if (window.NrStopVoice) {
          window.NrStopVoice();
        }
        break;
      }
    }
  };

  function onchat(align, content) {
    if (!SceneManager._scene || !SceneManager._scene._chatLogWindow) {
      return;
    }
    const chatWindow = SceneManager._scene._chatLogWindow;

    if (align !== "left" && align !== "right") {
      return;
    }

    if (typeof content === "string") {
      const normalizedContent = content.replace(/\\+n(?![\[\d])/g, "\n");
      chatWindow.addChatText(normalizedContent, align);
    } else if (typeof content === "object" && content.stamp) {
      chatWindow.addStamp(
        content.stamp,
        align,
        content.width,
        content.height
      );
    } else {
    }
  }
  window.onchat = onchat;

  window.OnChatStopVoice = function() {
    if (SceneManager._scene && SceneManager._scene._chatLogWindow) {
      const chatWindow = SceneManager._scene._chatLogWindow;
      if (chatWindow._currentVoiceBuffer && chatWindow._currentVoiceBuffer.isPlaying()) {
        chatWindow._currentVoiceBuffer.stop();
        chatWindow._currentVoiceBuffer = null;
      }
    }
    if (window.NrStopVoice) {
      window.NrStopVoice();
    }
  };


  const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
  Scene_Map.prototype.createAllWindows = function () {
    _Scene_Map_createAllWindows.call(this);
    this._chatLogWindow = new Window_ChatLog();
    const picContainer = this._spriteset._pictureContainer;
    picContainer.addChild(this._chatLogWindow);
    picContainer.setChildIndex(
      this._chatLogWindow,
      picContainer.children.length - 1
    );
    this._chatLogWindow.restoreFromSaveData();
  };

  const _Scene_Map_updateMain = Scene_Map.prototype.updateMain;
  Scene_Map.prototype.updateMain = function () {
    _Scene_Map_updateMain.call(this);

    if (this._chatLogWindow) {
      this._chatLogWindow.update(); 
      this._chatLogWindow.updateVisibility();
      this._chatLogWindow.updateDragScroll();
    }
  };

  Window_ChatLog.prototype.restoreFromSaveData = function () {
    if ($chatLogData.entries) {
      chatEntries = JSON.parse(JSON.stringify($chatLogData.entries));
    }
    scrollY = $chatLogData.scrollY || 0;
    totalLogHeight = 0;
    currentWindowX = $chatLogData.windowX || currentWindowX;
    currentWindowY = $chatLogData.windowY || currentWindowY;
    currentWindowWidth = $chatLogData.windowWidth || currentWindowWidth;
    currentWindowHeight = $chatLogData.windowHeight || currentWindowHeight;
    currentWindowOpacity = $chatLogData.windowOpacity || currentWindowOpacity;

    this.move(
      currentWindowX,
      currentWindowY,
      currentWindowWidth,
      currentWindowHeight
    );
    this.opacity = currentWindowOpacity;
    this.createContents();
    this._refreshChatMask();

    if ($chatLogData.wasVisible) {
      this.showWindow();
    } else {
      this.hideWindow();
    }
    this.redrawLog();

    windowMoveTween = null;
    lastTargetKey = null;
  };

  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function () {
    _Scene_Map_start.call(this);

    if ($chatLogData.wasVisible && this._chatLogWindow) {
      this._chatLogWindow.showWindow();
    }
  };

  const _Scene_Map_terminate = Scene_Map.prototype.terminate;
  Scene_Map.prototype.terminate = function () {
    if (this._chatLogWindow) {
      this._chatLogWindow.saveChatLogData();
    }
    _Scene_Map_terminate.call(this);
  };

  Window_ChatLog.prototype.saveChatLogData = function () {
    $chatLogData.entries = JSON.parse(JSON.stringify(chatEntries));
    $chatLogData.scrollY = scrollY;
    $chatLogData.wasVisible = this.visible;
    $chatLogData.windowX = currentWindowX;
    $chatLogData.windowY = currentWindowY;
    $chatLogData.windowWidth = currentWindowWidth;
    $chatLogData.windowHeight = currentWindowHeight;
    $chatLogData.windowOpacity = currentWindowOpacity;
  };

  const _DataManager_makeSaveContents = DataManager.makeSaveContents;
  DataManager.makeSaveContents = function () {
    const contents = _DataManager_makeSaveContents.call(this);
    contents.chatLogData = JSON.parse(JSON.stringify($chatLogData));
    return contents;
  };

  const _DataManager_extractSaveContents = DataManager.extractSaveContents;
  DataManager.extractSaveContents = function (contents) {
    _DataManager_extractSaveContents.call(this, contents);
    if (contents.chatLogData) {
      $chatLogData = JSON.parse(JSON.stringify(contents.chatLogData));
    }
  };

  Window_ChatLog.prototype.standardFontFace = function () {
    return fontName;
  };

  PluginManager.registerCommand("OnChat2", "ChatLog", (args) => {
    const action = args.action;
    const message = args.message || "";

    if (!SceneManager._scene || !SceneManager._scene._chatLogWindow) {
      console.warn("チャットログウィンドウが見つかりません。");
      return;
    }

    const chatWindow = SceneManager._scene._chatLogWindow;

    switch (action) {
      case "左にログを追加":
        chatWindow.addChatText(message, "left");
        break;

      case "右にログを追加":
        chatWindow.addChatText(message, "right");
        break;

      case "ログをクリア":
        chatWindow.clearLog();
        break;
    }
  });

  PluginManager.registerCommand("OnChat2", "ChatStamp", (args) => {
    const side = args.side === "右" ? "right" : "left";
    const stamp = args.stampName || "";
    const w = Number(args.width || 0);
    const h = Number(args.height || 0);
    const ceId = Number(args.commonEventId || 0);

    const chatWindow =
      SceneManager._scene && SceneManager._scene._chatLogWindow;
    chatWindow.addStamp(stamp, side, w, h, ceId);
  });

  PluginManager.registerCommand("OnChat2", "ChatWindow", (args) => {
    const x = Number(args.x || 0);
    const y = Number(args.y || 0);
    const width = Number(args.width || 480);
    const height = Number(args.height || 216);
    const opacity = Number(args.opacity || 255);
    
    const chatWindow =
      SceneManager._scene && SceneManager._scene._chatLogWindow;
    if (chatWindow) {
      chatWindow.setWindowSize(width, height, x, y, opacity);
      manualPositionOverride = true;
      
      if (Number(args.paddingTop) >= 0) {
        paddingTop = Number(args.paddingTop);
      }
      if (Number(args.paddingBottom) >= 0) {
        paddingBottom = Number(args.paddingBottom);
      }
      if (Number(args.paddingLeft) >= 0) {
        paddingLeft = Number(args.paddingLeft);
      }
      if (Number(args.paddingRight) >= 0) {
        paddingRight = Number(args.paddingRight);
      }
      if (Number(args.lineSpacing) >= 0) {
        lineSpacing = Number(args.lineSpacing);
      }
      if (Number(args.fontSize) >= 0) {
        fontSize = Number(args.fontSize);
      }
      if (Number(args.iconSize) >= 0) {
        iconSize = Number(args.iconSize);
      }
      if (Number(args.scrollAmount) >= 0) {
        scrollAmount = Number(args.scrollAmount);
      }
      if (Number(args.defaultStampWidth) >= 0) {
        parameters["DefaultStampWidth"] = String(args.defaultStampWidth);
      }
      if (Number(args.defaultStampHeight) >= 0) {
        parameters["DefaultStampHeight"] = String(args.defaultStampHeight);
      }
      if (args.customBackgroundImage !== undefined) {
        if (args.customBackgroundImage.trim() !== "") {
          customBackgroundImage = args.customBackgroundImage.trim();
        } else {
          customBackgroundImage = "";
        }
        chatWindow.updateBackgroundImage();
      }
      if (args.fontName && args.fontName.trim() !== "") {
        fontName = args.fontName.trim();
      }
      
      chatWindow.redrawLog();
    } else {
      console.warn("⚠️ チャットウィンドウがまだ初期化されていません");
    }
  });
})();

