//=============================================================================
// SpineTimeScale.js
//=============================================================================
/*:
 * @target MZ
 * @plugindesc PictureSpine.js 用 — 特定トラックの timeScale を安全に変更する拡張
 *
 * @command setTrackTimeScale
 * @text トラック再生速度設定
 * @desc 指定ピクチャの Spine の特定トラックの timeScale を変更します。
 *
 * @arg pictureId
 * @type number
 * @text ピクチャ ID
 * @default 1
 *
 * @arg trackIndex
 * @type number
 * @text トラック番号
 * @default 0
 *
 * @arg timeScale
 * @type number
 * @text 再生速度
 * @decimals 2
 * @default 1.0
 *
 * @help
 * 使用例（イベントのスクリプトからでもOK）:
 *   $gameScreen.spine(1).setTrackTimeScale(0, 0.5);
 * または
 *   プラグインコマンド: pictureId=1 trackIndex=0 timeScale=0.5
 *
 * 注意:
 * - PictureSpine.js が導入され、対象ピクチャが Spine 表示である必要があります。
 */

(() => {
    const pluginName = 'SpineTimeScale';

    if (typeof Game_Spine !== 'undefined' && !Game_Spine.prototype.setTrackTimeScale) {
        Game_Spine.prototype.setTrackTimeScale = function(trackIndex, scale) {
            trackIndex = Number(trackIndex);
            scale = Number(scale);
            if (!Number.isFinite(trackIndex) || !Number.isFinite(scale)) return this;
            this._trackTimeScales = this._trackTimeScales || {};
            this._trackTimeScales[trackIndex] = scale;
            return this;
        };

        Game_Spine.prototype.clearTrackTimeScale = function(trackIndex) {
            if (this._trackTimeScales) {
                delete this._trackTimeScales[trackIndex];
            }
            return this;
        };
    }

    const wrapSpriteSpineUpdateTimeScale = () => {
        if (typeof Sprite_Spine === 'undefined') return;

        const proto = Sprite_Spine.prototype;
        if (proto._sts_wrapped) return;
        proto._sts_wrapped = true;

        const _orig_updateTimeScale = proto.updateTimeScale || function() {};

        proto.updateTimeScale = function(spine) {
            _orig_updateTimeScale.call(this, spine);

            try {
                if (!spine || !this._data || !this._data.state) return;
                const map = spine._trackTimeScales;
                if (!map) return;
                for (const k in map) {
                    if (!Object.prototype.hasOwnProperty.call(map, k)) continue;
                    const idx = Number(k);
                    const scale = Number(map[k]);
                    if (!Number.isFinite(idx) || !Number.isFinite(scale)) continue;
                    const entry = this._data.state.getCurrent(idx);
                    if (entry) {
                        entry.timeScale = scale;
                    }
                }
            } catch (e) {
                console.error('SpineTimeScale: updateTimeScale patch error', e);
            }
        };
    };

    wrapSpriteSpineUpdateTimeScale();

    const _Scene_Boot_start = Scene_Boot.prototype.start;
    Scene_Boot.prototype.start = function() {
        _Scene_Boot_start.call(this);
        setTimeout(() => wrapSpriteSpineUpdateTimeScale(), 0);
    };

    PluginManager.registerCommand(pluginName, 'setTrackTimeScale', args => {
        const pictureId  = Number(args.pictureId || 0);
        const trackIndex = Number(args.trackIndex || 0);
        const timeScale  = Number(args.timeScale || 1.0);
        if (!Number.isFinite(pictureId) || pictureId <= 0) return;
        if (!Number.isFinite(trackIndex) || !Number.isFinite(timeScale)) return;

        const spine = $gameScreen.spine(pictureId);
        if (spine && typeof spine.setTrackTimeScale === 'function') {
            spine.setTrackTimeScale(trackIndex, timeScale);
            return;
        }

        try {
            const scene = SceneManager._scene;
            const sprite = scene && scene._spriteset && scene._spriteset._pictureContainer
                ? scene._spriteset._pictureContainer.children.find(s => s._pictureId === pictureId)
                : null;
            const entry = sprite && sprite._data && sprite._data.state ? sprite._data.state.getCurrent(trackIndex) : null;
            if (entry) entry.timeScale = timeScale;
        } catch (e) {
            console.error('SpineTimeScale: plugin command failed', e);
        }
    });

    window.SpineTimeScale = function(pictureId, trackIndex, scale) {
        try {
            const spine = $gameScreen.spine(Number(pictureId));
            if (spine && typeof spine.setTrackTimeScale === 'function') {
                spine.setTrackTimeScale(Number(trackIndex), Number(scale));
                return true;
            }
            const scene = SceneManager._scene;
            const sprite = scene && scene._spriteset && scene._spriteset._pictureContainer
                ? scene._spriteset._pictureContainer.children.find(s => s._pictureId === Number(pictureId))
                : null;
            const entry = sprite && sprite._data && sprite._data.state ? sprite._data.state.getCurrent(Number(trackIndex)) : null;
            if (entry) {
                entry.timeScale = Number(scale);
                return true;
            }
        } catch (e) {
            console.error('SpineTimeScale() error', e);
        }
        return false;
    };

})();
