/*:
 * @target MZ
 * @plugindesc マップにHP/MPゲージと所持金を表示するHUD（マップ除外対応） @author You
 *
 * @param HudX
 * @text HUDのX座標
 * @type number
 * @default 10
 *
 * @param HudY
 * @text HUDのY座標
 * @type number
 * @default 10
 *
 * @param Width
 * @text HUDの幅
 * @type number
 * @default 360
 *
 * @param GaugeHeight
 * @text ゲージの高さ
 * @type number
 * @default 10
 *
 * @param BackgroundColor
 * @text 背景色（CSS形式）
 * @type string
 * @default rgba(0,0,0,0.5)
 * @desc 例: rgba(0,0,0,0.5) / #000000 など
 *
 * @param ExcludedMapIds
 * @text HUDを表示しないマップID
 * @type number[]
 * @default []
 * @desc ここで指定したマップIDではHUDを表示しません。
 *
 * @help
 * マップ画面にシンプルなHUDを表示します。
 * ・HPゲージ＋数値
 * ・MPゲージ＋数値
 * ・右側に所持金（アイコン＋金額）
 *
 * アイコンは System/IconSet のインデックス314固定です。
 * 戦闘中には自動的に非表示になります。
 */

(() => {
    // プラグイン名はファイル名から自動取得（ファイル名と違っても動くように）
    const pluginName = document.currentScript
        ? document.currentScript.src.match(/([^\/]+)\.js$/)[1]
        : "HUD_Display";

    const params = PluginManager.parameters(pluginName);

    const hudX = Number(params["HudX"] || 10);
    const hudY = Number(params["HudY"] || 10);
    const hudWidth = Number(params["Width"] || 360);
    const gaugeHeight = Number(params["GaugeHeight"] || 10);
    const backgroundColor = String(params["BackgroundColor"] || "rgba(0,0,0,0.5)");

    // 除外マップID配列
    const excludedMapIds = (() => {
        const raw = params["ExcludedMapIds"];
        if (!raw) return [];
        try {
            const arr = JSON.parse(raw);
            return arr.map(n => Number(n)).filter(n => !Number.isNaN(n));
        } catch (e) {
            return [];
        }
    })();

    function isMapExcluded() {
        return excludedMapIds.includes($gameMap.mapId());
    }

    //===================================
    // ● HUD用スプライトクラス
    //===================================
    class Sprite_HUD extends Sprite {
        initialize() {
            super.initialize();
            this.x = hudX;
            this.y = hudY;

            // 高さは余白込みで適度に確保（上下余白20＋HP行＋MP行）
            const hudHeight = 120; // 必要なら調整
            this._padding = 10;

            this.bitmap = new Bitmap(hudWidth, hudHeight);
            this._lastHp = -1;
            this._lastMp = -1;
            this._lastGold = -1;
        }

        update() {
            super.update();
            const actor = $gameParty.leader();
            if (!actor) return;

            const hp = actor.hp;
            const mp = actor.mp;
            const gold = $gameParty.gold();

            if (hp !== this._lastHp || mp !== this._lastMp || gold !== this._lastGold) {
                this.refresh();
                this._lastHp = hp;
                this._lastMp = mp;
                this._lastGold = gold;
            }
        }

        refresh() {
            const actor = $gameParty.leader();
            if (!actor) return;

            const pad = this._padding;
            const innerWidth = hudWidth - pad * 2;
            const innerHeight = this.bitmap.height - pad * 5.5;

            this.bitmap.clear();

            // 背景
            this.bitmap.fillRect(0, 0, hudWidth, innerHeight, backgroundColor);

            // 左右レイアウト分割
            // 左側：HP/MP、右側：所持金
            const leftWidth = Math.floor(innerWidth * 0.4);
            const rightWidth = innerWidth - leftWidth;
            const leftX = pad;
            const rightX = pad + leftWidth + 20;
            const topY = pad;

            // ---- HP ----
            const lineHeight = 20;
            const gaugeWidth = leftWidth;
            let y = topY;

            const hpRate = actor.mhp > 0 ? actor.hp / actor.mhp : 0;
            this.drawGauge(leftX, y, gaugeWidth, gaugeHeight, hpRate, "#ff4444", "#552222");
            this.bitmap.fontSize = 14;
            this.bitmap.textColor = "#ffffff";
            this.bitmap.drawText(`HP ${actor.hp}/${actor.mhp}`, leftX, y + gaugeHeight - 10, gaugeWidth, lineHeight, "left");

            // ---- MP ----
            y += gaugeHeight + lineHeight; // HP行の分下へ
            const mpRate = actor.mmp > 0 ? actor.mp / actor.mmp : 0;
            this.drawGauge(leftX, y, gaugeWidth, gaugeHeight, mpRate, "#4488ff", "#223355");
            this.bitmap.drawText(`MP ${actor.mp}/${actor.mmp}`, leftX, y + gaugeHeight - 10, gaugeWidth, lineHeight, "left");

            // ---- 所持金（右カラム）----
            // 上：アイコン＋金額（左揃え）
            const goldY = topY + 11;
            const iconIndex = 314;
            const iconSize = 32;
            const iconX = rightX;
            const iconY = goldY; // 上寄せ

            this.drawIcon(iconIndex, iconX, iconY);

            const goldTextX = iconX + iconSize + 0;
            const goldTextWidth = rightWidth - (iconSize + 2);
            this.bitmap.fontSize = 18;
            this.bitmap.textColor = "#ffffff";
            this.bitmap.drawText(`${$gameParty.gold()}円`, goldTextX, goldY + 2, goldTextWidth, lineHeight, "left");
        }

        // ---------------------------
        // ゲージ描画
        // ---------------------------
        drawGauge(x, y, w, h, rate, color1, color2) {
            const fillW = Math.floor(w * rate);
            this.bitmap.fillRect(x, y, w, h, color2);
            this.bitmap.fillRect(x, y, fillW, h, color1);
        }

        // ---------------------------
        // アイコン描画（IconSetから）
        // ---------------------------
        drawIcon(iconIndex, x, y) {
            const bitmap = ImageManager.loadSystem("IconSet");
            const pw = ImageManager.iconWidth;
            const ph = ImageManager.iconHeight;
            const sx = (iconIndex % 16) * pw;
            const sy = Math.floor(iconIndex / 16) * ph;
            this.bitmap.blt(bitmap, sx, sy, pw, ph, x, y, 24, 24);
        }
    }

    //===================================
    // ● HUDをマップシーンに追加
    //===================================
    const _Scene_Map_createSpriteset = Scene_Map.prototype.createSpriteset;
    Scene_Map.prototype.createSpriteset = function () {
        _Scene_Map_createSpriteset.call(this);

        // 除外マップはHUDを作らない
        if (isMapExcluded()) {
            this._hudSprite = null;
        } else {
            this._hudSprite = new Sprite_HUD();
            this.addChild(this._hudSprite);
        }
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);

        if (this._hudSprite) {
            // バトル中 or 除外マップ なら非表示
            this._hudSprite.visible = !$gameParty.inBattle() && !isMapExcluded();
        }
    };
})();
