//=============================================================================
// RPGツクールMV - LL_StandingPictureBattleMV.js v1.8.0
//-----------------------------------------------------------------------------
// ルルの教会 (Lulu's Church)
// https://nine-yusha.com/
//
// URL below for license details.
// https://nine-yusha.com/plugin/
//=============================================================================

/*:
 * @target MV
 * @plugindesc 战斗中自动显示立绘。
 * @author ルルの教会
 * @url https://nine-yusha.com/plugin-sbpicture/
 *
 * @help LL_StandingPictureBattleMV.js
 *
 * 战斗中，在以下时机自动显示立绘。
 *   ・战斗开始时 (选择战斗・逃跑时)
 *   ・选择命令时
 *   ・受伤时
 *   ・回避时
 *   ・胜利时
 *   ・攻击、防御、技能发动时
 *   ・反击、魔法反射时
 *   ・使用物品时
 *   ・战斗胜利时
 *
 * 如下所示，可以定义多个在状态、开关、变量条件下显示的立绘。
 *   ・开关1打开时且处于毒药状态的立绘
 *   ・变量1为10以上且处于中毒状态的立绘
 *   ・开关1打开时的立绘
 *   ・中毒的立绘
 *   ・无开关状态变量条件的普通立绘(最低限度需要)
 *
 * 用剩余的HP％切换立绘:
 *   首先制作将「剩余的HP％」设置为「100」的立绘列表。
 *   复制上述内容，将「剩余的HP％」更改为「50」，然后复制立绘列表。
 *   こ这样，HP减少到一半以下时，将调用设置为「50」的立绘。
 *   也可以为每个剩余的HP％定义多个立绘。
 *
 * 图像文件的显示优先顺序:
 *   1. 与状态ID、开关ID、变量条件全部一致
 *   2. 与状态ID、开关ID两者一致
 *   3. 与状态ID、变量条件两者一致
 *   4. 仅与状态ID一致
 *   5. 与开关ID、变量条件两者一致
 *   6. 仅开关ID一致
 *   7. 仅与变量条件一致
 *   8. 无条件(状态ID、开关ID、变量条件全部无设置)
 *   (在上述内容中，剩余的HP％最低的优先显示)
 *
 * 如果要翻转图像:
 *   输入X放大率「-100」可反转图像。
 *   (原点位于左上方时，x坐标向左偏移图像宽度)
 *
 * 反击、魔法反射时的立绘显示:
 *   反击时，显示分配给攻击技能的立绘。
 *   魔法反射时，显示分配给反射技能的立绘。
 *
 * 插件命令:
 *   LL_StandingPictureBattleMV setEnabled true   # 将立绘设置为显示
 *   LL_StandingPictureBattleMV setEnabled false  # 将立绘设置为隐藏
 *
 * 使用条款:
 *   ・不需要版权声明。
 *   ・使用时不需要特别报告。
 *   ・商用・非商用不限。
 *   ・R18作品也没有使用限制。
 *   ・根据游戏自由修改没有问题。
 *   ・禁止作为插件素材重新分发(包括修改后)。
 *
 * 作者: ルルの教会
 * 作成日: 2022/6/3
 *
 * @command setEnabled
 * @text 立绘显示ON・OFF
 * @desc 统一控制立绘的显示、隐藏。
 *
 * @arg enabled
 * @text 立绘显示
 * @desc 关闭后，站立图将不再显示。
 * @default true
 * @type boolean
 *
 * @param pictureListSettings
 * @text 立绘列表
 * @desc ※不使用此项目
 *
 * @param sbCommandPictures
 * @text 选择命令时
 * @desc 定义在命令选择过程中显示的立绘。
 * 可以按状态、开关、剩余HP％定义多幅立绘。
 * @default []
 * @type struct<sbCommandPictures>[]
 * @parent pictureListSettings
 *
 * @param sbDamagePictures
 * @text 受伤时
 * @desc 定义受伤时显示的立绘。
 * 可以按状态、开关、剩余HP％定义多幅立绘。
 * @default []
 * @type struct<sbDamagePictures>[]
 * @parent pictureListSettings
 *
 * @param sbEvasionPictures
 * @text 回避时
 * @desc 定义回避时显示的立绘。
 * 可以按状态、开关、剩余HP％定义多幅立绘。
 * @default []
 * @type struct<sbEvasionPictures>[]
 * @parent pictureListSettings
 *
 * @param sbWinPictures
 * @text 胜利时
 * @desc 定义战斗胜利时显示的立绘。
 * 可以按状态、开关、剩余HP％定义多幅立绘。
 * @default []
 * @type struct<sbWinPictures>[]
 * @parent pictureListSettings
 *
 * @param sbActionPictures
 * @text 攻击、防御、技能、反击
 * @desc 定义攻击、技能、物品使用时显示的立绘。
 * 可以按状态、开关、剩余HP％定义多幅立绘。
 * @default []
 * @type struct<sbActionPictures>[]
 * @parent pictureListSettings
 *
 * @param sbItemPictures
 * @text 使用物品时
 * @desc 定义使用物品时显示的立绘。
 * 可以按状态、开关、剩余HP％定义多幅立绘。
 * @default []
 * @type struct<sbItemPictures>[]
 * @parent pictureListSettings
 *
 * @param counterSettings
 * @text 反击时的设置
 * @desc ※不使用此项目
 *
 * @param showCounterAttack
 * @text 反击时显示立绘
 * @desc 反击时显示立绘。
 * 显示分配给攻击技能的立绘。
 * @default true
 * @type boolean
 * @parent counterSettings
 *
 * @param showMagicReflection
 * @text 魔法反射时显示立绘
 * @desc 魔法反射时显示立绘。
 * 显示分配给反射技能的立绘。
 * @default true
 * @type boolean
 * @parent counterSettings
 *
 * @param startActorType
 * @text 战斗开始时的角色显示
 * @desc 请选择战斗・逃跑选择时显示的角色。
 * 显示选择命令时的立绘。
 * @type select
 * @default none
 * @option 不显示
 * @value none
 * @option 第一个角色
 * @value firstActor
 * @option 随机的
 * @value randomActor
 *
 * @param winActorType
 * @text 战斗胜利时的角色显示
 * @desc 请选择战斗胜利时显示的角色。
 * @type select
 * @default lastActor
 * @option 不显示
 * @value none
 * @option 最后行动的角色
 * @value lastActor
 * @option 第一个角色
 * @value firstActor
 * @option 随机的
 * @value randomActor
 *
 * @param hiddenEnemyWindow
 * @text 选择敌人时隐藏
 * @desc 选择敌人时隐藏立绘。
 * @default true
 * @type boolean
 *
 * @param hiddenActorWindow
 * @text 选择目标角色时隐藏
 * @desc 选择目标角色时隐藏立绘。
 * @default false
 * @type boolean
 *
 * @param deathBeforeStates
 * @text 即将死亡时的状态判定
 * @desc 在死亡时，立绘是在临死前的状态下判断的。
 * 如果想在死亡时显示特殊的立绘，请关闭它。
 * @default false
 * @type boolean
 */

/*~struct~sbCommandPictures:
 *
 * @param memo
 * @text 备注栏
 * @desc 为了便于在立绘列表中确认，可以备注标记。
 * 此项对功能没有任何影响。
 * @type string
 *
 * @param actorId
 * @text 角色ID
 * @desc 是角色ID。 请选择定义立绘的角色。
 * @type actor
 *
 * @param stateId
 * @text 状态ID
 * @desc 想在特定状态下变更立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type state
 *
 * @param switchId
 * @text 开关ID
 * @desc 打开开关，想更改立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type switch
 *
 * @param variableCase
 * @text 变量条件
 * @desc 在变量条件下想要变更立绘时使用。
 * @default
 * @type struct<variableCase>
 *
 * @param hpPercentage
 * @text 剩余HP％
 * @desc 在剩余HP％中想更改立绘时使用。
 * 请以100％设置通常时的立绘。
 * @default 100
 * @min 0
 * @max 100
 * @type number
 *
 * @param imageName
 * @text 图像文件名
 * @desc 请选择要显示为立绘的图像文件。
 * @dir img/pictures
 * @type file
 * @require 1
 *
 * @param origin
 * @text 原点
 * @desc 是立绘的原点。
 * @default upperleft
 * @type select
 * @option 左上
 * @value upperleft
 * @option 中心
 * @value center
 *
 * @param x
 * @text X坐标
 * @desc 立绘的显示位置((X)。
 * @default 464
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param y
 * @text Y坐标
 * @desc 立绘的显示位置(Y)。
 * @default 96
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param scaleX
 * @text X放大率
 * @desc 立绘的放大率(X)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param scaleY
 * @text Y放大率
 * @desc 立绘的放大率(Y)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param motion
 * @text 动作
 * @desc 请选择播放立绘的动作。
 * @default floatrightfast
 * @type select
 * @option 无
 * @value none
 * @option 从右侧漂浮(命令)
 * @value floatrightfast
 * @option 从左侧漂浮(命令)
 * @value floatleftfast
 * @option 点头
 * @value yes
 * @option 跳跃
 * @value jump
 * @option 重复跳跃
 * @value jumploop
 * @option 持续哆哆嗦嗦
 * @value shakeloop
 * @option 持续摇晃
 * @value noslowloop
 * @option 呼吸
 * @value breathing
 * @option 呼吸 (伸缩)
 * @value breathing2
 * @option 摇晃(受伤)
 * @value damage
 * @option 从右开始漂浮 (胜利)
 * @value floatright
 * @option 从左开始漂浮 (胜利)
 * @value floatleft
 * @option 向左滑动 (攻击)
 * @value stepleft
 * @option 向右滑动 (攻击)
 * @value stepright
 * @option 低头 (防御)
 * @value headdown
 */

/*~struct~sbDamagePictures:
 *
 * @param memo
 * @text 备注栏
 * @desc 为了便于在立绘列表中确认，可以备注标记。
 * 此项对功能没有任何影响。
 * @type string
 *
 * @param actorId
 * @text 角色ID
 * @desc 是角色ID。 请选择定义立绘的角色。
 * @type actor
 *
 * @param stateId
 * @text 状态ID
 * @desc 想在特定状态下变更立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type state
 *
 * @param switchId
 * @text 开关ID
 * @desc 打开开关，想更改立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type switch
 *
 * @param variableCase
 * @text 变量条件
 * @desc 在变量条件下想要变更立绘时使用。
 * @default
 * @type struct<variableCase>
 *
 * @param hpPercentage
 * @text 剩余HP％
 * @desc 在剩余HP％中想更改立绘时使用。
 * 请以100％设置通常时的立绘。
 * @default 100
 * @min 0
 * @max 100
 * @type number
 *
 * @param imageName
 * @text 图像文件名
 * @desc 请选择要显示为立绘的图像文件。
 * @dir img/pictures
 * @type file
 * @require 1
 *
 * @param origin
 * @text 原点
 * @desc 是立绘的原点。
 * @default upperleft
 * @type select
 * @option 左上
 * @value upperleft
 * @option 中心
 * @value center
 *
 * @param x
 * @text X坐标
 * @desc 立绘的显示位置((X)。
 * @default 464
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param y
 * @text Y坐标
 * @desc 立绘的显示位置(Y)。
 * @default 96
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param scaleX
 * @text X放大率
 * @desc 立绘的放大率(X)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param scaleY
 * @text Y放大率
 * @desc 立绘的放大率(Y)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param motion
 * @text 动作
 * @desc 请选择播放立绘的动作。
 * @default floatrightfast
 * @type select
 * @option 无
 * @value none
 * @option 从右侧漂浮(命令)
 * @value floatrightfast
 * @option 从左侧漂浮(命令)
 * @value floatleftfast
 * @option 点头
 * @value yes
 * @option 跳跃
 * @value jump
 * @option 重复跳跃
 * @value jumploop
 * @option 持续哆哆嗦嗦
 * @value shakeloop
 * @option 持续摇晃
 * @value noslowloop
 * @option 呼吸
 * @value breathing
 * @option 呼吸 (伸缩)
 * @value breathing2
 * @option 摇晃(受伤)
 * @value damage
 * @option 从右开始漂浮 (胜利)
 * @value floatright
 * @option 从左开始漂浮 (胜利)
 * @value floatleft
 * @option 向左滑动 (攻击)
 * @value stepleft
 * @option 向右滑动 (攻击)
 * @value stepright
 * @option 低头 (防御)
 * @value headdown
 */

/*~struct~sbEvasionPictures:
 *
 * @param memo
 * @text 备注栏
 * @desc 为了便于在立绘列表中确认，可以备注标记。
 * 此项对功能没有任何影响。
 * @type string
 *
 * @param actorId
 * @text 角色ID
 * @desc 是角色ID。 请选择定义立绘的角色。
 * @type actor
 *
 * @param stateId
 * @text 状态ID
 * @desc 想在特定状态下变更立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type state
 *
 * @param switchId
 * @text 开关ID
 * @desc 打开开关，想更改立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type switch
 *
 * @param variableCase
 * @text 变量条件
 * @desc 在变量条件下想要变更立绘时使用。
 * @default
 * @type struct<variableCase>
 *
 * @param hpPercentage
 * @text 剩余HP％
 * @desc 在剩余HP％中想更改立绘时使用。
 * 请以100％设置通常时的立绘。
 * @default 100
 * @min 0
 * @max 100
 * @type number
 *
 * @param imageName
 * @text 图像文件名
 * @desc 请选择要显示为立绘的图像文件。
 * @dir img/pictures
 * @type file
 * @require 1
 *
 * @param origin
 * @text 原点
 * @desc 是立绘的原点。
 * @default upperleft
 * @type select
 * @option 左上
 * @value upperleft
 * @option 中心
 * @value center
 *
 * @param x
 * @text X坐标
 * @desc 立绘的显示位置((X)。
 * @default 464
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param y
 * @text Y坐标
 * @desc 立绘的显示位置(Y)。
 * @default 96
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param scaleX
 * @text X放大率
 * @desc 立绘的放大率(X)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param scaleY
 * @text Y放大率
 * @desc 立绘的放大率(Y)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param motion
 * @text 动作
 * @desc 请选择播放立绘的动作。
 * @default floatrightfast
 * @type select
 * @option 无
 * @value none
 * @option 从右侧漂浮(命令)
 * @value floatrightfast
 * @option 从左侧漂浮(命令)
 * @value floatleftfast
 * @option 点头
 * @value yes
 * @option 跳跃
 * @value jump
 * @option 重复跳跃
 * @value jumploop
 * @option 持续哆哆嗦嗦
 * @value shakeloop
 * @option 持续摇晃
 * @value noslowloop
 * @option 呼吸
 * @value breathing
 * @option 呼吸 (伸缩)
 * @value breathing2
 * @option 摇晃(受伤)
 * @value damage
 * @option 从右开始漂浮 (胜利)
 * @value floatright
 * @option 从左开始漂浮 (胜利)
 * @value floatleft
 * @option 向左滑动 (攻击)
 * @value stepleft
 * @option 向右滑动 (攻击)
 * @value stepright
 * @option 低头 (防御)
 * @value headdown
 */

/*~struct~sbWinPictures:
 *
 * @param memo
 * @text 备注栏
 * @desc 为了便于在立绘列表中确认，可以备注标记。
 * 此项对功能没有任何影响。
 * @type string
 *
 * @param actorId
 * @text 角色ID
 * @desc 是角色ID。 请选择定义立绘的角色。
 * @type actor
 *
 * @param stateId
 * @text 状态ID
 * @desc 想在特定状态下变更立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type state
 *
 * @param switchId
 * @text 开关ID
 * @desc 打开开关，想更改立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type switch
 *
 * @param variableCase
 * @text 变量条件
 * @desc 在变量条件下想要变更立绘时使用。
 * @default
 * @type struct<variableCase>
 *
 * @param hpPercentage
 * @text 剩余HP％
 * @desc 在剩余HP％中想更改立绘时使用。
 * 请以100％设置通常时的立绘。
 * @default 100
 * @min 0
 * @max 100
 * @type number
 *
 * @param imageName
 * @text 图像文件名
 * @desc 请选择要显示为立绘的图像文件。
 * @dir img/pictures
 * @type file
 * @require 1
 *
 * @param origin
 * @text 原点
 * @desc 是立绘的原点。
 * @default upperleft
 * @type select
 * @option 左上
 * @value upperleft
 * @option 中心
 * @value center
 *
 * @param x
 * @text X坐标
 * @desc 立绘的显示位置((X)。
 * @default 464
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param y
 * @text Y坐标
 * @desc 立绘的显示位置(Y)。
 * @default 96
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param scaleX
 * @text X放大率
 * @desc 立绘的放大率(X)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param scaleY
 * @text Y放大率
 * @desc 立绘的放大率(Y)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param motion
 * @text 动作
 * @desc 请选择播放立绘的动作。
 * @default floatrightfast
 * @type select
 * @option 无
 * @value none
 * @option 从右侧漂浮(命令)
 * @value floatrightfast
 * @option 从左侧漂浮(命令)
 * @value floatleftfast
 * @option 点头
 * @value yes
 * @option 跳跃
 * @value jump
 * @option 重复跳跃
 * @value jumploop
 * @option 持续哆哆嗦嗦
 * @value shakeloop
 * @option 持续摇晃
 * @value noslowloop
 * @option 呼吸
 * @value breathing
 * @option 呼吸 (伸缩)
 * @value breathing2
 * @option 摇晃(受伤)
 * @value damage
 * @option 从右开始漂浮 (胜利)
 * @value floatright
 * @option 从左开始漂浮 (胜利)
 * @value floatleft
 * @option 向左滑动 (攻击)
 * @value stepleft
 * @option 向右滑动 (攻击)
 * @value stepright
 * @option 低头 (防御)
 * @value headdown
 */

/*~struct~sbActionPictures:
 *
 * @param memo
 * @text 备注栏
 * @desc 为了便于在立绘列表中确认，可以备注标记。
 * 此项对功能没有任何影响。
 * @type string
 *
 * @param actorId
 * @text 角色ID
 * @desc 是角色ID。 请选择定义立绘的角色。
 * @type actor
 *
 * @param itemId
 * @text 技能ID
 * @desc 使用此技能时，将显示立绘。
 * 没有设置时，包括攻击、防御在内的全部技能启动时都会显示。
 * @type skill
 *
 * @param stateId
 * @text 状态ID
 * @desc 想在特定状态下变更立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type state
 *
 * @param switchId
 * @text 开关ID
 * @desc 打开开关，想更改立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type switch
 *
 * @param variableCase
 * @text 变量条件
 * @desc 在变量条件下想要变更立绘时使用。
 * @default
 * @type struct<variableCase>
 *
 * @param hpPercentage
 * @text 剩余HP％
 * @desc 在剩余HP％中想更改立绘时使用。
 * 请以100％设置通常时的立绘。
 * @default 100
 * @min 0
 * @max 100
 * @type number
 *
 * @param imageName
 * @text 图像文件名
 * @desc 请选择要显示为立绘的图像文件。
 * @dir img/pictures
 * @type file
 * @require 1
 *
 * @param origin
 * @text 原点
 * @desc 是立绘的原点。
 * @default upperleft
 * @type select
 * @option 左上
 * @value upperleft
 * @option 中心
 * @value center
 *
 * @param x
 * @text X坐标
 * @desc 立绘的显示位置((X)。
 * @default 464
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param y
 * @text Y坐标
 * @desc 立绘的显示位置(Y)。
 * @default 96
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param scaleX
 * @text X放大率
 * @desc 立绘的放大率(X)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param scaleY
 * @text Y放大率
 * @desc 立绘的放大率(Y)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param motion
 * @text 动作
 * @desc 请选择播放立绘的动作。
 * @default floatrightfast
 * @type select
 * @option 无
 * @value none
 * @option 从右侧漂浮(命令)
 * @value floatrightfast
 * @option 从左侧漂浮(命令)
 * @value floatleftfast
 * @option 点头
 * @value yes
 * @option 跳跃
 * @value jump
 * @option 重复跳跃
 * @value jumploop
 * @option 持续哆哆嗦嗦
 * @value shakeloop
 * @option 持续摇晃
 * @value noslowloop
 * @option 呼吸
 * @value breathing
 * @option 呼吸 (伸缩)
 * @value breathing2
 * @option 摇晃(受伤)
 * @value damage
 * @option 从右开始漂浮 (胜利)
 * @value floatright
 * @option 从左开始漂浮 (胜利)
 * @value floatleft
 * @option 向左滑动 (攻击)
 * @value stepleft
 * @option 向右滑动 (攻击)
 * @value stepright
 * @option 低头 (防御)
 * @value headdown
 */

/*~struct~sbItemPictures:
 *
 * @param memo
 * @text 备注栏
 * @desc 为了便于在立绘列表中确认，可以备注标记。
 * 此项对功能没有任何影响。
 * @type string
 *
 * @param actorId
 * @text 角色ID
 * @desc 是角色ID。 请选择定义立绘的角色。
 * @type actor
 *
 * @param itemId
 * @text 物品ID
 * @desc 使用这个物品时会显示立绘。
 * 如果没有，则在使用所有物品时都会显示。
 * @type item
 *
 * @param stateId
 * @text 状态ID
 * @desc 想在特定状态下变更立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type state
 *
 * @param switchId
 * @text 开关ID
 * @desc 打开开关，想更改立绘时使用。
 * 请将通常时的立绘设置为空白(无)。
 * @type switch
 *
 * @param variableCase
 * @text 变量条件
 * @desc 在变量条件下想要变更立绘时使用。
 * @default
 * @type struct<variableCase>
 *
 * @param hpPercentage
 * @text 剩余HP％
 * @desc 在剩余HP％中想更改立绘时使用。
 * 请以100％设置通常时的立绘。
 * @default 100
 * @min 0
 * @max 100
 * @type number
 *
 * @param imageName
 * @text 图像文件名
 * @desc 请选择要显示为立绘的图像文件。
 * @dir img/pictures
 * @type file
 * @require 1
 *
 * @param origin
 * @text 原点
 * @desc 是立绘的原点。
 * @default upperleft
 * @type select
 * @option 左上
 * @value upperleft
 * @option 中心
 * @value center
 *
 * @param x
 * @text X坐标
 * @desc 立绘的显示位置((X)。
 * @default 464
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param y
 * @text Y坐标
 * @desc 立绘的显示位置(Y)。
 * @default 96
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param scaleX
 * @text X放大率
 * @desc 立绘的放大率(X)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param scaleY
 * @text Y放大率
 * @desc 立绘的放大率(Y)。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param motion
 * @text 动作
 * @desc 请选择播放立绘的动作。
 * @default floatrightfast
 * @type select
 * @option 无
 * @value none
 * @option 从右侧漂浮(命令)
 * @value floatrightfast
 * @option 从左侧漂浮(命令)
 * @value floatleftfast
 * @option 点头
 * @value yes
 * @option 跳跃
 * @value jump
 * @option 重复跳跃
 * @value jumploop
 * @option 持续哆哆嗦嗦
 * @value shakeloop
 * @option 持续摇晃
 * @value noslowloop
 * @option 呼吸
 * @value breathing
 * @option 呼吸 (伸缩)
 * @value breathing2
 * @option 摇晃(受伤)
 * @value damage
 * @option 从右开始漂浮 (胜利)
 * @value floatright
 * @option 从左开始漂浮 (胜利)
 * @value floatleft
 * @option 向左滑动 (攻击)
 * @value stepleft
 * @option 向右滑动 (攻击)
 * @value stepright
 * @option 低头 (防御)
 * @value headdown
 */

/*~struct~variableCase:
 *
 * @param id
 * @text 变量ID
 * @desc 条件中使用的变量ID。
 * @type variable
 *
 * @param type
 * @text 变量条件
 * @desc 与变量ID的比较条件。
 * @default equal
 * @type select
 * @option 一致
 * @value equal
 * @option 以上
 * @value higher
 * @option 以下
 * @value lower
 *
 * @param value
 * @text 变量比较数值
 * @desc 是要与变量ID进行比较的数值。
 * @default 0
 * @min -99999999
 * @max 99999999
 * @type number
 */

(function() {
	"use strict";
	var pluginName = "LL_StandingPictureBattleMV";

	var parameters = PluginManager.parameters(pluginName);
	var paramJsonParse = function(key, value) {
		try {
			return JSON.parse(value);
		} catch(e) {
			return value;
		}
	};

	var sbCommandPictures = String(parameters["sbCommandPictures"] || "[]");
	var sbDamagePictures = String(parameters["sbDamagePictures"] || "[]");
	var sbEvasionPictures = String(parameters["sbEvasionPictures"] || "[]");
	var sbWinPictures = String(parameters["sbWinPictures"] || "[]");
	var sbActionPictures = String(parameters["sbActionPictures"] || "[]");
	var sbItemPictures = String(parameters["sbItemPictures"] || "[]");
	var startActorType = String(parameters["startActorType"] || "none");
	var winActorType = String(parameters["winActorType"] || "lastActor");
	var hiddenEnemyWindow = eval(parameters["hiddenEnemyWindow"] || "true");
	var hiddenActorWindow = eval(parameters["hiddenActorWindow"] || "false");
	var deathBeforeStates = eval(parameters["deathBeforeStates"] || "false");

	// 反撃時の設定
	var showCounterAttack = eval(parameters["showCounterAttack"] || "true");
	var showMagicReflection = eval(parameters["showMagicReflection"] || "true");

	var sbCommandPictureLists = JSON.parse(JSON.stringify(sbCommandPictures, paramJsonParse));
	var sbDamagePictureLists = JSON.parse(JSON.stringify(sbDamagePictures, paramJsonParse));
	var sbEvasionPictureLists = JSON.parse(JSON.stringify(sbEvasionPictures, paramJsonParse));
	var sbWinPictureLists = JSON.parse(JSON.stringify(sbWinPictures, paramJsonParse));
	var sbActionPictureLists = JSON.parse(JSON.stringify(sbActionPictures, paramJsonParse));
	var sbItemPictureLists = JSON.parse(JSON.stringify(sbItemPictures, paramJsonParse));

	// Plugin Command (for MV)
	var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command === pluginName) {
            switch (args[0]) {
				case 'setEnabled':
					var enabled = eval(args[1] || "true");
					$gameSystem._StandingPictureBattleDisabled = !enabled;
					break;
            }
        }
	};

	// 独自変数定義
	var animationCount = 0;
	var refSbPicture = false;
	var motionSbPicture = "";
	var showDamageActorId = null;
	var showActionActorId = null;
	var showEvasionActorId = null;
	var activeCommandActorId = null;
	var activeDamageActorId = null;
	var activeActionActorId = null;
	var activeActionItemId = null;
	var activeActionDataClass = null;

	// アニメーションフレーム数定義
	var animationFrame = {
		"yes":            24,
		"yesyes":         48,
		"no":             24,
		"noslow":         48,
		"jump":           24,
		"jumpjump":       48,
		"jumploop":       48,
		"shake":          1,
		"shakeloop":      1,
		"runleft":        1,
		"runright":       1,
		"damage":         1,
		"floatrightfast": 12,
		"floatright":     48,
		"floatleftfast":  12,
		"floatleft":      48,
		"noslowloop":     96,
		"breathing":      96,
		"breathing2":     96,
		"stepleft":       24,
		"stepright":      24,
		"headdown":       12,
		"none":           0
	};

	//-----------------------------------------------------------------------------
	// 画像ファイル名を取得
	//-----------------------------------------------------------------------------
	var getImageName = function(actorId, pictureLists, itemId) {
		if (!pictureLists) return;

		// for Before ES5
		if (typeof itemId === 'undefined') {
			itemId = null;
		}

		// アクターのステート情報を取得
		var actorStates = [];
		if (actorId) {
			actorStates = $gameActors.actor(actorId)._states;

			// 死亡時に直前のステートIDで判定するか？
			if (actorStates.indexOf(1) !== -1 && deathBeforeStates) {
				actorStates = $gameActors.actor(actorId).beforeStates();
			}
		}
		var specificPicture = null;

		// アクターIDが一致する立ち絵を検索
		pictureLists = pictureLists.filter(function(item) {
			if (Number(item.actorId) == actorId) {
				return true;
			}
		});

		// アイテムID(スキルID)が指定されているか？
		// (処理を共通化するため、ここではskillIdも便宜的にitemIdとして扱う)
		if (itemId !== null) {
			pictureLists = pictureLists.filter(function(item) {
				if (Number(item.itemId) == itemId) {
					return true;
				}
			});
		}

		// ステートにかかっているか？
		if (actorStates.length) {
			// ステートID・スイッチID・変数IDが有効な立ち絵リストを検索
			specificPicture = pictureLists.filter(function(item) {
				if (item.variableCase) {
					if (
						actorStates.indexOf(Number(item.stateId)) !== -1 &&
						$gameSwitches.value(Number(item.switchId)) &&
						(
							String(item.variableCase.type) == "equal" && $gameVariables.value(Number(item.variableCase.id)) == Number(item.variableCase.value) ||
							String(item.variableCase.type) == "higher" && $gameVariables.value(Number(item.variableCase.id)) >= Number(item.variableCase.value) ||
							String(item.variableCase.type) == "lower" && $gameVariables.value(Number(item.variableCase.id)) <= Number(item.variableCase.value)
						)
					) {
						return true;
					}
				}
			});
			if (specificPicture.length) return checkHpPercentage(actorId, specificPicture);
			// ステートID・スイッチIDが有効な立ち絵リストを検索
			specificPicture = pictureLists.filter(function(item) {
				if (actorStates.indexOf(Number(item.stateId)) !== -1 && $gameSwitches.value(Number(item.switchId)) && !item.variableCase) {
					return true;
				}
			});
			if (specificPicture.length) return checkHpPercentage(actorId, specificPicture);
			// ステートID・変数IDが有効な立ち絵リストを検索
			specificPicture = pictureLists.filter(function(item) {
				if (item.variableCase) {
					if (
						actorStates.indexOf(Number(item.stateId)) !== -1 &&
						(Number(item.switchId) === 0 || !item.switchId) &&
						(
							String(item.variableCase.type) == "equal" && $gameVariables.value(Number(item.variableCase.id)) == Number(item.variableCase.value) ||
							String(item.variableCase.type) == "higher" && $gameVariables.value(Number(item.variableCase.id)) >= Number(item.variableCase.value) ||
							String(item.variableCase.type) == "lower" && $gameVariables.value(Number(item.variableCase.id)) <= Number(item.variableCase.value)
						)
					) {
						return true;
					}
				}
			});
			if (specificPicture.length) return checkHpPercentage(actorId, specificPicture);
			// ステートIDが有効な立ち絵リストを検索
			specificPicture = pictureLists.filter(function(item) {
				if (actorStates.indexOf(Number(item.stateId)) !== -1 && (Number(item.switchId) === 0 || !item.switchId) && !item.variableCase) {
					return true;
				}
			});
			if (specificPicture.length) return checkHpPercentage(actorId, specificPicture);
		}

		// スイッチID・変数IDが有効な立ち絵リストを検索
		specificPicture = pictureLists.filter(function(item) {
			if (item.variableCase) {
				if (
					(Number(item.stateId) === 0 || !item.stateId) &&
					$gameSwitches.value(Number(item.switchId)) &&
					(
						String(item.variableCase.type) == "equal" && $gameVariables.value(Number(item.variableCase.id)) == Number(item.variableCase.value) ||
						String(item.variableCase.type) == "higher" && $gameVariables.value(Number(item.variableCase.id)) >= Number(item.variableCase.value) ||
						String(item.variableCase.type) == "lower" && $gameVariables.value(Number(item.variableCase.id)) <= Number(item.variableCase.value)
					)
				) {
					return true;
				}
			}
		});
		if (specificPicture.length) return checkHpPercentage(actorId, specificPicture);
		// スイッチIDが有効な立ち絵リストを検索
		specificPicture = pictureLists.filter(function(item) {
			if ((Number(item.stateId) === 0 || !item.stateId) && $gameSwitches.value(Number(item.switchId)) && !item.variableCase) {
				return true;
			}
		});
		if (specificPicture.length) return checkHpPercentage(actorId, specificPicture);
		// 変数IDが有効な立ち絵リストを検索
		specificPicture = pictureLists.filter(function(item) {
			if (item.variableCase) {
				if (
					(Number(item.stateId) === 0 || !item.stateId) &&
					(Number(item.switchId) === 0 || !item.switchId) &&
					(
						String(item.variableCase.type) == "equal" && $gameVariables.value(Number(item.variableCase.id)) == Number(item.variableCase.value) ||
						String(item.variableCase.type) == "higher" && $gameVariables.value(Number(item.variableCase.id)) >= Number(item.variableCase.value) ||
						String(item.variableCase.type) == "lower" && $gameVariables.value(Number(item.variableCase.id)) <= Number(item.variableCase.value)
					)
				) {
					return true;
				}
			}
		});
		if (specificPicture.length) return checkHpPercentage(actorId, specificPicture);

		// 上記で見つからなかった場合、通常の立ち絵を検索
		let normalPicture = pictureLists.filter(function(item) {
			if ((Number(item.stateId) === 0 || !item.stateId) && (Number(item.switchId) === 0 || !item.switchId) && !item.variableCase) return true;
		});
		if (normalPicture.length) return checkHpPercentage(actorId, normalPicture);
	};

	var checkHpPercentage = function(actorId, pictureLists) {
		// アクターの残HP％を取得
		var hpRate = getHpRate(actorId);
		// 最もHP%が低い立ち絵を適用する
		var minHpRate = 100;
		var result = null;
		pictureLists.forEach(function(item) {
			if (hpRate <= Number(item.hpPercentage) && minHpRate >= Number(item.hpPercentage)) {
				result = item;
				minHpRate = Number(item.hpPercentage);
			} else if (!item.hpPercentage && minHpRate >= 100) {
				// プラグインパラメータが更新されていない場合、便宜的に100として扱う
				result = item;
				minHpRate = Number(item.hpPercentage);
			}
		});
		return result;
	}

	// アクターのHPレートを取得
	var getHpRate = function(actorId) {
		if (!$gameActors.actor(actorId)) return 0;
		return $gameActors.actor(actorId).mhp > 0 ? $gameActors.actor(actorId).hp / $gameActors.actor(actorId).mhp * 100 : 0;
	}

	// Battle Managerを拡張
	BattleManager.isPhase = function() {
		return this._phase;
	};

	// Game Partyを拡張
	Game_Party.prototype.aliveBattleMembers = function() {
		// return this.allMembers()
		// 	.slice(0, this.maxBattleMembers())
		// 	.filter(actor => actor.isAppeared())
		// 	.filter(actor => actor.isAlive());

		// for Ver.1.5.1
		return this.allMembers().slice(0, this.maxBattleMembers()).filter(function(actor) {
			return actor.isAppeared();
		}).filter(function(actor) {
			return actor.isAlive();
		});
	};
	// Game_Party.prototype.firstBattleMember = function() {
	// 	return this.allMembers()
	// 		.slice(0, 1)
	// 		.filter(actor => actor.isAppeared());
	// };
	// Game_Party.prototype.randomBattleMenber = function() {
	// 	var r = Math.randomInt(this.maxBattleMembers());
	// 	return this.allMembers()
	// 		.slice(r, r + 1)
	// 		.filter(actor => actor.isAppeared());
	// };

	var _Game_Battler_performActionStart = Game_Battler.prototype.performActionStart;
	Game_Battler.prototype.performActionStart = function(action) {
		_Game_Battler_performActionStart.apply(this, arguments);
		// スキルIDを取得
		activeActionItemId = action._item._itemId;
		activeActionDataClass = action._item._dataClass;
		// アクターIDを取得
		showActionActorId = action._subjectActorId;
		// リフレッシュ
		activeActionActorId = null;
	};

	var _Game_Battler_performDamage = Game_Battler.prototype.performDamage;
	Game_Battler.prototype.performDamage = function() {
		_Game_Battler_performDamage.apply(this, arguments);
		// ダメージを受けたアクターIDを取得
		showDamageActorId = this._actorId;
		showEvasionActorId = null;
		// リフレッシュ
		activeDamageActorId = null;
	};

	var _Game_Battler_performEvasion = Game_Battler.prototype.performEvasion;
	Game_Battler.prototype.performEvasion = function() {
		_Game_Battler_performEvasion.apply(this, arguments);
		// アクターIDを取得
		showEvasionActorId = this._actorId;
		showDamageActorId = null;
		// リフレッシュ
		activeDamageActorId = null;
	};

	// 死亡時の直前ステートIDを保持するため、Game_BattlerBaseを拡張
	var _Game_BattlerBase_initMembers = Game_BattlerBase.prototype.initMembers;
	Game_BattlerBase.prototype.initMembers = function() {
		_Game_BattlerBase_initMembers.apply(this, arguments);

		this._beforeStates = [];
	};

	var _Game_BattlerBase_die = Game_BattlerBase.prototype.die;
	Game_BattlerBase.prototype.die = function() {
		this._beforeStates = this._states;

		_Game_BattlerBase_die.apply(this, arguments);
	};

	Game_BattlerBase.prototype.beforeStates = function() {
		return this._beforeStates;
	};

	var _BattleManager_invokeCounterAttack = BattleManager.invokeCounterAttack;
	BattleManager.invokeCounterAttack = function(subject, target) {
		_BattleManager_invokeCounterAttack.apply(this, arguments);

		if (!showCounterAttack) return;

		if (target) {
			// スキルIDを取得
			activeActionItemId = target.attackSkillId();
			activeActionDataClass = "skill";
			// アクターIDを取得
			showActionActorId = target._actorId;
			showDamageActorId = null;
			// リフレッシュ
			activeActionActorId = null;
		}
	};

	var _BattleManager_invokeMagicReflection = BattleManager.invokeMagicReflection;
	BattleManager.invokeMagicReflection = function(subject, target) {
		_BattleManager_invokeMagicReflection.apply(this, arguments);

		if (!showMagicReflection) return;

		if (target) {
			// スキルIDを取得
			activeActionItemId = this._action._item._itemId;
			activeActionDataClass = this._action._item._dataClass;
			// アクターIDを取得
			showActionActorId = target._actorId;
			showDamageActorId = null;
			// リフレッシュ
			activeActionActorId = null;
		}
	};

	//-----------------------------------------------------------------------------
	// ExStandingPictureBattle
	//
	// 戦闘中立ち絵を表示する独自のクラスを追加定義します。

	class ExStandingPictureBattle {

		static create (elm) {
			// 立ち絵1
			elm._spbSprite = new Sprite_LL_SPicture();
			elm.addChild(elm._spbSprite);
			// バトル開始・終了フラグをオフ
			this._battleStart = false;
			this._battleEnd = false;
		}

		static update (elm) {
			// 初期設定
			var sbPicture = null;
			var isPhase = BattleManager.isPhase();
			var isInputting = BattleManager.isInputting();
			var isEscaped = BattleManager.isEscaped();
			var isAllDead = $gameParty.isAllDead();
			var commandActor = BattleManager.actor();
			if (BattleManager._action) {
				if (BattleManager._action._subjectActorId) {
					this._lastActionActorId = BattleManager._action._subjectActorId;
				}
			}
			//-----------------------------------------------------------------------------
			// ターン開始時
			//-----------------------------------------------------------------------------
			if (isPhase == "start") {
				if (!this._battleStart) {
					// 生存しているアクターを取得
					this._aliveBattleMembers = $gameParty.aliveBattleMembers();
					// 先頭アクターIDを取得
					this._firstActorId = this._aliveBattleMembers.length > 0 ? this._aliveBattleMembers[0]._actorId : null;
					// ランダムアクターID抽選
					this._randomActorId = this._aliveBattleMembers.length > 0 ? this._aliveBattleMembers[Math.floor(Math.random() * this._aliveBattleMembers.length)]._actorId : null;
					this._battleStart = true;
				}
			}
			//-----------------------------------------------------------------------------
			// ターン終了時
			//-----------------------------------------------------------------------------
			if (isPhase == "turnEnd") {
				// 生存しているアクターを取得
				this._aliveBattleMembers = $gameParty.aliveBattleMembers();
				// 先頭アクターIDを取得
				this._firstActorId = this._aliveBattleMembers.length > 0 ? this._aliveBattleMembers[0]._actorId : null;
				// ランダムアクターID抽選
				this._randomActorId = this._aliveBattleMembers.length > 0 ? this._aliveBattleMembers[Math.floor(Math.random() * this._aliveBattleMembers.length)]._actorId : null;
			}

			// 立ち絵を非表示に設定している場合、処理を中断
			if ($gameSystem._StandingPictureBattleDisabled) {
				elm._spbSprite.opacity = 0;
				return;
			}

			//-----------------------------------------------------------------------------
			// 戦闘終了時
			//-----------------------------------------------------------------------------
			if (isPhase == "battleEnd") {
				if (isEscaped) {
					// 逃走
				} else if (isAllDead) {
					// 全滅
				} else {
					if (!this._battleEnd) {
						// 生存しているアクターを取得
						var aliveBattleMembers = $gameParty.aliveBattleMembers();
						// 先頭アクターIDを取得
						this._firstActorId = aliveBattleMembers.length > 0 ? aliveBattleMembers[0]._actorId : null;
						// ランダムアクターID抽選
						this._randomActorId = aliveBattleMembers.length > 0 ? aliveBattleMembers[Math.floor(Math.random() * aliveBattleMembers.length)]._actorId : null;
					}
					if (winActorType == "lastActor") {
						sbPicture = getImageName(this._lastActionActorId, sbWinPictureLists);
					} else if (winActorType == "randomActor") {
						sbPicture = getImageName(this._randomActorId, sbWinPictureLists);
					} else if (winActorType == "firstActor") {
						sbPicture = getImageName(this._firstActorId, sbWinPictureLists);
					}
					if (!this._battleEnd) {
						if (sbPicture) {
							refSbPicture = true;
							motionSbPicture = sbPicture.motion;
							animationCount = animationFrame[motionSbPicture];
							elm._spbSprite.opacity = 0;
						}
					}
				}
				this._battleEnd = true;
			}
			//-----------------------------------------------------------------------------
			// 被ダメージ時
			//-----------------------------------------------------------------------------
			if (showDamageActorId) {
				if (isPhase == "action") {
					sbPicture = getImageName(showDamageActorId, sbDamagePictureLists);
					if (sbPicture) {
						if (activeDamageActorId != showDamageActorId) {
							activeDamageActorId = showDamageActorId;
							refSbPicture = true;
							motionSbPicture = sbPicture.motion;
							animationCount = animationFrame[motionSbPicture];
							elm._spbSprite.opacity = 0;
						}
					}
				} else {
					showDamageActorId = null;
					sbPicture = null;
				}
			}
			//-----------------------------------------------------------------------------
			// 回避時
			//-----------------------------------------------------------------------------
			if (showEvasionActorId) {
				if (isPhase == "action") {
					sbPicture = getImageName(showEvasionActorId, sbEvasionPictureLists);
					if (sbPicture) {
						if (activeDamageActorId != showEvasionActorId) {
							activeDamageActorId = showEvasionActorId;
							refSbPicture = true;
							motionSbPicture = sbPicture.motion;
							animationCount = animationFrame[motionSbPicture];
							elm._spbSprite.opacity = 0;
						}
					}
				} else {
					showEvasionActorId = null;
					sbPicture = null;
				}
			}
			//-----------------------------------------------------------------------------
			// アクション時
			//-----------------------------------------------------------------------------
			if (!showDamageActorId && !showEvasionActorId) {
				if (isPhase == "action") {
					// スキル発動時
					if (activeActionDataClass == "skill") {
						sbPicture = getImageName(showActionActorId, sbActionPictureLists, activeActionItemId);
						// 見つからなかった場合、スキルIDなしの立ち絵を再検索
						if (!sbPicture) sbPicture = getImageName(showActionActorId, sbActionPictureLists, 0);
					}
					// アイテム使用時
					if (activeActionDataClass == "item") {
						sbPicture = getImageName(showActionActorId, sbItemPictureLists, activeActionItemId);
						// 見つからなかった場合、アイテムIDなしの立ち絵を再検索
						if (!sbPicture) sbPicture = getImageName(showActionActorId, sbItemPictureLists, 0);
					}
					if (sbPicture) {
						if (activeActionActorId != showActionActorId) {
							activeActionActorId = showActionActorId;
							refSbPicture = true;
							motionSbPicture = sbPicture.motion;
							animationCount = animationFrame[motionSbPicture];
							elm._spbSprite.opacity = 0;

							activeCommandActorId = null;
						}
					}
				} else {
					showActionActorId = null;
				}
			}
			//-----------------------------------------------------------------------------
			// 戦う or 逃げる 選択時
			//-----------------------------------------------------------------------------
			if (isPhase == "turn" || isPhase == "input") {
				if (!commandActor && isInputting) {
					var selectStartActorId = null;
					if (startActorType == "firstActor") {
						sbPicture = getImageName(this._firstActorId, sbCommandPictureLists);
						selectStartActorId = this._firstActorId;
					} else if (startActorType == "randomActor") {
						sbPicture = getImageName(this._randomActorId, sbCommandPictureLists);
						selectStartActorId = this._randomActorId;
					}
					if (sbPicture) {
						sbPicture = JSON.parse(JSON.stringify(sbPicture));
						if (activeCommandActorId != selectStartActorId) {
							activeCommandActorId = selectStartActorId;
							refSbPicture = true;
							// 通常
							motionSbPicture = sbPicture.motion;
							animationCount = animationFrame[motionSbPicture];
							elm._spbSprite.opacity = 0;
						}
					}

				}
			}
			//-----------------------------------------------------------------------------
			// コマンド入力時
			//-----------------------------------------------------------------------------
			if (isPhase == "turn" || isPhase == "input") {
				if (commandActor) {
					sbPicture = getImageName(commandActor._actorId, sbCommandPictureLists);
					// HPレートを取得
					var hpRate = commandActor.mhp > 0 ? commandActor.hp / commandActor.mhp * 100 : 0;
					if (sbPicture) {
						sbPicture = JSON.parse(JSON.stringify(sbPicture));
						if (activeCommandActorId != commandActor._actorId) {
							activeCommandActorId = commandActor._actorId;
							refSbPicture = true;
							// 通常
							motionSbPicture = sbPicture.motion;
							animationCount = animationFrame[motionSbPicture];
							elm._spbSprite.opacity = 0;
						}
						// 敵選択時は非表示にする
						if (hiddenEnemyWindow) {
							if (elm._enemyWindow) {
								elm._spbSprite.visible = !elm._enemyWindow.active;
							}
						}

						// 対象アクター選択時は非表示にする
						if (hiddenActorWindow && elm._spbSprite.visible === true) {
							if (elm._actorWindow) {
								elm._spbSprite.visible = !elm._actorWindow.active;
							}
						}
					}
				}
			}

			// 立ち絵ピクチャ作成
			if (sbPicture && refSbPicture) {
				this.refresh(elm._spbSprite, sbPicture);
				refSbPicture = false;
			}

			// フェード処理
			if (sbPicture) {
				this.fadeIn(elm._spbSprite, sbPicture);
			} else {
				this.fadeOut(elm._spbSprite, sbPicture);
			}

			// 立ち絵モーション再生
			if (animationCount > 0) {
				animationCount = this.animation(elm._spbSprite, motionSbPicture, animationCount);
			}
		}

		static refresh (sSprite, sPicture) {
			sSprite.setPicture(sPicture);
			sSprite.showing = false;
			var calcScaleX = Number(sPicture.scaleX);
			var calcScaleY = Number(sPicture.scaleY);
			// 画像が読み込まれたあとに実行
			sSprite.bitmap.addLoadListener(function() {
				if (Number(sPicture.origin) != 1 && String(sPicture.origin) != "center") {
					// 左上原点
					sSprite.x = Number(sPicture.x);
					sSprite.y = Number(sPicture.y);
					sSprite.originX = Number(sPicture.x);
					sSprite.originY = Number(sPicture.y);
				} else {
					// 中央原点
					sSprite.x = Number(sPicture.x) - (sSprite.width * calcScaleX / 100) / 2;
					sSprite.y = Number(sPicture.y) - (sSprite.height * calcScaleY / 100) / 2;
					sSprite.originX = Number(sPicture.x) - (sSprite.width * calcScaleX / 100) / 2;
					sSprite.originY = Number(sPicture.y) - (sSprite.height * calcScaleY / 100) / 2;
				}
				// 切替効果
				if (sSprite.opacity == 0) {
					//
				}
				sSprite.scale.x = calcScaleX / 100;
				sSprite.scale.y = calcScaleY / 100;
				sSprite.showing = true;
				// 非表示状態リセット
				sSprite.visible = true;
			}.bind(this));
		}

		static fadeIn (sSprite, sPicture) {
			if (!sSprite.showing) return;
			if (sSprite.opacity >= 255) {
				sSprite.opening = false;
				sSprite.opacity = 255;
				return;
			}
			sSprite.opening = true;
			sSprite.closing = false;
			sSprite.opacity += 24;
		}

		static fadeOut (sSprite, sPicture) {
			if (sSprite.opacity == 0) {
				activeCommandActorId = null;
				activeDamageActorId = null;
				activeActionActorId = null;
				sSprite.closing = false;
				return;
			}
			sSprite.closing = true;
			if (!sPicture) {
				//sSprite.opacity = 0;
				//return;
			}
			sSprite.opacity -= 24;
		}

		static animation (sSprite, sMotion, animationCount) {
			if (!sSprite.showing) return animationCount;
			if (sMotion == "yes") {
				if (animationCount > 12) {
					sSprite.y += 2;
				} else {
					sSprite.y -= 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "yesyes") {
				if (animationCount > 36) {
					sSprite.y += 2;
				} else if (animationCount > 24) {
					sSprite.y -= 2;
				} else if (animationCount > 12) {
					sSprite.y += 2;
				} else {
					sSprite.y -= 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "no") {
				if (animationCount > 18) {
					sSprite.x += 2;
				} else if (animationCount > 6) {
					sSprite.x -= 2;
				} else {
					sSprite.x += 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "noslow") {
				if (animationCount > 36) {
					sSprite.x += 1;
				} else if (animationCount > 12) {
					sSprite.x -= 1;
				} else {
					sSprite.x += 1;
				}
				animationCount -= 1;
			}
			if (sMotion == "jump") {
				if (animationCount > 12) {
					sSprite.y -= 2;
				} else {
					sSprite.y += 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "jumpjump") {
				if (animationCount > 36) {
					sSprite.y -= 2;
				} else if (animationCount > 24) {
					sSprite.y += 2;
				} else if (animationCount > 12) {
					sSprite.y -= 2;
				} else {
					sSprite.y += 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "jumploop") {
				if (animationCount > 36) {
					sSprite.y -= 2;
				} else if (animationCount > 24) {
					sSprite.y += 2;
				}
				animationCount -= 1;
				if (animationCount == 0) animationCount = animationFrame["jumploop"];
			}
			if (sMotion == "shake") {
				if (animationCount <= 2) {
					sSprite.x -= 2;
					animationCount += 1;
				} else if (animationCount <= 4) {
					sSprite.y -= 2;
					animationCount += 1;
				} else if (animationCount <= 6) {
					sSprite.x += 4;
					sSprite.y += 4;
					animationCount += 1;
				} else if (animationCount <= 8) {
					sSprite.y -= 2;
					animationCount += 1;
				} else if (animationCount == 9) {
					sSprite.x -= 2;
					animationCount += 1;
				} else if (animationCount == 10) {
					sSprite.x -= 2;
					animationCount = 0;
				}
			}
			if (sMotion == "shakeloop") {
				if (animationCount <= 2) {
					sSprite.x -= 1;
					animationCount += 1;
				} else if (animationCount <= 4) {
					sSprite.y -= 1;
					animationCount += 1;
				} else if (animationCount <= 6) {
					sSprite.x += 2;
					sSprite.y += 2;
					animationCount += 1;
				} else if (animationCount <= 8) {
					sSprite.y -= 1;
					animationCount += 1;
				} else if (animationCount <= 10) {
					sSprite.x -= 1;
					animationCount += 1;
				}
				if (animationCount > 10) animationCount = 1;
			}
			if (sMotion == "runleft") {
				sSprite.x -= 16;
				if (sSprite.x < -2000) animationCount = 0;
			}
			if (sMotion == "runright") {
				sSprite.x += 16;
				if (sSprite.x > 2000) animationCount = 0;
			}
			//
			if (sMotion == "damage") {
				if (animationCount <= 2) {
					sSprite.x -= 4;
					animationCount += 1;
				} else if (animationCount <= 4) {
					sSprite.y -= 4;
					animationCount += 1;
				} else if (animationCount <= 6) {
					sSprite.x += 8;
					sSprite.y += 8;
					animationCount += 1;
				} else if (animationCount <= 8) {
					sSprite.y -= 4;
					animationCount += 1;
				} else if (animationCount == 9) {
					sSprite.x -= 4;
					animationCount += 1;
				} else if (animationCount == 10) {
					sSprite.x -= 4;
					animationCount = 0;
				}
			}
			if (sMotion == "floatrightfast") {
				if (animationCount == 12) {
					sSprite.x += 22;
				} else {
					sSprite.x -= 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "floatright") {
				if (animationCount == 48) {
					sSprite.x += 47;
				} else {
					sSprite.x -= 1;
				}
				animationCount -= 1;
			}
			if (sMotion == "floatleftfast") {
				if (animationCount == 12) {
					sSprite.x -= 22;
				} else {
					sSprite.x += 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "floatleft") {
				if (animationCount == 48) {
					sSprite.x -= 47;
				} else {
					sSprite.x += 1;
				}
				animationCount -= 1;
			}
			if (sMotion == "noslowloop") {
				if (animationCount > 72) {
					sSprite.x += 0.25;
				} else if (animationCount > 24) {
					sSprite.x -= 0.25;
				} else {
					sSprite.x += 0.25;
				}
				animationCount -= 1;
				if (animationCount == 0) animationCount = animationFrame["noslowloop"];
			}
			if (sMotion == "breathing") {
				if (animationCount > 72) {
					sSprite.y += 0.5;
				} else if (animationCount > 48) {
					sSprite.y -= 0.5;
				} else {
				}
				animationCount -= 1;
				if (animationCount == 0) animationCount = animationFrame["breathing"];
			}
			if (sMotion == "breathing2") {
				if (animationCount > 48) {
					// sSprite.anchor.y = 1;
					sSprite.y -= sSprite.height * 0.0003;
					sSprite.scale.y += 0.0003;
				} else {
					// sSprite.anchor.y = 1;
					sSprite.y += sSprite.height * 0.0003;
					sSprite.scale.y -= 0.0003;
				}
				animationCount -= 1;
				if (animationCount == 0) animationCount = animationFrame["breathing2"];
			}
			if (sMotion == "stepleft") {
				if (animationCount > 12) {
					sSprite.x -= 2;
				} else {
					sSprite.x += 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "stepright") {
				if (animationCount > 12) {
					sSprite.x += 2;
				} else {
					sSprite.x -= 2;
				}
				animationCount -= 1;
			}
			if (sMotion == "headdown") {
				sSprite.y += 2;
				animationCount -= 1;
			}
			return animationCount;
		}
	}

	var _Scene_Battle_update = Scene_Battle.prototype.update;
	Scene_Battle.prototype.update = function() {
		_Scene_Battle_update.apply(this, arguments);
		ExStandingPictureBattle.update(this);
	};

	var _Scene_Battle_createSpriteset = Scene_Battle.prototype.createSpriteset;
	Scene_Battle.prototype.createSpriteset = function() {
		_Scene_Battle_createSpriteset.apply(this, arguments);
		ExStandingPictureBattle.create(this);
	};

	//-----------------------------------------------------------------------------
	// Sprite_LL_SPicture
	//
	// 立ち絵を表示するための独自のスプライトを追加定義します。

	function Sprite_LL_SPicture() {
		this.initialize.apply(this, arguments);
	}

	Sprite_LL_SPicture.prototype = Object.create(Sprite.prototype);
	Sprite_LL_SPicture.prototype.constructor = Sprite_LL_SPicture;

	Sprite_LL_SPicture.prototype.initialize = function() {
		Sprite.prototype.initialize.call(this);

		this.bitmap = null;
		this.opacity = 0;
		this.opening = false;
		this.closing = false;
		this.originX = 0;
		this.originY = 0;
		this.showing = false;

		this.setOverlayBitmap();
		this.initMembers();
	};

	Sprite_LL_SPicture.prototype.setOverlayBitmap = function() {
		//
	};

	Sprite_LL_SPicture.prototype.initMembers = function() {
		//
	};

	Sprite_LL_SPicture.prototype.update = function() {
		Sprite.prototype.update.call(this);
	};

	Sprite_LL_SPicture.prototype.setPicture = function(sPicture) {
		// ベース画像
		this.bitmap = null;
		this.bitmap = ImageManager.loadPicture(sPicture.imageName);
	};

	Sprite_LL_SPicture.prototype.setBlendColor = function(color) {
		Sprite.prototype.setBlendColor.call(this, color);
	};

	Sprite_LL_SPicture.prototype.setColorTone = function(tone) {
		Sprite.prototype.setColorTone.call(this, tone);
	};

	Sprite_LL_SPicture.prototype.setBlendMode = function(mode) {
		this.blendMode = mode;
	};
})();
