/*:
 * @target MZ
 * @plugindesc Skill-Specific Portraits Plugin for RPG Maker MZ
 * @param Skills
 * @text 技能设置
 * @type struct<SkillConfig>[]
 * @default []
 * @help
 * 这个插件允许为每个技能绑定特定的立绘图片，技能使用时显示立绘。
 */

/*~struct~SkillConfig:
 * @param SkillID
 * @text 技能ID
 * @type number
 * @min 1
 * @desc 技能的ID
 * 
 * @param PortraitImage
 * @text 立绘图片
 * @type file
 * @dir img/pictures
 * @desc 技能的立绘图片
 * 
 * @param PortraitX
 * @text 立绘X位置
 * @type number
 * @desc 立绘图片的X坐标
 * 
 * @param PortraitY
 * @text 立绘Y位置
 * @type number
 * @desc 立绘图片的Y坐标
 * 
 * @param PortraitWidth
 * @text 立绘宽度
 * @type number
 * @desc 立绘图片的宽度
 * 
 * @param PortraitHeight
 * @text 立绘高度
 * @type number
 * @desc 立绘图片的高度
 * 
 * @param DisplayDuration
 * @text 显示时长
 * @type number
 * @min 1
 * @desc 立绘显示的时长（毫秒）
 */

(() => {
    const pluginName = "SkillSpecificPortraits";
    const parameters = PluginManager.parameters(pluginName);
    const skills = JSON.parse(parameters["Skills"] || "[]").map(skill => JSON.parse(skill));

    let currentPortraitSprite = null;
    let portraitTimeout = null;

    const _BattleManager_startAction = BattleManager.startAction;
    BattleManager.startAction = function() {
        _BattleManager_startAction.call(this);
        this.showSkillPortrait();
    };

    BattleManager.showSkillPortrait = function() {
        const subject = this._subject;
        if (!subject) return;

        const action = subject.currentAction();
        if (!action) return;

        const skillId = action.item().id;
        const skillConfig = skills.find(skill => skill.SkillID == skillId);
        if (skillConfig) {
            if (currentPortraitSprite) {
                SceneManager._scene.removeChild(currentPortraitSprite);
                clearTimeout(portraitTimeout);
            }

            const bitmap = ImageManager.loadPicture(skillConfig.PortraitImage);
            currentPortraitSprite = new Sprite(bitmap);
            currentPortraitSprite.x = Number(skillConfig.PortraitX);
            currentPortraitSprite.y = Number(skillConfig.PortraitY);
            currentPortraitSprite.z = 10;
            currentPortraitSprite.setFrame(0, 0, Number(skillConfig.PortraitWidth), Number(skillConfig.PortraitHeight));
            SceneManager._scene.addChild(currentPortraitSprite);

            const displayDuration = Number(skillConfig.DisplayDuration);
            portraitTimeout = setTimeout(() => {
                SceneManager._scene.removeChild(currentPortraitSprite);
                currentPortraitSprite = null;
            }, displayDuration);
        }
    };

    const _BattleManager_endAction = BattleManager.endAction;
    BattleManager.endAction = function() {
        _BattleManager_endAction.call(this);
        if (currentPortraitSprite) {
            SceneManager._scene.removeChild(currentPortraitSprite);
            currentPortraitSprite = null;
            clearTimeout(portraitTimeout);
        }
    };
})();
