/*:
 * @target MZ
 * @plugindesc 【完全安定版 MarkⅤ】クリックで閉じる通知（SE・部分色・高解像度・自動幅調整・逐次表示）@トリアコンタン MarkⅤ
 * @author トリアコンタン MarkⅤ
 *
 * @param PositionX
 * @text 表示X座標
 * @type number
 * @default 1080
 *
 * @param PositionY
 * @text 表示Y座標
 * @type number
 * @default 40
 *
 * @param MaxWidth
 * @text 最大幅
 * @type number
 * @default 360
 *
 * @param MinWidth
 * @text 最小幅
 * @type number
 * @default 120
 *
 * @param Height
 * @text 通知の高さ
 * @type number
 * @default 72
 *
 * @param BackgroundColor
 * @text 背景色
 * @default rgba(0, 0, 0, 0.7)
 *
 * @param FontFileName
 * @text フォントファイル名（fontsフォルダ内）
 * @type string
 * @default
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @default 22
 *
 * @param DefaultTextColor
 * @text デフォルト文字色
 * @default #ffffff
 *
 * @param SlideDirection
 * @text スライド方向
 * @type select
 * @option 上
 * @value up
 * @option 下
 * @value down
 * @default up
 *
 * @param SlideInDistance
 * @text スライドイン距離
 * @type number
 * @default 20
 *
 * @param SlideInDuration
 * @text スライドイン時間(フレーム)
 * @type number
 * @default 20
 *
 * @param FadeInSpeed
 * @text フェードイン速度
 * @type number
 * @default 15
 *
 * @param FadeOutSpeed
 * @text フェードアウト速度
 * @type number
 * @default 8
 *
 * @param SlideOutSpeed
 * @text スライドアウト速度
 * @type number
 * @default 1
 *
 * @param AutoDuration
 * @text 自動消滅時間
 * @type number
 * @default 0
 *
 * @param NotifySe
 * @text 通知音（SE）
 * @type file
 * @dir audio/se
 * @default
 *
 * @param NotifySeVolume
 * @text SE音量
 * @type number
 * @default 90
 *
 * @command show
 * @text 通知を表示
 * @desc 通知メッセージを表示します。
 *
 * @arg text
 * @text 表示テキスト
 * @type multiline_string
 * @default 通知メッセージ
 * @desc [red]警告[-] のように色指定できます。
 */

(() => {
'use strict';
const PLUGIN_NAME = 'ClickNotifyMessagePlusPlus_SmoothSlide_AutoWidth';
const p = PluginManager.parameters(PLUGIN_NAME);

// === パラメータ ===
const POS_X = Number(p.PositionX || 1080);
const POS_Y = Number(p.PositionY || 40);
const MAX_WIDTH = Number(p.MaxWidth || 360);
const MIN_WIDTH = Number(p.MinWidth || 120);
const HEIGHT = Number(p.Height || 72);
const BG_COLOR = p.BackgroundColor || 'rgba(0,0,0,0.7)';
const FONT_FILE = p.FontFileName || '';
const FONT_SIZE = Number(p.FontSize || 22);
const DEFAULT_COLOR = p.DefaultTextColor || '#ffffff';
const SLIDE_DIR = p.SlideDirection || 'up';
const SLIDEIN_DIST = Number(p.SlideInDistance || 20);
const SLIDEIN_DUR = Number(p.SlideInDuration || 20);
const FADEIN_SPEED = Number(p.FadeInSpeed || 15);
const FADEOUT_SPEED = Number(p.FadeOutSpeed || 8);
const SLIDEOUT_SPEED = Number(p.SlideOutSpeed || 1);
const AUTO_DURATION = Number(p.AutoDuration || 0);
const SE_NAME = p.NotifySe || '';
const SE_VOLUME = Number(p.NotifySeVolume || 90);

// === 通知キュー ===
const queue = [];
let current = null;
let showing = false;

// === コマンド登録 ===
PluginManager.registerCommand(PLUGIN_NAME, 'show', args => {
    queue.push({ text: args.text || '' });
});

// === Scene_Map拡張 ===
const _Scene_Map_update = Scene_Map.prototype.update;
Scene_Map.prototype.update = function() {
    _Scene_Map_update.call(this);
    this.updateClickNotify();
};

Scene_Map.prototype.updateClickNotify = function() {
    if (SceneManager.isSceneChanging() || this._destroyed) {
        this.clearNotify();
        return;
    }

    if (!showing && queue.length > 0) {
        const next = queue.shift();
        this.showNextNotify(next.text);
    }

    if (current) {
        current.update();
        if (current._isDone) {
            this.removeNotify(current);
            current = null;
            showing = false;
        }
    }
};

Scene_Map.prototype.showNextNotify = function(text) {
    showing = true;
    setTimeout(() => {
        if (this._destroyed || SceneManager.isSceneChanging()) {
            showing = false;
            return;
        }
        const parent = this.getNotifyParent();
        if (!parent) {
            showing = false;
            return;
        }
        current = new Sprite_ClickNotify(text);
        parent.addChild(current);

        // 🎵 通知音を再生
        if (SE_NAME) {
            AudioManager.playSe({
                name: SE_NAME,
                volume: SE_VOLUME,
                pitch: 100,
                pan: 0
            });
        }
    }, 80);
};

Scene_Map.prototype.removeNotify = function(sprite) {
    const parent = this.getNotifyParent();
    if (parent && parent.children && parent.children.includes(sprite)) {
        parent.removeChild(sprite);
    }
};

const _Scene_Map_terminate = Scene_Map.prototype.terminate;
Scene_Map.prototype.terminate = function() {
    this.clearNotify();
    _Scene_Map_terminate.call(this);
};

Scene_Map.prototype.clearNotify = function() {
    if (current) {
        try {
            const parent = this.getNotifyParent();
            if (parent && parent.children && parent.children.includes(current)) {
                parent.removeChild(current);
            }
        } catch (e) {}
        current = null;
    }
    showing = false;
};

Scene_Map.prototype.getNotifyParent = function() {
    if (this._spriteset && this._spriteset.children) return this._spriteset;
    if (this._windowLayer) return this._windowLayer;
    return SceneManager._scene;
};

// === 通知スプライト本体 ===
class Sprite_ClickNotify extends Sprite {
    constructor(text) {
        super();
        this._text = text;
        this._fadeIn = true;
        this._fadeOut = false;
        this._isDone = false;
        this._autoTimer = 0;
        this.opacity = 0;
        this._slideFrame = 0;
        this._baseY = POS_Y;
        this.x = POS_X;
        this.createContents();
        this.y = this._baseY + (SLIDE_DIR === 'up' ? SLIDEIN_DIST : -SLIDEIN_DIST);
    }

    createContents() {
        const scale = 2;
        this._content = new Sprite();
        this._content.bitmap = new Bitmap(MAX_WIDTH * scale, HEIGHT * scale);
        this._content.scale.set(1 / scale);
        this.addChild(this._content);
        this._content.bitmap.fontSize = FONT_SIZE * scale;
        this._content.bitmap.outlineWidth = 0;
        this.prepareFontAndDraw();
    }

    prepareFontAndDraw() {
        const bitmap = this._content.bitmap;
        if (FONT_FILE) {
            const fontName = FONT_FILE.replace(/\.[^/.]+$/, '');
            const fontPath = `fonts/${FONT_FILE}`;
            const font = new FontFace(fontName, `url(${fontPath})`);
            font.load().then(f => {
                document.fonts.add(f);
                bitmap.fontFace = fontName;
                this.adjustWidthAndDraw();
            }).catch(() => {
                bitmap.fontFace = $gameSystem.mainFontFace();
                this.adjustWidthAndDraw();
            });
        } else {
            bitmap.fontFace = $gameSystem.mainFontFace();
            this.adjustWidthAndDraw();
        }
    }

    parseTextSegments(text) {
        const regex = /\[(#[0-9A-Fa-f]{3,6}|red|blue|green|yellow|white|black)\]|\[-\]/g;
        const result = [];
        let lastIndex = 0;
        let currentColor = DEFAULT_COLOR;
        let match;
        while ((match = regex.exec(text)) !== null) {
            if (match.index > lastIndex) {
                result.push({ text: text.slice(lastIndex, match.index), color: currentColor });
            }
            if (match[0] === '[-]') {
                currentColor = DEFAULT_COLOR;
            } else {
                const table = {
                    red: '#ff4040', blue: '#4080ff', green: '#40ff40',
                    yellow: '#ffff40', white: '#ffffff', black: '#000000'
                };
                currentColor = table[match[1]] || match[1];
            }
            lastIndex = regex.lastIndex;
        }
        if (lastIndex < text.length) result.push({ text: text.slice(lastIndex), color: currentColor });
        return result;
    }

    adjustWidthAndDraw() {
        const bitmap = this._content.bitmap;
        const segs = this.parseTextSegments(this._text);
        let w = 0;
        for (const s of segs) w += bitmap.measureTextWidth(s.text) / 2;
        w += 24;
        this._width = Math.min(MAX_WIDTH, Math.max(MIN_WIDTH, w));
        bitmap.resize(this._width * 2, HEIGHT * 2);
        this.refreshText(segs);
    }

    refreshText(segs) {
        const bitmap = this._content.bitmap;
        bitmap.clear();
        bitmap.fillRect(0, 0, this._width * 2, HEIGHT * 2, BG_COLOR);
        let x = 0;
        const y = (HEIGHT * 2 - FONT_SIZE * 2) / 2;
        for (const s of segs) {
            bitmap.textColor = s.color;
            const w = bitmap.measureTextWidth(s.text);
            bitmap.drawText(s.text, x, y, w, HEIGHT * 2, 'left');
            x += w;
        }
    }

    update() {
        super.update();
        if (this._fadeIn) {
            this.opacity += FADEIN_SPEED;
            this._slideFrame++;
            const t = Math.min(this._slideFrame / SLIDEIN_DUR, 1);
            const ease = 1 - Math.pow(1 - t, 3);
            const offset = SLIDEIN_DIST * (1 - ease);
            this.y = this._baseY + (SLIDE_DIR === 'up' ? offset : -offset);
            if (this.opacity >= 255 && t >= 1) this._fadeIn = false;
        }

        if (!this._fadeOut && AUTO_DURATION > 0) {
            this._autoTimer++;
            if (this._autoTimer > AUTO_DURATION) this.startFadeOut();
        }

        if (TouchInput.isTriggered() && !this._fadeOut) this.startFadeOut();

        if (this._fadeOut) {
            this.opacity -= FADEOUT_SPEED;
            this.y += (SLIDE_DIR === 'up' ? -1 : 1) * SLIDEOUT_SPEED;
            if (this.opacity <= 0) this._isDone = true;
        }
    }

    startFadeOut() {
        this._fadeOut = true;
    }
}
})();
