/*:
 * @target MZ
 * @plugindesc [v1.1] 指定のBGSを複数同時再生します（パラメータ設定対応・安定版）
 * @author トリアコンタン
 * 
 * @param BGSList
 * @text BGSリスト
 * @type struct<BGS>[]
 * @desc 再生するBGSの設定リストです。
 * 
 * @param AutoPlay
 * @text 自動再生
 * @type boolean
 * @default true
 * @desc マップ読み込み時に自動でBGSを並行再生します。
 * 
 * @command PlayParallelBGS
 * @text 並行BGS再生
 * @desc 設定されたBGSを並行再生します。
 * 
 * @command StopParallelBGS
 * @text 並行BGS停止
 * @desc 再生中の並行BGSをすべて停止します。
 */

/*~struct~BGS:
 * @param name
 * @text ファイル名
 * @type file
 * @dir audio/bgs/
 * 
 * @param volume
 * @text 音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 * 
 * @param pitch
 * @text ピッチ
 * @type number
 * @min 50
 * @max 150
 * @default 100
 * 
 * @param pan
 * @text 位相
 * @type number
 * @min -100
 * @max 100
 * @default 0
 */

(() => {
    const pluginName = "ParallelBGS";
    const parameters = PluginManager.parameters(pluginName);
    const bgsList = JSON.parse(parameters["BGSList"] || "[]").map(e => JSON.parse(e));
    const autoPlay = parameters["AutoPlay"] === "true";

    // 各BGS専用プレイヤーを保持
    const parallelBgsPlayers = [];

    // 並行BGS再生処理
    function playParallelBGS() {
        stopParallelBGS(); // 既存を停止
        for (const data of bgsList) {
            if (!data.name) continue;
            const bgs = {
                name: data.name,
                volume: Number(data.volume),
                pitch: Number(data.pitch),
                pan: Number(data.pan)
            };
            const player = new WebAudio(`audio/bgs/${bgs.name}.ogg`);
            player.volume = bgs.volume / 100;
            player.pitch = bgs.pitch / 100;
            player.pan = bgs.pan / 100;
            player.loop = true;
            player.play(true);
            parallelBgsPlayers.push(player);
        }
    }

    // 並行BGS停止処理
    function stopParallelBGS() {
        for (const player of parallelBgsPlayers) {
            if (player && player.isPlaying()) {
                player.stop();
            }
        }
        parallelBgsPlayers.length = 0;
    }

    // プラグインコマンド登録
    PluginManager.registerCommand(pluginName, "PlayParallelBGS", playParallelBGS);
    PluginManager.registerCommand(pluginName, "StopParallelBGS", stopParallelBGS);

    // 自動再生
    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.apply(this, arguments);
        if (autoPlay) playParallelBGS();
    };
})();
