/*:
 * @target MZ
 * @plugindesc 【共有スイッチ】セーブデータ間で共通のスイッチ状態を保持します。ニューゲーム時にも引き継ぎます。
 * @author トリアコンタン MarkⅡ
 *
 * @help
 * このプラグインは、指定したスイッチの状態を
 * すべてのセーブデータで共有します。
 *
 * あるセーブでONにしたスイッチは、
 * 他のセーブデータやニューゲームでもONのままになります。
 *
 * --- 使用方法 ---
 *
 * 1. プラグインコマンド「共有スイッチを追加」で
 *    指定したスイッチを共有対象に登録します。
 *
 * 2. 登録したスイッチは、ON/OFFを切り替えるたびに
 *    自動的にローカルストレージへ保存されます。
 *
 * 3. ニューゲーム時や別セーブロード時にも状態が反映されます。
 *
 * --- コマンド一覧 ---
 * ・共有スイッチを追加（add）
 * ・共有スイッチを解除（remove）
 * ・共有スイッチ全削除（clearAll）
 *
 * --- 注意 ---
 * ・保存はセーブデータとは独立しています。
 * ・テスト中に初期化したい場合は「共有スイッチ全削除」を使用してください。
 *
 * @command add
 * @text 共有スイッチを追加
 * @desc 指定スイッチを共有対象にします。
 *
 * @arg switchId
 * @type switch
 * @text スイッチID
 * @desc 共有対象にするスイッチID
 *
 * @command remove
 * @text 共有スイッチを解除
 * @desc 指定スイッチを共有対象から外します。
 *
 * @arg switchId
 * @type switch
 * @text スイッチID
 * @desc 共有解除するスイッチID
 *
 * @command clearAll
 * @text 共有スイッチ全削除
 * @desc すべての共有スイッチ設定と状態を削除します。
 */

(() => {
  const PLUGIN_NAME = "SharedSwitch";
  const STORAGE_KEY = `${PLUGIN_NAME}_Data`;

  // --- データ管理 ---
  const loadSharedData = () => {
    try {
      return JSON.parse(localStorage.getItem(STORAGE_KEY) || "{}");
    } catch (e) {
      console.error(`${PLUGIN_NAME}: データ読み込み失敗`, e);
      return {};
    }
  };

  const saveSharedData = (data) => {
    localStorage.setItem(STORAGE_KEY, JSON.stringify(data));
  };

  let sharedSwitches = loadSharedData();

  // --- スイッチ反映 ---
  const applySharedSwitches = () => {
    for (const [id, value] of Object.entries(sharedSwitches)) {
      $gameSwitches.setValue(Number(id), value);
    }
  };

  // --- ニューゲーム時 ---
  const _DataManager_setupNewGame = DataManager.setupNewGame;
  DataManager.setupNewGame = function() {
    _DataManager_setupNewGame.call(this);
    applySharedSwitches();
  };

  // --- セーブロード時 ---
  const _DataManager_loadGame = DataManager.loadGame;
  DataManager.loadGame = async function(savefileId) {
    const result = await _DataManager_loadGame.call(this, savefileId);
    applySharedSwitches();
    return result;
  };

  // --- スイッチ監視 ---
  const _Game_Switches_setValue = Game_Switches.prototype.setValue;
  Game_Switches.prototype.setValue = function(switchId, value) {
    _Game_Switches_setValue.call(this, switchId, value);
    if (sharedSwitches.hasOwnProperty(switchId)) {
      sharedSwitches[switchId] = !!value;
      saveSharedData(sharedSwitches);
    }
  };

  // --- プラグインコマンド登録 ---
  PluginManager.registerCommand(PLUGIN_NAME, "add", (args) => {
    const id = String(args.switchId);
    sharedSwitches[id] = $gameSwitches.value(Number(id));
    saveSharedData(sharedSwitches);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "remove", (args) => {
    const id = String(args.switchId);
    delete sharedSwitches[id];
    saveSharedData(sharedSwitches);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "clearAll", () => {
    sharedSwitches = {};
    saveSharedData(sharedSwitches);
  });
})();
