// SwivelPicture.js Ver.1.1.1
// MIT License (C) 2024 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ
* @orderAfter PluginCommonBase
* @plugindesc Allows you to freely change the angle of the picture.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/503299840.html
* @help Ver.1.1.1
* [Scripts]
* $gameScreen.swivelPicture(pictureId, angle, duration, easingType);
* //Swivel
* $gameScreen.swivelPictureRelatively(pictureId, angle, duration, easingType);
* //Swivel Relatively
* $gameScreen.setPictureSwivelInertia(pictureId, inertia, reverse);
* //Set Inertia
* $gameScreen.fixPictureAngle(pictureId);
* //FixAngle
* Enter true/false for inertia and reverse.
*
* @command swivel
* @text Swivel
* @desc Rotates the picture to the specified angle.
* @arg pictureId
* @text Picture ID
* @default 1
* @arg angle
* @text Angle
* @desc Values over 360 and negative values can also be specified.
* @default 0
* @arg easingType
* @text Easing Type
* @type select 
* @option Constant speed
* @value 0
* @option Slow start
* @value 1
* @option Slow end
* @value 2
* @option Slow start and end
* @value 3
* @default 0
* @arg duration
* @text Duration
* @desc Apply immediately with 0
* @default 24
* @arg wait
* @text Wait for Completion
* @desc Wait until it completes.
* @type boolean
* @default true
*
* @command swivelRelatively
* @text Swivel Relatively
* @desc Rotates the picture by the specified angle.
* @arg pictureId
* @text Picture ID
* @default 1
* @arg angle
* @text Angle
* @desc Values over 360 and negative values can also be specified.
* @default 0
* @arg easingType
* @text Easing Type
* @type select 
* @option Constant speed
* @value 0
* @option Slow start
* @value 1
* @option Slow end
* @value 2
* @option Slow start and end
* @value 3
* @default 0
* @arg duration
* @text Duration
* @desc Apply immediately with 0
* @default 24
* @arg wait
* @text Wait for Completion
* @desc Wait until it completes.
* @type boolean
* @default true
*
* @command setSwivelInertia
* @text Set Swivel Inertia
* @desc Keep the last momentum and rotate.
* (Rotation speed) = (Last angle) - (Previous angle)
* @arg pictureId
* @text Picture ID
* @default 1
* @arg inertia
* @text Inertia
* @type select
* @option None
* @value 0
* @option Forward
* @value 1
* @option Reverse
* @value -1
* @default 0
*
* @command fixAngle
* @text Fix Angle
* @desc Correct the stored angle to within the range of -360 to 360.
* @arg pictureId
* @text Picture ID
* @default 1
*
* @command rotate
* @text Rotate Picture
* @desc This is an improved version of the event command "Rotate Picture".
* @arg pictureId
* @text Picture ID
* @default 1
* @arg speed
* @text Speed
* @desc How many degrees to rotate per frame.
* Values over 360 and negative values can also be specified.
* @default 0
*
* @command swivelSV
* @text Swivel (Short Version)
* @desc Rotates the picture to the specified angle.
* It can be written in fewer lines.
* @arg params
* @text Parameters
* @desc From left, Picture ID, Angle, Frames, Easing Type
*  # can be omitted.
* @default #1,0,24,0
* @arg wait
* @text Wait for Completion
* @desc Wait until it completes.
* @type boolean
* @default true
*
* @command swivelRelativelySV
* @text Swivel Relatively (Short Version)
* @desc Rotates the picture by the specified angle.
* It can be written in fewer lines.
* @arg params
* @text Parameters
* @desc From left, Picture ID, Angle, Frames, Easing Type
*  # can be omitted.
* @default #1,0,24,0
* @arg wait
* @text Wait for Completion
* @desc Wait until it completes.
* @type boolean
* @default true
*
*/

/*:ja
* @target MZ
* @orderAfter PluginCommonBase
* @plugindesc ピクチャの角度を自由に変更可能にします。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/503299840.html
* @help [スクリプト]
* $gameScreen.swivelPicture(pictureId, angle, duration, easingType);
* //回転
* $gameScreen.swivelPictureRelatively(pictureId, angle, duration, easingType);
* //相対的に回転
* $gameScreen.setPictureSwivelInertia(pictureId, inertia, reverse);
* //回転慣性の設定
* $gameScreen.fixPictureAngle(pictureId);
* //角度の修正
* inertia、reverseはtrue/falseを入力。
*
* [更新履歴]
* 2024/05/12：Ver.1.0.0　公開。
* 2024/05/19：Ver.1.1.0　慣性、角度の修正を追加。PictureControlExtendに対応。
* 2024/05/27：Ver.1.1.1　短縮版コマンドを追加。
*
* @command swivel
* @text 回転
* @desc ピクチャを指定した角度まで回転します。
* @arg pictureId
* @text ピクチャID
* @default 1
* @arg angle
* @text 角度
* @desc 360を超える値や負の値も指定可能。
* @default 0
* @arg easingType
* @text イージング
* @type select 
* @option 一定速度
* @value 0
* @option ゆっくり始まる
* @value 1
* @option ゆっくり終わる
* @value 2
* @option ゆっくり始まってゆっくり終わる
* @value 3
* @default 0
* @arg duration
* @text 時間
* @desc 0で即時適用
* @default 24
* @arg wait
* @text 完了までウェイト
* @desc 完了するまで待ちます。
* @type boolean
* @default true
*
* @command swivelRelatively
* @text 相対的に回転
* @desc ピクチャを指定した角度ぶん回転します。
* @arg pictureId
* @text ピクチャID
* @default 1
* @arg angle
* @text 角度
* @desc 360を超える値や負の値も指定可能。
* @default 0
* @arg easingType
* @text イージング
* @type select 
* @option 一定速度
* @value 0
* @option ゆっくり始まる
* @value 1
* @option ゆっくり終わる
* @value 2
* @option ゆっくり始まってゆっくり終わる
* @value 3
* @default 0
* @arg duration
* @text 時間
* @desc 0で即時適用
* @default 24
* @arg wait
* @text 完了までウェイト
* @desc 完了するまで待ちます。
* @type boolean
* @default true
*
* @command setSwivelInertia
* @text 回転慣性の設定
* @desc 最後の勢いを維持して回転します。
* （回転速度）＝（最後の角度）－（ひとつ前の角度）
* @arg pictureId
* @text ピクチャID
* @default 1
* @arg inertia
* @text 慣性
* @type select
* @option なし
* @value 0
* @option 正転
* @value 1
* @option 逆転
* @value -1
* @default 0
*
* @command fixAngle
* @text 角度の修正
* @desc 格納されている角度を-360～360の範囲内に修正します。
* @arg pictureId
* @text ピクチャID
* @default 1
*
* @command rotate
* @text ピクチャの回転
* @desc イベントコマンド「ピクチャの回転」の改善版です。
* @arg pictureId
* @text ピクチャID
* @default 1
* @arg speed
* @text 速度
* @desc 1フレーム毎に何度回転するか。
* 360を超える値や負の値も指定可能。
* @default 0
*
* @command swivelSV
* @text 回転（短縮版）
* @desc ピクチャを指定した角度まで回転します。
* より少ない行数で表記可能です。
* @arg params
* @text パラメータ
* @desc 左からﾋﾟｸﾁｬID、角度、フレーム数、イージング
* 記号#は無くても問題ないです。
* @default #1,0,24,0
* @arg wait
* @text 完了までウェイト
* @desc 完了するまで待ちます。
* @type boolean
* @default true
*
* @command swivelRelativelySV
* @text 相対的に回転（短縮版）
* @desc ピクチャを指定した角度ぶん回転します。
* より少ない行数で表記可能です。
* @arg params
* @text パラメータ
* @desc 左からﾋﾟｸﾁｬID、角度、フレーム数、イージング
* 記号#は無くても問題ないです。
* @default #1,0,24,0
* @arg wait
* @text 完了までウェイト
* @desc 完了するまで待ちます。
* @type boolean
* @default true
*
*/

'use strict';
{
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const hasPictureControlExtend = PluginManager._scripts.includes("PictureControlExtend");
	const _PluginManager = window.PluginManagerEx || window.PluginManager;
	const script = window.PluginManagerEx ? document.currentScript : pluginName;

	//-----------------------------------------------------------------------------
	// _PluginManager

	_PluginManager.registerCommand(script, "swivel", function(args) {
		$gameScreen.swivelPicture(+args.pictureId, +args.angle, +args.duration, +args.easingType);
		if (String(args.wait) === "true") {
			this.wait(+args.duration);
		}
	});

	_PluginManager.registerCommand(script, "swivelRelatively", function(args) {
		$gameScreen.swivelPictureRelatively(+args.pictureId, +args.angle, +args.duration, +args.easingType);
		if (String(args.wait) === "true") {
			this.wait(+args.duration);
		}
	});

	_PluginManager.registerCommand(script, "setSwivelInertia", function(args) {
		$gameScreen.setPictureSwivelInertia(+args.pictureId, +args.inertia);
	});

	_PluginManager.registerCommand(script, "fixAngle", function(args) {
		$gameScreen.fixPictureAngle(+args.pictureId);
	});

	_PluginManager.registerCommand(script, "rotate", function(args) {
		$gameScreen.rotatePicture(+args.pictureId, args.speed*2);
	});

	_PluginManager.registerCommand(script, "swivelSV", function(args) {
		const params = args.params.replace(/#/g, "").split(",").map(Number);
		$gameScreen.swivelPicture(...params);
		if (String(args.wait) === "true") {
			this.wait(params[2]);
		}
	});

	_PluginManager.registerCommand(script, "swivelRelativelySV", function(args) {
		const params = args.params.replace(/#/g, "").split(",").map(Number);
		$gameScreen.swivelPictureRelatively(...params);
		if (String(args.wait) === "true") {
			this.wait(params[2]);
		}
	});

	//-----------------------------------------------------------------------------
	// Game_Screen

	Game_Screen.prototype.swivelPicture = function(pictureId, angle, duration, easingType) {
		const picture = this.picture(pictureId);
		if (picture) {
			picture.swivel(angle, duration, easingType);
		}
	};

	Game_Screen.prototype.swivelPictureRelatively = function(pictureId, angle, duration, easingType) {
		const picture = this.picture(pictureId);
		if (picture) {
			picture.swivelRelatively(angle, duration, easingType);
		}
	};

	Game_Screen.prototype.setPictureSwivelInertia = function(pictureId, inertia) {
		const picture = this.picture(pictureId);
		if (picture) {
			picture.setSwivelInertia(inertia);
		}
	};

	Game_Screen.prototype.fixPictureAngle = function(pictureId) {
		const picture = this.picture(pictureId);
		if (picture) {
			picture.fixAngle();
		}
	};

	if (hasPictureControlExtend) {
		const _Game_Screen_swivelPicture = Game_Screen.prototype.swivelPicture;
		Game_Screen.prototype.swivelPicture = function() {
			this.iteratePictures(_Game_Screen_swivelPicture, arguments);
		};

		const _Game_Screen_swivelPictureRelatively = Game_Screen.prototype.swivelPictureRelatively;
		Game_Screen.prototype.swivelPictureRelatively = function() {
			this.iteratePictures(_Game_Screen_swivelPictureRelatively, arguments);
		};

		const _Game_Screen_setPictureSwivelInertia = Game_Screen.prototype.setPictureSwivelInertia;
		Game_Screen.prototype.setPictureSwivelInertia = function() {
			this.iteratePictures(_Game_Screen_setPictureSwivelInertia, arguments);
		};

		const _Game_Screen_fixPictureAngle = Game_Screen.prototype.fixPictureAngle;
		Game_Screen.prototype.fixPictureAngle = function() {
			this.iteratePictures(_Game_Screen_fixPictureAngle, arguments);
		};
	}

	//-----------------------------------------------------------------------------
	// Game_Picture

	const _Game_Picture_initialize = Game_Picture.prototype.initialize;
	Game_Picture.prototype.initialize = function() {
		_Game_Picture_initialize.call(this);
		this.initSwivel();
	};

	Game_Picture.prototype.initSwivel = function() {
		this._targetAngle = 0;
		this._swivelDuration = 0;
		this._swivelWholeDuration = 0;
		this._swivelEasingType = 0;
		this._swivelEasingExponent = 0;
		this._swivelLastSpeed = 0;
		this._swivelInertia = 0;
	};

	const _Game_Picture_show = Game_Picture.prototype.show;
	Game_Picture.prototype.show = function(name, origin, x, y, scaleX, scaleY, opacity, blendMode) {
		_Game_Picture_show.apply(this, arguments);
		this.initSwivel();
	};

	Game_Picture.prototype.swivelInertia = function() {
		return this._swivelInertia;
	};

	Game_Picture.prototype.setSwivelInertia = function(inertia) {
		this._swivelInertia = inertia;
	};

	Game_Picture.prototype.fixAngle = function() {
		this._angle = this._angle % 360;
	};

	Game_Picture.prototype.swivel = function(angle, duration, easingType) {
		this._targetAngle = angle;
		this._swivelDuration = duration || 1;
		this._swivelWholeDuration = duration || 1;
		this._swivelEasingType = easingType;
		this._swivelEasingExponent = 2;
		if (!duration) {
			this.updateSwivel();
		}
	};

	Game_Picture.prototype.swivelRelatively = function(angle, duration, easingType) {
		this.swivel(angle + this._angle, duration, easingType);
	};

	const _Game_Picture_update =  Game_Picture.prototype.update
	Game_Picture.prototype.update = function() {
		_Game_Picture_update.call(this);
		this.updateSwivel();
	};

	Game_Picture.prototype.updateSwivel = function() {
		if (this._swivelDuration > 0) {
			//元の値を保存
			const d = this._duration;
			const wd = this._wholeDuration;
			const easingType = this._easingType;
			const exponent = this._easingExponent;
			const angle = this._angle;
			//回転イージング用に差し替え
			this._duration = this._swivelDuration;
			this._wholeDuration = this._swivelWholeDuration;
			this._easingType = this._swivelEasingType;
			this._easingExponent = this._swivelEasingExponent;
			this._angle = this.applyEasing(this._angle, this._targetAngle);
			//戻す
			this._duration = d;
			this._wholeDuration = wd;
			this._easingType = easingType;
			this._easingExponent = exponent;
			this._swivelDuration--;
			//速度を保存する
			this._swivelLastSpeed = this._angle - angle;
		} else if (this._swivelInertia) {
			this._angle += this._swivelLastSpeed * this._swivelInertia;
		}
	};

}