/*:
 * @target MZ
 * @plugindesc 変数を画面上に監視・表示するプラグイン（背景画像付き・ウィンドウなし）v1.3
 * @author トリアコンタン MarkⅡ
 *
 * @param Variable1
 * @text 変数1
 * @type variable
 * @default 0
 *
 * @param Variable2
 * @text 変数2
 * @type variable
 * @default 0
 *
 * @param Variable3
 * @text 変数3
 * @type variable
 * @default 0
 *
 * @param Variable4
 * @text 変数4
 * @type variable
 * @default 0
 *
 * @param Variable5
 * @text 変数5
 * @type variable
 * @default 0
 *
 * @param X
 * @text 表示X座標
 * @type number
 * @default 10
 *
 * @param Y
 * @text 表示Y座標
 * @type number
 * @default 10
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @default 20
 *
 * @param TextColor
 * @text テキストカラー
 * @type string
 * @default #FFFFFF
 *
 * @param BackImage
 * @text 背景画像名
 * @type file
 * @dir img/pictures
 * @default
 * @desc 背景に使用する画像を指定します。指定しない場合は背景なし。
 *
 * @param BackOpacity
 * @text 背景の不透明度
 * @type number
 * @default 100
 * @min 0
 * @max 255
 *
 * @help
 * ウィンドウを使わずに、指定した変数を画面上へ表示します。
 * 背景に任意の画像を薄く敷くことができます。
 *
 * 【使い方】
 * 1. 「変数1〜5」に監視したい変数を設定。
 * 2. 「背景画像名」に img/pictures 内の画像ファイル名（拡張子不要）を指定。
 * 3. 不透明度を設定して薄い背景にできます。
 *
 * 【注意事項】
 * - 背景画像を指定しない場合は文字のみ表示。
 * - テキストに合わせて背景画像が自動リサイズされます。
 */

(() => {
    const PLUGIN_NAME = "TKR_VariableMonitor";
    const p = PluginManager.parameters(PLUGIN_NAME);

    const variableIds = [];
    for (let i = 1; i <= 5; i++) {
        const id = Number(p[`Variable${i}`] || 0);
        if (id > 0) variableIds.push(id);
    }

    const posX = Number(p["X"] || 10);
    const posY = Number(p["Y"] || 10);
    const fontSize = Number(p["FontSize"] || 20);
    const textColor = p["TextColor"] || "#FFFFFF";
    const backImage = p["BackImage"] || "";
    const backOpacity = Number(p["BackOpacity"] || 100);

    class Sprite_VariableMonitor extends Sprite {
        initialize() {
            super.initialize();
            this.x = posX;
            this.y = posY;
            this.bitmap = new Bitmap(Graphics.width, Graphics.height);
            this.bitmap.fontSize = fontSize;
            this.createBackground();
            this.refresh();
        }

        createBackground() {
            if (backImage) {
                this._backSprite = new Sprite(ImageManager.loadPicture(backImage));
                this._backSprite.opacity = backOpacity;
                this.addChild(this._backSprite);
            }
        }

        update() {
            super.update();
            const values = this.currentValues();
            if (!this._lastValues || !this.equals(values, this._lastValues)) {
                this.refresh();
            }
        }

        currentValues() {
            return variableIds.map(id => $gameVariables.value(id));
        }

        equals(a, b) {
            return JSON.stringify(a) === JSON.stringify(b);
        }

        refresh() {
            this.bitmap.clear();
            this.bitmap.textColor = textColor;

            let maxWidth = 0;
            variableIds.forEach((id, i) => {
                const name = $dataSystem.variables[id] || `変数${id}`;
                const value = $gameVariables.value(id);
                const text = `${name} : ${value}`;
                this.bitmap.drawText(text, 0, i * (fontSize + 4), 400, fontSize + 4);
                const width = this.bitmap.measureTextWidth(text);
                if (width > maxWidth) maxWidth = width;
            });

            // 背景画像のサイズをテキストに合わせる
            if (this._backSprite && this._backSprite.bitmap && this._backSprite.bitmap.isReady()) {
                const totalHeight = variableIds.length * (fontSize + 4);
                this._backSprite.scale.x = (maxWidth + 20) / this._backSprite.bitmap.width;
                this._backSprite.scale.y = (totalHeight + 10) / this._backSprite.bitmap.height;
                this._backSprite.x = -10;
                this._backSprite.y = -5;
            }

            this._lastValues = this.currentValues();
        }
    }

    // マップ表示
    const _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
    Scene_Map.prototype.createDisplayObjects = function() {
        _Scene_Map_createDisplayObjects.apply(this, arguments);
        if (variableIds.length > 0) {
            this._variableMonitorSprite = new Sprite_VariableMonitor();
            this.addChild(this._variableMonitorSprite);
        }
    };

    // バトル表示
    const _Scene_Battle_createDisplayObjects = Scene_Battle.prototype.createDisplayObjects;
    Scene_Battle.prototype.createDisplayObjects = function() {
        _Scene_Battle_createDisplayObjects.apply(this, arguments);
        if (variableIds.length > 0) {
            this._variableMonitorSprite = new Sprite_VariableMonitor();
            this.addChild(this._variableMonitorSprite);
        }
    };
})();
