//=============================================================================
// RPG Maker MZ - Battle Command Custom
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 戦闘シーンのコマンドをカスタマイズします。
 * @author sun_dandy
 * @url https://star-write-dream.com/
 *
 * @help BattleCommandCustom.js(ver1.0.0)
 *
 * 戦闘シーンのコマンドをカスタマイズします。
 * プラグインリストの下の方に配置してください。
 *
 * ・パーティコマンド：非表示
 * ・アクターコマンド：ウィンドウを拡張して表示（最大10件）
 * ・「逃げる」：アクターコマンドに追加
 *               使用可否をスイッチ制御
 *               100%成功
 *
 * --------------------------
 * ■利用規約
 *
 * Copyright (c) 2023 sun_dandy
 * https://star-write-dream.com/
 *
 * 以下に定める条件に従い、本プラグインファイル（以下「ソフトウェア」）の
 * 制作依頼者（以下、購入者）に対し、ソフトウェアを変更し、
 * 購入者の制作する作品（以下、制作物）に組み込むことを許可します。
 * 制作物の内容および公開形式に制約はありません。
 *
 * ソフトウェア単体で掲載、頒布、共有することはできません。
 * 上記の著作権表示および本許諾表示は変更できず、削除もできません。
 * 別途、制作物の重要な箇所にソフトウェアの著作権表示をする必要はありません。
 *
 * ソフトウェアは「現状のまま」で、明示であるか暗黙であるかを問わず、
 * 何らの保証もなく提供されます。ここでいう保証とは、商品性、
 * 特定の目的への適合性、および権利非侵害についての保証も含みますが、
 * それに限定されるものではありません。作者または著作権者は、契約行為、
 * 不法行為、またはそれ以外であろうと、ソフトウェアに起因または関連し、
 * あるいはソフトウェアの使用またはその他の扱いによって生じる一切の請求、
 * 損害、その他の義務について何らの責任も負わないものとします。
 * --------------------------
 *
 * @param escapeDisableSwitch
 * @type switch
 * @text 逃げる無効化スイッチ
 * @desc このスイッチがONだと「逃げる」が無効化されます。
 * @default 0
 *
 */

(() => {
    "use strict";

    const pluginName = "BattleCommandCustom";
    const parameters = PluginManager.parameters(pluginName);
    const escapeDisableSwitch = Number(parameters["escapeDisableSwitch"]) || 0;

    // -------------
    // BattleManager

    BattleManager.canEscape = function () {
        return !$gameSwitches.value(escapeDisableSwitch);
    };

    BattleManager.processEscape = function () {
        $gameParty.performEscape();
        SoundManager.playEscape();
        this.onEscapeSuccess();
        return true;
    };

    // -------------
    // Scene_Battle

    Scene_Battle.prototype.isAnyInputWindowActive = function () {
        return (
            this._actorCommandWindow.active || this._skillWindow.active || this._itemWindow.active || this._actorWindow.active || this._enemyWindow.active
        );
    };

    Scene_Battle.prototype.stop = function () {
        Scene_Message.prototype.stop.call(this);
        if (this.needsSlowFadeOut()) {
            this.startFadeOut(this.slowFadeSpeed(), false);
        } else {
            this.startFadeOut(this.fadeSpeed(), false);
        }
        this._statusWindow.close();
        this._actorCommandWindow.close();
    };

    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function () {
        _Scene_Battle_createAllWindows.apply(this, arguments);
        this._partyCommandWindow.hide();
    };

    Scene_Battle.prototype.startPartyCommandSelection = function () {
        this._statusWindow.deselect();
        this._statusWindow.show();
        this._statusWindow.open();
        this.selectNextCommand();
    };

    Scene_Battle.prototype.closeCommandWindows = function () {
        this._actorCommandWindow.deactivate();
        this._actorCommandWindow.close();
    };

    const _Scene_Battle_createActorCommandWindow = Scene_Battle.prototype.createActorCommandWindow;
    Scene_Battle.prototype.createActorCommandWindow = function () {
        _Scene_Battle_createActorCommandWindow.apply(this, arguments);
        this._actorCommandWindow.setHandler("escape", this.commandEscape.bind(this));
    };

    Scene_Battle.prototype.actorCommandWindowRect = function() {
        const ww = 192;
        const wh = this.calcWindowHeight(10, true);
        const wx = this.isRightInputMode() ? Graphics.boxWidth - ww : 0;
        const wy = Graphics.boxHeight - wh;
        return new Rectangle(wx, wy, ww, wh);
    };

    // -----------------------
    // Window_ActorCommand

    const _Window_ActorCommand_makeCommandList = Window_ActorCommand.prototype.makeCommandList;
    Window_ActorCommand.prototype.makeCommandList = function () {
        _Window_ActorCommand_makeCommandList.apply(this, arguments);
        this.addEscapeCommand();
    };

    Window_ActorCommand.prototype.addEscapeCommand = function () {
        this.addCommand(TextManager.escape, "escape", BattleManager.canEscape());
    };

    const _Window_ActorCommand_refresh = Window_ActorCommand.prototype.refresh;
    Window_ActorCommand.prototype.refresh = function () {
        _Window_ActorCommand_refresh.apply(this, arguments);
        const length = Math.max(4, this._list.length);
        const height = SceneManager._scene.calcWindowHeight(length, true);
        this.move(this.x, Graphics.boxHeight - height, this.width, height);
        Window_Selectable.prototype.refresh.call(this);
    }

})();
