//=============================================================================
// RPG Maker MZ - BattleSceneCustom2
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 戦闘シーンの配置をカスタマイズします。(1920*1080)
 * @author sun_dundy
 * @url https://star-write-dream.com/
 *
 * @help BattleSceneCustom2.js(ver1.0.0)
 *
 * 戦闘シーンの配置をカスタマイズします。
 *
 *
 * --------------------------
 * ■利用規約
 *
 * Copyright (c) 2024 sun_dandy
 * https://star-write-dream.com/
 *
 * 以下に定める条件に従い、本プラグインファイル（以下「ソフトウェア」）の
 * 制作依頼者（以下、購入者）に対し、ソフトウェアを変更し、
 * 購入者の制作する作品（以下、制作物）に組み込むことを許可します。
 * 制作物の内容および公開形式に制約はありません。
 *
 * ソフトウェア単体で掲載、頒布、共有することはできません。
 * 上記の著作権表示および本許諾表示は変更できず、削除もできません。
 * 別途、制作物の重要な箇所にソフトウェアの著作権表示をする必要はありません。
 *
 * ソフトウェアは「現状のまま」で、明示であるか暗黙であるかを問わず、
 * 何らの保証もなく提供されます。ここでいう保証とは、商品性、
 * 特定の目的への適合性、および権利非侵害についての保証も含みますが、
 * それに限定されるものではありません。作者または著作権者は、契約行為、
 * 不法行為、またはそれ以外であろうと、ソフトウェアに起因または関連し、
 * あるいはソフトウェアの使用またはその他の扱いによって生じる一切の請求、
 * 損害、その他の義務について何らの責任も負わないものとします。
 * --------------------------
 *
 * @param labelWidth
 * @text ゲージラベルの幅
 * @type number
 * @desc バトルステータスのゲージラベルの幅を設定します。
 * @default 100
 *
 * @param gaugeColorTp1
 * @text TPゲージの色（左）
 * @type string
 * @desc TPゲージのグラデーションカラー（左側）をHEX値で指定します。
 * @default #FF69B4
 *
 * @param gaugeColorTp2
 * @text TPゲージの色（右）
 * @type string
 * @desc TPゲージのグラデーションカラー（右側）をHEX値で指定します。
 * @default #FFC0CB
 *
 */

// 更新履歴
// v1.0.0
// 初版

(() => {
    "use strict";
    const pluginName = "BattleSceneCustom2";

    const parameters = PluginManager.parameters(pluginName);
    const labelWidth = Number(parameters["labelWidth"]) || 100;
    const gaugeColorTp1 = parameters["gaugeColorTp1"] || "#FF69B4";
    const gaugeColorTp2 = parameters["gaugeColorTp2"] || "#FFC0CB";

    // Scene_Battle

    Scene_Battle.prototype.statusWindowRect = function () {
        const extra = 10;
        const ww = Graphics.boxWidth / 2 - 192;
        const wh = this.windowAreaHeight() + extra;
        const wx = 192;
        const wy = Graphics.boxHeight - wh + extra - 4;
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_Battle.prototype.enemyWindowRect = function () {
        const wx = this._statusWindow.x;
        const ww = Graphics.boxWidth - 192;
        const wh = this.windowAreaHeight();
        const wy = Graphics.boxHeight - wh;
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_Battle.prototype.isRightInputMode = function () {
        return false;
    };

    Scene_Battle.prototype.windowAreaHeight = function () {
        return this.calcWindowHeight(4, true) + 90;
    };

    // ----------------------
    // Window_BattleStatus

    const _Window_BattleStatus_initialize = Window_BattleStatus.prototype.initialize;
    Window_BattleStatus.prototype.initialize = function (rect) {
        _Window_BattleStatus_initialize.apply(this, arguments);
        this._currentDrawIndex = 0;
    };

    Window_BattleStatus.prototype.maxCols = function () {
        return 2;
    };

    Window_StatusBase.prototype.placeGauge = function (actor, type, x, y) {
        const key = "actor%1-gauge-%2".format(actor.actorId(), type);
        const sprite = actor._actorId > 0 ? this.createInnerSprite(key, Sprite_Gauge_BattleActor) : this.createInnerSprite(key, Sprite_Gauge);
        sprite.setup(actor, type);
        sprite.move(x, y);
        sprite.show();
    };

    Window_BattleStatus.prototype.gaugeLineHeight = function () {
        return 30;
    };

    const _Window_BattleStatus_drawItemStatus = Window_BattleStatus.prototype.drawItemStatus;
    Window_BattleStatus.prototype.drawItemStatus = function (index) {
        this._currentDrawIndex = index;
        _Window_BattleStatus_drawItemStatus.apply(this, arguments);
    };

    Window_BattleStatus.prototype.faceRect = function (index) {
        const rect = this.itemRect(index);
        rect.pad(-1);
        rect.height = this.nameY(rect) + this.gaugeLineHeight() / 2 - rect.y;
        rect.y -= 16;
        return rect;
    };

    Window_BattleStatus.prototype.placeActorName = function (actor, x, y) {
        const key = "actor%1-name".format(actor.actorId());
        const sprite = this.createInnerSprite(key, Sprite_NameForBattleStatus);
        sprite.setup(actor);
        const rect = this.itemRect(this._currentDrawIndex);
        const wx = rect.x + (rect.width / 2 - sprite.bitmapWidth() / 2);
        sprite.move(wx, y - 10);
        sprite.show();
    };

    // ----------------------
    // ステータス用の少し長いゲージ

    class Sprite_Gauge_BattleActor extends Sprite_Gauge {
        bitmapWidth() {
            return 340;
        }

        labelWidth() {
            return labelWidth;
        }

        labelFontSize() {
            return $gameSystem.mainFontSize() - 10;
        }

        measureLabelWidth() {
            this.setupLabelFont();
            const labels = [TextManager.hpA, TextManager.mpA, TextManager.tpA];
            const width = labels.map((str) => this.bitmap.measureTextWidth(str));
            if (width.some((el) => el > 0)) {
                return this.labelWidth();
            } else {
                return 0;
            }
        }

        drawLabel() {
            const label = this.label();
            const x = this.labelOutlineWidth() / 2;
            const y = this.labelY();
            const width = this.labelWidth();
            const height = this.textHeight();
            this.setupLabelFont();
            this.bitmap.paintOpacity = this.labelOpacity();
            this.bitmap.drawText(label, x, y, width, height, "left");
            this.bitmap.paintOpacity = 255;
        }

        gaugeColor1() {
            if (this._statusType === "tp") {
                return gaugeColorTp1;
            }
            return super.gaugeColor1();
        }

        gaugeColor2() {
            if (this._statusType === "tp") {
                return gaugeColorTp2;
            }
            return super.gaugeColor2();
        }
    }

    class Sprite_NameForBattleStatus extends Sprite_Name {
        bitmapHeight() {
            return 44;
        }

        redraw() {
            const name = this.name();
            const width = this.bitmapWidth();
            const height = this.bitmapHeight();
            this.setupFont();
            this.bitmap.clear();
            this.bitmap.drawText(name, 0, 0, width, height, "center");
        }
    }
})();
