﻿//=============================================================================
// Ground.js
//=============================================================================

/*:
@target MV MZ
@plugindesc display ground tiling sprite
@author Lib

@help
*/

/*:ja
@target MV MZ
@plugindesc 遠景、近景、水、フォグ等を全て統合したグラウンドシステム
@author Lib

@help 
※このプラグインの他作品への転用、単体での頒布および販売を固く禁じます。

プラグインコマンド:
RemakeGround 番号 画像ファイル名 #番号の画像を差し替える

使用する画像ファイルは全て、img/parallaxes に置いてください。
※ブラウザでの動作は保証していません。

マップのメモ:
<gName番号:画像ファイル名,優先,(合成),(ループXの可否),(ループYの可否),(X速度),(Y速度),(視差補正X),(視差補正Y),(シフトX),(シフトY)>

番号 1～9

優先 0～5
0:遠景の下 1:マップチップの下 2:キャラの下 3:シャドウの下 4:天気の下 5:天気の上

合成 0～3
キャラクターと同じ

視差補正X,視差補正Y
2が通常と同じで大きいほど動きにくくなる

例：
<gName1:!Sample_U,0,1> //一般的な水中の表現
<gName2:!Sample_F,3,0> //一般的な近景の表現
<gName3:!Sample_C,4,0,0,true,true,-1,-1> //一般的なフォグの表現

@command RemakeGround
@text 遠景をリメイクする
@desc 遠景をリメイクする
@arg arg1
@text 遠景ID
@desc 遠景ID
@arg arg2
@text 遠景画像名
@desc 遠景画像名

*/

(function() {

"use strict";

//=======================
// PluginManager
//  プラグインコマンドを追加定義します。
//=======================
PluginManager.registerCommand("Ground", "RemakeGround", args => {
	var GroundNo = args.arg1;
	var GroundName = args.arg2;
	$gameMap._ground[GroundNo]._groundName = GroundName;
});

var GroundParam = {};
GroundParam.max = 10;	//重い場合は数を減らす。

var _DataManager_extractMetadata = DataManager.extractMetadata;
DataManager.extractMetadata      = function(data) {
	_DataManager_extractMetadata.apply(this, arguments);
		this.extractMetadataArray(data);
	};

	DataManager.extractMetadataArray = function(data) {
		var re         = /<([^<>:]+)(:?)([^>]*)>/g;
		data.metaArray = {};
		var match      = true;
		while (match) {
			match = re.exec(data.note);
			if (match) {
				var metaName             = match[1];
				data.metaArray[metaName] = data.metaArray[metaName] || [];
				data.metaArray[metaName].push(match[2] === ':' ? match[3] : true);
			}
	}
};

var getArgArrayString = function(args, upperFlg) {
	var values = getArgString(args, upperFlg).split(',');
	for (var i = 0; i < values.length; i++) values[i] = values[i].trim();
	return values;
};

var getArgString = function(args, upperFlg) {
	args = convertEscapeCharacters(args);
	return upperFlg ? args.toUpperCase() : args;
};

var getArgNumber = function(arg, min, max) {
	if (arguments.length < 2) min = -Infinity;
	if (arguments.length < 3) max = Infinity;
	return (parseInt(convertEscapeCharacters(arg), 10) || 0).clamp(min, max);
};

var convertEscapeCharacters = function(text) {
	if (text == null) text = '';
		text = text.replace(/\\/g, '\x1b');
		text = text.replace(/\x1b\x1b/g, '\\');
		text = text.replace(/\x1bV\[(\d+)\]/gi, function() {
			return $gameVariables.value(parseInt(arguments[1], 10));
		}.bind(this));
		text = text.replace(/\x1bV\[(\d+)\]/gi, function() {
		return $gameVariables.value(parseInt(arguments[1], 10));
	}.bind(this));
		text = text.replace(/\x1bN\[(\d+)\]/gi, function() {
		var actor = parseInt(arguments[1], 10) >= 1 ? $gameActors.actor(parseInt(arguments[1], 10)) : null;
		return actor ? actor.name() : '';
	}.bind(this));
		text = text.replace(/\x1bP\[(\d+)\]/gi, function() {
		var actor = parseInt(arguments[1], 10) >= 1 ? $gameParty.members()[parseInt(arguments[1], 10) - 1] : null;
		return actor ? actor.name() : '';
	}.bind(this));
		text = text.replace(/\x1bG/gi, TextManager.currencyUnit);
		return text;
};

//getArgStringは使わないBoolean変換
var getBoolean = function(str) {
	if(typeof str != 'string'){ 
		return Boolean(str); 
	}
	try{
		var obj = JSON.parse(str.toLowerCase());
		return obj == true;
	}catch(e){
		return str != '';
	}
};

//////////////////////////////////////////////////////////////
//Game_Map
Game_Map.prototype.setDisplayPosSolo = function(no, x, y) {
	if ($gameMap.isLoopHorizontal()) {
		this._ground[no]._groundX = x;
	} else {
		this._ground[no]._groundX = $gameMap._displayX;
	}
	if ($gameMap.isLoopVertical()) {
		this._ground[no]._groundY = y;
	} else {
		this._ground[no]._groundY = $gameMap._displayY;
	}
};

Game_Map.prototype.groundOxSolo = function(no) {
	if (this._ground[no]._groundZero) {
		return this._ground[no]._groundX * $gameMap.tileWidth();
//	} else if (this._ground[no]._groundLoopX) {
	} else if (this._ground[no]._groundSisaX != 0) {
		return this._ground[no]._groundX * $gameMap.tileWidth() / this._ground[no]._groundSisaX - this._ground[no]._groundCenteringX;
	} else {
		return 0;
	}
};

Game_Map.prototype.groundOySolo = function(no) {
	if (this._ground[no]._groundZero) {
		return this._ground[no]._groundY * $gameMap.tileHeight();
//	} else if (this._ground[no]._groundLoopY) {
	} else if (this._ground[no]._groundSisaY != 0) {
		return this._ground[no]._groundY * $gameMap.tileHeight() / this._ground[no]._groundSisaY - this._ground[no]._groundCenteringY;
	} else {
		return 0;
	}
};

Game_Map.prototype.scrollDownSolo = function(no,distance,lastY) {

	if ($gameMap.isLoopVertical()) {
		if (this._ground[no]._groundLoopY) {
			this._ground[no]._groundY += distance;
		}
	} else if ($gameMap.height() >= $gameMap.screenTileY()) {
		var displayY = Math.min(lastY + distance, $gameMap.height() - $gameMap.screenTileY());
		this._ground[no]._groundY += displayY - lastY;
	}
};

Game_Map.prototype.scrollLeftSolo = function(no,distance,lastX) {
	
	if ($gameMap.isLoopHorizontal()) {
		if (this._ground[no]._groundLoopX) {
			this._ground[no]._groundX -= distance;
		}
	} else if ($gameMap.width() >= $gameMap.screenTileX()) {
		var displayX = Math.max(lastX - distance, 0);
		this._ground[no]._groundX += displayX - lastX;
	}
};

Game_Map.prototype.scrollRightSolo = function(no,distance,lastX) {
	
	if ($gameMap.isLoopHorizontal()) {
		if (this._ground[no]._groundLoopX) {
			this._ground[no]._groundX += distance;
		}
	} else if ($gameMap.width() >= $gameMap.screenTileX()) {
		var displayX = Math.min(lastX + distance, $gameMap.width() - $gameMap.screenTileX());
		this._ground[no]._groundX += displayX - lastX;
	}
};

Game_Map.prototype.scrollUpSolo = function(no,distance,lastY) {
	
	if ($gameMap.isLoopVertical()) {
		if (this._ground[no]._groundLoopY) {
			this._ground[no]._groundY -= distance;
		}
	} else if ($gameMap.height() >= $gameMap.screenTileY()) {
		var displayY = Math.max(lastY - distance, 0);
		this._ground[no]._groundY += displayY - lastY;
	}
};

var _Game_Map_update = Game_Map.prototype.update;
Game_Map.prototype.update = function(sceneActive) {
	_Game_Map_update.call(this, sceneActive);

	var rx = $gameMap.displayX();
	var ry = $gameMap.displayY();
	this._charSpeedX = rx - this._displayX;
	this._charSpeedY = ry - this._displayY;
	this.lastX = rx;
	this.lastY = ry;

	for (var i=0; i<GroundParam.max; i++) {
		this.guardGround(i);
		this.updateSolo(i);
	}
};

Game_Map.prototype.updateSolo = function(No) {

	if(this._ground[No]._groundName){
		var speedX = this._ground[No]._groundSx;
		var speedY = this._ground[No]._groundSy;
		if( this._ground[No]._groundSx != 0 ){
			speedX -= this._charSpeedX;
		}
		if( this._ground[No]._groundSy != 0 ){
			speedY -= this._charSpeedY;
		}

		if (this._ground[No]._groundLoopX) {
	//		this._ground[No]._groundX += speedX / $gameMap.tileWidth() / this._ground[no]._groundSisaX;
			this._ground[No]._groundX += this.groundSpeedX(No);
		}
		if (this._ground[No]._groundLoopY) {
	//		this._ground[No]._groundY += speedY / $gameMap.tileHeight() / this._ground[no]._groundSisaY;
			this._ground[No]._groundY += this.groundSpeedY(No);
		}
	}
};

Game_Map.prototype.groundSpeedX = function(No) {

	var _ground = this._ground[No];

	if( Math.abs(_ground._groundAddX) < 1.0 ){	//少数の場合
		_ground._groundSx = (_ground._groundSx*100 + _ground._groundAddX*100)/100;
	}else{
		_ground._groundSx = _ground._groundAddX;
	}
	
	var r = parseInt(_ground._groundSx) / $gameMap.tileWidth() / _ground._groundSisaX;

	if( Math.abs(_ground._groundAddX) < 1.0 && Math.abs(_ground._groundSx) >= 1.0 ){
		_ground._groundSx = 0.0;
	}

	return Math.floor( r * Math.pow( 10, 2 ) ) / Math.pow( 10, 2 );
};

Game_Map.prototype.groundSpeedY = function(No) {

	var _ground = this._ground[No];

	if( Math.abs(_ground._groundAddY) < 1.0 ){	//少数の場合
		_ground._groundSy = (_ground._groundSy*100 + _ground._groundAddY*100)/100;
	}else{
		_ground._groundSy = _ground._groundAddY;
	}
	
	var r = parseInt(_ground._groundSy) / $gameMap.tileHeight() / _ground._groundSisaY;

	if( Math.abs(_ground._groundAddY) < 1.0 && Math.abs(_ground._groundSy) >= 1.0 ){
		_ground._groundSy = 0.0;
	}

	return Math.floor( r * Math.pow( 10, 2 ) ) / Math.pow( 10, 2 );

//	var r = this._ground[No]._groundSy / $gameMap.tileHeight() / this._ground[No]._groundSisaY;
//	return Math.floor( r * Math.pow( 10, 2 ) ) / Math.pow( 10, 2 );
};


ImageManager.isZeroForeground = function(filename) {
	return filename.charAt(0) === '!';
};

var _Game_Map_initialize = Game_Map.prototype.initialize;
Game_Map.prototype.initialize = function() {
	_Game_Map_initialize.call(this);
	this.initGround();
};

Game_Map.prototype.initGround = function(){
	this._ground = [];
	for (var i = 0; i < GroundParam.max; i++) {
		this._ground[i] = {};
		this.clearGroundSolo(i);
	}
};

Game_Map.prototype.guardGround = function(No){
	if(!this._ground[No] && this._ground[No]._groundName){
		this._ground[No] = {};
		this.clearGroundSolo(No);
	}
};

Game_Map.prototype.groundName = function(No) {
    this.guardGround(No);
    return this._ground[No]._groundName;
};

var _Game_Map_setup = Game_Map.prototype.setup;
Game_Map.prototype.setup = function(mapId) {
	_Game_Map_setup.call(this, mapId);
	this.setupGround();
};

Game_Map.prototype.getMetaCg = function(names) {
	if (!$dataMap.metaArray) {
		return null;
	}
	if (!Array.isArray(names)) names = [names];
	var metaParams = this.getMetaParameter(names);
	if (!metaParams) return null;
	var result = null;
	metaParams.some(function(metaParam) {
		var params = getArgArrayString(metaParam);
		result = params;
		return result;
	}.bind(this));
	return result;
};

Game_Map.prototype.getMetaParameter = function(names) {
	var metaParams = null;
	names.some(function(name) {
		if (!metaParams || metaParams[0] === '') {
			metaParams = $dataMap.metaArray[names];
		}
		return metaParams;
	}.bind(this));
	return metaParams;
};

Game_Map.prototype.setupGround = function() {
	this._waterEffect = false;
	for (var No=0; No<GroundParam.max; No++) {
		this.setupGroundSolo(No);
		if(this._ground[No]._groundFil == 1){
			this._waterEffect = true;
		}
	}
};

Game_Map.prototype.setupGroundSolo = function(No) {
	var name = "gName" + No;
	var mapParams = this.getMetaCg([name]);
	if( !mapParams ){
		this.clearGroundSolo(No);
		return;
	}
	if( !this._ground[No] ) return;

	this.clearGroundSolo(No);
	this._ground[No]._groundName = mapParams[0];
	this._ground[No]._groundPri = getArgNumber(mapParams[1]) || 0;			//プライオリティ
	this._ground[No]._groundBle = getArgNumber(mapParams[2]) || 0;			//合成
	this._ground[No]._groundLoopX = getBoolean(mapParams[3]) || false;	//横ループの可否
	this._ground[No]._groundLoopY = getBoolean(mapParams[4]) || false;	//縦ループの可否
//	this._ground[No]._groundSx = Number(mapParams[5]) || 0.0;				//Xの速度
//	this._ground[No]._groundSy = Number(mapParams[6]) || 0.0;				//Yの速度

	this._ground[No]._groundAddX = Number(mapParams[5]) || 0.0;				//Xの速度
	this._ground[No]._groundAddY = Number(mapParams[6]) || 0.0;				//Yの速度

	this._ground[No]._groundSisaX = Number(mapParams[7]) || 1.0;			//視差補正値X
	this._ground[No]._groundSisaY = Number(mapParams[8]) || 1.0;			//視差補正値Y
	this._ground[No]._groundCenteringX = getArgNumber(mapParams[9]) || 0;	//位置ずらしX
	this._ground[No]._groundCenteringY = getArgNumber(mapParams[10]) || 0;	//位置ずらしX
	this._ground[No]._groundZero = ImageManager.isZeroForeground(this._ground[No]._groundName);
	this._ground[No]._groundX = 0;
	this._ground[No]._groundY = 0;
};

Game_Map.prototype.clearGroundSolo = function(No){
	this._ground[No]._groundName = '';
	this._ground[No]._groundPri = 0;
	this._ground[No]._groundBle = 0;
	this._ground[No]._groundLoopX = false;
	this._ground[No]._groundLoopY = false;
	this._ground[No]._groundSx = 0.0;
	this._ground[No]._groundSy = 0.0;
	this._ground[No]._groundZero = true;
	this._ground[No]._groundSisaX = 1;
	this._ground[No]._groundSisaY = 1;
	this._ground[No]._groundX = 0;
	this._ground[No]._groundY = 0;
	this._ground[No]._groundCenteringX = 0;
	this._ground[No]._groundCenteringY = 0;
	this._ground[No]._groundOpa = 256;

	this._ground[No]._groundAddX = 0.0;
	this._ground[No]._groundAddY = 0.0;
};

var _Game_Map_setDisplayPos = Game_Map.prototype.setDisplayPos;
Game_Map.prototype.setDisplayPos = function(x, y) {
	_Game_Map_setDisplayPos.call(this, x, y);

	for (var i=0; i<GroundParam.max; i++) {
		this.guardGround(i);
		this.setDisplayPosSolo(i,x,y);
	}
};

var _Game_Map_scrollDown = Game_Map.prototype.scrollDown;
Game_Map.prototype.scrollDown = function(distance) {
	var lastY = this._displayY;
	_Game_Map_scrollDown.call(this, distance);

	for (var i=0; i<GroundParam.max; i++) {
		this.guardGround(i);
		this.scrollDownSolo(i,distance,lastY);
	}
};

var _Game_Map_scrollLeft = Game_Map.prototype.scrollLeft;
Game_Map.prototype.scrollLeft = function(distance) {
	var lastX = this._displayX;
	_Game_Map_scrollLeft.call(this, distance);

	for (var i=0; i<GroundParam.max; i++) {
		this.guardGround(i);
		this.scrollLeftSolo(i,distance,lastX);
	}
};

var _Game_Map_scrollRight = Game_Map.prototype.scrollRight;
Game_Map.prototype.scrollRight = function(distance) {
	var lastX = this._displayX;
	_Game_Map_scrollRight.call(this, distance);

	for (var i=0; i<GroundParam.max; i++) {
		this.guardGround(i);
		this.scrollRightSolo(i,distance,lastX);
	}
};

var _Game_Map_scrollUp = Game_Map.prototype.scrollUp;
Game_Map.prototype.scrollUp = function(distance) {
	var lastY = this._displayY;
	_Game_Map_scrollUp.call(this, distance);

	for (var i=0; i<GroundParam.max; i++) {
		this.guardGround(i);
		this.scrollUpSolo(i,distance,lastY);
	}
};


Game_Map.prototype.differenceX = function() {
	this.lastX = $gameMap._displayX;
	this.lastY = $gameMap._displayY;
};

/////////////////////////////////////////////////////////////////////////////////
//Spriteset_Map
var _Spriteset_Map_prototype_initialize = Spriteset_Map.prototype.initialize;
Spriteset_Map.prototype.initialize = function() {
	this._groundName = [];
	_Spriteset_Map_prototype_initialize.call(this);
};

var _Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
Spriteset_Map.prototype.createLowerLayer = function() {
	_Spriteset_Map_createLowerLayer.call(this);
	this.createGround();
};

var _Spriteset_Map_update = Spriteset_Map.prototype.update;
	Spriteset_Map.prototype.update = function() {
	_Spriteset_Map_update.call(this);
	this.updateGround();
};

Spriteset_Map.prototype.createGround = function() {

	//めんどくさいので一回全部リムーブ
	this._baseSprite.removeChild(this._parallax);	//遠景
	this._baseSprite.removeChild(this._tilemap);		//タイル
	for (var i = 0; i < this._characterSprites.length; i++) {	//キャラ
		this._tilemap.removeChild(this._characterSprites[i]);
	}
	this._tilemap.removeChild(this._shadowSprite);	//シャドウ
	this._baseSprite.removeChild(this._weather);		//天気

	this._groundSpr = [];
	for (var i = 0; i < GroundParam.max; i++) {
		if( $gameMap._ground[i]._groundName ){			//まだ生成されていないのでマップのメモ欄から取ったパラメータを参照する
			this._groundSpr[i] = new TilingSprite();
		   this._groundSpr[i].move(0, 0, Graphics.width, Graphics.height);
			this._groundSpr[i].blendMode = $gameMap._ground[i]._groundBle;
		}
	}

	this.setPriority(0);
	this._baseSprite.addChild(this._parallax);	//遠景
	this.setPriority(1);
	this._baseSprite.addChild(this._tilemap);		//タイル
	this.setPriority(2);
	for (var i = 0; i < this._characterSprites.length; i++) {	//キャラ
		if(this._characterSprites[i]._character._attachGroundNo == null){
			this._tilemap.addChild(this._characterSprites[i]);
		}
	}

	this.setPriority(3);
	this._tilemap.addChild(this._shadowSprite);	//シャドウ
	this.setPriority(4);
	this._baseSprite.addChild(this._weather);		//天気
	this.setPriority(5);
};

Spriteset_Map.prototype.charAttachGround = function(no,fb) {
	for (var j = 0; j < this._characterSprites.length; j++) {	//キャラ
		var chara = this._characterSprites[j];
		if(chara._character._attachGroundNo == no && chara._character._attachGroundFb == fb){
			this._baseSprite.addChild(chara);
//			this._groundSpr[no].addChild(chara);
		}
	}
};

Spriteset_Map.prototype.setPriority = function(pri) {
	for (var i = 0; i < GroundParam.max; i++) {
		if( this._groundSpr[i] && $gameMap._ground[i]._groundPri == pri ){
			this.charAttachGround(i,0);	//遠景の裏側にアタッチ
			this._baseSprite.addChild(this._groundSpr[i]);
			this.charAttachGround(i,1);	//遠景の表側にアタッチ
		}
	}
};

Spriteset_Map.prototype.updateGround = function() {

	for (var i = 0; i < GroundParam.max; i++) {
		if( this._groundSpr[i] ){
			if (this._groundName[i] !== $gameMap.groundName(i)) {
				this._groundName[i] = $gameMap.groundName(i);
				this._groundSpr[i].bitmap = ImageManager.loadParallax(this._groundName[i]);
			}

			if (this._groundSpr[i].bitmap) {
				this._groundSpr[i].origin.x = $gameMap.groundOxSolo(i);
				this._groundSpr[i].origin.y = $gameMap.groundOySolo(i);
			}
		}
	}
};

//アニメーションの優先順位対応
// prettier-ignore
var _Spriteset_Base_prototype_createAnimationSprite = Spriteset_Base.prototype.createAnimationSprite;
Spriteset_Base.prototype.createAnimationSprite = function(
    targets, animation, mirror, delay
) {
	_Spriteset_Base_prototype_createAnimationSprite.call(this,targets,animation,mirror,delay);

	var sprite = this._animationSprites[this._animationSprites.length-1];
	if(sprite.targetObjects[0]._attachGroundNo){
		 var no = sprite.targetObjects[0]._attachGroundNo;
		 SceneManager._scene._spriteset._groundSpr[no].addChild(sprite);
	}
};

})();
