/*:ja
 * @plugindesc 自作サーチ
 * @author kido
 * @help
 *   ARGシステムのサーチ機能
 *   使用方法: 
 *    

 *    キーワードを検索するとき				: ARG search;
 *    Addしたキーワードをチェックするとき	: ARG check;
 *    キーワードを追加するときは、[スクリプト]からthis.ARG_add();を呼び出す
 *    単語の追加：this.ARG_Add("I'm Ken");
 *    複数単語の追加：this.ARG_Add(["Hello","Nice to meet you"]);
 */

(function(){
	'use strict';
	
	let correctAnswers = [];

	let pluginName = "ARG";

	Game_Interpreter.prototype.ARG_Add = function( value ) {
        correctAnswers.push(value);
    };

	//プラグインコマンド定義
	var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;

	Game_Interpreter.prototype.pluginCommand = function(command, args) {
	_Game_Interpreter_pluginCommand.call(this, command, args);

		if( command === pluginName ) 
		{
			switch(args[0])
			{
				case 'init':
					init();
					break;
				case 'search':
					search();
					break;
				
				case 'check':
					check();
					break;
			}
		}
	}
	
	/// <summary>
    /// 正解リストの初期化
    /// </summary>
    /// <param name="label1">選択肢1のテキスト</param>
    /// <param name="label2">選択肢2のテキスト</param>
    /// <returns></returns>
	//	
	const init = function()
	{
		correctAnswers.splice(0);
	}
	
	/// <summary>
    /// 正解リストに追加
    /// </summary>
    /// <param name="args">追加する正解キーワード</param>
	const add = function(args)
	{
		//	キーワードが設定されていない
		if(args.length <= 1) return;
		
		if(args.length == 2)
		{
			correctAnswers.push(args[1]);
		}
		else
		{
			correctAnswers.push([args[1], args[2]]);
		}
	}

	/// <summary>
    /// 入力した検索キーワードを単語に分割する
    /// </summary>
    /// <param name="inputWords">入力した検索キーワード</param>
    /// <returns name = "matches">入力したキーワードを単語に分割し直した配列</returns>
	function parseKeywords(inputWords) 
	{
		// ダブルクォーテーション対応
		// "the brave story" prot → ["the brave story", "prot"]
		let regex = /"([^"]+)"|(\S+)/g;
		let matches = [];
		let match;

		while ((match = regex.exec(inputWords)) !== null) 
		{
			//	どちらかnullでない方をプッシュする
			//	全部null or undefinedの場合、先頭の方が入る
			//	なお、null,undefined,数値をtoLowerCase()しようとするとエラーになる
			matches.push((match[1] || match[2]).toLowerCase());
		}

		return matches;
	}

	/// <summary>
    /// 入力した検索キーワードが正解と一致しているかを判定する
    /// 単語は順不同
    /// </summary>
	function checkSearch()
	{
		let inputWords = $gameVariables.value(10);
		let words = parseKeywords(inputWords.trim());

		// 0単語チェック
		if (words.length === 0)
		{
			//	空欄のままは検索失敗
			setOutcome(0);
			return;
		}

		// 3語以上チェック（フレーズを考慮した後の数で判定）
		if (words.length > 2)
		{
			//	3つ以上のキーワードは検索失敗
			setOutcome(-3);
			return;
		}

		for (let answer of correctAnswers)
		{
			if (words.length !== answer.length) continue;

			//	[...words]	=>	スプレッド構文
			//	[words]の中身を先頭から順に引数に渡している。
			//	fonc(x, y)	に	func(words[0], words[1])と同じ意味になる
			//	配列ごと渡した各単語をアルファベット昇順にソートする
			let sortedInput = [...words].sort();
			let sortedAnswer = [...answer].sort();

			//	JSON.stringify()は、引数をJSONにエンコードする関数
			//	["prot","the brave story"]が入った引数を渡すと、["prot","the brave story"]でJSONファイルが作られる
			if (JSON.stringify(sortedInput) === JSON.stringify(sortedAnswer))
			{
				//	キーワード一致
				setOutcome(1);
				return;
			}
			
			//	惜しいチェック（片方でも含まれていれば）
			let someMatch = words.some(w => answer.includes(w));
			if (someMatch) 
			{
				//	片方のキーワードが一致
				setOutcome(-2);
				return;
			}
		}

		//	何も見つからなかった
		setOutcome(1);
		return;
	}

	/// <summary>
    /// 結果を判定用変数に保存する
    /// <param name="judgeType">判定タイプ：[-3: 3語エラー][-2: 片方一致][-1: 発見できず][0: 未入力][1: 発見]</param>
    /// </summary>
	const setOutcome = function(judgeType)
	{
		$gameVariables.setValue(11, judgeType);
	}

	/// <summary>
    /// デバッグ用。登録単語のコンソール出力。
    /// </summary>
	const check = function()
	{
		for(let i = 0; i < correctAnswers.length; i++)
		{
			console.log(correctAnswers[i]);
		}
	}	

	})();
