/*:
 * @target MZ
 * @plugindesc キャラカード（2〜4番）＋ATK合計ウィンドウ表示プラグイン。コスト・5人目表示は削除。
 * @author C&B
 *
 * @command ShowNoteWindows
 * @text 表示ON
 *
 * @command HideNoteWindows
 * @text 表示OFF
 *
 * @command RefreshNoteWindows
 * @text キャラカード再構築
 *
 * @command ReturnWindow
 * @text ウインドウ戻し
 * @desc ウインドウを元の位置に戻します
 *
 * @command Attack
 * @text 攻撃
 * @desc 一番上のキャラが回転しながら中央に移動します
 */

(() => {
  let showNoteWindows = false;
  let isAttacking = false;

  PluginManager.registerCommand("ActorNoteDisplay", "ShowNoteWindows", () => {
    showNoteWindows = true;
  });

  PluginManager.registerCommand("ActorNoteDisplay", "HideNoteWindows", () => {
    showNoteWindows = false;
  });

  PluginManager.registerCommand("ActorNoteDisplay", "RefreshNoteWindows", () => {
    if (SceneManager._scene?.refreshNoteWindows) {
      SceneManager._scene.refreshNoteWindows();
    }
  });

  PluginManager.registerCommand("ActorNoteDisplay", "ReturnWindow", () => {
    if (SceneManager._scene?.returnWindows) {
      SceneManager._scene.returnWindows();
    }
  });

  PluginManager.registerCommand("ActorNoteDisplay", "Attack", () => {
    if (SceneManager._scene?.performAttack) {
      SceneManager._scene.performAttack();
    }
  });

  class Window_ActorNote extends Window_Base {
    constructor(index) {
      const margin = 10;
      const width = 380;
      const height = 145;
      const x = 1920 - width - margin;
      const y = margin + index * (height + margin) - 100;
      super(new Rectangle(x, y, width, height));
      this._index = index;
      this._fontLarge = 28;
      this._fontSmall = 24;
      this._fontTiny = 22; 
      this._bgSprite = null;
      this._originalX = x;
      this._originalY = y;
      this._isAnimating = false;
      this._animationFrames = 0;
      this._rotation = 0;
      this.refresh();
    }

    loadBackgroundBitmap(name) {
      const sprite = new Sprite(ImageManager.loadPicture(name));
      
      const padding = 12;
      
      const contentWidth = this.width - padding * 2;
      const contentHeight = this.height - padding * 2;
      
      sprite.scale.x = contentWidth / sprite.bitmap.width;
      sprite.scale.y = contentHeight / sprite.bitmap.height;
      
      sprite.x = padding;
      sprite.y = padding;
      
      sprite.opacity = this.backOpacity;
      this.addChildToBack(sprite);
      return sprite;
    }

    clearBackground() {
      if (this._bgSprite) {
        this.removeChild(this._bgSprite);
        this._bgSprite = null;
      }
    }

    refresh() {
      this.contents.clear();
      const actor = $gameParty.members()[this._index];
      if (!actor) return;

      const level = actor.level;
      const rarityIndex = Math.min(level - 1, 3);
      const rarityLabel = ["N", "R", "SR", "SSR"][rarityIndex] ?? "N";
      const bgName = ["mojiiro00", "mojiiro02", "mojiiro03", "mojiiro01"][rarityIndex] ?? "mojiiro00";

      this.clearBackground();
      this._bgSprite = this.loadBackgroundBitmap(bgName);

      const charName = actor.characterName();
      const charIndex = actor.characterIndex();
      const bitmap = ImageManager.loadCharacter(charName);
      const pw = bitmap.width / 12;
      const ph = bitmap.height / 8;
      const sx = (charIndex % 4) * 3 * pw;
      const sy = Math.floor(charIndex / 4) * 4 * ph;
      bitmap.addLoadListener(() => {
        this.contents.blt(bitmap, sx, sy, pw, ph, 0, 0);
      });

      const startX = pw + 10;
      const classId = actor.currentClass().id;
      const iconId = classId === 1 ? 64 : classId === 2 ? 67 : classId === 3 ? 72 : 0;
      if (iconId > 0) this.drawIcon(iconId, startX, 2);

      this.contents.fontSize = this._fontLarge;
      this.changeTextColor(ColorManager.normalColor());
      const name = actor.name().slice(0, 10);
      this.drawText(`${name} [${rarityLabel}]`, startX + 36, 0, this.width - startX - 48);
      this.resetTextColor();

      const weapon = actor.weapons()[0];
      const atk = weapon ? weapon.params[2] : 0;

      this.contents.fontSize = this._fontSmall;
      this.changeTextColor(ColorManager.textColor(10));
      this.drawText(`atk ${atk}`, 8, 36, 100);
      this.resetTextColor();

      const notes = actor.actor().note.split(/[\r\n]/).filter(l => l.trim()).slice(0, 2);
      this.contents.fontSize = this._fontTiny; 
      for (let i = 0; i < notes.length; i++) {
        this.drawText(notes[i].slice(0, 30), 4, 66 + i * 22, this.width - 16); 
      }
    }

    startAttackAnimation() {
      this._isAnimating = true;
      this._animationFrames = 0;
      this._rotation = 0;
    }

    updateAnimation() {
      if (!this._isAnimating) return;

      this._animationFrames++;
      const totalFrames = 60; 
      const progress = this._animationFrames / totalFrames;

      if (progress <= 1) {
        const targetX = Graphics.width / 2 - this.width / 2;
        const targetY = Graphics.height / 2 - this.height / 2;
        
        this.x = this._originalX + (targetX - this._originalX) * progress;
        this.y = this._originalY + (targetY - this._originalY) * progress;
        this._rotation = progress * Math.PI * 8; 
        
        this.rotation = this._rotation;
      } else {
        this._isAnimating = false;
        this.x = Graphics.width / 2 - this.width / 2;
        this.y = Graphics.height / 2 - this.height / 2;
        this.rotation = 0;
        return true; 
      }
      
      return false;
    }

    returnToOriginalPosition() {
      this.x = this._originalX;
      this.y = this._originalY;
      this.rotation = 0;
      this._isAnimating = false;
      this._animationFrames = 0;
      this.visible = true;
    }
  }

  class Window_AtkTotal extends Window_Base {
    constructor() {
      const margin = 10;
      const width = 120;
      const height = 70;
      const x = 1920 - width - margin;
      const y = margin + 3 * (145 + margin) + 60;
      super(new Rectangle(x, y, width, height));
      this._fontSize = 22;
      this.refresh();
    }

    padding() {
      return 0;
    }

    refresh() {
      this.contents.clear();
      const party = $gameParty.members().slice(1, 4);
      let atkTotal = 0;

      for (const actor of party) {
        const weapon = actor.weapons()[0];
        if (weapon) atkTotal += weapon.params[2];
      }

      this.contents.fontSize = this._fontSize;
      this.changeTextColor(ColorManager.textColor(2));
      this.drawText(`atk ${atkTotal}`, 0, 10, this.width - 10, 'left');
      this.resetTextColor();
    }
  }

  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function () {
    _Scene_Map_start.call(this);
    this.refreshNoteWindows();
  };

  Scene_Map.prototype.refreshNoteWindows = function () {
    if (this._noteWindows) {
      this._noteWindows.forEach(win => this.removeChild(win));
    }
    this._noteWindows = [];

    for (let i = 1; i <= 3; i++) {
      const win = new Window_ActorNote(i);
      this.addChild(win);
      this._noteWindows.push(win);
    }

    if (this._atkTotalWindow) this.removeChild(this._atkTotalWindow);
    this._atkTotalWindow = new Window_AtkTotal();
    this.addChild(this._atkTotalWindow);
  };

  Scene_Map.prototype.performAttack = function () {
    if (isAttacking || !this._noteWindows || this._noteWindows.length === 0) return;
    
    isAttacking = true;
    const topWindow = this._noteWindows[0];
    
    topWindow.startAttackAnimation();
  };

  Scene_Map.prototype.returnWindows = function () {
    if (!this._noteWindows) return;
    
    this._noteWindows.forEach(win => {
      win.returnToOriginalPosition();
    });
    
    isAttacking = false;
    
    this.refreshNoteWindows();
  };


  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function () {
    _Scene_Map_update.call(this);
    if (!this._noteWindows) return;

    if (isAttacking) {
      this._noteWindows.forEach(win => {
        if (win._isAnimating) {
          win.updateAnimation();
        }
      });
      return;
    }

    const needsRefresh = this.checkForChanges();
    if (needsRefresh) {
      this.refreshNoteWindows();
    }

    const px = $gamePlayer.screenX();
    const py = $gamePlayer.screenY() - 24;

    const overlap = this._noteWindows.some(win => {
      const wx = win.x;
      const wy = win.y;
      const ww = win.width;
      const wh = win.height;
      return px >= wx && px <= wx + ww && py >= wy && py <= wy + wh;
    });

    this._noteWindows.forEach(win => {
      win.visible = showNoteWindows;
      if (!needsRefresh) win.refresh(); 
      const alpha = overlap ? 40 : 255;
      win.opacity = overlap ? 40 : 200;
      win.backOpacity = overlap ? 40 : 160;
      win.contentsOpacity = alpha;
      if (win._bgSprite) win._bgSprite.opacity = win.backOpacity;
    });

    if (this._atkTotalWindow) {
      this._atkTotalWindow.visible = showNoteWindows;
      if (!needsRefresh) this._atkTotalWindow.refresh();
      const alpha = overlap ? 40 : 255;
      this._atkTotalWindow.opacity = overlap ? 40 : 200;
      this._atkTotalWindow.backOpacity = overlap ? 40 : 160;
      this._atkTotalWindow.contentsOpacity = alpha;
    }
  };

  Scene_Map.prototype.checkForChanges = function () {
    if (!this._lastPartyState) {
      this._lastPartyState = this.getCurrentPartyState();
      return false;
    }

    const currentState = this.getCurrentPartyState();
    const hasChanged = JSON.stringify(this._lastPartyState) !== JSON.stringify(currentState);
    
    if (hasChanged) {
      this._lastPartyState = currentState;
      return true;
    }
    
    return false;
  };

  Scene_Map.prototype.getCurrentPartyState = function () {
    const party = $gameParty.members().slice(1, 4);
    return party.map(actor => ({
      id: actor.actorId(),
      name: actor.name(),
      level: actor.level,
      classId: actor.currentClass().id,
      weaponId: actor.weapons()[0] ? actor.weapons()[0].id : 0,
      atk: actor.weapons()[0] ? actor.weapons()[0].params[2] : 0,
      note: actor.actor().note
    }));
  };
})();