/*:
 * @target MZ
 * @plugindesc イベント注釈コメント表示プラグイン（▲表示のみ検出・アイコン対応した）
 * @author C&B
 *
 * @command ShowComment
 * @text コメント表示
 * @desc コメントウィンドウを表示します。
 *
 * @command HideComment
 * @text コメント非表示
 * @desc コメントウィンドウを非表示にします。
 *
 * @help
 * 使用方法：
 * - イベント内の注釈で以下のように記述
 * ▲表示 注釈タイトル
 * 表示したい文章
 * 複数行でも可能
 * 
 * アイコン表示：
 * - \I[n] または ￥I[n] でアイコンID nのアイコンを表示（例：￥I[64] でポーションアイコン）
 * - \I[アイコンID] または ￥I[アイコンID] の形式で使用
 *
 * - プラグインコマンド「ShowComment」で表示、「HideComment」で非表示
 */
(() => {
  const pluginName = "CommentDisplay";
  let commentWindow = null;
  let cachedCommentText = null; 

  PluginManager.registerCommand(pluginName, "ShowComment", () => {
    const text = getCommentText();
    if (text) {
      showCommentWindow(text);
    }
  });

  PluginManager.registerCommand(pluginName, "HideComment", () => {
    hideCommentWindow();
  });

  function getCommentText() {
    const interpreter = $gameMap._interpreter;
    if (!interpreter || !interpreter._list) return cachedCommentText;

    const list = interpreter._list;
    
    for (let i = 0; i < list.length; i++) {
      if (list[i].code === 108 && list[i].parameters[0].includes("▲表示")) {
        const lines = [list[i].parameters[0]]; 
        
        for (let j = i + 1; j < list.length; j++) {
          if (list[j].code === 408) {
            lines.push(list[j].parameters[0]);
          } else {
            break;
          }
        }
        
        const text = lines.join("\n").replace(/▲表示/g, "");
        cachedCommentText = text;
        return text;
      }
    }
    
    const events = $dataMap.events;
    if (events) {
      for (let eventId = 1; eventId < events.length; eventId++) {
        const event = events[eventId];
        if (!event) continue;
        
        for (const page of event.pages) {
          const pageList = page.list;
          for (let i = 0; i < pageList.length; i++) {
            if (pageList[i].code === 108 && pageList[i].parameters[0].includes("▲表示")) {
              const lines = [pageList[i].parameters[0]];
              
              for (let j = i + 1; j < pageList.length; j++) {
                if (pageList[j].code === 408) {
                  lines.push(pageList[j].parameters[0]);
                } else {
                  break;
                }
              }
              
              const text = lines.join("\n").replace(/▲表示/g, "");
              cachedCommentText = text;
              return text;
            }
          }
        }
      }
    }
    
    return cachedCommentText;
  }

  // 元のDOM要素版のprocessIconCodes関数は削除し、ウィンドウシステム用に作り直し
  
  // ウィンドウシステムベースのコメントウィンドウクラス
  class Window_Comment extends Window_Base {
    constructor(text) {
      // ウィンドウシステムでの座標系に合わせて調整（元のDOM版の位置に近づける）
      const x = 42;
      const y = 60;
      const width = Math.min(600, Graphics.width * 0.4);
      const height = 200;
      
      super(new Rectangle(x, y, width, height));
      this._text = text;
      this._fontSize = 26; // 元の18に戻す
      this.opacity = 0;           // ウィンドウ背景を透明に
      this.backOpacity = 0;       // ウィンドウ背景を透明に
      this.frameVisible = false;  // ウィンドウ枠を非表示
      this.loadCustomFont();
      this.refresh();
    }

    itemPadding() {
      return 2; // 元のpadding: "2px"を再現
    }
    
    standardPadding() {
      return 2;
    }

    loadCustomFont() {
      // YDWaosagiフォントを使わず通常フォントのままにする
    }

    refresh() {
      this.contents.clear();
      if (!this._text) return;

      // YDWaosagiフォント設定を削除し、通常フォント + 黒色に変更
      this.contents.fontSize = this._fontSize;
      this.changeTextColor('#000000');
      
      // テキストの描画（アイコン対応）
      this.drawTextWithIcons(this._text, 0, 0);
    }

    drawTextWithIcons(text, x, y) {
      if (!text) return;
      
      const lines = text.split('\n');
      const lineHeight = Math.floor(this._fontSize * 1.5); // 元の lineHeight: "1.5" を再現
      let currentY = y;
      
      // ウィンドウの高さを内容に合わせて調整
      const requiredHeight = lines.length * lineHeight + this.standardPadding() * 2 + 20;
      if (this.height < requiredHeight) {
        this.height = requiredHeight;
        this.contents = new Bitmap(this.contentsWidth(), this.contentsHeight());
        this.contents.fontFace = 'YDWaosagi, sans-serif';
        this.contents.fontSize = this._fontSize;
      }
      
      for (const line of lines) {
        this.drawLineWithIcons(line, x, currentY);
        currentY += lineHeight;
      }
    }

    drawLineWithIcons(line, startX, y) {
      let currentX = startX;
      const iconRegex = /[\\￥]I\[(\d+)\]/g;
      let lastIndex = 0;
      let match;
      
      while ((match = iconRegex.exec(line)) !== null) {
        // アイコンの前のテキストを描画
        const beforeText = line.substring(lastIndex, match.index);
        if (beforeText) {
          this.drawText(beforeText, currentX, y, this.contentsWidth() - currentX);
          currentX += this.textWidth(beforeText);
        }
        
        // アイコンを描画（位置を左に10px移動）
        const iconId = parseInt(match[1]);
        this.drawScaledIcon(iconId, currentX - 0, y + 2);
        currentX += 16; // 元の margin-right:-16px を考慮した調整
        
        lastIndex = iconRegex.lastIndex;
      }
      
      // 残りのテキストを描画
      const remainingText = line.substring(lastIndex);
      if (remainingText) {
        this.drawText(remainingText, currentX, y, this.contentsWidth() - currentX);
      }
    }

    drawScaledIcon(iconIndex, x, y) {
      const bitmap = ImageManager.loadSystem("IconSet");
      const pw = 32;
      const ph = 32;
      const sx = (iconIndex % 16) * pw;
      const sy = Math.floor(iconIndex / 16) * ph;
      // アイコンサイズを2倍に変更（16から32へ）
      const scaledSize = 32;
      this.contents.blt(bitmap, sx, sy, pw, ph, x, y, scaledSize, scaledSize);
    }
  }

  function showCommentWindow(text) {
    hideCommentWindow();
    
    if (SceneManager._scene && SceneManager._scene instanceof Scene_Map) {
      commentWindow = new Window_Comment(text);
      SceneManager._scene.addChild(commentWindow);
    }
  }

  function hideCommentWindow() {
    if (commentWindow && SceneManager._scene) {
      SceneManager._scene.removeChild(commentWindow);
      commentWindow = null;
    }
  }

  const _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
  Scene_Map.prototype.onMapLoaded = function() {
    _Scene_Map_onMapLoaded.call(this);
    cachedCommentText = null;
  };
})();