/*:
 * @target MZ
 * @plugindesc 注釈に{actorId:X}でアクター名＋レア＋ATKを2列で表示（v1.6）できるはず
 * @author C&B
 *
 * @command RefreshAnnotations
 * @text 注釈再チェック
 * @desc イベントページを再確認し注釈表示を更新します
 *
 * @command ShowAnnotations
 * @text 注釈表示ON
 * @desc 全ての注釈を表示します
 *
 * @command HideAnnotations
 * @text 注釈表示OFF
 * @desc 全ての注釈を非表示にします
 *
 * @command ToggleAnnotations
 * @text 注釈表示切り替え
 * @desc 注釈の表示・非表示を切り替えます
 */

(() => {
  const FONT_SIZE = 20;
  const TEXT_COLOR = "#ffffff";
  const OUTLINE_COLOR = "#000000";
  const OUTLINE_WIDTH = 4;
  const OFFSET_Y = -36;

  // グローバル表示状態を管理
  let annotationsVisible = true;

  function parseActorToken(token) {
    const id = parseInt(token);
    const actor = $gameActors.actor(id);
    if (!actor) return "Unknown\n---";

    const name = actor.name().slice(0, 10);
    const atk = actor.param(2);
    const level = actor.level;
    const rarityIndex = Math.min(level - 1, 3);
    const rarity = ["N", "R", "SR", "SSR"][rarityIndex] ?? "N";

    return `${name}\n[${rarity}] ATK ${atk}`;
  }

  class Sprite_Annotation extends Sprite {
    constructor(character) {
      super(new Bitmap(400, 48));
      this._character = character;
      this._rawText = "";
      this.anchor.x = 0.5;
      this.anchor.y = 1;
      this.z = 8;
      this.refresh();
    }

    parseText(raw) {
      return raw.replace(/\{actorId:(\d+)\}/g, (_, id) => {
        return parseActorToken(id);
      });
    }

    refresh() {
      const bmp = this.bitmap;
      bmp.clear();
      bmp.fontSize = FONT_SIZE;
      bmp.textColor = TEXT_COLOR;
      bmp.outlineColor = OUTLINE_COLOR;
      bmp.outlineWidth = OUTLINE_WIDTH;

      const parsed = this.parseText(this._rawText);
      const [line1, line2] = parsed.split("\n");
      bmp.drawText(line1 ?? "", 0, 0, bmp.width, 24, "center");
      bmp.drawText(line2 ?? "", 0, 22, bmp.width, 24, "center");
    }

    update() {
      super.update();
      this.x = this._character.screenX();
      this.y = this._character.screenY() + OFFSET_Y;
      this.updateFromEvent();
    }

    updateFromEvent() {
      const page = this._character.page();
      
      // イベント名が「ca」で始まるかチェック
      const eventName = this._character.event().name || "";
      const shouldShowForName = eventName.toLowerCase().startsWith("ca");
      
      // イベントページの最初のコマンドが注釈（コード108）かどうかをチェックしよう
      const firstCommand = page?.list?.[0];
      const isFirstCommandComment = firstCommand?.code === 108;
      
      const newRaw = isFirstCommandComment ? firstCommand.parameters[0] ?? "" : "";
      const hasContent = isFirstCommandComment;

      if (this._rawText !== newRaw) {
        this._rawText = newRaw;
        this.refresh();
      }

      // グローバル表示状態、注釈内容、イベント名の全てをチェック
      this.visible = hasContent && annotationsVisible && shouldShowForName;
    }
  }

  const _Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
  Spriteset_Map.prototype.createLowerLayer = function () {
    _Spriteset_Map_createLowerLayer.call(this);
    this._annotationSprites = [];
    this.createAnnotationSprites();
  };

  Spriteset_Map.prototype.createAnnotationSprites = function () {
    this._characterSprites.forEach(charSprite => {
      const ev = charSprite._character;
      if (ev instanceof Game_Event) {
        // イベント名が「ca」で始まる場合のみスプライトを作成
        const eventName = ev.event().name || "";
        if (eventName.toLowerCase().startsWith("ca")) {
          const sprite = new Sprite_Annotation(ev);
          this.addChild(sprite);
          this._annotationSprites.push(sprite);
        }
      }
    });
  };

  const _Spriteset_Map_update = Spriteset_Map.prototype.update;
  Spriteset_Map.prototype.update = function () {
    _Spriteset_Map_update.call(this);
    this._annotationSprites?.forEach(sprite => sprite.update());
  };

  // 表示状態を更新する関数
  function updateAnnotationsVisibility() {
    const spriteset = SceneManager._scene?._spriteset;
    if (spriteset?._annotationSprites) {
      spriteset._annotationSprites.forEach(sprite => sprite.updateFromEvent());
    }
  }

  // プラグインコマンドの登録
  PluginManager.registerCommand("CommentDisplayOnEventPage", "RefreshAnnotations", () => {
    const spriteset = SceneManager._scene._spriteset;
    if (spriteset?.createAnnotationSprites) {
      spriteset._annotationSprites?.forEach(sprite => spriteset.removeChild(sprite));
      spriteset._annotationSprites = [];
      spriteset.createAnnotationSprites();
    }
  });

  PluginManager.registerCommand("CommentDisplayOnEventPage", "ShowAnnotations", () => {
    annotationsVisible = true;
    updateAnnotationsVisibility();
  });

  PluginManager.registerCommand("CommentDisplayOnEventPage", "HideAnnotations", () => {
    annotationsVisible = false;
    updateAnnotationsVisibility();
  });

  PluginManager.registerCommand("CommentDisplayOnEventPage", "ToggleAnnotations", () => {
    annotationsVisible = !annotationsVisible;
    updateAnnotationsVisibility();
  });
})();