/*:
 * @target MZ
 * @plugindesc 文章表示コマンドで特殊記法を使ってイベント実行
 * @author C&B
 * @help MessageCommandExecutor.js
 *
 * 文章の表示で特殊な記法を書くと、自動的にコマンドを実行します。
 * 
 * 記法:
 * name:勇者@あいうえお1  → 名前欄に「勇者」、本文に「あいうえお1」を表示
 * name:勇者              → 名前欄に「勇者」のみ設定（次の行から本文）
 * 変数:1=100            → 変数1に100を代入（画面に表示せず飛ばす）
 * コモン:5              → コモンイベント5を実行（飛ばす）
 * SE:Coin,90,100        → SE再生（飛ばす）
 * BGM:Theme1,90,100     → BGM再生（飛ばす）
 * BGM停止:3             → BGMフェードアウト3秒（飛ばす）
 * 待ち:60               → 60フレーム待機（飛ばす）
 * 画像:1,name,0,0,255,0,100,100  → ピクチャ表示（番号,名前,X,Y,不透明度,原点,拡大率X,拡大率Y）
 *                                  原点: 0=左上 1=中央
 * 画像移動:1,100,200,60,255,100,100,0  → ピクチャ移動（番号,X,Y,時間,不透明度,拡大率X,拡大率Y,イージング）
 *                                        イージング: 0=一定速度 1=ゆっくり始まる 2=ゆっくり終わる
 * 画像消去:1            → ピクチャ1を消去
 * アニメ:10             → 画面中央にアニメーション10を表示
 * アニメ対象:1,10       → イベント1にアニメーション10を表示
 *
 * これらは会話イベント中に即座に実行されます。
 */

(() => {
    'use strict';

    const _Game_Message_add = Game_Message.prototype.add;
    Game_Message.prototype.add = function(text) {
        // name:で始まる場合は名前欄に設定
        if (text.startsWith('name:')) {
            const content = text.substring(5);
            const atIndex = content.indexOf('@');
            
            if (atIndex !== -1) {
                // name:勇者@あいうえお の形式
                const name = content.substring(0, atIndex).trim();
                const message = content.substring(atIndex + 1);
                this.setSpeakerName(name);
                _Game_Message_add.call(this, message);
                return;
            } else {
                // name:勇者 の形式（本文なし）
                const name = content.trim();
                this.setSpeakerName(name);
                return;
            }
        }

        // 変数:で始まる場合
        if (text.startsWith('変数:')) {
            const match = text.match(/変数:(\d+)=(\d+)/);
            if (match) {
                $gameVariables.setValue(parseInt(match[1]), parseInt(match[2]));
            }
            return; // 画面に表示せず飛ばす
        }

        // コモン:で始まる場合
        if (text.startsWith('コモン:')) {
            const match = text.match(/コモン:(\d+)/);
            if (match) {
                const commonEventId = parseInt(match[1]);
                const commonEvent = $dataCommonEvents[commonEventId];
                if (commonEvent) {
                    // 新しいインタープリターで独立して実行（親に影響を与えない）
                    const tempInterpreter = new Game_Interpreter(1);
                    tempInterpreter.setup(commonEvent.list, 0);
                    
                    // 条件分岐を正しく評価しながら実行
                    while (tempInterpreter.isRunning()) {
                        tempInterpreter.update();
                    }
                }
            }
            return;
        }

        // SE:で始まる場合
        if (text.startsWith('SE:')) {
            const match = text.match(/SE:([^,]+),(\d+),(\d+)/);
            if (match) {
                AudioManager.playSe({
                    name: match[1],
                    volume: parseInt(match[2]),
                    pitch: parseInt(match[3]),
                    pan: 0
                });
            }
            return;
        }

        // BGM停止:で始まる場合
        if (text.startsWith('BGM停止:')) {
            const match = text.match(/BGM停止:(\d+)/);
            if (match) {
                AudioManager.fadeOutBgm(parseInt(match[1]));
            }
            return;
        }

        // BGM:で始まる場合
        if (text.startsWith('BGM:')) {
            const match = text.match(/BGM:([^,]+),(\d+),(\d+)/);
            if (match) {
                AudioManager.playBgm({
                    name: match[1],
                    volume: parseInt(match[2]),
                    pitch: parseInt(match[3]),
                    pan: 0
                });
            }
            return;
        }

        // 待ち:で始まる場合
        if (text.startsWith('待ち:')) {
            const match = text.match(/待ち:(\d+)/);
            if (match) {
                const interpreter = $gameMap._interpreter;
                if (interpreter) {
                    interpreter.wait(parseInt(match[1]));
                }
            }
            return;
        }

        // 画像:で始まる場合（ピクチャ表示）
        if (text.startsWith('画像:')) {
            const match = text.match(/画像:(\d+),([^,]+),(-?\d+),(-?\d+),(\d+),(\d+),(\d+),(\d+)/);
            if (match) {
                const pictureId = parseInt(match[1]);
                const name = match[2];
                const x = parseInt(match[3]);
                const y = parseInt(match[4]);
                const opacity = parseInt(match[5]);
                const origin = parseInt(match[6]);
                const scaleX = parseInt(match[7]);
                const scaleY = parseInt(match[8]);
                $gameScreen.showPicture(pictureId, name, origin, x, y, scaleX, scaleY, opacity, 0);
            }
            return;
        }

        // 画像移動:で始まる場合（ピクチャ移動）
        if (text.startsWith('画像移動:')) {
            const match = text.match(/画像移動:(\d+),(-?\d+),(-?\d+),(\d+),(\d+),(\d+),(\d+),(\d+)/);
            if (match) {
                const pictureId = parseInt(match[1]);
                const x = parseInt(match[2]);
                const y = parseInt(match[3]);
                const duration = parseInt(match[4]);
                const opacity = parseInt(match[5]);
                const scaleX = parseInt(match[6]);
                const scaleY = parseInt(match[7]);
                const easingType = parseInt(match[8]);
                
                const picture = $gameScreen.picture(pictureId);
                if (picture) {
                    const origin = picture.origin();
                    const blendMode = picture.blendMode();
                    $gameScreen.movePicture(pictureId, origin, x, y, scaleX, scaleY, opacity, blendMode, duration, easingType);
                }
            }
            return;
        }

        // 画像消去:で始まる場合
        if (text.startsWith('画像消去:')) {
            const match = text.match(/画像消去:(\d+)/);
            if (match) {
                const pictureId = parseInt(match[1]);
                $gameScreen.erasePicture(pictureId);
            }
            return;
        }

        // アニメ対象:で始まる場合（特定の対象にアニメーション）
        if (text.startsWith('アニメ対象:')) {
            const match = text.match(/アニメ対象:(-?\d+),(\d+)/);
            if (match) {
                const targetId = parseInt(match[1]);
                const animationId = parseInt(match[2]);
                
                if (targetId === 0) {
                    // プレイヤー
                    $gamePlayer.requestAnimation(animationId);
                } else if (targetId > 0) {
                    // イベント
                    const event = $gameMap.event(targetId);
                    if (event) {
                        event.requestAnimation(animationId);
                    }
                }
            }
            return;
        }

        // アニメ:で始まる場合（画面中央にアニメーション）
        if (text.startsWith('アニメ:')) {
            const match = text.match(/アニメ:(\d+)/);
            if (match) {
                const animationId = parseInt(match[1]);
                $gameTemp.requestAnimation([$gamePlayer], animationId);
            }
            return;
        }

        // 通常の文章はそのまま追加
        _Game_Message_add.call(this, text);
    };

})();