﻿/*:
 * @author Synrec/Kylestclr
 * @plugindesc v1.5 Allows for creation of custom menus
 * @target MZ
 * @url https://synrec.itch.io/
 * 
 * @help
 * Use the plugin parameters to setup custom menus.
 * Ensure menu identifier names are unique.
 * 
 * Menus can be "nested"/lead further into more menus.
 * 
 * @param Menu Configurations
 * @desc Setup custom menus
 * @type struct<custMenu>[]
 * @default []
 * 
 */
/*~struct~staticGfx:
 * 
 * @param File
 * @desc Graphic file
 * @type file
 * @dir img/pictures/
 * 
 * @param X
 * @desc Position Setting
 * @type text
 * @default  0
 * 
 * @param Y
 * @desc Position Setting
 * @type text
 * @default  0
 * 
 * @param Scrolling X
 * @desc Scroll image consistently on origin.
 * @type number
 * @default 0
 * 
 * @param Scrolling Y
 * @desc Scroll image consistently on origin.
 * @type number
 * @default 0
 * 
 * @param Anchor X
 * @desc Modify image pivot point.
 * @type number
 * @decimals 3
 * @min -999999
 * @default 0.000
 * 
 * @param Anchor Y
 * @desc Modify image pivot point.
 * @type number
 * @decimals 3
 * @min -999999
 * @default 0.000
 * 
 * @param Rotation
 * @desc Apply rotation to image.
 * @type number
 * @decimals 3
 * @min -999999
 * @default 0.000
 * 
 * @param Constant Rotation
 * @desc Repeatedly apply rotation
 * @type boolean
 * @default false
 * 
 */
/*~struct~animGfx:
 * 
 * @param File
 * @desc Graphic file
 * @type file
 * @dir img/pictures/
 * 
 * @param X
 * @desc Position Setting
 * @type number
 * @default  0
 * 
 * @param Y
 * @desc Position Setting
 * @type number
 * @default  0
 * 
 * @param Max Frames
 * @desc Number of frames the graphic uses
 * @type number
 * @min 1
 * @default 1
 * 
 * @param Frame Rate
 * @desc Speed at which frames update
 * @type number
 * @default 0
 * 
 */
/*~struct~winText:
 * 
 * @param Text
 * @desc Text to draw
 * @type note
 * @default ""
 * 
 * @param X
 * @desc Starting position of text
 * @type text
 * @default 0
 * 
 * @param Y
 * @desc Starting position of text
 * @type text
 * @default 0
 *
 */
/*~struct~gfx:
 * 
 * @param Picture
 * @desc Picture file to use
 * @type file
 * @dir img/pictures/
 * 
 * @param X
 * @desc Starting position of picture
 * @type text
 * @default 0
 * 
 * @param Y
 * @desc Starting position of picture
 * @type text
 * @default 0
 * 
 * @param Width
 * @desc Size of the picture
 * @type text
 * @default 1
 * 
 * @param Height
 * @desc Size of the picture
 * @type text
 * @default 1
 *
 */
/*~struct~locSize:
 * 
 * @param X
 * @desc Position setting.
 * @type number
 * @default 0
 * 
 * @param Y
 * @desc Position setting.
 * @type number
 * @default 0
 * 
 * @param Width
 * @desc Size setting.
 * @type number
 * @default 1
 * 
 * @param Height
 * @desc Size setting.
 * @type number
 * @default 1
 * 
 */
/*~struct~windowStyle:
 * 
 * @param Font Settings
 * @desc Setup child parameters
 * 
 * @param Font Size
 * @parent Font Settings
 * @desc Size of font.
 * @type number
 * @default 16
 * 
 * @param Font Face
 * @parent Font Settings
 * @desc Font face used for the window.
 * @type text
 * @default sans-serif
 * 
 * @param Base Font Color
 * @parent Font Settings
 * @desc Default font color for window
 * @type text
 * @default #ffffff
 * 
 * @param Font Outline Color
 * @parent Font Settings
 * @desc Default font color for window
 * @type text
 * @default rgba(0, 0, 0, 0.5)
 * 
 * @param Font Outline Thickness
 * @parent Font Settings
 * @desc The thickness of the text outline
 * @type number
 * @default 3
 * 
 * @param Window Skin
 * @desc Image file used for the window skin.
 * @type file
 * @dir img/system/
 * @default Window
 * 
 * @param Window Opacity
 * @desc 0 = Fully transparent, 255 = Fully opaque.
 * @type number
 * @default 255
 * 
 * @param Show Window Dimmer
 * @desc Hides window skin
 * @type boolean
 * @default false
 * 
 */
/*~struct~gaugeDraw:
 * 
 * @param Label
 * @desc Label text for gauge
 * @type text
 * @default gauge
 * 
 * @param Label X
 * @desc Position of the label text in window
 * @type text
 * @default 0
 * 
 * @param Label Y
 * @desc Position of the label text in window
 * @type text
 * @default 0
 * 
 * @param Gauge Current Value
 * @desc How to set gauge current value
 * Evaluated value.
 * @type text
 * @default
 * 
 * @param Gauge Max Value
 * @desc How to set gauge max value
 * Evaluated value.
 * @type text
 * @default
 * 
 * @param Gauge X
 * @desc Position of the gauge in window
 * @type text
 * @default 0
 * 
 * @param Gauge Y
 * @desc Position of the gauge in window
 * @type text
 * @default 0
 * 
 * @param Gauge Width
 * @desc Size of the gauge
 * @type text
 * @default 1
 * 
 * @param Gauge Height
 * @desc Size of the gauge
 * @type text
 * @default 1
 * @default 1
 * 
 * @param Gauge Border
 * @desc Border size indent of the gauge
 * @type text
 * @default 2
 * 
 * @param Gauge Border Color
 * @desc Color for gauge border
 * @type text
 * @default #000000
 * 
 * @param Gauge Background Color
 * @desc Color for gauge background
 * @type text
 * @default #666666
 * 
 * @param Gauge Color
 * @desc Color for gauge background
 * @type text
 * @default #aaffaa
 * 
 */
/*~struct~basicWin:
 * 
 * @param Name
 * @desc No function
 * @type text
 * @default Window
 * 
 * @param Dimension Configuration
 * @desc Setup position and width of the window
 * @type struct<locSize>
 * @default {"X":"0","Y":"0","Width":"1","Height":"1"}
 * 
 * @param Window Font and Style Configuration
 * @desc Custom style the window
 * @type struct<windowStyle>
 * @default {"Font Settings":"","Font Size":"16","Font Face":"sans-serif","Base Font Color":"#ffffff","Font Outline Color":"rgba(0, 0, 0, 0.5)","Font Outline Thickness":"3","Window Skin":"Window","Window Opacity":"255","Show Window Dimmer":"false"}
 * 
 * @param Draw Texts
 * @desc Draw various text
 * @type struct<winText>[]
 * @default []
 * 
 * @param Text References
 * @parent Draw Texts
 * @desc Set code references for draw text
 * %1 = first, %2 = second, etc...
 * @type text
 * @default []
 * 
 * @param Draw Pictures
 * @desc Draw various pictures
 * @type struct<gfx>[]
 * @default []
 * 
 * @param Draw Gauges
 * @desc Draw gauges on the window
 * @type struct<gaugeDraw>[]
 * @default []
 * 
 */
/*~struct~gaugeDraw:
 * 
 * @param Label
 * @desc Label text for gauge
 * @type text
 * @default gauge
 * 
 * @param Label X
 * @desc Position of the label text in window
 * @type text
 * @default 0
 * 
 * @param Label Y
 * @desc Position of the label text in window
 * @type text
 * @default 0
 * 
 * @param Gauge Current Value
 * @desc How to set gauge current value
 * @type text
 * @default
 * 
 * @param Gauge Max Value
 * @desc How to set gauge max value
 * @type text
 * @default
 * 
 * @param Gauge X
 * @desc Position of the gauge in window
 * @type text
 * @default 0
 * 
 * @param Gauge Y
 * @desc Position of the gauge in window
 * @type text
 * @default 0
 * 
 * @param Gauge Width
 * @desc Size of the gauge
 * @type text
 * @default 1
 * 
 * @param Gauge Height
 * @desc Size of the gauge
 * @type text
 * @default 1
 * @default 1
 * 
 * @param Gauge Border
 * @desc Border size indent of the gauge
 * @type text
 * @default 2
 * 
 * @param Gauge Border Color
 * @desc Color for gauge border
 * @type text
 * @default #000000
 * 
 * @param Gauge Background Color
 * @desc Color for gauge background
 * @type text
 * @default #666666
 * 
 * @param Gauge Color
 * @desc Color for gauge background
 * @type text
 * @default #aaffaa
 * 
 */
/*~struct~stockVarReq:
 * 
 * @param Name
 * @desc No function
 * @type text
 * @default Variable
 * 
 * @param Variable
 * @desc The game variable considered
 * @type variable
 * @default 0
 * 
 * @param Min Value
 * @desc Smallest value of the variable
 * @type text
 * @default 0
 * 
 * @param Max Value
 * @desc Largest value of the variable
 * @type text
 * @default 0
 * 
 */
/*~struct~stockItmReq:
 * 
 * @param Name
 * @desc No function
 * @type text
 * @default Variable
 * 
 * @param Item
 * @desc The game item considered
 * @type item
 * @default 0
 * 
 * @param Min Value
 * @desc Smallest amount of the item.
 * @type text
 * @default 0
 * 
 * @param Max Value
 * @desc Largest amount of the item.
 * @type text
 * @default 0
 * 
 */
/*~struct~stockWepReq:
 * 
 * @param Name
 * @desc No function
 * @type text
 * @default Variable
 * 
 * @param Weapon
 * @desc The game weapon considered
 * @type weapon
 * @default 0
 * 
 * @param Min Value
 * @desc Smallest amount of the weapon.
 * @type text
 * @default 0
 * 
 * @param Max Value
 * @desc Largest amount of the weapon.
 * @type text
 * @default 0
 * 
 */
/*~struct~stockArmReq:
 * 
 * @param Name
 * @desc No function
 * @type text
 * @default Variable
 * 
 * @param Armor
 * @desc The game weapon considered
 * @type armor
 * @default 0
 * 
 * @param Min Value
 * @desc Smallest amount of the armor.
 * @type text
 * @default 0
 * 
 * @param Max Value
 * @desc Largest amount of the armor.
 * @type text
 * @default 0
 * 
 */
/*~struct~optReq:
 * 
 * @param Variable Requirements
 * @desc Setup multiple variable requirements
 * @type struct<stockVarReq>[]
 * @default []
 * 
 * @param Switch Requirements
 * @desc The listed switches must be enabled/ON.
 * @type switch[]
 * @default []
 * 
 * @param Item Requirements
 * @desc Setup multiple item requirements.
 * @type struct<stockItmReq>[]
 * @default []
 * 
 * @param Weapon Requirements
 * @desc Setup multiple weapon requirements.
 * @type struct<stockWepReq>[]
 * @default []
 * 
 * @param Armor Requirements
 * @desc Setup multiple armor requirements.
 * @type struct<stockArmReq>[]
 * @default []
 * 
 */
/*~struct~selcOpt:
 * 
 * @param Name
 * @desc Name of the option
 * @type text
 * @default Option
 * 
 * @param Alternative Name
 * @desc Alternative name of the option for display use
 * @type text
 * @default Alternative Name
 * 
 * @param Display Requirements
 * @desc Requirements to list option in menu
 * @type struct<optReq>
 * 
 * @param Select Requirements
 * @desc Requirements to list option in menu
 * @type struct<optReq>
 * 
 * @param Description
 * @desc How to describe the selection
 * @type note
 * @default ""
 * 
 * @param Pictures
 * @desc Picture images used to represent selection
 * @type file[]
 * @dir img/pictures/
 * @default []
 * 
 * @param Static Graphic
 * @desc Image layered over scene background and just below back graphics
 * @type struct<staticGfx>
 * 
 * @param Animated Graphic
 * @desc Animated graphic to represent selection
 * @type struct<animGfx>
 * 
 * @param Scene Button
 * @desc Add button to scene to link to option
 * @type struct<menuButton>
 * 
 * @param Event Execution
 * @desc Event to run on selecting the option
 * Takes priority over code execution.
 * @type common_event
 * @default 0
 * 
 * @param Code Execution
 * @desc Code to run on selecting the option
 * @type note
 * @default ""
 * 
 */
/*~struct~selcWindow:
 * 
 * @param Dimension Configuration
 * @desc Setup position and width of the window
 * @type struct<locSize>
 * @default {"X":"0","Y":"0","Width":"1","Height":"1"}
 * 
 * @param Window Font and Style Configuration
 * @desc Custom style the window
 * @type struct<windowStyle>
 * @default {"Font Settings":"","Font Size":"16","Font Face":"sans-serif","Base Font Color":"#ffffff","Font Outline Color":"rgba(0, 0, 0, 0.5)","Font Outline Thickness":"3","Window Skin":"Window","Window Opacity":"255","Show Window Dimmer":"false"}
 * 
 * @param Item Width
 * @desc Size of the selection area
 * @type text
 * @default 1
 * 
 * @param Item Height
 * @desc Size of the selection area
 * @type text
 * @default 1
 * 
 * @param Max Columns
 * @desc Number of horizontal columns
 * @type text
 * @default 1
 * 
 * @param Selection Options
 * @desc A list of options for the window
 * @type struct<selcOpt>[]
 * @default []
 * 
 * @param Draw Gauges
 * @desc Setup gauges for the window
 * @type struct<gaugeDraw>[]
 * @default []
 * 
 * @param Draw Name
 * @desc Draw Option Name?
 * @type boolean
 * @default false
 * 
 * @param Name Text
 * @parent Draw Name
 * @desc Text for name. %1 = Name.
 * @type text
 * @default %1
 * 
 * @param Name X
 * @parent Draw Name
 * @desc Position of text
 * @type text
 * @default 0
 * 
 * @param Name Y
 * @parent Draw Name
 * @desc Position of text
 * @type text
 * @default 0
 * 
 * @param Draw Alternative Name
 * @desc Draw Option Alternative Name?
 * @type boolean
 * @default false
 * 
 * @param Alternative Name Text
 * @parent Draw Alternative Name
 * @desc Text for name. %1 = Alternative Name.
 * @type text
 * @default %1
 * 
 * @param Alternative Name X
 * @parent Draw Alternative Name
 * @desc Position of text
 * @type text
 * @default 0
 * 
 * @param Alternative Name Y
 * @parent Draw Alternative Name
 * @desc Position of text
 * @type text
 * @default 0
 * 
 * @param Draw Description
 * @desc Draw Option Description?
 * @type boolean
 * @default false
 * 
 * @param Description X
 * @parent Draw Description
 * @desc Position of text
 * @type text
 * @default 0
 * 
 * @param Description Y
 * @parent Draw Description
 * @desc Position of text
 * @type text
 * @default 0
 * 
 * @param Draw Picture
 * @desc Draw Option Picture?
 * @type boolean
 * @default false
 * 
 * @param Picture Index
 * @parent Draw Picture
 * @desc Index of picture to draw (start at zero (0))
 * @type text
 * @default $gameVariables.value(1)
 * 
 * @param Picture X
 * @parent Draw Picture
 * @desc Position of picture
 * @type text
 * @default 0
 * 
 * @param Picture Y
 * @parent Draw Picture
 * @desc Position of picture
 * @type text
 * @default 0
 * 
 * @param Picture Width
 * @parent Draw Picture
 * @desc Size of picture
 * @type text
 * @default 0
 * 
 * @param Picture Height
 * @parent Draw Picture
 * @desc Size of picture
 * @type text
 * @default 0
 * 
 */
/*~struct~selcDataWindow:
 * 
 * @param Name
 * @desc No function
 * @type text
 * @default Data Window
 * 
 * @param Dimension Configuration
 * @desc Setup position and width of the window
 * @type struct<locSize>
 * @default {"X":"0","Y":"0","Width":"1","Height":"1"}
 * 
 * @param Window Font and Style Configuration
 * @desc Custom style the window
 * @type struct<windowStyle>
 * @default {"Font Settings":"","Font Size":"16","Font Face":"sans-serif","Base Font Color":"#ffffff","Font Outline Color":"rgba(0, 0, 0, 0.5)","Font Outline Thickness":"3","Window Skin":"Window","Window Opacity":"255","Show Window Dimmer":"false"}
 * 
 * @param Draw Gauges
 * @desc Setup gauges for the window
 * @type struct<gaugeDraw>[]
 * @default []
 * 
 * @param Draw Option Name
 * @desc Draw Name of option?
 * @type boolean
 * @default false
 * 
 * @param Name X
 * @parent Draw Option Name
 * @desc Position of name
 * @type text
 * @default 0
 * 
 * @param Name Y
 * @parent Draw Option Name
 * @desc Position of name
 * @type text
 * @default 0
 * 
 * @param Draw Alternative Option Name
 * @desc Draw Name of option?
 * @type boolean
 * @default false
 * 
 * @param Alternative Name X
 * @parent Draw Alternative Option Name
 * @desc Position of name
 * @type text
 * @default 0
 * 
 * @param Alternative Name Y
 * @parent Draw Alternative Option Name
 * @desc Position of name
 * @type text
 * @default 0
 * 
 * @param Draw Description
 * @desc Draw Description of option?
 * @type boolean
 * @default false
 * 
 * @param Description X
 * @parent Draw Description
 * @desc Position of name
 * @type text
 * @default 0
 * 
 * @param Description Y
 * @parent Draw Description
 * @desc Position of name
 * @type text
 * @default 0
 * 
 * @param Draw Picture
 * @desc Draw Description of option?
 * @type boolean
 * @default false
 * 
 * @param Picture Index
 * @parent Draw Picture
 * @desc Index of picture used.
 * @type text
 * @default 1
 * 
 * @param Picture X
 * @parent Draw Picture
 * @desc Position of picture
 * @type text
 * @default 0
 * 
 * @param Picture Y
 * @parent Draw Picture
 * @desc Position of picture
 * @type text
 * @default 0
 * 
 * @param Picture Width
 * @parent Draw Picture
 * @desc Size of picture
 * @type text
 * @default 0
 * 
 * @param Picture Height
 * @parent Draw Picture
 * @desc Size of picture
 * @type text
 * @default 0
 * 
 */
/*~struct~menuButton:
 * 
 * @param Name
 * @desc No function
 * @type text
 * @default Button
 * 
 * @param X
 * @desc Position on screen.
 * @type text
 * @default 0
 * 
 * @param Y
 * @desc Position on screen.
 * @type text
 * @default 0
 * 
 * @param Cold Graphic
 * @desc Graphic when not mouse over.
 * @type struct<animGfx>
 * 
 * @param Hot Graphic
 * @desc Graphic when mouse over.
 * @type struct<animGfx>
 * 
 */
/*~struct~custMenu:
 *
 * @param Identifier Name
 * @desc Name used to identify menu data.
 * @type text
 * @default menu
 * 
 * @param Preload Backgrounds
 * @desc Picture images used for scene preload background.
 * @type struct<gfx>[]
 * @default []
 * 
 * @param Backgrounds
 * @desc Picture images used for scene background.
 * @type struct<staticGfx>[]
 * @default []
 * 
 * @param Back Graphics
 * @desc Animated pictures layered just above scene background.
 * @type struct<animGfx>[]
 * @default []
 * 
 * @param Selection Window
 * @desc Window used for selection options
 * @type struct<selcWindow>
 * 
 * @param Selection Data Windows
 * @parent Selection Window
 * @desc Display data based on window selection
 * @type struct<selcDataWindow>[]
 * @default []
 * 
 * @param Basic Windows
 * @desc Windows to display basic data. (Always shown)
 * @type struct<basicWin>[]
 * @default []
 * 
 * @param Fore Graphics
 * @desc Animated pictures layered just below scene foreground.
 * @type struct<animGfx>[]
 * @default []
 * 
 * @param Foregrounds
 * @desc Picture images used for scene foreground.
 * @type struct<staticGfx>[]
 * @default []
 * 
 */
/*:ja
 * @author Synrec/Kylestclr
 * @plugindesc v1.5 カスタムメニューの作成が可能
 * @target MZ
 * @url https://synrec.itch.io/
 * 
 * @help
 * カスタムメニューを設定するには、プラグインパラメータを使用します。
 * メニュー識別子の名前がユニークであることを確認します。
 * 
 * メニューは 「ネスト」（入れ子）することができ、さらに他のメニューに導くことができる。
 * 
 * @param Menu Configurations
 * @desc カスタムメニューの設定
 * @type struct<custMenu>[]
 * @default []
 * 
 */
/*~struct~staticGfx:ja
 * 
 * @param File
 * @desc 画像ファイル
 * @type file
 * @dir img/pictures/
 * 
 * @param X
 * @desc X位置設定
 * @type text
 * @default  0
 * 
 * @param Y
 * @desc Y位置設定
 * @type text
 * @default  0
 * 
 * @param Scrolling X
 * @desc 画像をX原点上で一貫して画像をスクロールする。
 * @type number
 * @default 0
 * 
 * @param Scrolling Y
 * @desc 画像をY原点上で一貫して画像をスクロールする。
 * @type number
 * @default 0
 * 
 * @param Anchor X
 * @desc 画像のXピボットポイントを変更する。
 * @type number
 * @decimals 3
 * @min -999999
 * @default 0.000
 * 
 * @param Anchor Y
 * @desc 画像のYピボットポイントを変更する。
 * @type number
 * @decimals 3
 * @min -999999
 * @default 0.000
 * 
 * @param Rotation
 * @desc 画像に回転をかける。
 * @type number
 * @decimals 3
 * @min -999999
 * @default 0.000
 * 
 * @param Constant Rotation
 * @desc 回転の繰り返し
 * @type boolean
 * @default false
 * 
 */
/*~struct~animGfx:ja
 * 
 * @param File
 * @desc 画像ファイル
 * @type file
 * @dir img/pictures/
 * 
 * @param X
 * @desc X位置設定
 * @type number
 * @default  0
 * 
 * @param Y
 * @desc Y位置設定
 * @type number
 * @default  0
 * 
 * @param Max Frames
 * @desc グラフィックが使用するフレーム数
 * @type number
 * @min 1
 * @default 1
 * 
 * @param Frame Rate
 * @desc フレームの更新速度
 * @type number
 * @default 0
 * 
 */
/*~struct~winText:ja
 * 
 * @param Text
 * @desc 描くテキスト
 * @type note
 * @default ""
 * 
 * @param X
 * @desc テキストのX開始位置
 * @type text
 * @default 0
 * 
 * @param Y
 * @desc テキストのY開始位置
 * @type text
 * @default 0
 *
 */
/*~struct~gfx:ja
 * 
 * @param Picture
 * @desc 使用する画像ファイル
 * @type file
 * @dir img/pictures/
 * 
 * @param X
 * @desc 画像のX開始位置
 * @type text
 * @default 0
 * 
 * @param Y
 * @desc 画像のY開始位置
 * @type text
 * @default 0
 * 
 * @param Width
 * @desc 画像の幅
 * @type text
 * @default 1
 * 
 * @param Height
 * @desc 画像の高さ
 * @type text
 * @default 1
 *
 */
/*~struct~locSize:ja
 * 
 * @param X
 * @desc X位置設定
 * @type number
 * @default 0
 * 
 * @param Y
 * @desc Y位置設定.
 * @type number
 * @default 0
 * 
 * @param Width
 * @desc 幅の設定
 * @type number
 * @default 1
 * 
 * @param Height
 * @desc 高さの設定
 * @type number
 * @default 1
 * 
 */
/*~struct~windowStyle:ja
 * 
 * @param Font Settings
 * @desc 子パラメーターの設定
 * 
 * @param Font Size
 * @parent Font Settings
 * @desc フォントサイズ
 * @type number
 * @default 16
 * 
 * @param Font Face
 * @parent Font Settings
 * @desc ウィンドウに使用されるフォント
 * @type text
 * @default sans-serif
 * 
 * @param Base Font Color
 * @parent Font Settings
 * @desc ウィンドウのデフォルトフォントの色
 * @type text
 * @default #ffffff
 * 
 * @param Font Outline Color
 * @parent Font Settings
 * @desc ウィンドウのデフォルトフォントの輪郭色
 * @type text
 * @default rgba(0, 0, 0, 0.5)
 * 
 * @param Font Outline Thickness
 * @parent Font Settings
 * @desc テキストの輪郭の太さ
 * @type number
 * @default 3
 * 
 * @param Window Skin
 * @desc ウィンドウスキンに使用される画像ファイル
 * @type file
 * @dir img/system/
 * @default Window
 * 
 * @param Window Opacity
 * @desc 0 = 完全に透明、255 = 完全に不透明
 * @type number
 * @default 255
 * 
 * @param Show Window Dimmer
 * @desc ウィンドウのスキンを隠す
 * @type boolean
 * @default false
 * 
 */
/*~struct~gaugeDraw:ja
 * 
 * @param Label
 * @desc ゲージのラベルテキスト
 * @type text
 * @default gauge
 * 
 * @param Label X
 * @desc ウィンドウ内のラベルテキストのX位置
 * @type text
 * @default 0
 * 
 * @param Label Y
 * @desc ウィンドウ内のラベルテキストのY位置
 * @type text
 * @default 0
 * 
 * @param Gauge Current Value
 * @desc ゲージ電流値の設定方法
 * Evaluated value.
 * @type text
 * @default
 * 
 * @param Gauge Max Value
 * @descゲージの最大値の設定方法
 * Evaluated value.
 * @type text
 * @default
 * 
 * @param Gauge X
 * @desc ウィンドウ内のゲージのX位置
 * @type text
 * @default 0
 * 
 * @param Gauge Y
 * @desc ウィンドウ内のゲージのY位置
 * @type text
 * @default 0
 * 
 * @param Gauge Width
 * @desc ゲージの幅
 * @type text
 * @default 1
 * 
 * @param Gauge Height
 * @desc ゲージの高さ
 * @type text
 * @default 1
 * @default 1
 * 
 * @param Gauge Border
 * @desc ゲージの枠の大きさ
 * @type text
 * @default 2
 * 
 * @param Gauge Border Color
 * @desc ゲージの縁取りの色
 * @type text
 * @default #000000
 * 
 * @param Gauge Background Color
 * @desc ゲージ背景の色
 * @type text
 * @default #666666
 * 
 * @param Gauge Color
 * @desc ゲージ背景の色
 * @type text
 * @default #aaffaa
 * 
 */
/*~struct~basicWin:ja
 * 
 * @param Name
 * @desc 機能なし
 * @type text
 * @default Window
 * 
 * @param Dimension Configuration
 * @desc ウィンドウの位置と幅の設定
 * @type struct<locSize>
 * @default {"X":"0","Y":"0","Width":"1","Height":"1"}
 * 
 * @param Window Font and Style Configuration
 * @desc ウィンドウのカスタムスタイル
 * @type struct<windowStyle>
 * @default {"Font Settings":"","Font Size":"16","Font Face":"sans-serif","Base Font Color":"#ffffff","Font Outline Color":"rgba(0, 0, 0, 0.5)","Font Outline Thickness":"3","Window Skin":"Window","Window Opacity":"255","Show Window Dimmer":"false"}
 * 
 * @param Draw Texts
 * @desc 様々なテキストを描画
 * @type struct<winText>[]
 * @default []
 * 
 * @param Text References
 * @parent テキストを描画
 * @desc Set code references for draw text
 * %1 = first, %2 = second, etc...
 * @type text
 * @default []
 * 
 * @param Draw Pictures
 * @desc様々な絵を描画
 * @type struct<gfx>[]
 * @default []
 * 
 * @param Draw Gauges
 * @desc ウィンドウにゲージを描画
 * @type struct<gaugeDraw>[]
 * @default []
 * 
 */
/*~struct~gaugeDraw:ja
 * 
 * @param Label
 * @desc ゲージのラベルテキスト
 * @type text
 * @default gauge
 * 
 * @param Label X
 * @desc ウィンドウ内のラベルテキストのX位置
 * @type text
 * @default 0
 * 
 * @param Label Y
 * @desc ウィンドウ内のラベルテキストのY位置
 * @type text
 * @default 0
 * 
 * @param Gauge Current Value
 * @desc ゲージ電流値の設定方法
 * @type text
 * @default
 * 
 * @param Gauge Max Value
 * @desc Hゲージ最大値の設定方法
 * @type text
 * @default
 * 
 * @param Gauge X
 * @desc ウィンドウ内のゲージのX位置
 * @type text
 * @default 0
 * 
 * @param Gauge Y
 * @desc ウィンドウ内のゲージのY位置
 * @type text
 * @default 0
 * 
 * @param Gauge Width
 * @desc ゲージの幅
 * @type text
 * @default 1
 * 
 * @param Gauge Height
 * @desc ゲージの高さ
 * @type text
 * @default 1
 * @default 1
 * 
 * @param Gauge Border
 * @desc ゲージの枠の大きさ
 * @type text
 * @default 2
 * 
 * @param Gauge Border Color
 * @desc ゲージの縁取りの色
 * @type text
 * @default #000000
 * 
 * @param Gauge Background Color
 * @desc ゲージ背景の色
 * @type text
 * @default #666666
 * 
 * @param Gauge Color
 * @desc ゲージ背景の色
 * @type text
 * @default #aaffaa
 * 
 */
/*~struct~stockVarReq:ja
 * 
 * @param Name
 * @desc 機能なし
 * @type text
 * @default Variable
 * 
 * @param Variable
 * @desc ゲーム変数
 * @type variable
 * @default 0
 * 
 * @param Min Value
 * @desc 変数の最小値
 * @type text
 * @default 0
 * 
 * @param Max Value
 * @desc 変数の最大値
 * @type text
 * @default 0
 * 
 */
/*~struct~stockItmReq:ja
 * 
 * @param Name
 * @desc 機能なし
 * @type text
 * @default Variable
 * 
 * @param Item
 * @desc ゲームアイテム
 * @type item
 * @default 0
 * 
 * @param Min Value
 * @desc アイテムの最小量
 * @type text
 * @default 0
 * 
 * @param Max Value
 * @desc アイテムの最大量
 * @type text
 * @default 0
 * 
 */
/*~struct~stockWepReq:ja
 * 
 * @param Name
 * @desc 機能なし
 * @type text
 * @default Variable
 * 
 * @param Weapon
 * @desc ゲームの武器
 * @type weapon
 * @default 0
 * 
 * @param Min Value
 * @desc 武器の最小量
 * @type text
 * @default 0
 * 
 * @param Max Value
 * @desc 武器の最大量
 * @type text
 * @default 0
 * 
 */
/*~struct~stockArmReq:ja
 * 
 * @param Name
 * @desc 機能なし
 * @type text
 * @default Variable
 * 
 * @param Armor
 * @desc ゲームの鎧
 * @type armor
 * @default 0
 * 
 * @param Min Value
 * @desc 鎧の最小値
 * @type text
 * @default 0
 * 
 * @param Max Value
 * @desc 鎧の最大値
 * @type text
 * @default 0
 * 
 */
/*~struct~optReq:ja
 * 
 * @param Variable Requirements
 * @desc 複数の可変要件を設定する
 * @type struct<stockVarReq>[]
 * @default []
 * 
 * @param Switch Requirements
 * @desc 記載されているスイッチは有効/ONでなければならない。
 * @type switch[]
 * @default []
 * 
 * @param Item Requirements
 * @desc 複数のアイテムの必要要件を設定します。
 * @type struct<stockItmReq>[]
 * @default []
 * 
 * @param Weapon Requirements
 * @desc 複数の武器の必要要件を設定します。
 * @type struct<stockWepReq>[]
 * @default []
 * 
 * @param Armor Requirements
 * @desc 複数の鎧の必要要件を設定します。
 * @type struct<stockArmReq>[]
 * @default []
 * 
 */
/*~struct~selcOpt:ja
 * 
 * @param Name
 * @desc オプション名
 * @type text
 * @default Option
 * 
 * @param Alternative Name
 * @desc 表示用のオプションの別名
 * @type text
 * @default Alternative Name
 * 
 * @param Display Requirements
 * @desc メニューにオプションを表示するための条件
 * @type struct<optReq>
 * 
 * @param Select Requirements
 * @desc メニューにオプションを表示するための条件
 * @type struct<optReq>
 * 
 * @param Description
 * @desc セレクションの説明方法
 * @type note
 * @default ""
 * 
 * @param Pictures
 * @desc セレクションを表す画像
 * @type file[]
 * @dir img/pictures/
 * @default []
 * 
 * @param Static Graphic
 * @desc 画像は背景の上にレイヤーされ、バックグラフィックのすぐ下にある。
 * @type struct<staticGfx>
 * 
 * @param Animated Graphic
 * @desc 選択を表すアニメーショングラフィック
 * @type struct<animGfx>
 * 
 * @param Scene Button
 * @descオプションにリンクするボタンをシーンに追加する
 * @type struct<menuButton>
 * 
 * @param Event Execution
 * @desc オプションを選択すると実行されるイベント
 * Takes priority over code execution.
 * @type common_event
 * @default 0
 * 
 * @param Code Execution
 * @desc オプション選択時に実行されるコード
 * @type note
 * @default ""
 * 
 */
/*~struct~selcWindow:ja
 * 
 * @param Dimension Configuration
 * @desc ウィンドウの位置と幅の設定
 * @type struct<locSize>
 * @default {"X":"0","Y":"0","Width":"1","Height":"1"}
 * 
 * @param Window Font and Style Configuration
 * @desc ウィンドウのカスタムスタイル
 * @type struct<windowStyle>
 * @default {"Font Settings":"","Font Size":"16","Font Face":"sans-serif","Base Font Color":"#ffffff","Font Outline Color":"rgba(0, 0, 0, 0.5)","Font Outline Thickness":"3","Window Skin":"Window","Window Opacity":"255","Show Window Dimmer":"false"}
 * 
 * @param Item Width
 * @desc 選択領域の幅
 * @type text
 * @default 1
 * 
 * @param Item Height
 * @desc 選択領域の高さ
 * @type text
 * @default 1
 * 
 * @param Max Columns
 * @desc 横列の数
 * @type text
 * @default 1
 * 
 * @param Selection Options
 * @desc ウィンドウのオプションのリスト
 * @type struct<selcOpt>[]
 * @default []
 * 
 * @param Draw Gauges
 * @descウィンドウのゲージを設定する
 * @type struct<gaugeDraw>[]
 * @default []
 * 
 * @param Draw Name
 * @desc ドローオプション名は？
 * @type boolean
 * @default false
 * 
 * @param Name Text
 * @parent Draw Name
 * @desc 名前のテキスト。%1 = 名前。
 * @type text
 * @default %1
 * 
 * @param Name X
 * @parent Draw Name
 * @desc テキストのX位置
 * @type text
 * @default 0
 * 
 * @param Name Y
 * @parent Draw Name
 * @desc テキストのY位置
 * @type text
 * @default 0
 * 
 * @param Draw Alternative Name
 * @desc ドローオプションの別名は？
 * @type boolean
 * @default false
 * 
 * @param Alternative Name Text
 * @parent Draw Alternative Name
 * @desc 代替名のテキスト。%1 = 代替名。
 * @type text
 * @default %1
 * 
 * @param Alternative Name X
 * @parent Draw Alternative Name
 * @desc テキストのX位置
 * @type text
 * @default 0
 * 
 * @param Alternative Name Y
 * @parent Draw Alternative Name
 * @desc テキストのY位置
 * @type text
 * @default 0
 * 
 * @param Draw Description
 * @desc ドローオプションの説明
 * @type boolean
 * @default false
 * 
 * @param Description X
 * @parent Draw Description
 * @desc テキストのX位置
 * @type text
 * @default 0
 * 
 * @param Description Y
 * @parent Draw Description
 * @desc テキストのY位置
 * @type text
 * @default 0
 * 
 * @param Draw Picture
 * @desc オプション画像を描く？
 * @type boolean
 * @default false
 * 
 * @param Picture Index
 * @parent Draw Picture
 * @desc 描画する画像のインデックス（0から始まる）
 * @type text
 * @default $gameVariables.value(1)
 * 
 * @param Picture X
 * @parent Draw Picture
 * @desc 画像のX位置
 * @type text
 * @default 0
 * 
 * @param Picture Y
 * @parent Draw Picture
 * @desc 画像のY位置
 * @type text
 * @default 0
 * 
 * @param Picture Width
 * @parent Draw Picture
 * @desc 画像の幅
 * @type text
 * @default 0
 * 
 * @param Picture Height
 * @parent Draw Picture
 * @desc 画像の高さ
 * @type text
 * @default 0
 * 
 */
/*~struct~selcDataWindow:ja
 * 
 * @param Name
 * @desc 機能なし
 * @type text
 * @default Data Window
 * 
 * @param Dimension Configuration
 * @desc ウィンドウの位置と幅の設定
 * @type struct<locSize>
 * @default {"X":"0","Y":"0","Width":"1","Height":"1"}
 * 
 * @param Window Font and Style Configuration
 * @desc ウィンドウのカスタムスタイル
 * @type struct<windowStyle>
 * @default {"Font Settings":"","Font Size":"16","Font Face":"sans-serif","Base Font Color":"#ffffff","Font Outline Color":"rgba(0, 0, 0, 0.5)","Font Outline Thickness":"3","Window Skin":"Window","Window Opacity":"255","Show Window Dimmer":"false"}
 * 
 * @param Draw Gauges
 * @desc ウィンドウのゲージを設定する
 * @type struct<gaugeDraw>[]
 * @default []
 * 
 * @param Draw Option Name
 * @desc オプション名は？
 * @type boolean
 * @default false
 * 
 * @param Name X
 * @parent Draw Option Name
 * @desc 名前のX位置
 * @type text
 * @default 0
 * 
 * @param Name Y
 * @parent Draw Option Name
 * @desc 名前のY位置
 * @type text
 * @default 0
 * 
 * @param Draw Alternative Option Name
 * @desc オプションの代替名は？
 * @type boolean
 * @default false
 * 
 * @param Alternative Name X
 * @parent Draw Alternative Option Name
 * @desc 代替名のX位置
 * @type text
 * @default 0
 * 
 * @param Alternative Name Y
 * @parent Draw Alternative Option Name
 * @desc 代替名のY位置
 * @type text
 * @default 0
 * 
 * @param Draw Description
 * @desc オプションの説明は？
 * @type boolean
 * @default false
 * 
 * @param Description X
 * @parent Draw Description
 * @desc 説明のX位置
 * @type text
 * @default 0
 * 
 * @param Description Y
 * @parent Draw Description
 * @desc 説明のY位置
 * @type text
 * @default 0
 * 
 * @param Draw Picture
 * @desc オプションの説明を描く？
 * @type boolean
 * @default false
 * 
 * @param Picture Index
 * @parent Draw Picture
 * @desc 使用画像のインデックス
 * @type text
 * @default 1
 * 
 * @param Picture X
 * @parent Draw Picture
 * @desc 画像のX位置
 * @type text
 * @default 0
 * 
 * @param Picture Y
 * @parent Draw Picture
 * @desc 画像のY位置
 * @type text
 * @default 0
 * 
 * @param Picture Width
 * @parent Draw Picture
 * @desc 画像の幅
 * @type text
 * @default 0
 * 
 * @param Picture Height
 * @parent Draw Picture
 * @desc 画像の高さ
 * @type text
 * @default 0
 * 
 */
/*~struct~menuButton:ja
 * 
 * @param Name
 * @desc 機能なし
 * @type text
 * @default Button
 * 
 * @param X
 * @desc 画面上のX位置
 * @type text
 * @default 0
 * 
 * @param Y
 * @desc 画面上のY位置
 * @type text
 * @default 0
 * 
 * @param Cold Graphic
 * @desc マウスオーバーしていない時のグラフィック
 * @type struct<animGfx>
 * 
 * @param Hot Graphic
 * @desc マウスオーバー時のグラフィック
 * @type struct<animGfx>
 * 
 */
/*~struct~custMenu:ja
 *
 * @param Identifier Name
 * @desc メニューデータを識別するための名前。
 * @type text
 * @default menu
 * 
 * @param Preload Backgrounds
 * @desc シーンのプリロードの背景に使用される画像。
 * @type struct<gfx>[]
 * @default []
 * 
 * @param Backgrounds
 * @desc シーンの背景に使用される画像。
 * @type struct<staticGfx>[]
 * @default []
 * 
 * @param Back Graphics
 * @desc シーンの背景のすぐ上にレイヤーされたアニメーション画像。
 * @type struct<animGfx>[]
 * @default []
 * 
 * @param Selection Window
 * @desc 選択オプションに使用されるウィンドウ
 * @type struct<selcWindow>
 * 
 * @param Selection Data Windows
 * @parent Selection Window
 * @desc ウィンドウの選択に基づいてデータを表示する
 * @type struct<selcDataWindow>[]
 * @default []
 * 
 * @param Basic Windows
 * @desc 基本データを表示するウィンドウ。(常時表示）
 * @type struct<basicWin>[]
 * @default []
 * 
 * @param Fore Graphics
 * @desc シーンの前景のすぐ下にレイヤーされたアニメーション画像
 * @type struct<animGfx>[]
 * @default []
 * 
 * @param Foregrounds
 * @desc シーンの前景に使用される画像。
 * @type struct<staticGfx>[]
 * @default []
 * 
 */

const Syn_MenuBuildr = {};
Syn_MenuBuildr.Plugin = PluginManager.parameters(`Synrec_MenuBuilder`);

function WINDOW_TEXT_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        try{
            obj['Text'] = JSON.parse(obj['Text']);
        }catch(e){
            obj['Text'] = "";
        }
        return obj
    }catch(e){
        return;
    }
}

function GRAPHIC_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        return obj
    }catch(e){
        return;
    }
}

function STATIC_GRAPHIC_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        return obj
    }catch(e){
        return;
    }
}

function ANIMATED_GRAPHIC_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        return obj
    }catch(e){
        return;
    }
}

function DIMENSION_CONFIGURATION_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        obj['X'] = eval(obj['X']);
        obj['Y'] = eval(obj['Y']);
        obj['Width'] = eval(obj['Width']);
        obj['Height'] = eval(obj['Height']);
    }catch(e){
        console.warn(`Failed to parse dimension configuration. ${e}`);
        const obj = {};
        obj['X'] = 0;
        obj['Y'] = 0;
        obj['Width'] = 1;
        obj['Height'] = 1;
    }
    return obj;
}

function WINDOW_STYLE_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj)
        obj['Font Size'] = eval(obj['Font Size']);
        obj['Font Outline Thickness'] = eval(obj['Font Outline Thickness']);
        obj['Window Opacity'] = eval(obj['Window Opacity']);
        obj['Show Window Dimmer'] = eval(obj['Show Window Dimmer']);
    }catch(e){
        console.warn(`Failed to parse window style. ${e}`);
        const obj = {};
        obj['Font Size'] = 16;
        obj['Font Face'] = 'sans-serif';
        obj['Base Font Color'] = '#ffffff';
        obj['Font Outline Color'] = 'rgba(0, 0, 0, 0.5)';
        obj['Font Outline Thickness'] = 3;
        obj['Window Skin'] = 'Window';
        obj['Window Opacity'] = 255;
        obj['Show Window Dimmer'] = false;
    }
    return obj;
}

function GAUGE_DRAW_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        return obj;
    }catch(e){
        return;
    }
}

function BASIC_WINDOW_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        obj['Dimension Configuration'] = DIMENSION_CONFIGURATION_PARSER_MNUBLD(obj['Dimension Configuration']);
        obj['Window Font and Style Configuration'] = WINDOW_STYLE_PARSER_MNUBLD(obj['Window Font and Style Configuration']);
        try{
            const texts = JSON.parse(obj['Draw Texts']).map((text_config)=>{
                return WINDOW_TEXT_PARSER_MNUBLD(text_config);
            }).filter(Boolean);
            obj['Draw Texts'] = texts;
        }catch(e){
            obj['Draw Texts'] = [];
        }
        try{
            const codes = JSON.parse(obj['Text References'])
            obj['Text References'] = codes;
        }catch(e){
            obj['Text References'] = [];
        }
        try{
            const pictures = JSON.parse(obj['Draw Pictures']).map((pic_config)=>{
                return GRAPHIC_PARSER_MNUBLD(pic_config);
            }).filter(Boolean);
            obj['Draw Pictures'] = pictures;
        }catch(e){
            obj['Draw Pictures'] = [];
        }
        try{
            obj['Draw Gauges'] = JSON.parse(obj['Draw Gauges']).map((gauge_draw_config)=>{
                return GAUGE_DRAW_PARSER_MNUBLD(gauge_draw_config);
            }).filter(Boolean)
        }catch(e){
            obj['Draw Gauges'] = [];
        }
        return obj;
    }catch(e){
        return;
    }
}

function VAR_REQ_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        return obj;
    }catch(e){
        return;
    }
}

function ITM_REQ_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        return obj;
    }catch(e){
        return;
    }
}

function WEP_REQ_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        return obj;
    }catch(e){
        return;
    }
}

function ARM_REQ_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        return obj;
    }catch(e){
        return;
    }
}

function SELC_OPT_REQ_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        try{
            obj['Variable Requirements'] = JSON.parse(obj['Variable Requirements']).map((config)=>{
                return VAR_REQ_PARSER_MNUBLD(config);
            }).filter(Boolean)
        }catch(e){
            obj['Variable Requirements'] = [];
        }
        try{
            obj['Switch Requirements'] = JSON.parse(obj['Switch Requirements']);
        }catch(e){
            obj['Switch Requirements'] = [];
        }
        try{
            obj['Item Requirements'] = JSON.parse(obj['Item Requirements']).map((config)=>{
                return ITM_REQ_PARSER_MNUBLD(config);
            }).filter(Boolean)
        }catch(e){
            obj['Item Requirements'] = [];
        }
        try{
            obj['Weapon Requirements'] = JSON.parse(obj['Weapon Requirements']).map((config)=>{
                return WEP_REQ_PARSER_MNUBLD(config);
            }).filter(Boolean)
        }catch(e){
            obj['Weapon Requirements'] = [];
        }
        try{
            obj['Armor Requirements'] = JSON.parse(obj['Armor Requirements']).map((config)=>{
                return ARM_REQ_PARSER_MNUBLD(config);
            }).filter(Boolean)
        }catch(e){
            obj['Armor Requirements'] = [];
        }
        return obj;
    }catch(e){
        return;
    }
}

function SELC_OPT_BUTTON_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        obj['Cold Graphic'] = ANIMATED_GRAPHIC_PARSER_MNUBLD(obj['Cold Graphic']);
        obj['Hot Graphic'] = ANIMATED_GRAPHIC_PARSER_MNUBLD(obj['Hot Graphic']);
        return obj;
    }catch(e){
        return;
    }
}

function SELC_OPT_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        obj['Display Requirements'] = SELC_OPT_REQ_PARSER_MNUBLD(obj['Display Requirements']);
        obj['Select Requirements'] = SELC_OPT_REQ_PARSER_MNUBLD(obj['Select Requirements']);
        obj['Static Graphic'] = STATIC_GRAPHIC_PARSER_MNUBLD(obj['Static Graphic']);
        obj['Animated Graphic'] = ANIMATED_GRAPHIC_PARSER_MNUBLD(obj['Animated Graphic']);
        obj['Scene Button'] = SELC_OPT_BUTTON_PARSER_MNUBLD(obj['Scene Button']);
        try{
            obj['Description'] = JSON.parse(obj['Description']);
        }catch(e){
            obj['Description'] = "";
        }
        try{
            obj['Code Execution'] = JSON.parse(obj['Code Execution']);
        }catch(e){
            obj['Code Execution'] = "";
        }
        return obj;
    }catch(e){
        return;
    }
}

function SELC_WINDOW_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        obj['Dimension Configuration'] = DIMENSION_CONFIGURATION_PARSER_MNUBLD(obj['Dimension Configuration']);
        obj['Window Font and Style Configuration'] = WINDOW_STYLE_PARSER_MNUBLD(obj['Window Font and Style Configuration']);
        try{
            obj['Selection Options'] = JSON.parse(obj['Selection Options']).map((opt_config)=>{
                return SELC_OPT_PARSER_MNUBLD(opt_config);
            }).filter(Boolean)
        }catch(e){
            obj['Selection Options'] = [];
        }
        try{
            obj['Draw Gauges'] = JSON.parse(obj['Draw Gauges']).map((gauge_draw_config)=>{
                return GAUGE_DRAW_PARSER_MNUBLD(gauge_draw_config);
            }).filter(Boolean)
        }catch(e){
            obj['Draw Gauges'] = [];
        }
        return obj;
    }catch(e){
        return console.error(e);
    }
}

function SELC_DATA_WINDOW_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        obj['Dimension Configuration'] = DIMENSION_CONFIGURATION_PARSER_MNUBLD(obj['Dimension Configuration']);
        obj['Window Font and Style Configuration'] = WINDOW_STYLE_PARSER_MNUBLD(obj['Window Font and Style Configuration']);
        try{
            obj['Draw Gauges'] = JSON.parse(obj['Draw Gauges']).map((gauge_draw_config)=>{
                return GAUGE_DRAW_PARSER_MNUBLD(gauge_draw_config);
            }).filter(Boolean)
        }catch(e){
            obj['Draw Gauges'] = [];
        }
        return obj;
    }catch(e){
        return;
    }
}

function MENU_SCENE_BUILD_PARSER_MNUBLD(obj){
    try{
        obj = JSON.parse(obj);
        if(!obj['Identifier Name']){
            console.error(`No identifier name detected. Parse aborted.`);
            return;
        }
        try{
            obj['Preload Backgrounds'] = JSON.parse(obj['Preload Backgrounds']);
        }catch(e){
            obj['Preload Backgrounds'] = [];
        }
        try{
            obj['Backgrounds'] = JSON.parse(obj['Backgrounds']).map((bg_config)=>{
                return STATIC_GRAPHIC_PARSER_MNUBLD(bg_config);
            }).filter(Boolean);
        }catch(e){
            obj['Backgrounds'] = [];
        }
        try{
            obj['Back Graphics'] = JSON.parse(obj['Back Graphics']).map((bg_config)=>{
                return ANIMATED_GRAPHIC_PARSER_MNUBLD(bg_config);
            }).filter(Boolean);
        }catch(e){
            obj['Back Graphics'] = [];
        }
        obj['Selection Window'] = SELC_WINDOW_PARSER_MNUBLD(obj['Selection Window']);
        try{
            const data_windows = JSON.parse(obj['Selection Data Windows']).map((config)=>{
                return SELC_DATA_WINDOW_PARSER_MNUBLD(config);
            }).filter(Boolean);
            obj['Selection Data Windows'] = data_windows;
        }catch(e){
            obj['Selection Data Windows'] = [];
        }
        try{
            const basic_windows = JSON.parse(obj['Basic Windows']).map((window_config)=>{
                return BASIC_WINDOW_PARSER_MNUBLD(window_config);
            }).filter(Boolean)
            obj['Basic Windows'] = basic_windows;
        }catch(e){
            obj['Basic Windows'] = [];
        }
        try{
            obj['Foregrounds'] = JSON.parse(obj['Foregrounds']).map((fg_config)=>{
                return STATIC_GRAPHIC_PARSER_MNUBLD(fg_config);
            }).filter(Boolean);
        }catch(e){
            obj['Foregrounds'] = [];
        }
        try{
            obj['Fore Graphics'] = JSON.parse(obj['Fore Graphics']).map((fg_config)=>{
                return ANIMATED_GRAPHIC_PARSER_MNUBLD(fg_config);
            }).filter(Boolean);
        }catch(e){
            obj['Fore Graphics'] = [];
        }
        return obj;
    }catch(e){
        return;
    }
}

try{
    const menus = JSON.parse(Syn_MenuBuildr.Plugin['Menu Configurations']).map((config)=>{
        return MENU_SCENE_BUILD_PARSER_MNUBLD(config);
    }).filter(Boolean);
    Syn_MenuBuildr.MENU_CONFIGURATIONS = menus;
}catch(e){
    Syn_MenuBuildr.MENU_CONFIGURATIONS = [];
}

SceneManager.snapForSynMenuBackground = function() {
    if (this._synMenuBackgroundBitmap) {
        this._synMenuBackgroundBitmap.destroy();
    }
    this._synMenuBackgroundBitmap = this.snap();
}

SceneManager.synMenuBackgroundBitmap = function() {
    return this._synMenuBackgroundBitmap;
}

Game_Temp.prototype.openedMenu = function(){
    return this._current_menu;
}

Game_Temp.prototype.openMenu = function(menu_name){
    if(!Array.isArray(this._last_menu)){
        this._last_menu = [];
    }
    const new_menu = Syn_MenuBuildr.MENU_CONFIGURATIONS.find((menu_config)=>{
        return menu_config['Identifier Name'] == menu_name;
    })
    if(!new_menu)return;
    const current_menu = this.openedMenu();
    this._last_menu.push(current_menu);
    if(!current_menu){
        SceneManager.snapForSynMenuBackground();
    }
    this._current_menu = JsonEx.makeDeepCopy(new_menu);
    SceneManager.push(Scene_SynrecMenuPreload);
}

Game_Temp.prototype.closeMenu = function(){
    if(!Array.isArray(this._last_menu)){
        this._last_menu = [];
    }
    const old_menu = this._last_menu.pop();
    if(old_menu){
        this._current_menu = old_menu;
    }else{
        this._current_menu = null;
    }
}

Syn_MenuBuildr_GmIntrpr_SetWaitMode = Game_Interpreter.prototype.setWaitMode
Game_Interpreter.prototype.setWaitMode = function(waitMode) {
    Syn_MenuBuildr_GmIntrpr_SetWaitMode.call(this, ...arguments);
    if(waitMode == 'transfer'){
        const scene = SceneManager._scene;
        if(scene instanceof Scene_SynrecMenu){
            scene.closeMenu();
        }
    }
}

function Sprite_SynMenuStaticGfx(){
    this.initialize(...arguments);
}

Sprite_SynMenuStaticGfx.prototype = Object.create(TilingSprite.prototype);
Sprite_SynMenuStaticGfx.prototype.constructor = Sprite_SynMenuStaticGfx;

Sprite_SynMenuStaticGfx.prototype.initialize = function(data){
    TilingSprite.prototype.initialize.call(this);
    this._gfx_data = data;
    this.setupGfx();
}

Sprite_SynMenuStaticGfx.prototype.setupGfx = function(gfx_data){
    const gfx_config = gfx_data || this._gfx_data;
    if(!gfx_config)return;
    const file_name = gfx_config['File'];
    if(!file_name)return;
    const bitmap = ImageManager.loadPicture(file_name);
    this._scroll_x = eval(gfx_config['Scrolling X']) || 0;
    this._scroll_y = eval(gfx_config['Scrolling Y']) || 0;
    this._rotation = eval(gfx_config['Rotation']) || 0;
    this._constant_rot = !!eval(gfx_config['Constant Rotation']);
    this.bitmap = bitmap;
    const mx = eval(gfx_config['X']) || 0;
    const my = eval(gfx_config['Y']) || 0;
    const mw = bitmap.width;
    const mh = bitmap.height;
    this.move(mx,my,mw,mh);
    this.rotation = this._rotation;
}

Sprite_SynMenuStaticGfx.prototype.update = function(){
    TilingSprite.prototype.update.call(this);
    this.updateScrolling();
    this.updateRotation();
}

Sprite_SynMenuStaticGfx.prototype.updateScrolling = function(){
    this.origin.x += this._scroll_x || 0;
    this.origin.y += this._scroll_y || 0;
}

Sprite_SynMenuStaticGfx.prototype.updateRotation = function(){
    if(this._constant_rot){
        this.rotation += this._rotation || 0;
    }
}

function Sprite_SynMenuAnimGfx(){
    this.initialize(...arguments);
}

Sprite_SynMenuAnimGfx.prototype = Object.create(Sprite.prototype);
Sprite_SynMenuAnimGfx.prototype.constructor = Sprite_SynMenuAnimGfx;

Sprite_SynMenuAnimGfx.prototype.initialize = function(data){
    Sprite.prototype.initialize.call(this);
    this._gfx_data = data;
    this.setupGfx();
}

Sprite_SynMenuAnimGfx.prototype.setupGfx = function(gfx_data){
    const gfx_config = gfx_data || this._gfx_data;
    if(!gfx_config)return;
    const file_name = gfx_config['File'];
    if(!file_name)return;
    const bitmap = ImageManager.loadPicture(file_name);
    this._cur_frame = 0;
    this._max_frames = eval(gfx_config['Max Frames']);
    this._frame_rate = eval(gfx_config['Frame Rate']);
    this._frame_time = eval(gfx_config['Frame Rate']);
    this.bitmap = bitmap;
    const mx = eval(gfx_config['X']);
    const my = eval(gfx_config['Y']);
    this.move(mx,my);
    this.updateFrames();
}

Sprite_SynMenuAnimGfx.prototype.update = function(){
    Sprite.prototype.update.call(this);
    this.updateFrames();
}

Sprite_SynMenuAnimGfx.prototype.updateFrames = function(){
    const bitmap = this.bitmap;
    if(!bitmap)return;
    if(isNaN(this._frame_time) || this._frame_time >= this._frame_rate){
        this._frame_time = 0;
        const frames = this._max_frames;
        const w = bitmap.width / frames;
        const h = bitmap.height;
        const x = w * this._cur_frame;
        const y = 0;
        this.setFrame(x,y,w,h);
        this._cur_frame++;
        if(this._cur_frame >= frames){
            this._cur_frame = 0;
        }
    }else{
        this._frame_time++;
    }
}

function Sprite_SynMenuButton(){
    this.initialize(...arguments);
}

Sprite_SynMenuButton.prototype = Object.create(Sprite.prototype);
Sprite_SynMenuButton.prototype.constructor = Sprite_SynMenuButton;

Sprite_SynMenuButton.prototype.initialize = function(data, selc_window, index){
    Sprite.prototype.initialize.call(this);
    this._button_data = data;
    this._select_window = selc_window;
    this._selc_index = index;
    this.createColdGraphic();
    this.createHotGraphic();
    this.setPosition();
}

Sprite_SynMenuButton.prototype.createColdGraphic = function(){
    const sprite = new Sprite_SynMenuAnimGfx();
    const btn_data = this._button_data;
    if(btn_data){
        const cold_graphic_data = btn_data['Cold Graphic'];
        sprite.setupGfx(cold_graphic_data);
    }
    this.addChild(sprite);
    this._cold_gfx = sprite;
}

Sprite_SynMenuButton.prototype.createHotGraphic = function(){
    const sprite = new Sprite_SynMenuAnimGfx();
    sprite.visible = false;
    const btn_data = this._button_data;
    if(btn_data){
        const hot_graphic_data = btn_data['Hot Graphic'];
        sprite.setupGfx(hot_graphic_data);
    }
    this.addChild(sprite);
    this._hot_gfx = sprite;
}

Sprite_SynMenuButton.prototype.setPosition = function(){
    const btn_data = this._button_data;
    if(btn_data){
        const x = eval(btn_data['X']);
        const y = eval(btn_data['Y']);
        this.move(x,y);
    }
}

Sprite_SynMenuButton.prototype.update = function(){
    Sprite.prototype.update.call(this);
    this.updateSelected();
    this.updateGraphic();
    this.updateOnClick();
}

Sprite_SynMenuButton.prototype.updateSelected = function(){
    const selc_window = this._select_window;
    if(!selc_window)return;
    const selc_index = this._selc_index;
    const tx = TouchInput.x;
    const ty = TouchInput.y;
    const gfx = this._cold_gfx;
    const x = this.x;
    const y = this.y;
    const w = gfx.width;
    const h = gfx.height;
    if(
        tx >= x &&
        tx <= x + w &&
        ty >= y &&
        ty <= y + h
    ){
        this._hover = true;
        if(selc_window.index() != selc_index){
            selc_window.select(selc_index);
            SoundManager.playCursor();
        }
    }else{
        this._hover = false;
    }
    const win_indx = selc_window.index();
    this._selected = win_indx == selc_index;
}

Sprite_SynMenuButton.prototype.updateGraphic = function(){
    if(this._selected){
        this._cold_gfx.visible = false;
        this._hot_gfx.visible = true;
    }else{
        this._cold_gfx.visible = true;
        this._hot_gfx.visible = false;
    }
}

Sprite_SynMenuButton.prototype.updateOnClick = function(){
    if(this._hover && TouchInput.isTriggered()){
        const scene = SceneManager._scene;
        SoundManager.playOk();
        scene.doSelectAction();
    }
}

function Window_SynMenuBasic(){
    this.initialize(...arguments);
}

Window_SynMenuBasic.prototype = Object.create(Window_Base.prototype);
Window_SynMenuBasic.prototype.constructor = Window_SynMenuBasic;

Window_SynMenuBasic.prototype.initialize = function(data){
    const mz_mode = Utils.RPGMAKER_NAME == "MZ";
    const rect = this.createRect(data);
    this._window_data = data;
    this._style_data = data['Window Font and Style Configuration'];
    if(mz_mode){
        Window_Base.prototype.initialize.call(this, rect);
    }else{
        const x = rect.x;
        const y = rect.y;
        const w = rect.width;
        const h = rect.height;
        Window_Base.prototype.initialize.call(this,x,y,w,h);
    }
    this.setOpacityAndDimmer();
    this.drawData();
}

Window_SynMenuBasic.prototype.createRect = function(data){
    const dimension_config = data['Dimension Configuration'];
    const x = dimension_config['X'];
    const y = dimension_config['Y'];
    const w = dimension_config['Width'];
    const h = dimension_config['Height'];
    return new Rectangle(x,y,w,h);
}

Window_SynMenuBasic.prototype.standardPadding = function() {
    return 8;
}

Window_SynMenuBasic.prototype.loadWindowskin = function(){
    const base = Window_Base.prototype.loadWindowskin.call(this);
    const custom_config = this._style_data;
    if(!custom_config)return base;
    const skin_name = custom_config['Window Skin'];
    if(!skin_name)return base;
    this.windowskin = ImageManager.loadSystem(skin_name);
}

Window_SynMenuBasic.prototype.resetFontSettings = function() {
    const base = Window_Base.prototype.resetFontSettings;
    const custom_config = this._style_data;
    if(!custom_config)return base.call(this);
    const font_face = custom_config['Font Face'] || "sans-serif";
    const font_size = custom_config['Font Size'] || 16;
    const font_outline_size = custom_config['Font Outline Thickness'] || 3;
    this.contents.fontFace = font_face;
    this.contents.fontSize = font_size;
    this.contents.outlineWidth = font_outline_size;
    this.resetTextColor();
}

Window_SynMenuBasic.prototype.resetTextColor = function() {
    const base = Window_Base.prototype.resetTextColor;
    const custom_config = this._style_data;
    if(!custom_config)return base.call(this);
    const text_color = custom_config['Base Font Color'] || "#ffffff";
    const outline_color = custom_config['Font Outline Color'] || "rgba(0, 0, 0, 0.5)";
    this.changeTextColor(text_color);
    this.contents.outlineColor = outline_color;
}

Window_SynMenuBasic.prototype.setOpacityAndDimmer = function(){
    const custom_config = this._style_data;
    if(!custom_config)return;
    const show_dimmer = custom_config['Show Window Dimmer'] || false;
    const win_opacity = custom_config['Window Opacity'] || 0;
    this.opacity = win_opacity;
    show_dimmer ? this.showBackgroundDimmer() : this.hideBackgroundDimmer();
}

Window_SynMenuBasic.prototype.drawData = function(){
    this.drawPictures();
    this.drawGauges();
    this.drawTexts();
}

Window_SynMenuBasic.prototype.drawPictures = function(){
    const data = this._window_data;
    const pictures = data['Draw Pictures'];
    if(Array.isArray(pictures)){
        const win = this;
        pictures.forEach((config)=>{
            const image_name = config['Picture'];
            if(image_name){
                const bitmap = ImageManager.loadPicture(image_name);
                const dx = eval(config['X']);
                const dy = eval(config['Y']);
                const dw = eval(config['Width']);
                const dh = eval(config['Height']);
                const bx = 0;
                const by = 0;
                const bw = bitmap.width || dw;
                const bh = bitmap.height || dh;
                win.contents.blt(bitmap, bx, by, bw, bh, dx, dy, dw, dh);
            }
        })
    }
}

Window_SynMenuBasic.prototype.drawGauges = function(){
    const window = this;
    const data = this._window_data;
    console.log(data)
    const gauge_draw_configs = data['Draw Gauges'];
    gauge_draw_configs.forEach((config)=>{
        const label = config['Label'];
        const lx = eval(config['Label X']);
        const ly = eval(config['Label Y']);
        window.drawTextEx(label, lx, ly);
        const cur_val = eval(config['Gauge Current Value']) || 0;
        const max_val = eval(config['Gauge Max Value']) || 1;
        const ratio = Math.max(0, Math.min(1, cur_val / max_val));
        const gx = eval(config['Gauge X']);
        const gy = eval(config['Gauge Y']);
        const gw = eval(config['Gauge Width']);
        const gh = eval(config['Gauge Height']);
        const gb = eval(config['Gauge Border']);
        const border_color = config['Gauge Border Color'];
        const background_color = config['Gauge Background Color'];
        const fill_color = config['Gauge Color'];
        window.contents.fillRect(gx,gy,gw,gh,border_color);
        window.contents.fillRect(gx + gb, gy + gb, gw - (gb * 2), gh - (gb * 2), background_color);
        window.contents.fillRect(gx + gb, gy + gb, (gw - (gb * 2)) * ratio, gh - (gb * 2), fill_color);
    })
}

Window_SynMenuBasic.prototype.drawTexts = function(){
    const data = this._window_data;
    const references = (data['Text References'] || []).map((code)=>{
        try{
            code = eval(code);
            return code;
        }catch(e){
            return '';
        }
    })
    const texts = data['Draw Texts'];
    if(Array.isArray(texts)){
        const win = this;
        texts.forEach((config)=>{
            const text = (config['Text'] || "").format(...references);
            const tx = eval(config['X']);
            const ty = eval(config['Y']);
            win.drawTextEx(text, tx, ty);
        })
    }
}

function Window_SynMenuSelc(){
    this.initialize(...arguments);
}

Window_SynMenuSelc.prototype = Object.create(Window_Selectable.prototype);
Window_SynMenuSelc.prototype.constructor = Window_SynMenuSelc;

Window_SynMenuSelc.prototype.initialize = function(data){
    this._list = [];
    const mz_mode = Utils.RPGMAKER_NAME == "MZ";
    const rect = this.createRect(data);
    this._window_data = data;
    this._style_data = data['Window Font and Style Configuration'];
    if(mz_mode){
        Window_Selectable.prototype.initialize.call(this, rect);
    }else{
        const x = rect.x;
        const y = rect.y;
        const w = rect.width;
        const h = rect.height;
        Window_Selectable.prototype.initialize.call(this,x,y,w,h);
    }
    this.setOpacityAndDimmer();
    this.generateList();
    this.refresh();
    this.select(0);
    this.activate();
}

Window_SynMenuSelc.prototype.generateList = function(){
    const scene = SceneManager._scene;
    scene.clearButtons();
    const win = this;
    const window_data = this._window_data;
    const selc_options = (window_data['Selection Options'] || []).filter((option)=>{
        const display_req = option['Display Requirements'];
        if(win.checkRequirements(display_req)){
            return true;
        }
    });
    let index = 0;
    selc_options.forEach((option)=>{
        const button_data = option['Scene Button'];
        if(button_data){
            const button = new Sprite_SynMenuButton(button_data, win, index);
            scene.addChild(button);
            scene._scene_buttons.push(button);
        }
        index++;
    })
    this._list = selc_options;
}

Window_SynMenuSelc.prototype.createRect = function(data){
    const dimension_config = data['Dimension Configuration'];
    const x = dimension_config['X'];
    const y = dimension_config['Y'];
    const w = dimension_config['Width'];
    const h = dimension_config['Height'];
    return new Rectangle(x,y,w,h);
}

Window_SynMenuSelc.prototype.standardPadding = function() {
    return 8;
}

Window_SynMenuSelc.prototype.loadWindowskin = function(){
    const base = Window_Selectable.prototype.loadWindowskin.call(this);
    const custom_config = this._style_data;
    if(!custom_config)return base;
    const skin_name = custom_config['Window Skin'];
    if(!skin_name)return base;
    this.windowskin = ImageManager.loadSystem(skin_name);
}

Window_SynMenuSelc.prototype.resetFontSettings = function() {
    const base = Window_Selectable.prototype.resetFontSettings;
    const custom_config = this._style_data;
    if(!custom_config)return base.call(this);
    const font_face = custom_config['Font Face'] || "sans-serif";
    const font_size = custom_config['Font Size'] || 16;
    const font_outline_size = custom_config['Font Outline Thickness'] || 3;
    this.contents.fontFace = font_face;
    this.contents.fontSize = font_size;
    this.contents.outlineWidth = font_outline_size;
    this.resetTextColor();
}

Window_SynMenuSelc.prototype.resetTextColor = function() {
    const base = Window_Selectable.prototype.resetTextColor;
    const custom_config = this._style_data;
    if(!custom_config)return base.call(this);
    const text_color = custom_config['Base Font Color'] || "#ffffff";
    const outline_color = custom_config['Font Outline Color'] || "rgba(0, 0, 0, 0.5)";
    this.changeTextColor(text_color);
    this.contents.outlineColor = outline_color;
}

Window_SynMenuSelc.prototype.setOpacityAndDimmer = function(){
    const custom_config = this._style_data;
    if(!custom_config)return;
    const show_dimmer = custom_config['Show Window Dimmer'] || false;
    const win_opacity = custom_config['Window Opacity'] || 0;
    this.opacity = win_opacity;
    show_dimmer ? this.showBackgroundDimmer() : this.hideBackgroundDimmer();
}

Window_SynMenuSelc.prototype.maxItems = function(){
    return this._list.length;
}

Window_SynMenuSelc.prototype.maxCols = function(){
    const window_data = this._window_data;
    const cols = eval(window_data['Max Columns']) || 1;
    return Math.max(1, cols);
}

Window_SynMenuSelc.prototype.itemWidth = function(){
    const window_data = this._window_data;
    return eval(window_data['Item Width']) || 1;
}

Window_SynMenuSelc.prototype.itemHeight = function(){
    const window_data = this._window_data;
    return eval(window_data['Item Height']) || 1;
}

Window_SynMenuSelc.prototype.drawItem = function(i){
    const rect = this.itemRect(i);
    const data = this._list[i];
    if(data){
        this.drawPicture(rect, data);
        this.drawGauges(rect, data);
        this.drawName(rect, data);
        this.drawAltName(rect, data);
        this.drawDesc(rect, data);
    }
}

Window_SynMenuSelc.prototype.drawPicture = function(rect, data){
    const rx = rect.x;
    const ry = rect.y;
    const window_data = this._window_data;
    const draw_pic = eval(window_data['Draw Picture']);
    if(!draw_pic)return;
    const pic_names = JSON.parse(data['Pictures'] || []);
    const pic_index = eval(window_data['Picture Index']);
    const pic_name = pic_names[pic_index];
    if(!pic_name)return;
    const bitmap = ImageManager.loadPicture(pic_name);
    const bx = 0;
    const by = 0;
    const bw = bitmap.width;
    const bh = bitmap.height;
    const dx = rx + eval(window_data['Picture X']);
    const dy = ry + eval(window_data['Picture Y']);
    const dw = eval(window_data['Picture Width']);
    const dh = eval(window_data['Picture Height']);
    this.contents.blt(bitmap,bx,by,bw,bh,dx,dy,dw,dh);
}

Window_SynMenuSelc.prototype.drawGauges = function(rect, data){
    const rx = rect.x;
    const ry = rect.y;
    const window = this;
    const window_data = this._window_data;
    const gauge_draw_configs = window_data['Draw Gauges'];
    gauge_draw_configs.forEach((config)=>{
        const label = config['Label'];
        const lx = rx + eval(config['Label X']);
        const ly = ry + eval(config['Label Y']);
        window.drawTextEx(label, lx, ly);
        const cur_val = eval(config['Gauge Current Value']) || 0;
        const max_val = eval(config['Gauge Max Value']) || 1;
        const ratio = Math.max(0, Math.min(1, cur_val / max_val));
        const gx = rx + eval(config['Gauge X']);
        const gy = ry + eval(config['Gauge Y']);
        const gw = eval(config['Gauge Width']);
        const gh = eval(config['Gauge Height']);
        const gb = eval(config['Gauge Border']);
        const border_color = config['Gauge Border Color'];
        const background_color = config['Gauge Background Color'];
        const fill_color = config['Gauge Color'];
        window.contents.fillRect(gx,gy,gw,gh,border_color);
        window.contents.fillRect(gx + gb, gy + gb, gw - (gb * 2), gh - (gb * 2), background_color);
        window.contents.fillRect(gx + gb, gy + gb, (gw - (gb * 2)) * ratio, gh - (gb * 2), fill_color);
    })
}

Window_SynMenuSelc.prototype.drawName = function(rect, data){
    const rx = rect.x;
    const ry = rect.y;
    const window_data = this._window_data;
    const draw_name = eval(window_data['Draw Name']);
    if(!draw_name)return;
    const name = data['Name'];
    const tx = rx + eval(window_data['Name X']);
    const ty = ry + eval(window_data['Name Y']);
    const text = (window_data['Name Text'] || "").format(name);
    this.drawTextEx(text,tx,ty);
}

Window_SynMenuSelc.prototype.drawAltName = function(rect, data){
    const rx = rect.x;
    const ry = rect.y;
    const window_data = this._window_data;
    const draw_alt_name = eval(window_data['Draw Alternative Name']);
    if(!draw_alt_name)return;
    const alt_name = data['Alternative Name'];
    const tx = rx + eval(window_data['Alternative Name X']);
    const ty = ry + eval(window_data['Alternative Name Y']);
    const text = (window_data['Alternative Name Text'] || "").format(alt_name);
    this.drawTextEx(text,tx,ty);
}

Window_SynMenuSelc.prototype.drawDesc = function(rect, data){
    const rx = rect.x;
    const ry = rect.y;
    const window_data = this._window_data;
    const draw_desc = eval(window_data['Draw Description']);
    if(!draw_desc)return;
    const desc = data['Description'];
    const tx = rx + eval(window_data['Description X']);
    const ty = ry + eval(window_data['Description Y']);
    this.drawTextEx(desc,tx,ty);
}

Window_SynMenuSelc.prototype.currentSelection = function(){
    const index = this.index();
    return this._list[index];
}

Window_SynMenuSelc.prototype.meetSelectRequirements = function(){
    const option = this.currentSelection();
    if(!option)return false;
    const requirements = option['Select Requirements'];
    const valid_reqs = this.checkRequirements(requirements);
    return valid_reqs;
}

Window_SynMenuSelc.prototype.checkRequirements = function(requirements){
    if(!requirements)return true;
    const var_reqs = requirements['Variable Requirements'];
    if(Array.isArray(var_reqs)){
        if(var_reqs.some((req)=>{
            const id = eval(req['Variable']);
            const value = $gameVariables.value(id);
            const min = eval(req['Min Value']);
            const max = eval(req['Max Value']);
            if(
                value < min ||
                value > max
            ){
                return true;
            }
        })){
            return false;
        }
    }
    const sw_reqs = requirements['Switch Requirements'];
    if(Array.isArray(sw_reqs)){
        if(sw_reqs.some((sw_id)=>{
            const switch_value = $gameSwitches.value(sw_id);
            return !switch_value;
        })){
            return false;
        }
    }
    const itm_reqs = requirements['Item Requirements'];
    if(Array.isArray(itm_reqs)){
        if(itm_reqs.some((req)=>{
            const id = eval(req['Item']);
            const data = $dataWeapons[id];
            if(!data)throw new Error(`${id} is not a valid item ID.`);
            const min = eval(req['Min Value']);
            const max = eval(req['Max Value']);
            const num_bag = $gameParty.numItems(data);
            if(
                num_bag < min ||
                num_bag > max
            ){
                return true;
            }
        })){
            return false;
        }
    }
    const wep_reqs = requirements['Weapon Requirements'];
    if(Array.isArray(wep_reqs)){
        if(wep_reqs.some((req)=>{
            const id = eval(req['Weapon']);
            const data = $dataWeapons[id];
            if(!data)throw new Error(`${id} is not a valid weapon ID.`);
            const min = eval(req['Min Value']);
            const max = eval(req['Max Value']);
            const num_bag = $gameParty.numItems(data);
            if(
                num_bag < min ||
                num_bag > max
            ){
                return true;
            }
        })){
            return false;
        }
    }
    const arm_reqs = requirements['Armor Requirements'];
    if(Array.isArray(arm_reqs)){
        if(arm_reqs.some((req)=>{
            const id = eval(req['Armor']);
            const data = $dataArmors[id];
            if(!data)throw new Error(`${id} is not a valid armor ID.`);
            const min = eval(req['Min Value']);
            const max = eval(req['Max Value']);
            const num_bag = $gameParty.numItems(data);
            if(
                num_bag < min ||
                num_bag > max
            ){
                return true;
            }
        })){
            return false;
        }
    }
    return true;
}

function Window_SynMenuSelcData(){
    this.initialize(...arguments);
}

Window_SynMenuSelcData.prototype = Object.create(Window_Base.prototype);
Window_SynMenuSelcData.prototype.constructor = Window_SynMenuSelcData;

Window_SynMenuSelcData.prototype.initialize = function(data){
    const mz_mode = Utils.RPGMAKER_NAME == "MZ";
    const rect = this.createRect(data);
    this._window_data = data;
    this._style_data = data['Window Font and Style Configuration'];
    if(mz_mode){
        Window_Base.prototype.initialize.call(this, rect);
    }else{
        const x = rect.x;
        const y = rect.y;
        const w = rect.width;
        const h = rect.height;
        Window_Base.prototype.initialize.call(this,x,y,w,h);
    }
    this.setOpacityAndDimmer();
}

Window_SynMenuSelcData.prototype.createRect = function(data){
    const dimension_config = data['Dimension Configuration'];
    const x = dimension_config['X'];
    const y = dimension_config['Y'];
    const w = dimension_config['Width'];
    const h = dimension_config['Height'];
    return new Rectangle(x,y,w,h);
}

Window_SynMenuSelcData.prototype.standardPadding = function() {
    return 8;
}

Window_SynMenuSelcData.prototype.loadWindowskin = function(){
    const base = Window_Base.prototype.loadWindowskin.call(this);
    const custom_config = this._style_data;
    if(!custom_config)return base;
    const skin_name = custom_config['Window Skin'];
    if(!skin_name)return base;
    this.windowskin = ImageManager.loadSystem(skin_name);
}

Window_SynMenuSelcData.prototype.resetFontSettings = function() {
    const base = Window_Base.prototype.resetFontSettings;
    const custom_config = this._style_data;
    if(!custom_config)return base.call(this);
    const font_face = custom_config['Font Face'] || "sans-serif";
    const font_size = custom_config['Font Size'] || 16;
    const font_outline_size = custom_config['Font Outline Thickness'] || 3;
    this.contents.fontFace = font_face;
    this.contents.fontSize = font_size;
    this.contents.outlineWidth = font_outline_size;
    this.resetTextColor();
}

Window_SynMenuSelcData.prototype.resetTextColor = function() {
    const base = Window_Base.prototype.resetTextColor;
    const custom_config = this._style_data;
    if(!custom_config)return base.call(this);
    const text_color = custom_config['Base Font Color'] || "#ffffff";
    const outline_color = custom_config['Font Outline Color'] || "rgba(0, 0, 0, 0.5)";
    this.changeTextColor(text_color);
    this.contents.outlineColor = outline_color;
}

Window_SynMenuSelcData.prototype.setOpacityAndDimmer = function(){
    const custom_config = this._style_data;
    if(!custom_config)return;
    const show_dimmer = custom_config['Show Window Dimmer'] || false;
    const win_opacity = custom_config['Window Opacity'] || 0;
    this.opacity = win_opacity;
    show_dimmer ? this.showBackgroundDimmer() : this.hideBackgroundDimmer();
}

Window_SynMenuSelcData.prototype.update = function(){
    Window_Base.prototype.update.call(this);
    this.updateSelected();
}

Window_SynMenuSelcData.prototype.updateSelected = function(){
    const selc_window = this._select_window;
    if(!selc_window)return;
    const selc_index = selc_window.index();
    if(this._selc_index != selc_index){
        this._selc_index = selc_index;
        this._data = selc_window.currentSelection();
        this.drawData();
    }
}

Window_SynMenuSelcData.prototype.drawData = function(){
    this.contents.clear();
    const data = this._data;
    if(data){
        this.drawPicture(data);
        this.drawGauges(data);
        this.drawName(data);
        this.drawAltName(data);
        this.drawDesc(data);
    }
}

Window_SynMenuSelcData.prototype.drawPicture = function(data){
    const window_data = this._window_data;
    const draw_pic = eval(window_data['Draw Picture']);
    if(!draw_pic)return;
    const pic_names = JSON.parse(data['Pictures'] || [])
    const pic_index = eval(window_data['Picture Index']);
    const pic_name = pic_names[pic_index];
    if(!pic_name)return;
    const bitmap = ImageManager.loadPicture(pic_name);
    const bx = 0;
    const by = 0;
    const bw = bitmap.width;
    const bh = bitmap.height;
    const dx = eval(window_data['Picture X']);
    const dy = eval(window_data['Picture Y']);
    const dw = eval(window_data['Picture Width']);
    const dh = eval(window_data['Picture Height']);
    this.contents.blt(bitmap,bx,by,bw,bh,dx,dy,dw,dh);
}

Window_SynMenuSelcData.prototype.drawGauges = function(data){
    const window = this;
    const window_data = this._window_data;
    const gauge_draw_configs = window_data['Draw Gauges'];
    gauge_draw_configs.forEach((config)=>{
        const label = config['Label'];
        const lx = eval(config['Label X']);
        const ly = eval(config['Label Y']);
        window.drawTextEx(label, lx, ly);
        const cur_val = eval(config['Gauge Current Value']) || 0;
        const max_val = eval(config['Gauge Max Value']) || 1;
        const ratio = Math.max(0, Math.min(1, cur_val / max_val));
        const gx = eval(config['Gauge X']);
        const gy = eval(config['Gauge Y']);
        const gw = eval(config['Gauge Width']);
        const gh = eval(config['Gauge Height']);
        const gb = eval(config['Gauge Border']);
        const border_color = config['Gauge Border Color'];
        const background_color = config['Gauge Background Color'];
        const fill_color = config['Gauge Color'];
        window.contents.fillRect(gx,gy,gw,gh,border_color);
        window.contents.fillRect(gx + gb, gy + gb, gw - (gb * 2), gh - (gb * 2), background_color);
        window.contents.fillRect(gx + gb, gy + gb, (gw - (gb * 2)) * ratio, gh - (gb * 2), fill_color);
    })
}

Window_SynMenuSelcData.prototype.drawName = function(data){
    const window_data = this._window_data;
    const draw_name = eval(window_data['Draw Name']);
    if(!draw_name)return;
    const name = data['Name'];
    const tx = eval(window_data['Name X']);
    const ty = eval(window_data['Name Y']);
    const text = (window_data['Name Text'] || "").format(name);
    this.drawTextEx(text,tx,ty);
}

Window_SynMenuSelcData.prototype.drawAltName = function(data){
    const window_data = this._window_data;
    const draw_alt_name = eval(window_data['Draw Alternative Name']);
    if(!draw_alt_name)return;
    const alt_name = data['Alternative Name'];
    const tx = eval(window_data['Alternative Name X']);
    const ty = eval(window_data['Alternative Name Y']);
    const text = (window_data['Alternative Name Text'] || "").format(alt_name);
    this.drawTextEx(text,tx,ty);
}

Window_SynMenuSelcData.prototype.drawDesc = function(data){
    const window_data = this._window_data;
    const draw_desc = eval(window_data['Draw Description']);
    if(!draw_desc)return;
    const desc = data['Description'];
    const tx = eval(window_data['Description X']);
    const ty = eval(window_data['Description Y']);
    this.drawTextEx(desc,tx,ty);
}

function Scene_SynrecMenuPreload(){
    this.initialize(...arguments);
}

Scene_SynrecMenuPreload.prototype = Object.create(Scene_Base.prototype);
Scene_SynrecMenuPreload.prototype.constructor = Scene_SynrecMenuPreload;

Scene_SynrecMenuPreload.prototype.initialize = function(){
    Scene_Base.prototype.initialize.call(this);
    if(this.checkMenu()){
        this._rsvp_exit = true;
        this.clearSceneStack();
        return;
    }
    this._menu_data = $gameTemp.openedMenu();
    this.setupPreloadGfx();
}

Scene_SynrecMenuPreload.prototype.checkMenu = function(){
    const current_menu = $gameTemp.openedMenu();
    return !current_menu;
}

Scene_SynrecMenuPreload.prototype.clearSceneStack = function(){
    const scene_stack = SceneManager._stack;
    for(let i = 0; i < scene_stack.length; i++){
        const func = scene_stack[i];
        if(
            func == Scene_SynrecMenuPreload ||
            func == Scene_SynrecMenu
        ){
            scene_stack.splice(i, 1);
            i--;
        }
    }
}

Scene_SynrecMenuPreload.prototype.setupPreloadGfx = function(){
    const preload_pictures = [];
    const UI = $gameTemp.openedMenu();
    const backgrounds = UI['Backgrounds'];
    if(Array.isArray(backgrounds)){
        backgrounds.forEach((background)=>{
            const bitmap_name = background['File'];
            if(bitmap_name){
                preload_pictures.push(bitmap_name);
            }
        })
    }
    const back_gfxs = UI['Back Graphics'];
    if(Array.isArray(back_gfxs)){
        back_gfxs.forEach((background)=>{
            const bitmap_name = background['File'];
            if(bitmap_name){
                preload_pictures.push(bitmap_name);
            }
        })
    }
    const foregrounds = UI['Foregrounds'];
    if(Array.isArray(foregrounds)){
        foregrounds.forEach((foreground)=>{
            const bitmap_name = foreground['File'];
            if(bitmap_name){
                preload_pictures.push(bitmap_name);
            }
        })
    }
    const fore_gfxs = UI['Fore Graphics'];
    if(Array.isArray(fore_gfxs)){
        fore_gfxs.forEach((foreground)=>{
            const bitmap_name = foreground['File'];
            if(bitmap_name){
                preload_pictures.push(bitmap_name);
            }
        })
    }
    const basic_windows = UI['Basic Windows'];
    if(Array.isArray(basic_windows)){
        basic_windows.forEach((window_config)=>{
            const picture_draws = window_config['Draw Pictures'];
            if(Array.isArray(picture_draws)){
                picture_draws.forEach((pic_config)=>{
                    const pic_name = pic_config['Picture'];
                    preload_pictures.push(pic_name);
                })
            }
        })
    }
    const selc_window = UI['Selection Window'];
    if(!selc_window){
        throw new Error(`No selection window setup.`);
    }
    const selc_opts = selc_window['Selection Options']
    if(Array.isArray(selc_opts)){
        selc_opts.forEach((option)=>{
            const pics = JSON.parse(option['Pictures'] || []);
            if(Array.isArray(pics)){
                pics.forEach((pic)=>{
                    if(pic){
                        preload_pictures.push(pic);
                    }
                })
            }
            const static_gfx = option['Static Graphic'];
            if(static_gfx){
                const pic_name = static_gfx['File'];
                if(pic_name){
                    preload_pictures.push(pic_name);
                }
            }
            const anim_gfx = option['Animated Graphic'];
            if(anim_gfx){
                const pic_name = anim_gfx['File'];
                if(pic_name){
                    preload_pictures.push(pic_name);
                }
            }
            const button = option['Scene Button'];
            if(button){
                const cold_gfx = button['Cold Graphic'];
                if(cold_gfx){
                    const pic_name = cold_gfx['File'];
                    if(pic_name){
                        preload_pictures.push(pic_name);
                    }
                }
                const hot_gfx = button['Hot Graphic'];
                if(hot_gfx){
                    const pic_name = hot_gfx['File'];
                    if(pic_name){
                        preload_pictures.push(pic_name);
                    }
                }
            }
        })
    }
    this._preload_pictures = preload_pictures;
    this._load_index = 0;
}

Scene_SynrecMenuPreload.prototype.create = function(){
    Scene_Base.prototype.create.call(this);
    if(!this._rsvp_exit){
        this.createBackgrounds();
    }else{
        this.createBasicBackground();
    }
}

Scene_SynrecMenuPreload.prototype.createBackgrounds = function(){
    const scene = this;
    const UI = $gameTemp.openedMenu();
    const backgrounds = UI['Preload Backgrounds'];
    if(!Array.isArray(backgrounds)){
        return this._backgrounds = [];
    }
    const sprites = [];
    const base_bg = new Sprite();
    base_bg.bitmap = SceneManager.synMenuBackgroundBitmap();
    this.addChild(base_bg);
    sprites.push(base_bg);
    backgrounds.forEach((bg_config)=>{
        const sprite = new Sprite();
        sprite._blt_nme = bg_config['Picture'];
        sprite._px = eval(bg_config['X']);
        sprite._py = eval(bg_config['Y']);
        sprite._sw = eval(bg_config['Width']);
        sprite._sh = eval(bg_config['Height']);
        sprite.aliasUpdt = sprite.update;
        sprite.update = function(){
            this.aliasUpdt.call(this);
            if(this._blt_nme){
                this.bitmap = ImageManager.loadPicture(this._blt_nme);
            }
            const x = this._px || 0;
            const y = this._py || 0;
            const w = this._sw || 1;
            const h = this._sh || 1;
            this.setFrame(x,y,w,h);
        }
        scene.addChild(sprite);
        sprites.push(sprite);
    })
    this._backgrounds = sprites;
}

Scene_SynrecMenuPreload.prototype.createBasicBackground = function(){
    const base_bg = new Sprite();
    base_bg.bitmap = SceneManager.synMenuBackgroundBitmap();
    this.addChild(base_bg);
}

Scene_SynrecMenuPreload.prototype.update = function(){
    if(this._rsvp_exit && !SceneManager.isSceneChanging()){
        SceneManager.pop();
        return;
    }
    if(SceneManager.isSceneChanging())return;
    Scene_Base.prototype.update.call(this);
    this.updateLoadGraphics();
    this.updateLoadMenu();
}

Scene_SynrecMenuPreload.prototype.updateLoadGraphics = function(){
    if(this._rsvp_exit || SceneManager.isSceneChanging())return;
    if(!ImageManager.isReady())return;
    for(let i = 0; i < 5; i++){
        const bitmap_name = this._preload_pictures[this._load_index];
        if(!bitmap_name)break;
        ImageManager.loadPicture(bitmap_name);
        this._load_index++;
    }
}

Scene_SynrecMenuPreload.prototype.updateLoadMenu = function(){
    if(this._rsvp_exit || SceneManager.isSceneChanging())return;
    if(!ImageManager.isReady())return;
    if(this._preload_pictures.length <= 0){
        return SceneManager.push(Scene_SynrecMenu);
    }
    const ratio = this._load_index / this._preload_pictures.length;
    if(ratio >= 1){
        SceneManager.push(Scene_SynrecMenu);
    }
}

function Scene_SynrecMenu(){
    this.initialize(...arguments);
}

Scene_SynrecMenu.prototype = Object.create(Scene_Base.prototype);
Scene_SynrecMenu.prototype.constructor = Scene_SynrecMenu;

Scene_SynrecMenu.prototype.create = function(){
    this.createBackgrounds();
    this.createBackgfx();
    this.createStaticGfx();
    this.createWindowLayer();
    this.createWindows();
    this.createEventHandlers();
    this.createAnimGfx();
    this.createForegfx();
    this.createForegrounds();
}

Scene_SynrecMenu.prototype.clearButtons = function(){
    if(!Array.isArray(this._scene_buttons)){
        this._scene_buttons = [];
        return;
    }
    for(let i = 0; i < this._scene_buttons.length; i++){
        const btn = this._scene_buttons[i];
        if(btn){
            if(btn.parent){
                btn.parent.removeChild(btn);
            }
            if(btn.destroy){
                btn.destroy();
            }
            this._scene_buttons.splice(i, 1);
            i--;
        }
    }
    this._scene_buttons = this._scene_buttons.filter(Boolean);
}

Scene_SynrecMenu.prototype.createBackgrounds = function(){
    const scene = this;
    const UI = $gameTemp.openedMenu();
    const data = UI['Backgrounds'];
    const arr = [];
    const base_bg = new Sprite();
    base_bg.bitmap = SceneManager.synMenuBackgroundBitmap();
    this.addChild(base_bg);
    arr.push(base_bg);
    data.forEach((gfx)=>{
        const sprite = new Sprite_SynMenuStaticGfx(gfx);
        scene.addChild(sprite);
        arr.push(sprite);
    })
    this._backgrounds = arr;
}

Scene_SynrecMenu.prototype.createBackgfx = function(){
    const scene = this;
    const UI = $gameTemp.openedMenu();
    const data = UI['Back Graphics'];
    const arr = [];
    data.forEach((gfx)=>{
        const sprite = new Sprite_SynMenuAnimGfx(gfx);
        scene.addChild(sprite);
        arr.push(sprite);
    })
    this._back_gfxs = arr;
}

Scene_SynrecMenu.prototype.createWindows = function(){
    this.createSelectionWindow();
    this.createSelectionDataWindows();
    this.createBasicWindows();
}

Scene_SynrecMenu.prototype.createSelectionWindow = function(){
    this._scene_buttons = [];
    const UI = $gameTemp.openedMenu();
    const data = UI['Selection Window'];
    if(!data){
        throw new Error(`No selection window setup. Check menu setup.`);
    }
    const window = new Window_SynMenuSelc(data);
    window.setHandler(`ok`, this.doSelectAction.bind(this));
    window.setHandler(`cancel`, this.closeMenu.bind(this));
    this.addWindow(window);
    this._selection_window = window;
}

Scene_SynrecMenu.prototype.createSelectionDataWindows = function(){
    const UI = $gameTemp.openedMenu();
    const data = UI['Selection Data Windows'];
    if(!Array.isArray(data))return;
    const scene = this;
    const windows = [];
    data.forEach((config)=>{
        const window = new Window_SynMenuSelcData(config);
        window._select_window = scene._selection_window;
        scene.addWindow(window);
        windows.push(window);
    })
    this._data_windows = windows;
}

Scene_SynrecMenu.prototype.createBasicWindows = function(){
    const UI = $gameTemp.openedMenu();
    const data = UI['Basic Windows'];
    if(!Array.isArray(data))return;
    const scene = this;
    const windows = [];
    data.forEach((config)=>{
        const window = new Window_SynMenuBasic(config);
        scene.addWindow(window);
        windows.push(window);
    })
    this._basic_windows = windows;
}

Scene_SynrecMenu.prototype.createEventHandlers = function(){
    const scene = this;
    scene.createPictures();
    this._event_handler = new Game_Interpreter();
    const msg_w = Graphics.boxWidth;
    const msg_h = Window_Base.prototype.fittingHeight(4);
    const msg_x = (Graphics.boxWidth * 0.5) - (msg_w * 0.5);
    const msg_y = 0;
    const rect_text = new Rectangle(msg_x, msg_y, msg_w, msg_h);
    this._messageWindow = new Window_Message(rect_text);
    if(Utils.RPGMAKER_NAME == 'MZ'){
        this._messageWindow.subWindows = function(){
            return [
                this._goldWindow, 
                this._choiceListWindow,
                this._numberInputWindow, 
                this._eventItemWindow, 
                this._nameBoxWindow
            ];
        }
        this._messageWindow.createSubWindows = function(){
            const gold_rect = Scene_Message.prototype.goldWindowRect();
            this._goldWindow = new Window_Gold(gold_rect);
            this._goldWindow.openness = 0;
            this._choiceListWindow = new Window_ChoiceList();
            this._numberInputWindow = new Window_NumberInput();
            const item_rect = Scene_Message.prototype.eventItemWindowRect();
            this._eventItemWindow = new Window_EventItem(item_rect);
            this._nameBoxWindow = new Window_NameBox();
            this._nameBoxWindow.setMessageWindow(this);
            this._choiceListWindow.setMessageWindow(this);
            this._numberInputWindow.setMessageWindow(this);
            this._eventItemWindow.setMessageWindow(this);
        }
        this._messageWindow.createSubWindows();
        this._messageWindow.close();
    }
    this.addWindow(this._messageWindow);
    if(this._messageWindow.subWindows){
        this._messageWindow.subWindows().forEach((window)=>{
            scene.addWindow(window);
        });
    }
    const rect_scroll = new Rectangle(0, 0, Graphics.width, Graphics.height);
    this._scrollTextWindow = new Window_ScrollText(rect_scroll);
    this.addWindow(this._scrollTextWindow);
}

Scene_SynrecMenu.prototype.createPictures = function(){
    if(Utils.RPGMAKER_NAME == 'MZ'){
        const rect = new Rectangle(0, 0, Graphics.width, Graphics.height);
        this._pictureContainer = new Sprite();
        this._pictureContainer.setFrame(rect.x, rect.y, rect.width, rect.height);
        for (let i = 1; i <= $gameScreen.maxPictures(); i++) {
            this._pictureContainer.addChild(new Sprite_Picture(i));
        }
        this.addChild(this._pictureContainer);
    }else{
        const width = Graphics.boxWidth;
        const height = Graphics.boxHeight;
        const x = (Graphics.width - width) / 2;
        const y = (Graphics.height - height) / 2;
        this._pictureContainer = new Sprite();
        this._pictureContainer.setFrame(x, y, width, height);
        for (let i = 1; i <= $gameScreen.maxPictures(); i++) {
            this._pictureContainer.addChild(new Sprite_Picture(i));
        }
        this.addChild(this._pictureContainer);
    }
}

Scene_SynrecMenu.prototype.createStaticGfx = function(){
    const sprite = new Sprite_SynMenuStaticGfx();
    this.addChild(sprite);
    this._select_static_gfx = sprite;
}

Scene_SynrecMenu.prototype.createAnimGfx = function(){
    const sprite = new Sprite_SynMenuAnimGfx();
    this.addChild(sprite);
    this._select_anim_gfx = sprite;
}

Scene_SynrecMenu.prototype.createForegfx = function(){
    const scene = this;
    const UI = $gameTemp.openedMenu();
    const data = UI['Fore Graphics'];
    const arr = [];
    data.forEach((gfx)=>{
        const sprite = new Sprite_SynMenuAnimGfx(gfx);
        scene.addChild(sprite);
        arr.push(sprite);
    })
    this._fore_gfxs = arr;
}

Scene_SynrecMenu.prototype.createForegrounds = function(){
    const scene = this;
    const UI = $gameTemp.openedMenu();
    const data = UI['Foregrounds'];
    const arr = [];
    data.forEach((gfx)=>{
        const sprite = new Sprite_SynMenuStaticGfx(gfx);
        scene.addChild(sprite);
        arr.push(sprite);
    })
    this._foregrounds = arr;
}

Scene_SynrecMenu.prototype.doSelectAction = function(){
    const selc_window = this._selection_window;
    const option_data = selc_window.currentSelection();
    if(!option_data){
        SoundManager.playBuzzer();
        selc_window.activate();
        return;
    }
    if(!selc_window.meetSelectRequirements()){
        SoundManager.playBuzzer();
        selc_window.activate();
        return;
    }
    try{
        const event = eval(option_data['Event Execution']) || 0;
        if(event){
            this._rsvp_evnt = event;
            return;
        }
    }catch(e){}
    const code = option_data['Code Execution'];
    if(code){
        eval(code);
    }else{
        SoundManager.playBuzzer();
        selc_window.activate();
    }
}

Scene_SynrecMenu.prototype.closeMenu = function(){
    $gameTemp.closeMenu();
    SceneManager.pop();
}

Scene_SynrecMenu.prototype.update = function(){
    Scene_Base.prototype.update.call(this);
    this.updateStaitcGfx();
    this.updateAnimGfx();
    if(this.updateEvent())return;
}

Scene_SynrecMenu.prototype.updateStaitcGfx = function(){
    const selc_window = this._selection_window;
    const static_gfx = this._select_static_gfx
    if(!static_gfx)return;
    const selc_index = selc_window.index();
    if(this._static_selc === selc_index)return;
    this._static_selc = selc_index;
    if(!selc_window){
        static_gfx.visible = false;
        return;
    }
    const option_data = selc_window.currentSelection();
    if(!option_data){
        static_gfx.visible = false;
        return;
    }
    const gfx_data = option_data['Static Graphic'];
    if(gfx_data){
        static_gfx.setupGfx(gfx_data);
        static_gfx.visible = true;
    }else{
        static_gfx.visible = false;
    }
}

Scene_SynrecMenu.prototype.updateAnimGfx = function(){
    const selc_window = this._selection_window;
    const anim_gfx = this._select_anim_gfx
    if(!anim_gfx)return;
    const selc_index = selc_window.index();
    if(this._anim_selc === selc_index)return;
    this._anim_selc = selc_index;
    if(!selc_window){
        anim_gfx.visible = false;
        return;
    }
    const option_data = selc_window.currentSelection();
    if(!option_data){
        anim_gfx.visible = false;
        return;
    }
    const gfx_data = option_data['Animated Graphic'];
    if(gfx_data){
        anim_gfx.setupGfx(gfx_data);
        anim_gfx.visible = true;
    }else{
        anim_gfx.visible = false;
    }
}

Scene_SynrecMenu.prototype.updateEvent = function(){
    if(this._rsvp_evnt){
        const event = $dataCommonEvents[this._rsvp_evnt];
        if(event){
            this._event_handler.setup(event.list);
            if(event.list.length <= 0){
                this.activateSelcWindow();
            }
            this._reactivate_on_false = true;
        }
        this._rsvp_evnt = undefined;
    }
    this._event_handler.update();
    if(
        this._event_handler.isRunning() || 
        $gameMessage.isBusy() ||
        this._messageWindow.isOpen() ||
        this._scrollTextWindow.visible
    ){
        delete this._reactivate_on_false;
        this._event_busy = true;
        this.deactivateSelcWindow();
        return true;
    }else if(this._event_busy || this._reactivate_on_false){
        this._event_busy = false;
        this.activateSelcWindow();
    }
    return false;
}

Scene_SynrecMenu.prototype.deactivateSelcWindow = function(){
    this._selection_window.deactivate();
    this._selection_window.hide();
    this._data_windows.forEach((window)=>{
        window.hide();
    })
    this._basic_windows.forEach((window)=>{
        window.hide();
    })
    this._select_static_gfx.visible = false;
    this._select_anim_gfx.visible = false;
}

Scene_SynrecMenu.prototype.activateSelcWindow = function(){
    this._selection_window.activate();
    this._selection_window.show();
    this._data_windows.forEach((window)=>{
        window.show();
    })
    this._basic_windows.forEach((window)=>{
        window.show();
    })
    this._select_static_gfx.visible = true;
    this._select_anim_gfx.visible = true;
}