// SelectableWindow.js Ver.3.0.0
// MIT License (C) 2022 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ
* @plugindesc Sets the background of the selection to the window.
* @orderAfter FesCursor
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/491065758.html
* @help Ver.3.0.0
*
* @param windowSkin
* @text Window Skin
* @desc Specify a file name.
* @default 
* @type file
* @dir img/system
*
* @param windowOpacity
* @text Window Opacity
* @desc Specifies the opacity of the window.
* -1 is system default.
* @default -1
* @type number
* @min -1
*
* @param windowThrough
* @text Window Through
* @desc Makes behind window visible.
* @default false
* @type boolean
*
* @param gradientWidth
* @text Gradient Width
* @desc The width of the gradient used for decoration.
* @default 16
* @type number
*
* @param cursorPad
* @text Cursor Width Expansion
* @desc Extends the visible range of the window's cursor.
* @default 0
* @type number
* @min -999999
*
* @param offsetW1
* @text Item Width
* @desc Item width correction value
* @default -8
* @type number
* @min -999999
*
* @param offsetH1
* @text Item Height
* @desc Item height correction value
* @default -8
* @type number
* @min -999999
*
* @param offsetW2
* @text Window Width
* @desc Window width correction value
* @default 2
* @type number
* @min -999999
*
* @param offsetH2
* @text Window Height
* @desc Window Height correction value
* @default 2
* @type number
* @min -999999
*
* @param inclusionList
* @text Inclusion List
* @desc Sets the window that makes the choice background a window.
* @type string[]
* @default ["Window_SavefileList","Window_MenuCommand","Window_MenuStatus","Window_MenuActor","Window_ItemCategory","Window_EquipCommand","Window_ShopCommand","Window_ProfileTab"]
*
* @param decorationList
* @text Decoration List
* @desc Sets the window that decorates the choice background cursor.
* @type string[]
* @default ["Window_SavefileList"]
*
*/

/*:ja
* @target MZ
* @plugindesc 選択肢背景をウィンドウにします。
* @orderAfter FesCursor
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/491065758.html
* @help [更新履歴]
* 2022/08/28：Ver.1.0.0　公開。
* 2022/12/22：Ver.1.1.0　背後にあるウィンドウを透過する機能を追加。
* 2023/01/28：Ver.1.1.1　クリック判定を修正。
* 2023/02/05：Ver.2.0.0　任意のウィンドウを装飾できる機能を追加。
* 2023/08/02：Ver.3.0.0　難解だったパラメータの調整方法を修正。
*
* @param windowSkin
* @text ウィンドウスキン
* @desc ファイル名を指定します。
* @default 
* @type file
* @dir img/system
*
* @param windowOpacity
* @text ウィンドウ不透明度
* @desc ウィンドウの不透明度を指定します。
* -1でシステムの初期値。
* @default -1
* @type number
* @min -1
*
* @param windowThrough
* @text ウィンドウ透過
* @desc 背後にあるウィンドウが見えるようにします。
* @default false
* @type boolean
*
* @param gradientWidth
* @text カーソル装飾幅
* @desc 装飾に使うグラデーションの幅です。
* @default 16
* @type number
*
* @param cursorPad
* @text カーソル幅拡張
* @desc ウィンドウのカーソルの表示範囲を拡張します。
* @default 0
* @type number
* @min -999999
*
* @param offsetW1
* @text 要素幅
* @desc 要素の幅の補正値
* @default -8
* @type number
* @min -999999
*
* @param offsetH1
* @text 要素高さ
* @desc 要素の高さの補正値
* @default -8
* @type number
* @min -999999
*
* @param offsetW2
* @text ウィンドウ幅
* @desc ウィンドウの幅の補正値
* @default 2
* @type number
* @min -999999
*
* @param offsetH2
* @text ウィンドウ高さ
* @desc ウィンドウの高さの補正値
* @default 2
* @type number
* @min -999999
*
* @param inclusionList
* @text 包含リスト
* @desc 選択肢背景をウィンドウにするウィンドウを設定します。
* @type string[]
* @default ["Window_SavefileList","Window_MenuCommand","Window_MenuStatus","Window_MenuActor","Window_ItemCategory","Window_EquipCommand","Window_ShopCommand","Window_ProfileTab"]
*
* @param decorationList
* @text 装飾リスト
* @desc 選択肢背景カーソルを装飾するウィンドウを設定します。
* @type string[]
* @default ["Window_SavefileList"]
*
*/

'use strict';

{
	//プラグイン名取得。
	const script = document.currentScript;
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];

	const parameters = PluginManager.parameters(pluginName);

	const windowSkin = parameters["windowSkin"];
	const windowOpacity = Number(parameters["windowOpacity"]);
	const windowGradient = parameters["windowGradient"] === "true";
	const gradientWidth = Number(parameters["gradientWidth"]);
	const windowThrough = parameters["windowThrough"] === "true";
	const cursorPad = Number(parameters["cursorPad"]);
	const offsetW1 = Number(parameters["offsetW1"]);
	const offsetW2 = Number(parameters["offsetW2"])-offsetW1;
	const offsetH1 = Number(parameters["offsetH1"]);
	const offsetH2 = Number(parameters["offsetH2"])-offsetH1;
	const inclusionList = new Set(JSON.parse(parameters["inclusionList"] || "[]"));
	const decorationList = new Set(JSON.parse(parameters["decorationList"] || "[]"));

	//-----------------------------------------------------------------------------
	// Window_Selectable

	const _Window_Selectable_initialize = Window_Selectable.prototype.initialize;
	Window_Selectable.prototype.initialize = function(rect) {
		this._selectableWindowEnabled = inclusionList.has(this.constructor.name);
		this._selectableWindowGradientEnabled = decorationList.has(this.constructor.name);
		_Window_Selectable_initialize.call(this, rect);
		if (this.selectableWindowEnabled()) {
			this.opacity = 0;
			this._isWindow = !windowThrough;
		}
	};

	let hitTest = false;
	const _Window_Selectable_hitTest = Window_Selectable.prototype.hitTest;
	Window_Selectable.prototype.hitTest = function(x, y) {
		hitTest = this.selectableWindowEnabled();
		const r = _Window_Selectable_hitTest.call(this, x, y);
		hitTest = false;
		return r;
	};

	const _Window_Selectable_itemRect = Window_Selectable.prototype.itemRect;
	Window_Selectable.prototype.itemRect = function(index) {
		if (hitTest) {
			hitTest = false;
			const r = this.itemWindowRect(index);
			hitTest = true;
			return r;
		}
		const rect = _Window_Selectable_itemRect.call(this, index);
		if (this.selectableWindowEnabled()) {
			rect.x -= Math.floor(offsetW1/2);
			rect.y -= Math.floor(offsetH1/2);
			rect.width += offsetW1;
			rect.height += offsetH1;
		}
		return rect;
	};

	Window_Selectable.prototype.itemWindowRect = function(index) {
		const rect = this.itemRect(index);
		rect.x -= Math.floor(offsetW2/2);
		rect.y -= Math.floor(offsetH2/2);
		rect.width += offsetW2;
		rect.height += offsetH2;
		return rect;
	};

	Window_Selectable.prototype.selectableWindowEnabled = function() {
		return this._selectableWindowEnabled;
	};

	Window_Selectable.prototype.selectableWindowGradientEnabled = function() {
		return this._selectableWindowGradientEnabled;
	};

	const _Window_Selectable__createContentsBackSprite = Window_Selectable.prototype._createContentsBackSprite;
	Window_Selectable.prototype._createContentsBackSprite = function() {
		if (this.selectableWindowEnabled()) {
			this._createSelectableWindowContainer();
		}
		_Window_Selectable__createContentsBackSprite.call(this);
	};

	Window_Selectable.prototype._createSelectableWindowContainer = function() {
		this._selectableWindowContainer = new Sprite();
		this._selectableWindows = [];
		this._clientArea.addChild(this._selectableWindowContainer);
	};

	Window_Selectable.prototype.updateSelectableWindow = function() {
		if (this.selectableWindowEnabled()) {
			this.adjustSelectableWindowSpriteSize();
			this.updateSelectableWindowPosition();
		}
	};

	Window_Selectable.prototype.adjustSelectableWindowSpriteSize = function() {
		const max = this.maxVisibleItems();
		const dif = max - this._selectableWindows.length;

		if (dif > 0) {
			while (max !== this._selectableWindows.length) {
				const sprite = new Window_Background(new Rectangle());
				this._selectableWindows.push(sprite);
				this._selectableWindowContainer.addChild(sprite);
				sprite.openness = 255;
				sprite._padding = 4;
				sprite.show();
				sprite.deactivate();
				sprite._container.children.indexOf(sprite._frameSprite);
				sprite.windowskin = this.windowskin;
				sprite.backOpacity = this.backOpacity;
				if (windowSkin) sprite.windowskin = ImageManager.loadSystem(windowSkin);
				if (windowOpacity !== -1) sprite.backOpacity = windowOpacity;
			}
		} else if (dif < 0) {
			while (max !== this._selectableWindows.length) {
				const sprite = this._selectableWindows.pop();
				sprite.destroy();
			}
		}
	};

	Window_Selectable.prototype.updateSelectableWindowPosition = function() {
		let maxItems;
		try {
			maxItems = this.maxItems();
		} catch (e) {
			maxItems = 0;
		}
		this._selectableWindows.forEach((sprite, index) => {
			const itemIndex = index + this.topIndex();
			const rect = this.itemWindowRect(itemIndex);
			sprite.move(rect.x, rect.y, rect.width, rect.height);
			sprite.setCursorRect(0,0,0,0);
			if (this.selectableWindowCursorRectEnabled(itemIndex)) {
				const rect = sprite.innerRect;
				rect.pad(cursorPad);
				sprite.setCursorRect(rect.x, rect.y, rect.width, rect.height);
			}
			sprite.show();
			if (itemIndex + 1 > maxItems) {
				sprite.hide();
			}
		});
	};

	Window_Selectable.prototype.selectableWindowCursorRectEnabled = function(index) {
		return this.cursorAll() || this.index() === index;
	};

	const _Window_Selectable_refreshCursor = Window_Selectable.prototype.refreshCursor;
	Window_Selectable.prototype.refreshCursor = function() {
		_Window_Selectable_refreshCursor.call(this);
		this.updateSelectableWindow();
	};

	const _Window_Selectable_updateScrollBase = Window_Selectable.prototype.updateScrollBase;
	Window_Selectable.prototype.updateScrollBase = function(baseX, baseY) {
		_Window_Selectable_updateScrollBase.apply(this, arguments);
		this.updateSelectableWindow();
	};

	const _Window_Selectable__updateCursor = Window_Selectable.prototype._updateCursor;
	Window_Selectable.prototype._updateCursor = function() {
		_Window_Selectable__updateCursor.call(this);
		if (this.selectableWindowEnabled()) {
			this._cursorSprite.visible = false;
			if (this.selectableWindowGradientEnabled()) {
				this._selectableWindows.forEach(sprite => {
					sprite._cursorSprite.alpha = sprite.contentsOpacity / 255;
					sprite._cursorGradientSprite.alpha = this.active ? Window_Background.prototype._makeCursorGradientAlpha.call(this) : 0;
					//sprite._cursorGradientSprite.alpha = Window_Background.prototype._makeCursorGradientAlpha.call(this);
				});
			} else {
				this._selectableWindows.forEach(sprite => {
					sprite._cursorSprite.alpha = this._cursorSprite.alpha
					sprite._cursorGradientSprite.alpha = 0;
				});
			}
		}
	};

	const _Window_Selectable_drawBackgroundRect = Window_Selectable.prototype.drawBackgroundRect;
	Window_Selectable.prototype.drawBackgroundRect = function(rect) {
		if (this.selectableWindowEnabled()) {
			rect.width = 0;
			rect.height = 0;
		}
		_Window_Selectable_drawBackgroundRect.call(this, rect);
	};

	//-----------------------------------------------------------------------------
	// Window_Command

	const _Window_Command_arrowCursorPadding = Window_Command.prototype.arrowCursorPadding;
	Window_Command.prototype.arrowCursorPadding = function() {
		return !this.selectableWindowEnabled() && _Window_Command_arrowCursorPadding.call(this);
	};

	//-----------------------------------------------------------------------------
	// Window_Background

	function Window_Background() {
		this.initialize(...arguments);
	}

	Window_Background.prototype = Object.create(Window_Base.prototype);
	Window_Background.prototype.constructor = Window_Background;

	Window_Background.prototype.initialize = function(rect) {
		this._cursorVignetteSprite = null;
		Window_Base.prototype.initialize.call(this, rect);
	};

	const _Window_Background__createCursorSprite = Window_Background.prototype._createCursorSprite;
	Window_Background.prototype._createCursorSprite = function() {
		_Window_Background__createCursorSprite.call(this);
		const index = this._container.children.indexOf(this._frameSprite);
		this._container.addChildAt(this._cursorSprite, index);
		this._createCursorGradientSprite();
	};

	Window_Background.prototype._createCursorGradientSprite = function() {
		this._cursorGradientSprite = new Sprite();
		const index = this._container.children.indexOf(this._frameSprite);
		this._container.addChildAt(this._cursorGradientSprite, index);
	};

	Window_Background.prototype._setRectPartsGeometry = function(sprite, srect, drect, m) {
		Window_Base.prototype._setRectPartsGeometry.apply(this, arguments);
		if (sprite !== this._cursorSprite) return;
		const dx = drect.x;
		const dy = drect.y;
		const dw = drect.width;
		const dh = drect.height;
		const grad = this._cursorGradientSprite;
		grad.setFrame(0, 0, dw, dh);
		grad.move(dx, dy);
		if (grad.bitmap) {
			grad.bitmap.destroy();
		}
		const bitmap = grad.bitmap = new Bitmap(dw, dh);
		if (!dw || !dh) return;
		const gw = gradientWidth;
		const col1 = "rgba(255, 255, 255, 0.4)";
		const col2 = "rgba(255, 255, 255, 0)";

		bitmap.gradientFillRect(0, gw, gw, dh - gw * 2, col1, col2);
		bitmap.gradientFillRect(dw - gw, gw, gw, dh - gw * 2, col2, col1);
		bitmap.gradientFillRect(gw, 0, dw - gw * 2, gw, col1, col2, true);
		bitmap.gradientFillRect(gw, dh - gw, dw - gw * 2, gw, col2, col1, true);

		bitmap.drawGradForSelectableWindow(0, 0, gw, 0, col2, col1);
		bitmap.drawGradForSelectableWindow(0, dh, gw, 1, col2, col1);
		bitmap.drawGradForSelectableWindow(dw, 0, gw, 2, col2, col1);
		bitmap.drawGradForSelectableWindow(dw, dh, gw, 3, col2, col1);

	};

	Window_Background.prototype._updateCursor = function() {
		Window_Base.prototype._updateCursor.call(this);
		this._cursorGradientSprite.visible = this._cursorSprite.visible;
	};

	Window_Background.prototype._makeCursorAlpha = function() {
		return this._cursorSprite.alpha;
	};

	Window_Background.prototype._makeCursorGradientAlpha = function() {
		const blinkCount = this._animationCount % 60;
		const baseAlpha = this.contentsOpacity / 255;
		if (this.active) {
			if (blinkCount < 30) {
				return baseAlpha - blinkCount / 48;
			} else {
				return baseAlpha - (60 - blinkCount) / 48;
			}
		}
		return baseAlpha;
	};

	//-----------------------------------------------------------------------------
	// Bitmap

	const posMap = [
	//upper-left
	{ "x": 1, "y": 1 },
	//bottom-left
	{ "x": 1, "y": -1 },
	//upper-right
	{ "x": -1, "y": 1 },
	//bottom-right
	{ "x": -1, "y": -1 }
	];
	Bitmap.prototype.drawGradForSelectableWindow = function(x, y, radius, pos, color1, color2) {
		const offsetX = radius * posMap[pos].x;
		const offsetY = radius * posMap[pos].y;
		const context = this.context;
		const grad = context.createRadialGradient(x + offsetX, y + offsetY, 0, x + offsetX, y + offsetY, radius);
		grad.addColorStop(0, color1);
		grad.addColorStop(1, color2);
		context.save();
		context.fillStyle = grad;
		context.fillRect(x, y, offsetX, offsetY);
		context.restore();
		this._baseTexture.update();
	};

}