/*:
 * @plugindesc 战斗衣服破损图片显示 v2.2.0（层级ID即图片ID + 简化配置）
 * @author Grok
 *
 * @help BattleClothDisplay.js
 *
 * 改进点（v2.2.0）：
 * 1. 配置中层级ID（1,2,3...）直接作为图片ID偏移量，无需额外映射。
 *    示例：ID=1 的衣服 → 图片编号 = BasePictureId + 1
 * 2. 配置结构简化：只需 { id: { cloth: "文件名" } } 和 rates
 * 3. 排错更方便：层级数字直观对应图片ID
 * 4. 保留纯事件驱动调用 + 公开函数
 *
 * 调用方式（事件 → 脚本）：
 *   BattleClothManager.refresh();   // 更新衣服显示（HP/装备变化后调用）
 *   BattleClothManager.hideAll();   // 隐藏所有衣服图片
 *   BattleClothManager.eraseAll();  // 彻底删除（战斗结束自动）
 *
 * @param BasePictureId
 * @text 图片ID起始值
 * @desc 衣服图片实际ID = 此值 + 层级ID（例如 Base=50，层级1 → 图片51）
 * @type number
 * @min 1
 * @max 90
 * @default 50
 *
 * @param PictureXOffset
 * @text X偏移（相对角色中心）
 * @type number
 * @min -1000
 * @max 1000
 * @default 0
 *
 * @param PictureYOffset
 * @text Y偏移（相对角色中心）
 * @type number
 * @min -1000
 * @max 1000
 * @default -50
 *
 * @param PictureScale
 * @text 图片缩放（%）
 * @type number
 * @min 10
 * @max 300
 * @default 100
 *
 * @param DefaultClothConfig
 * @text 默认衣服配置（JSON，多行）
 * @desc 无匹配装备时使用。层级ID为数字！
 * 示例：{"1":{"cloth":"default_base"},"rates":{"100":[1]}}
 * @type note
 * @default {"1":{"cloth":"default_base"},"rates":{"100":[1]}}
 */

(function() {
    'use strict';

    const params = PluginManager.parameters('BattleClothDisplay');
    const basePicId = Number(params['BasePictureId'] || 50);
    const xOffset = Number(params['PictureXOffset'] || 0);
    const yOffset = Number(params['PictureYOffset'] || -50);
    const scale = Number(params['PictureScale'] || 100);

    // ====================== 配置数据（层级ID即为图片偏移ID） ======================
    const battle_cloth_map = {
        
        51: {
            1: { cloth: "lumine_base" },
            2: { cloth: "lumine_short" },
            3: { cloth: "lumine_bra" },
			4: { cloth: "lumine_top" },
			5: { cloth: "lumine_bot" },
            "rates": {
                80: [1, 2, 3,4,5],
                70: [1, 2,3,5],
                60: [1,2,3],
				50: [1,2],
				40: [1],
            }
        }



    };

    let defaultConfig = {};
    try {
        const jsonStr = params['DefaultClothConfig'] || '{}';
        const parsed = JSON.parse(jsonStr);
        defaultConfig = convertKeysToNumber(parsed);
    } catch (e) {
        console.warn('默认配置解析失败', e);
        defaultConfig = {};
    }

    function convertKeysToNumber(obj) {
        if (typeof obj !== 'object' || obj === null) return obj;
        if (Array.isArray(obj)) return obj.map(convertKeysToNumber);
        const newObj = {};
        for (const key in obj) {
            const numKey = isNaN(key) ? key : Number(key);
            newObj[numKey] = convertKeysToNumber(obj[key]);
        }
        return newObj;
    }

    // ====================== 全局管理器 ======================
    window.BattleClothManager = {

        // 获取当前配置
        getCurrentConfig: function() {
            const leader = $gameParty.leader();
            if (!leader) return battle_cloth_map[51];

            const armor = leader.equips()[3]; // 铠甲槽位
            if (armor && battle_cloth_map[armor.id]) {
                return battle_cloth_map[armor.id];
            }
            return battle_cloth_map[51];
        },	

        getShowIds: function(config) {
            const leader = $gameParty.leader();
            if (!leader) return [];

            const hpRate = Math.floor(leader.hpRate() * 100);
            const rates = config.rates || {};
            const thresholds = Object.keys(rates).map(Number).sort((a, b) => b - a);

            if (thresholds.length === 0) return [];

            for (const t of thresholds) {
                if (hpRate >= t) {
                    return rates[t] || [];
                }
            }
            const minT = thresholds[thresholds.length - 1];
            return rates[minT] || [];
        },

        refresh: function() {
            if (!SceneManager._scene instanceof Scene_Battle) return;

            const config = this.getCurrentConfig();
            const showIds = this.getShowIds(config);

            this.eraseAll(); // 先清空

            const spriteset = SceneManager._scene._spriteset;
            if (!spriteset || !spriteset._actorSprites || spriteset._actorSprites.length === 0) return;

            const actorSprite = spriteset._actorSprites[0];
            if (!actorSprite) return;

            const baseX =  xOffset;
            const baseY = yOffset;

            // 直接使用层级ID作为图片ID偏移
            showIds.forEach(id => {
                const layer = config[id];
                if (!layer || !layer.cloth) return;

                const picId = basePicId + id;  // 关键优化：id直接加base

                $gameScreen.showPicture(
                    picId,
                    layer.cloth,
                    0,
                    baseX,
                    baseY,
                    scale,
                    scale,
                    255,
                    0
                );
            });
        },

        hideAll: function() {
            for (let i = 1; i <= 30; i++) {  // 增加范围防漏
                $gameScreen.erasePicture(basePicId + i);
            }
        },

        eraseAll: function() {
            this.hideAll();
        },

        eraseAllClothPictures: function() {
            this.eraseAll();
        },

        showCloth: function() {
            this.refresh();
        }
    };

    // ====================== 战斗自动调用 ======================
    const _Scene_Battle_start = Scene_Battle.prototype.start;
    Scene_Battle.prototype.start = function() {
        _Scene_Battle_start.call(this);
        BattleClothManager.refresh();
    };

    const _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
    Scene_Battle.prototype.terminate = function() {
        _Scene_Battle_terminate.call(this);
        BattleClothManager.eraseAll();
    };

    // 战斗中换装备自动刷新（可选）
    const _Game_Actor_changeEquip = Game_Actor.prototype.changeEquip;
    Game_Actor.prototype.changeEquip = function(slotId, item) {
        _Game_Actor_changeEquip.call(this, slotId, item);
        if (slotId === 1 && $gameParty.inBattle()) {
            BattleClothManager.refresh();
        }
    };

})();