/*:
 * @plugindesc Safe Executor - 安全执行函数并返回是否成功
 * @author Grok (generated)
 * @help
 * 这个插件提供一个全局工具，用于安全执行函数并判断是否成功。
 *
 * 使用方法（脚本调用）：
 * var success = SafeExecutor.try(function() {
 *     // 这里写可能出错的代码
 *     $gameVariables.setValue(10, $gameParty.gold() / 0);  // 示例：故意除零出错
 * });
 *
 * if (success) {
 *     $gameMessage.add("执行成功！");
 * } else {
 *     $gameMessage.add("执行失败，已安全捕获错误。");
 * }
 *
 * 特点：
 * - 任何异常都会被捕获，不会崩溃游戏
 * - 异常信息会输出到浏览器控制台（F8 或 F12）
 * - 返回 true / false 表示是否成功执行
 */

var SafeExecutor = SafeExecutor || {};

(function() {
    'use strict';

    /**
     * 安全执行函数
     * @param {Function} func 要执行的函数
     * @returns {Boolean} true 表示执行成功无异常，false 表示捕获到异常
     */
    SafeExecutor.try = function(func) {
        if (typeof func !== "function") {
            console.warn("SafeExecutor.try: 传入的参数不是函数");
            return false;
        }

        try {
            func();
            return true;  // 正常执行完成
        } catch (error) {
            // 捕获所有异常（包括 throw new Error、语法错误运行时表现等）
            console.error("SafeExecutor: 执行函数时捕获到异常:", error);
            // 可选：如果你想在游戏中显示错误提示，可以取消下面注释
            // if ($gameMessage) $gameMessage.add("脚本执行出错（已安全处理）");
            return false;  // 有异常
        }
    };

    // 可选：支持异步函数（await），但 RPG Maker MV 脚本通常是同步的
    // 如果你需要支持 async function，请使用 SafeExecutor.tryAsync
    SafeExecutor.tryAsync = async function(asyncFunc) {
        if (typeof asyncFunc !== "function") {
            console.warn("SafeExecutor.tryAsync: 传入的参数不是函数");
            return false;
        }
        try {
            await asyncFunc();
            return true;
        } catch (error) {
            console.error("SafeExecutor: 异步执行捕获到异常:", error);
            return false;
        }
    };

})();