//=============================================================================
// ScreenShotToFile.js
// 作者：Grok（终极防黑屏版）
// 功能：完整游戏画面截图（地图+UI+对话框），保存 PNG 到游戏根目录
// 用法：插件命令 ScreenShot save | 脚本 ScreenShot.save("名字");
// 仅打包版有效；完美兼容 WebGL/Canvas，所有场景无黑屏
//=============================================================================

var ScreenShot = ScreenShot || {};

(() => {
    'use strict';

    //=========================================================================
    // 工具函数：时间戳文件名
    //=========================================================================
    function getTimestamp() {
        const d = new Date();
        return d.toISOString().replace(/[-:T.]/g, '').slice(0, -1); // 20251117T195623999
    }

    //=========================================================================
    // 主函数：使用 SceneManager.snap() 合成完整画面
    //=========================================================================
    ScreenShot.save = function(customName = null) {
        // 场景检查
        const scene = SceneManager._scene;
        if (!scene || scene.isBusy()) {
            console.warn("ScreenShot: 场景忙碌，1秒后重试...");
            setTimeout(() => ScreenShot.save(customName), 1000);
            return;
        }

        console.log("ScreenShot: 开始截图...");

        // ★关键：使用官方 snap() 获取完整合成 Bitmap
        const bitmap = SceneManager.snap();
        if (!bitmap || !bitmap.canvas) {
            console.error("ScreenShot: snap() 失败");
            return;
        }

        // 生成文件名
        const filename = customName ? `${customName}.png` : `${getTimestamp()}.png`;

        // NW.js 保存
        if (typeof require === 'undefined') {
            console.warn("ScreenShot: 浏览器模式无法保存");
            return;
        }

        const path = require('path');
        const fs = require('fs');
        const gameDir = path.dirname(process.mainModule.filename);
        const fullPath = path.join(gameDir, filename);

        try {
            // Bitmap 直接转为 PNG base64
            const dataUrl = bitmap.canvas.toDataURL('image/png');
            const base64 = dataUrl.split(',')[1];
            fs.writeFileSync(fullPath, Buffer.from(base64, 'base64'));
            
            console.log(`✅ 截图成功：${fullPath}`);
            console.log(`   画面尺寸：${bitmap.width} x ${bitmap.height}`);
            
            // 游戏内提示
            if ($gameMessage) {
                $gameMessage.add(`📸 已保存：${filename}`);
            }
        } catch (e) {
            console.error("❌ 保存失败：", e.message);
        }
    };

    //=========================================================================
    // 插件命令（MV）
    //=========================================================================
    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command.toLowerCase() === "screenshot" || command === "ScreenShot") {
            if (args[0] === "save") {
                ScreenShot.save(args.slice(1).join('_') || null);
            }
        }
    };

    // MZ 插件命令
    if (PluginManager.registerCommand) {
        PluginManager.registerCommand("ScreenShotToFile", "save", args => {
            ScreenShot.save(args.name || null);
        });
    }

    // 全局访问
    window.ScreenShot = ScreenShot;

})();