/*:
 * @plugindesc Boss BGM Switch v6.0.0 (Fixed Default BGM Override)
 * @target MV
 * @help BossBGM_FinalFixed.js
 * 
 * == 介绍 ==
 * 通过“战斗处理”事件命令拦截Boss BGM切换。
 * 完全阻止敌群默认BGM覆盖，Boss BGM从敌人介绍阶段持续到战斗结束。
 * 
 * == 使用方法 ==
 * 在Boss敌群任意事件页注释中写：
 * <BossBGM: Battle3>
 * 或 <BossBGM: BossTheme, 80, 120>
 * 
 * == 参数 ==
 * @param DefaultBossBGM
 * @text 默认Boss BGM
 * @default 
 * 
 * @param DefaultVolume
 * @text 默认音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 * 
 * @param DefaultPitch
 * @text 默认音调
 * @type number
 * @min 50
 * @max 150
 * @default 100
 */

(function() {
    'use strict';

    var parameters = PluginManager.parameters('BossBGM_FinalFixed');
    var defaultBGM = parameters['DefaultBossBGM'] || '';
    var defaultVolume = Number(parameters['DefaultVolume'] || 90);
    var defaultPitch = Number(parameters['DefaultPitch'] || 100);

    var bossBGMData = null;  // 保存Boss BGM数据
    var originalBGM = null;  // 保存进入战斗前的BGM

    // 解析事件页注释
    function parseBossBGMFromPage(page) {
        if (!page || !page.list) return null;
        for (var i = 0; i < page.list.length; i++) {
            var command = page.list[i];
            if (command && command.code === 108 && command.parameters && command.parameters[0]) {
                var comment = command.parameters[0];
                var match = comment.match(/<BossBGM:\s*(.+?)(?:\s*,\s*(\d+))?(?:\s*,\s*(\d+))?>/i);
                if (match) {
                    var bgmName = match[1].trim();
                    var volume = Number(match[2] || defaultVolume);
                    var pitch = Number(match[3] || defaultPitch);
                    return { name: bgmName, volume: volume, pitch: pitch };
                }
            }
        }
        return null;
    }

    // 遍历敌群所有事件页找BossBGM
    function findBossBGMInTroop(troop) {
        if (!troop || !troop.pages) return null;
        for (var i = 0; i < troop.pages.length; i++) {
            var bgmData = parseBossBGMFromPage(troop.pages[i]);
            if (bgmData) return bgmData;
        }
        return null;
    }

    // 拦截战斗处理（获取敌群ID并设置Boss BGM）
    var _Game_Interpreter_command301 = Game_Interpreter.prototype.command301;
    Game_Interpreter.prototype.command301 = function() {
        if (!$gameParty.inBattle()) {
            var troopId = this._params[1];  // 第二个参数是敌群ID
            if (troopId > 0) {
                var troop = $dataTroops[troopId];
                bossBGMData = findBossBGMInTroop(troop);
                if (bossBGMData || defaultBGM) {
                    // 保存进入战斗前的BGM
                    originalBGM = AudioManager.saveBgm();

                    // 立即播放Boss BGM
                    var bgm = bossBGMData || { name: defaultBGM, volume: defaultVolume, pitch: defaultPitch };
                    AudioManager.playBgm(bgm);
                }
            }
        }
        return _Game_Interpreter_command301.call(this);
    };

    // 拦截BattleManager.playBattleBgm，阻止默认敌群BGM覆盖
    var _BattleManager_playBattleBgm = BattleManager.playBattleBgm;
    BattleManager.playBattleBgm = function() {
        if (bossBGMData || defaultBGM) {
            // Boss战：不播放默认BGM，直接保持我们设置的Boss BGM
			if(bossBGMData){
				AudioManager.playBgm(bossBGMData);
			}
            return;
        }
        _BattleManager_playBattleBgm.call(this);
    };

    // 战斗结束恢复原BGM
    var _BattleManager_endBattle = BattleManager.endBattle;
    BattleManager.endBattle = function(result) {
        _BattleManager_endBattle.call(this, result);
        if (originalBGM) {
            AudioManager.playBgm(originalBGM);
            originalBGM = null;
            bossBGMData = null;
        }
    };

})();