/*:
 * @plugindesc 自定义选项菜单（动态适配宽度高度、专业美观、强制居中、完全健壮）- 终极稳定版（修复：选择后立即执行公共事件并退出）
 *
 * @param MenuOptions
 * @type text
 * @desc 选项配置 JSON，格式为 { "menuId": [选项数组] }
 * @default {"transform":[{"id":"novice_village","name":"新手村","commonEventId":1},{"id":"mondstadt","name":"蒙德城","commonEventId":2,"condition":"switch:1"},{"id":"inazuma","name":"稻妻","commonEventId":3,"condition":"variable:1>5"},{"id":"sumeru","name":"须弥","commonEventId":4,"condition":"switch:2"}]}

 * @command ShowCustomMenu
 * @text 显示自定义菜单
 * @desc 显示指定 ID 的自定义菜单
 * @arg menuId
 * @type text
 * @desc 菜单 ID（如 transform）
 * @default transform

 * @command HideOption
 * @text 隐藏选项
 * @desc 根据选项 ID 隐藏
 * @arg menuId
 * @type text
 * @arg optionId
 * @type text

 * @command ShowOption
 * @text 显示选项
 * @desc 根据选项 ID 显示
 * @arg menuId
 * @type text
 * @arg optionId
 * @type text
 */

(function() {
    // ==================== 参数与全局数据 ====================
    var parameters = PluginManager.parameters('CustomOptionMenu');
    var menuOptions = {};
    try {
        menuOptions = JSON.parse(parameters['MenuOptions'] || '{}');
    } catch (e) {
        console.error('CustomOptionMenu: JSON 解析错误', e);
        menuOptions = {};
    }

    var optionVisibility = {}; // 选项隐藏状态
    var menuTemp = { menuId: 'transform', page: 0 }; // 临时传参

    // ==================== 条件判断 ====================
    function evaluateCondition(condition) {
        if (!condition || condition.trim() === '') return true;

        if (condition.indexOf('switch:') === 0) {
            var id = parseInt(condition.substring(7));
            return !isNaN(id) && $gameSwitches.value(id);
        }

        if (condition.indexOf('variable:') === 0) {
            var expr = condition.substring(9).trim();
            var match = expr.match(/^(\d+)\s*([><=!]+)\s*(\d+)$/);
            if (!match) return false;
            var varId = parseInt(match[1]);
            var op = match[2].trim();
            var value = parseInt(match[3]);
            var current = $gameVariables.value(varId);

            switch (op) {
                case '>': return current > value;
                case '<': return current < value;
                case '>=': return current >= value;
                case '<=': return current <= value;
                case '==':
                case '===': return current === value;
            }
        }
        return false;
    }

    // ==================== 自定义窗口 ====================
    function Window_CustomMenu() {
        this.initialize.apply(this, arguments);
    }

    Window_CustomMenu.prototype = Object.create(Window_Selectable.prototype);
    Window_CustomMenu.prototype.constructor = Window_CustomMenu;

    Window_CustomMenu.prototype.initialize = function(x, y, width, height) {
        Window_Selectable.prototype.initialize.call(this, x, y, width, height);
        this._itemList = [];
        this.refresh();
        this.select(0);
        this.activate();
    };

    Window_CustomMenu.prototype.setMenuData = function(menuId, page) {
        menuTemp.menuId = menuId || 'transform';
        menuTemp.page = page || 0;
        this.refresh();
    };

    Window_CustomMenu.prototype.calculateMaxTextWidth = function() {
        var texts = [gl_translate('{返回}')];

        var menuConfig = menuOptions[menuTemp.menuId];
        if (menuConfig && Array.isArray(menuConfig) && menuConfig.length > 0) {
            var start = menuTemp.page * 6;
            var end = Math.min(start + 6, menuConfig.length);

            for (var i = start; i < end; i++) {
                var opt = menuConfig[i];
                if (opt && opt.name) {
                    var visible = optionVisibility[menuTemp.menuId] ?
                                  optionVisibility[menuTemp.menuId][opt.id] !== false : true;
                    if (visible && evaluateCondition(opt.condition)) {
                        texts.push(opt.name);
                    }
                }
            }

            if (menuConfig.length > 6) {
                texts.push(gl_translate('{下一页} →'));
                texts.push(gl_translate('← {上一页}'));
            }
        }

        var maxWidth = 0;
        for (var i = 0; i < texts.length; i++) {
            maxWidth = Math.max(maxWidth, this.textWidth(texts[i]));
        }
        return maxWidth;
    };

    Window_CustomMenu.prototype.buildItemList = function() {
        this._itemList = [];

        var menuConfig = menuOptions[menuTemp.menuId];
        if (!menuConfig || !Array.isArray(menuConfig) || menuConfig.length === 0) {
            this._itemList.push({ name: gl_translate('({无选项})'), type: 'empty', enabled: false });
            this._itemList.push({ name: gl_translate('{返回}'), type: 'return', enabled: true });
            return;
        }

        var start = menuTemp.page * 6;
        var end = Math.min(start + 6, menuConfig.length);

        for (var i = start; i < end; i++) {
            var opt = menuConfig[i];
            if (opt) {
                var visible = optionVisibility[menuTemp.menuId] ?
                              optionVisibility[menuTemp.menuId][opt.id] !== false : true;
                if (visible && evaluateCondition(opt.condition)) {
                    this._itemList.push({
                        name: gl_translate(opt.name) || gl_translate('{未知选项}'),
                        type: 'option',
                        enabled: true,
                        commonEventId: opt.commonEventId || 0
                    });
                }
            }
        }

        if (menuConfig.length > 6) {
            this._itemList.push({ name: gl_translate('{下一页} →'), type: 'nextPage', enabled: true });
            this._itemList.push({ name: gl_translate('← {上一页}'), type: 'prevPage', enabled: true });
        }

        this._itemList.push({ name: gl_translate('{返回}'), type: 'return', enabled: true });
    };

    Window_CustomMenu.prototype.refresh = function() {
        this.buildItemList();
        this.contents.clear();
        this.drawAllItems();
    };

    Window_CustomMenu.prototype.maxItems = function() {
        return this._itemList ? this._itemList.length : 1;
    };

    Window_CustomMenu.prototype.item = function(index) {
        return (this._itemList && index >= 0 && index < this._itemList.length) ? this._itemList[index] : null;
    };

    Window_CustomMenu.prototype.isCommandEnabled = function(index) {
        var item = this.item(index);
        return item ? item.enabled : false;
    };

    Window_CustomMenu.prototype.drawItem = function(index) {
        var item = this.item(index);
        if (!item) return;

        var rect = this.itemRect(index);
        this.changePaintOpacity(this.isCommandEnabled(index));
        this.drawText(gl_translate(item.name), rect.x + 24, rect.y, rect.width - 48, 'left');
        this.changePaintOpacity(true);
    };

    Window_CustomMenu.prototype.itemHeight = function() {
        return this.lineHeight();
    };

    // ==================== 场景 ====================
    function Scene_CustomMenu() {
        this.initialize.apply(this, arguments);
    }

    Scene_CustomMenu.prototype = Object.create(Scene_MenuBase.prototype);
    Scene_CustomMenu.prototype.constructor = Scene_CustomMenu;

    Scene_CustomMenu.prototype.create = function() {
        Scene_MenuBase.prototype.create.call(this);
        this.createCommandWindow();
    };

    Scene_CustomMenu.prototype.createCommandWindow = function() {
        var tempWindow = new Window_CustomMenu(0, 0, 240, 480);
        tempWindow.setMenuData(menuTemp.menuId, menuTemp.page);

        var contentWidth = tempWindow.calculateMaxTextWidth() || 120;
        var windowWidth = Math.max(240, contentWidth + 96);

        var itemCount = tempWindow._itemList ? tempWindow._itemList.length : 2;
        var windowHeight = Window_Base.prototype.fittingHeight(itemCount);

        var x = Math.floor((Graphics.boxWidth - windowWidth) / 2);
        var y = Math.floor((Graphics.boxHeight - windowHeight) / 2);

        this._commandWindow = new Window_CustomMenu(x, y, windowWidth, windowHeight);
        this._commandWindow.setHandler('ok', this.onItemOk.bind(this));
        this._commandWindow.setHandler('cancel', this.popScene.bind(this));
        this.addWindow(this._commandWindow);

        this._commandWindow.setMenuData(menuTemp.menuId, menuTemp.page);
    };

    // 关键修复：选择真实选项后执行公共事件并立即退出场景
    Scene_CustomMenu.prototype.onItemOk = function() {
        var index = this._commandWindow.index();
        var item = this._commandWindow.item(index);
        if (!item || !item.enabled) return;

        if (item.type === 'option') {
            // 执行公共事件
            if (item.commonEventId > 0) {
                $gameTemp.reserveCommonEvent(item.commonEventId);
            }
            // 立即关闭菜单（关键修复）
            this.popScene();
        } else if (item.type === 'nextPage') {
            var menuConfig = menuOptions[menuTemp.menuId];
            var total = menuConfig && Array.isArray(menuConfig) ? menuConfig.length : 0;
            var maxPage = total > 0 ? Math.ceil(total / 6) - 1 : 0;
            menuTemp.page = (menuTemp.page + 1) % (maxPage + 1);
            this.recreateWindow();
        } else if (item.type === 'prevPage') {
            var menuConfig = menuOptions[menuTemp.menuId];
            var total = menuConfig && Array.isArray(menuConfig) ? menuConfig.length : 0;
            var maxPage = total > 0 ? Math.ceil(total / 6) - 1 : 0;
            menuTemp.page = menuTemp.page <= 0 ? maxPage : menuTemp.page - 1;
            this.recreateWindow();
        } else if (item.type === 'return') {
            this.popScene();
        }
    };

    Scene_CustomMenu.prototype.recreateWindow = function() {
        if (this._commandWindow) {
            this.removeWindow(this._commandWindow);
        }
        this.createCommandWindow();
    };

    // ==================== 插件命令 ====================
    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);

        if (command === 'ShowCustomMenu') {
            menuTemp.menuId = args[0] || 'transform';
            menuTemp.page = 0;
            SceneManager.push(Scene_CustomMenu);
        } else if (command === 'HideOption') {
            var menuId = args[0];
            var optionId = args[1];
            if (menuId && optionId) {
                if (!optionVisibility[menuId]) optionVisibility[menuId] = {};
                optionVisibility[menuId][optionId] = false;
            }
        } else if (command === 'ShowOption') {
            var menuId = args[0];
            var optionId = args[1];
            if (menuId && optionId) {
                if (!optionVisibility[menuId]) optionVisibility[menuId] = {};
                optionVisibility[menuId][optionId] = true;
            }
        }
    };

})();