/*:
 * @plugindesc Delayed Function Executor (支持存在判断 + 自动/手动删除)
 *  (generated)
 * @help
 * 随机 minMs ~ maxMs 毫秒后执行函数。
 * 同一个 key 永远只对应一个任务（原子性保证）。
 *
 * 新增功能：
 * DelayedExecutor.has(key) → 返回 true/false，表示该 key 是否存在
 *
 * 使用：
 * DelayedExecutor.scheduleMs(key, minMs, maxMs, func, autoDelete);
 *   autoDelete 可选，默认 true（执行后自动删除 key）
 *
 * 判断是否存在：
 * if (DelayedExecutor.has("myKey")) { ... }
 *
 * 取消：
 * DelayedExecutor.cancel(key);
 * DelayedExecutor.cancelAll();
 */

var DelayedExecutor = DelayedExecutor || {};

(function() {
    'use strict';

    // 任务存储：key -> {frames: number, func: function, autoDelete: boolean}
    var _tasks = {};

    // 每帧更新
    var _SceneManager_update = SceneManager.update;
    SceneManager.update = function() {
        _SceneManager_update.call(this);
        DelayedExecutor._update();
    };

    // 安全更新：先收集到期任务，再统一处理
    DelayedExecutor._update = function() {
        var expired = [];

        for (var key in _tasks) {
            if (_tasks.hasOwnProperty(key)) {
                var task = _tasks[key];
                task.frames--;
                if (task.frames <= 0) {
                    expired.push({key: key, func: task.func, autoDelete: task.autoDelete});
                }
            }
        }

        // 统一执行并根据 autoDelete 决定是否删除 key
        for (var i = 0; i < expired.length; i++) {
            var item = expired[i];
            try {
                item.func();
            } catch (e) {
                console.error("DelayedExecutor error [" + item.key + "]:", e);
            }
            if (item.autoDelete !== false) {  // 默认 true 时删除
                delete _tasks[item.key];
            }
            // autoDelete === false 时保留 key（但不会再触发）
        }
    };

    /**
     * 毫秒级调度
     * @param {String|Number} key 唯一键
     * @param {Number} minMs 最小毫秒
     * @param {Number} maxMs 最大毫秒
     * @param {Function} func 执行函数
     * @param {Boolean} [autoDelete=true] 执行后是否自动删除 key（可选）
     */
    DelayedExecutor.scheduleMs = function(key, minMs, maxMs, func, autoDelete) {
        if (key == null || key === "") {
            console.warn("DelayedExecutor: key 不能为空");
            return;
        }
        if (typeof func !== "function") {
            console.warn("DelayedExecutor: func 必须是函数");
            return;
        }

        // key 已存在 → 直接忽略（保证唯一）
        if (_tasks.hasOwnProperty(key)) {
            return;
        }

        if (minMs > maxMs) {
            var t = minMs; minMs = maxMs; maxMs = t;
        }

        var delayMs = minMs + Math.random() * (maxMs - minMs);
        var frames = Math.max(1, Math.round(delayMs * 60 / 1000));

        if (autoDelete === undefined) autoDelete = true;

        _tasks[key] = {
            frames: frames,
            func: func,
            autoDelete: autoDelete
        };
    };

    // 兼容秒级接口
    DelayedExecutor.schedule = function(key, minSec, maxSec, func, autoDelete) {
        this.scheduleMs(key, minSec * 1000, maxSec * 1000, func, autoDelete);
    };

    /**
     * 判断指定 key 是否存在（正在等待执行的任务）
     * @param {String|Number} key
     * @returns {Boolean} true 表示存在，false 表示不存在
     */
    DelayedExecutor.has = function(key) {
        return key != null && _tasks.hasOwnProperty(key);
    };

    /**
     * 手动取消单个任务
     */
    DelayedExecutor.cancel = function(key) {
        if (key != null && _tasks.hasOwnProperty(key)) {
            delete _tasks[key];
        }
    };

    /**
     * 取消所有任务
     */
    DelayedExecutor.cancelAll = function() {
        _tasks = {};
    };

    // 调试用：当前任务数量
    DelayedExecutor.getTaskCount = function() {
        return Object.keys(_tasks).length;
    };

})();