/*:
 * @plugindesc Delivery Task System (生产者-顾客运送系统) v4.5 - 调试增强版
 * @author Grok
 *
 * @param Debug Mode
 * @text 调试模式
 * @type boolean
 * @on 开启
 * @off 关闭
 * @default true   // 建议调试时打开，正式发布改成 false
 *
 * @param Demand Display Mode
 * @text 顾客需求显示模式
 * @type select
 * @option Event Name (事件名)
 * @value name
 * @option Sprite Icon (物品图标)
 * @value icon
 * @default name
 *
 * @help
 * === v4.5 更新内容 ===
 * - 按确定键时，仅对玩家正前方的 <producer>（已完成）或 <customer>（有需求）强制触发当前满足条件的页面
 * - 其他事件完全走原版逻辑
 * - 增加大量关键调试日志（仅在 Debug Mode = true 时输出）
 */

(function() {

    var parameters = PluginManager.parameters('DeliveryTaskSystem');
    var DEBUG = parameters['Debug Mode'] === 'true';
    var DISPLAY_MODE = parameters['Demand Display Mode'] || 'name';

    function log(msg) {
        if (DEBUG) console.log("[DeliveryTaskSystem] " + msg);
    }

    // 初始化日志
    log("插件加载完成 | Debug Mode: " + DEBUG + " | Display Mode: " + DISPLAY_MODE);

    var DemandIconCache = {};

    var DeliveryTemp = {
        active: false,
        tasks: {},
        allProducerIds: [],
        customerNeeds: {},
        currentCarry: null,
        score: 0
    };

    // ==================== 工具方法 ====================

    function getFramesFromNote(note) {
        var match = note.match(/<frames:\s*(\d+)>/i);
        return match ? Number(match[1]) : 1200;
    }

    function clearDemandIconCache() {
        //Object.keys(DemandIconCache).forEach(function(key) {
        //    if (DemandIconCache[key]) DemandIconCache[key].destroy();
        //});
        DemandIconCache = {};
        log("需求图标缓存已清除");
    }

    function resetAllSelfSwitches() {
        var events = $gameMap.events();
        for (var i = 0; i < events.length; i++) {
            var ev = events[i];
            if (ev) {
                ['A','B','C','D'].forEach(function(sw) {
                    $gameSelfSwitches.setValue([$gameMap.mapId(), ev.eventId(), sw], false);
                });
            }
        }
        DeliveryTemp.allProducerIds = [];
        DeliveryTemp.customerNeeds = {};
        DeliveryTemp.currentCarry = null;
        clearDemandIconCache();
        log("所有自开关重置，任务数据清空");
    }

    function registerAllProducers() {
        DeliveryTemp.tasks = {};
        DeliveryTemp.allProducerIds = [];
        var events = $gameMap.events();
        var count = 0;
        for (var i = 0; i < events.length; i++) {
            var ev = events[i];
            if (ev && ev.event()) {
                var note = ev.event().note.toLowerCase();
                if (note.contains('<producer>')) {
                    var frames = getFramesFromNote(ev.event().note);
                    var evId = ev.eventId();
                    var producerName = ev.event().name || "物品";
                    DeliveryTemp.tasks[evId] = {
                        framesNeeded: frames,
                        currentFrames: 0,
                        name: producerName
                    };
                    DeliveryTemp.allProducerIds.push(evId);
                    count++;
                    log("注册生产者 ID:" + evId + " 名称:" + producerName + " 需要帧数:" + frames);
                }
            }
        }
        log("生产者注册完成，共 " + count + " 个");
    }

    function createOrUpdateProgressBar(sp, task) {
        var bar = sp.progressBar || (function() {
            var b = new Sprite();
            b.bitmap = new Bitmap(40, 8);
            b.anchor.x = 0.5;
            b.anchor.y = 1;
            this._tilemap.addChild(b);
            sp.progressBar = b;
            return b;
        }.call(this));

        bar.x = sp.x;
        bar.y = sp.y - sp.patternHeight() - 5;

        var progress = Math.min(task.currentFrames / task.framesNeeded, 1);
        bar.bitmap.clear();
        bar.bitmap.fillRect(0, 0, 40, 8, '#000000');
        bar.bitmap.fillRect(2, 2, 36, 4, '#333333');
        bar.bitmap.fillRect(2, 2, Math.floor(36 * progress), 4, '#00ff88');
        bar.visible = true;
    }

    function createCachedDemandIcon(producerSprite, producerId) {
        var cacheKey = 'demand_icon_' + producerId;
        if (DemandIconCache[cacheKey]) return DemandIconCache[cacheKey];

        if (producerSprite && producerSprite.bitmap && producerSprite.bitmap.isReady()) {
            var cacheBitmap = new Bitmap(48, 48);
            cacheBitmap.blt(producerSprite.bitmap, producerSprite._frame.x, producerSprite._frame.y, 48, 48, 0, 0);
            DemandIconCache[cacheKey] = cacheBitmap;
            log("缓存顾客需求图标 - Producer ID: " + producerId);
            return cacheBitmap;
        }
        return null;
    }

    function showCustomerDemand(sp, producerTask, producerSprite, producerId) {
        if (DISPLAY_MODE === 'name') {
            var textSprite = sp.headText || (function() {
                var ts = new Sprite();
                ts.bitmap = new Bitmap(160, 36);
                ts.anchor.x = 0.5;
                ts.anchor.y = 1;
                this._tilemap.addChild(ts);
                sp.headText = ts;
                return ts;
            }.call(this));

            textSprite.bitmap.clear();
            textSprite.bitmap.fontSize = 26;
            textSprite.bitmap.outlineWidth = 5;
            textSprite.bitmap.outlineColor = 'black';
            textSprite.bitmap.textColor = '#ffffff';
            textSprite.bitmap.drawText(producerTask.name, 0, 0, 160, 36, 'center');

            textSprite.x = sp.x;
            textSprite.y = sp.y - sp.patternHeight() - 6;
            textSprite.visible = true;

        } else if (DISPLAY_MODE === 'icon') {
            var cachedBitmap = createCachedDemandIcon(producerSprite, producerId);
            if (cachedBitmap) {
                if (sp.headIcon) this._tilemap.removeChild(sp.headIcon);

                var icon = new Sprite(cachedBitmap);
                icon.scale.x = icon.scale.y = 0.6;
                icon.anchor.x = 0.5;
                icon.anchor.y = 1;
                this._tilemap.addChild(icon);
                sp.headIcon = icon;

                icon.x = sp.x;
                icon.y = sp.y - sp.patternHeight() - 4;
                icon.visible = true;
            }
        }
    }

    function hideDemandDisplay(sp) {
        if (sp.headText) sp.headText.visible = false;
        if (sp.headIcon) {
            this._tilemap.removeChild(sp.headIcon);
            sp.headIcon = null;
        }
    }

    // ==================== 左上角携带物品HUD ====================

    function Window_DeliveryCarryHUD() {
        this.initialize.apply(this, arguments);
    }

    Window_DeliveryCarryHUD.prototype = Object.create(Window_Base.prototype);
    Window_DeliveryCarryHUD.prototype.constructor = Window_DeliveryCarryHUD;

    Window_DeliveryCarryHUD.prototype.initialize = function() {
        var width = 300;
        var height = 160;
        var x = 10;
        var y = 10;
        Window_Base.prototype.initialize.call(this, x, y, width, height);
        this.opacity = 180;
        this.refresh();
    };

    Window_DeliveryCarryHUD.prototype.refresh = function() {
        this.contents.clear();

        var actor = $gameParty.leader();
        if (actor) {
            this.drawActorFace(actor, 10, 10);
        }

        var itemX = 170;
        var itemY = 48;

        var hasItem = !!DeliveryTemp.currentCarry;

        var borderColor = hasItem ? '#000000' : '#ff0000';

        this.contents.fillRect(itemX - 4, itemY - 4, 72, 72, borderColor);
        this.contents.fillRect(itemX - 2, itemY - 2, 68, 68, '#ffffff');
        this.contents.fillRect(itemX, itemY, 64, 64, '#888888');

        if (hasItem) {
            var producerId = DeliveryTemp.currentCarry.producerId;
            var producerEvent = $gameMap.event(producerId);
            if (producerEvent) {
                var characterSprite = SceneManager._scene._spriteset.findCharacterSprite(producerEvent);
                if (characterSprite && characterSprite.bitmap && characterSprite.bitmap.isReady()) {
                    this.contents.blt(
                        characterSprite.bitmap,
                        characterSprite._frame.x,
                        characterSprite._frame.y,
                        48, 48,
                        itemX + 8, itemY + 8
                    );
                }
            }
        }
    };

    Spriteset_Map.prototype.findCharacterSprite = function(character) {
        return this._characterSprites.find(function(sprite) {
            return sprite._character === character;
        });
    };

    // ==================== 主逻辑 ====================

    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command === 'DeliveryTask') {
            if (args[0] === 'start') {
                DeliveryTemp.active = true;
                resetAllSelfSwitches();
                registerAllProducers();
                if (SceneManager._scene instanceof Scene_Map) {
                    SceneManager._scene.createDeliveryCarryHUD();
                }
                log("=== 任务系统启动 ===");
            } else if (args[0] === 'stop') {
                DeliveryTemp.active = false;
                DeliveryTemp.tasks = {};
                resetAllSelfSwitches();
                if (SceneManager._scene instanceof Scene_Map && SceneManager._scene._deliveryCarryHUD) {
                    SceneManager._scene.removeChild(SceneManager._scene._deliveryCarryHUD);
                    SceneManager._scene._deliveryCarryHUD = null;
                }
                log("=== 任务系统停止 ===");
            }
        }
    };

    var _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        if (DeliveryTemp.active) {
            this.createDeliveryCarryHUD();
        }
    };

    Scene_Map.prototype.createDeliveryCarryHUD = function() {
        if (this._deliveryCarryHUD) {
            this.removeChild(this._deliveryCarryHUD);
        }
        this._deliveryCarryHUD = new Window_DeliveryCarryHUD();
        this.addChild(this._deliveryCarryHUD);
        log("左上角携带HUD已创建");
    };

    var _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (this._deliveryCarryHUD) {
            this._deliveryCarryHUD.refresh();
        }
    };

    var _Spriteset_Map_update = Spriteset_Map.prototype.update;
    Spriteset_Map.prototype.update = function() {
        _Spriteset_Map_update.call(this);
        if (DeliveryTemp.active) {
            this.updateDeliveryTasks();
            this.updateDeliveryVisuals();
        }
    };

    Spriteset_Map.prototype.updateDeliveryTasks = function() {
        var events = $gameMap.events();

        // 生产者进度更新
        for (var i = 0; i < events.length; i++) {
            var ev = events[i];
            if (!ev || !ev.event()) continue;
            var evId = ev.eventId();
            var task = DeliveryTemp.tasks[evId];
            if (!task) continue;
			if(task._isFinished == undefined || task._isFinished == null)
			{
				task._isFinished = false;
			}
            var allOff = ['A','B','C','D'].every(function(sw) {
                return !$gameSelfSwitches.value([$gameMap.mapId(), evId, sw]);
            });

            if (allOff && !task._isFinished) {
                task.currentFrames++;
                if (task.currentFrames >= task.framesNeeded) {
					task._isFinished = true;
                    $gameSelfSwitches.setValue([$gameMap.mapId(), evId, 'A'], true);
                    log("生产者 ID:" + evId + " 生产完成！自开关A开启");
                }
            }
        }

        // 顾客需求生成
        for (var i = 0; i < events.length; i++) {
            var ev = events[i];
            if (!ev || !ev.event()) continue;
            var evId = ev.eventId();
            var note = ev.event().note.toLowerCase();
            if (!note.contains('<customer>')) continue;

            var allOff = ['A','B','C','D'].every(function(sw) {
                return !$gameSelfSwitches.value([$gameMap.mapId(), evId, sw]);
            });

            if (allOff && DeliveryTemp.allProducerIds.length > 0) {
                var randomId = DeliveryTemp.allProducerIds[Math.randomInt(DeliveryTemp.allProducerIds.length)];
                DeliveryTemp.customerNeeds[evId] = { producerId: randomId };

                $gameSelfSwitches.setValue([$gameMap.mapId(), evId, 'B'], true);
                ['A','C','D'].forEach(function(sw) {
                    $gameSelfSwitches.setValue([$gameMap.mapId(), evId, sw], false);
                });
                log("顾客 ID:" + evId + " 生成需求 → 需要生产者 ID:" + randomId);
            }
        }
    };

    Spriteset_Map.prototype.updateDeliveryVisuals = function() {
        var characterSprites = this._characterSprites || [];
        for (var i = 0; i < characterSprites.length; i++) {
            var sp = characterSprites[i];
            if (!sp._character || !sp._character.eventId) continue;
            var evId = sp._character.eventId();
            var ev = $gameMap.event(evId);
            if (!ev) continue;
            var note = ev.event().note.toLowerCase();
            var isProducer = note.contains('<producer>');
            var isCustomer = note.contains('<customer>');

            if (isProducer && DeliveryTemp.tasks[evId]) {
                createOrUpdateProgressBar.call(this, sp, DeliveryTemp.tasks[evId]);

                if ($gameSelfSwitches.value([$gameMap.mapId(), evId, 'A'])) {
                    var pulse = 0.05 * Math.sin(Graphics.frameCount / 15);
                    sp.scale.x = 1 + pulse;
                    sp.scale.y = 1 + pulse;
                } else {
                    sp.scale.x = sp.scale.y = 1;
                }
            }

            if (isCustomer) {
                // 新增：如果顾客自开关 A 已开启（通常表示已交付完成），强制隐藏需求显示
                var switchAOn = $gameSelfSwitches.value([$gameMap.mapId(), evId, 'A']);
                if (switchAOn) {
                    hideDemandDisplay.call(this, sp);
                    log("顾客 ID:" + evId + " 自开关A已开启，强制隐藏需求显示");
                    continue; // 跳过后续显示逻辑
                }

                // 原有逻辑：只有在有需求且开关A未开启时才显示
                if (DeliveryTemp.customerNeeds[evId]) {
                    var need = DeliveryTemp.customerNeeds[evId];
                    var producerId = need.producerId;
                    var producerTask = DeliveryTemp.tasks[producerId];
                    if (!producerTask) {
                        hideDemandDisplay.call(this, sp);
                        continue;
                    }

                    var producerSprite = characterSprites.find(function(s) {
                        return s._character && s._character.eventId && s._character.eventId() === producerId;
                    });

                    showCustomerDemand.call(this, sp, producerTask, producerSprite, producerId);
                } else {
                    hideDemandDisplay.call(this, sp);
                }
            }
        }
    };

    // ==================== 关键功能：玩家正前方强制交互 ====================

    var _Game_Player_checkEventTriggerThere = Game_Player.prototype.checkEventTriggerThere;

    Game_Player.prototype.checkEventTriggerThere = function(triggers) {
        if ($gameMap.isEventRunning()) {
            return;
        }

        var direction = this.direction();
        var x2 = $gameMap.roundXWithDirection(this.x, direction);
        var y2 = $gameMap.roundYWithDirection(this.y, direction);
        var events = $gameMap.eventsXy(x2, y2);
        var event = events[0]; // 取最上层事件

        if (event) {
            var evId = event.eventId();
            var note = event.event().note.toLowerCase();
            var isProducer = note.contains('<producer>');
            var isCustomer = note.contains('<customer>');

            log("玩家按确定键 → 正前方检测到事件 ID:" + evId + " | 类型: " + (isProducer ? "Producer" : isCustomer ? "Customer" : "普通事件"));

            var shouldForce = false;

            if (isProducer && $gameSelfSwitches.value([$gameMap.mapId(), evId, 'A'])) {
                shouldForce = true;
                log("→ 生产者已完成，强制触发当前页面");
            } else if (isCustomer && $gameSelfSwitches.value([$gameMap.mapId(), evId, 'B'])) {
                shouldForce = true;
                log("→ 顾客有需求，强制触发当前页面");
            }

            if (shouldForce) {
                event.start();  // 正常检查页面条件，执行当前可运行页面
                return;
            } else {
                log("→ 非强制目标，走原版触发逻辑");
            }
        } else {
            log("玩家按确定键 → 正前方无事件");
        }

        // 非特殊目标或不满足条件 → 走原版逻辑
        _Game_Player_checkEventTriggerThere.call(this, triggers);
    };

    // ==================== 辅助方法 ====================

    Game_System.prototype.isProducerCompleted = function(eventId) {
        if (!DeliveryTemp.active || !DeliveryTemp.tasks[eventId]) {
            return false;
        }
        var task = DeliveryTemp.tasks[eventId];
        var switchA = $gameSelfSwitches.value([$gameMap.mapId(), eventId, 'A']);
        return switchA && task.currentFrames >= task.framesNeeded;
    };

    Game_System.prototype.setDeliveryCarry = function(producerId) {
        if (producerId === 0) producerId = null;
        DeliveryTemp.currentCarry = producerId ? { producerId: producerId } : null;
        
        if (SceneManager._scene instanceof Scene_Map && SceneManager._scene._deliveryCarryHUD) {
            SceneManager._scene._deliveryCarryHUD.refresh();
        }
        
        log("玩家携带物品更新 → Producer ID: " + (producerId || "无"));
    };

	Game_System.prototype.resetDeliveryCarry = function(producerId) {
        if (producerId === 0) producerId = null;	
        //log("玩家携带物品更新 → Producer ID: " + (producerId || "无"));
		var task = DeliveryTemp.tasks[producerId];
		if(task == null || task == undefined)
		{
			return;
		}
		if(true == task._isFinished)
		{
			task._isFinished = false;
			task.currentFrames = 0;
			log("玩家携带rest → Producer ID: " + (producerId || "无"));
		}
    };

	 // ==================== 新增：携带物品判断与清空方法 ====================

    /**
     * 判断玩家当前携带的物品是否满足指定顾客的需求
     * @param {number} customerEventId - 顾客的事件ID
     * @returns {boolean} 如果携带的正是该顾客所需的生产者物品，返回 true；否则 false
     */
    Game_System.prototype.isCarryingCorrectItemForCustomer = function(customerEventId) {
        if (!DeliveryTemp.active) {
            log("任务系统未激活，无法判断携带物品");
            return false;
        }

        var need = DeliveryTemp.customerNeeds[customerEventId];
        if (!need || !need.producerId) {
            log("顾客 ID:" + customerEventId + " 当前无需求");
            return false;
        }

        var carrying = DeliveryTemp.currentCarry;
        if (!carrying || !carrying.producerId) {
            log("玩家当前未携带任何物品");
            return false;
        }

        var isMatch = carrying.producerId === need.producerId;
        log("判断携带物品是否匹配顾客 ID:" + customerEventId + 
            " | 所需生产者:" + need.producerId + 
            " | 当前携带:" + carrying.producerId + 
            " | 结果: " + (isMatch ? "匹配" : "不匹配"));

        return isMatch;
    };

})();