/*:
 * @target MV
 * @plugindesc [v3.3.1] 合金装备式守卫扇形视野插件（完整扇形显示 + Bresenham 精准 Region 阻挡判定）
 *
 * @param DefaultAngle
 * @text 默认视野角度
 * @default 90
 * @type number
 *
 * @param DefaultDistance
 * @text 默认视野距离
 * @default 5
 * @type number
 *
 * @param DefaultLostSeconds
 * @text 默认脱离警戒秒数
 * @default 3
 * @type number
 *
 * @param DefaultTouchRange
 * @text 默认触碰距离
 * @default 1.2
 * @type number
 *
 * @param DefaultCommonEventId
 * @text 默认公共事件ID
 * @default 0
 * @type number
 *
 * @param DefaultReturnOnLost
 * @text 默认警戒解除后返回原位
 * @default false
 * @type boolean
 *
 * @param BlockRegionId
 * @text 阻挡视野的 Region ID
 * @desc 守卫视线无法穿过的 Region ID（0 = 不启用阻挡）
 * @default 255
 * @type number
 * @min 0
 * @max 255
 *
 * @param DefaultNormalColor
 * @text 默认正常视野颜色
 * @default #00ff00
 * @type string
 *
 * @param DefaultNormalA
 * @text 默认正常视野透明度
 * @default 100
 * @type number
 * @min 0
 * @max 255
 *
 * @help GuardFov_AlertOnly.js
 *
 * v3.3.1 更新：
 * - 警戒判定使用标准 Bresenham 算法，确保不遗漏任何路径格子
 * - 解决近距离、对角线方向阻挡失效的问题
 * - 视野扇形始终完整显示
 * - 路径上任意格子为 BlockRegionId 即阻挡警戒
 */

(function() {
    'use strict';

    var parameters = PluginManager.parameters('GuardFov_AlertOnly');
    var defAngle = Number(parameters['DefaultAngle']) || 90;
    var defDist = Number(parameters['DefaultDistance']) || 5;
    var defLost = Number(parameters['DefaultLostSeconds']) || 3;
    var defTouch = Number(parameters['DefaultTouchRange']) || 1.2;
    var defCommon = Number(parameters['DefaultCommonEventId']) || 0;
    var defReturn = parameters['DefaultReturnOnLost'] === 'true';
    var blockRegionId = Number(parameters['BlockRegionId']) || 255;
    var defNormalColor = parameters['DefaultNormalColor'] || '#00ff00';
    var defNormalA = Number(parameters['DefaultNormalA']) || 100;

    var NORMAL_OPACITY = 100;
    var BATTLE_OPACITY = 0;

    // ==================================================
    // Game_Map 管理视野精灵（保持原样）
    // ==================================================
    var _Game_Map_initialize = Game_Map.prototype.initialize;
    Game_Map.prototype.initialize = function() {
        _Game_Map_initialize.call(this);
        this._guardVisionSprites = [];
    };

    Game_Map.prototype.addGuardVisionSprite = function(sprite) {
        if (sprite && !this._guardVisionSprites.includes(sprite)) {
            this._guardVisionSprites.push(sprite);
        }
    };

    Game_Map.prototype.clearGuardVisionSprites = function() {
        this._guardVisionSprites = [];
    };

    Game_Map.prototype.recreateAllGuardVisions = function() {
        this.clearGuardVisionSprites();
        this.events().forEach(function(event) {
            if (event && event.page()) {
                event._guardParsed = false;
                event.parseGuardConfig();
                event._guardParsed = true;
                if (event._guardData) {
                    $gameSelfSwitches.setValue([event._mapId, event._eventId, 'A'], false);
                    $gameSelfSwitches.setValue([event._mapId, event._eventId, 'B'], false);
                    $gameSelfSwitches.setValue([event._mapId, event._eventId, 'C'], false);
                    $gameSelfSwitches.setValue([event._mapId, event._eventId, 'D'], false);
                    event.createVisionSprite();
                }
            }
        });
    };

    var _BattleManager_startBattle = BattleManager.startBattle;
    BattleManager.startBattle = function() {
        if ($gameMap._guardVisionSprites) {
            $gameMap._guardVisionSprites.forEach(function(sprite) {
                if (sprite) sprite.opacity = BATTLE_OPACITY;
            });
        }
        _BattleManager_startBattle.call(this);
    };

    var _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
    Scene_Map.prototype.createDisplayObjects = function() {
        _Scene_Map_createDisplayObjects.call(this);
        $gameMap.recreateAllGuardVisions();
    };

    var _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
    Scene_Map.prototype.onMapLoaded = function() {
        _Scene_Map_onMapLoaded.call(this);
        $gameMap.recreateAllGuardVisions();
    };

    // ==================================================
    // 事件配置解析
    // ==================================================
    Game_Event.prototype.extractPageComments = function() {
        var page = this.page();
        if (!page || !page.list) return '';
        var comments = [];
        for (var i = 0; i < page.list.length; i++) {
            var cmd = page.list[i];
            if (cmd.code === 108 || cmd.code === 408) {
                comments.push(cmd.parameters[0]);
            } else if (comments.length > 0) {
                break;
            }
        }
        return comments.join('\n');
    };

    Game_Event.prototype.parseGuardConfig = function() {
        if (this._visionSprite && this._visionSprite.parent) {
            this._visionSprite.parent.removeChild(this._visionSprite);
            this._visionSprite = null;
        }
        this._guardData = null;

        var note = this.extractPageComments();
        if (!note) return;

        var guardMatch = note.match(/<guard\b([^>]*)>/i);
        if (!guardMatch) return;

        var configStr = guardMatch[1].trim();

        var angle = defAngle;
        var dist = defDist;
        var lostTime = defLost;
        var commonId = defCommon;
        var touchRange = defTouch;
        var returnOnLost = defReturn;
        var normalColor = defNormalColor;
        var normalA = defNormalA;

        if (configStr) {
            var regex = /(\w+):\s*([^,\s]+)/gi;
            var match;
            while ((match = regex.exec(configStr)) !== null) {
                var key = match[1].toLowerCase();
                var value = isNaN(match[2]) ? match[2].toLowerCase() : Number(match[2]);
                switch (key) {
                    case 'angle': angle = Number(value); break;
                    case 'dist': case 'distance': dist = Number(value); break;
                    case 'lost': case 'losttime': case 'lostseconds': lostTime = Number(value); break;
                    case 'common': case 'event': case 'commonid': commonId = Number(value); break;
                    case 'touch': case 'touchrange': touchRange = Number(value); break;
                    case 'return': returnOnLost = (value === true || value === 'true' || value === 1); break;
                    case 'normalcolor': normalColor = value || defNormalColor; break;
                    case 'normala': normalA = Number(value); break;
                }
            }
        }

        this._guardData = {
            angle: angle,
            dist: dist,
            lostTime: lostTime,
            commonId: commonId,
            touchRange: touchRange,
            returnOnLost: returnOnLost,
            normalColor: normalColor,
            normalA: normalA
        };

        if (this._guardData.returnOnLost && !this._originalPatrol) {
            var page = this.page();
            this._originalPatrol = {
                moveRoute: page.moveRoute ? JsonEx.makeDeepCopy(page.moveRoute) : null,
                moveSpeed: this._moveSpeed,
                moveFrequency: this._moveFrequency,
                startX: this.x,
                startY: this.y,
                startDir: this.direction()
            };
        }

        this._guardLostFrames = 0;
        this._isAlert = false;
    };

    var _Game_Map_setupEvents = Game_Map.prototype.setupEvents;
    Game_Map.prototype.setupEvents = function() {
        _Game_Map_setupEvents.call(this);
        this.events().forEach(function(event) {
            if (event && !event._guardParsed) {
                event.parseGuardConfig();
                event._guardParsed = true;
            }
        });
    };

    var _Game_Event_update = Game_Event.prototype.update;
    Game_Event.prototype.update = function(sceneActive) {
        _Game_Event_update.call(this, sceneActive);
        if (this._guardData) {
            this.createVisionSprite();
            this.updateGuardAlert();
        }
    };

    // ==================================================
    // 精准 Bresenham 直线算法判定（不遗漏任何格子）
    // ==================================================
    Game_Event.prototype.isInSight = function(character) {
        var sx = this.x;
        var sy = this.y;
        var ex = character.x;
        var ey = character.y;

        var distance = Math.hypot(ex - sx, ey - sy);
        if (distance > this._guardData.dist) return false;

        // 角度判断
        var playerAngle = Math.atan2(ey - sy, ex - sx) * 180 / Math.PI;
        if (playerAngle < 0) playerAngle += 360;
        var dirAngle = this.directionToAngle();
        var half = this._guardData.angle / 2;
        var diff = Math.abs(playerAngle - dirAngle);
        if (diff > 180) diff = 360 - diff;
        if (diff > half) return false;

        // 无阻挡或极近
        if (blockRegionId === 0 || distance <= 1.0) return true;

        // Bresenham 算法
        var dx = Math.abs(ex - sx);
        var dy = -Math.abs(ey - sy);
        var stepX = (sx < ex) ? 1 : -1;
        var stepY = (sy < ey) ? 1 : -1;
        var err = dx + dy;

        var x = sx;
        var y = sy;

        var loopCount = 0;
        var maxLoops = 10000;

        while (true) {
            loopCount++;
            if (loopCount > maxLoops) return false;

            if (x === ex && y === ey) break;

            var e2 = 2 * err;
            if (e2 >= dy) {
                err += dy;
                x += stepX;
            }
            if (e2 <= dx) {
                err += dx;
                y += stepY;
            }

            // 检查当前步进后的格子（中间格子）
            if ($gameMap.regionId(x, y) === blockRegionId) {
                return false;
            }
        }

        return true;
    };

    Game_Event.prototype.directionToAngle = function() {
        switch (this.direction()) {
            case 2: return 90;
            case 4: return 180;
            case 6: return 0;
            case 8: return 270;
        }
        return 0;
    };

    Game_Event.prototype.directionToRotation = function() {
        switch (this.direction()) {
            case 2: return Math.PI / 2;
            case 4: return Math.PI;
            case 6: return 0;
            case 8: return -Math.PI / 2;
        }
        return 0;
    };

    Game_Event.prototype.updateGuardAlert = function() {
        var player = $gamePlayer;
        var inSight = this.isInSight(player);

        if (inSight) {
            this._guardLostFrames = 0;
            if (!this._isAlert) {
                this._isAlert = true;
                $gameSelfSwitches.setValue([this._mapId, this._eventId, 'A'], true);
            }
        } else {
            this._guardLostFrames++;
            if (this._isAlert && this._guardLostFrames >= this._guardData.lostTime * 60) {
                this._isAlert = false;
                $gameSelfSwitches.setValue([this._mapId, this._eventId, 'A'], false);
                $gameSelfSwitches.setValue([this._mapId, this._eventId, 'B'], false);
                $gameSelfSwitches.setValue([this._mapId, this._eventId, 'C'], false);
                $gameSelfSwitches.setValue([this._mapId, this._eventId, 'D'], false);

                if (this._guardData.returnOnLost && this._originalPatrol) {
                    this.locate(this._originalPatrol.startX, this._originalPatrol.startY);
                    this.setDirection(this._originalPatrol.startDir);
                    this._moveSpeed = this._originalPatrol.moveSpeed;
                    this.setMoveFrequency(this._originalPatrol.moveFrequency);
                    if (this._originalPatrol.moveRoute) {
                        this.setMoveRoute(JsonEx.makeDeepCopy(this._originalPatrol.moveRoute));
                        this._moveRouteForcing = false;
                    }
                }
            }
        }

        this.checkTouch();
    };

    Game_Event.prototype.checkTouch = function() {
        if (this._guardData.commonId <= 0) return;
        var dist = Math.hypot($gamePlayer.x - this.x, $gamePlayer.y - this.y);
        if (dist <= this._guardData.touchRange) {
            $gameTemp.reserveCommonEvent(this._guardData.commonId);
        }
    };

    Game_Event.prototype.isGuardAlert = function() {
        return this._isAlert;
    };

    Game_Event.prototype.createVisionSprite = function() {
        if (this._visionSprite) {
            $gameMap.addGuardVisionSprite(this._visionSprite);
            return;
        }
        var sprite = new Sprite_Vision(this);
        this._visionSprite = sprite;
        $gameMap.addGuardVisionSprite(sprite);

        var scene = SceneManager._scene;
        if (scene instanceof Scene_Map && scene._spriteset) {
            scene._spriteset._tilemap.addChild(sprite);
        }
    };

    function Sprite_Vision() { this.initialize.apply(this, arguments); }
    Sprite_Vision.prototype = Object.create(Sprite.prototype);
    Sprite_Vision.prototype.constructor = Sprite_Vision;

    Sprite_Vision.prototype.initialize = function(character) {
        Sprite.prototype.initialize.call(this);
        this._character = character;
        this.z = 1;
        this.anchor.x = 0.5;
        this.anchor.y = 0.5;
        this.opacity = NORMAL_OPACITY;
        this.bitmap = new Bitmap(512, 512);
    };

    Sprite_Vision.prototype.update = function() {
        Sprite.prototype.update.call(this);
        if (!this._character._guardData) {
            this.visible = false;
            return;
        }
        this.visible = !this._character._erased;

        this.x = this._character.screenX();
        this.y = this._character.screenY() - 24;
        this.rotation = this._character.directionToRotation();

        var data = this._character._guardData;
        var al = this._character.isGuardAlert() ? 100 : data.normalA;
        var color = this._character.isGuardAlert() ? '#ff0000' : data.normalColor;

        var radius = data.dist * 48;
        var baseAngle = this._character.directionToAngle();
        var halfAngle = data.angle / 2;

        this.bitmap.clear();
        this.bitmap.fillFan(256, 256, radius, -halfAngle, halfAngle, color, al);
    };

    Bitmap.prototype.fillFan = function(cx, cy, radius, angleFrom, angleTo, color, alpha) {
        if (angleTo - angleFrom <= 0.01) return;
        var ctx = this._context;
        ctx.save();
        ctx.globalAlpha = alpha / 255;
        ctx.fillStyle = color;
        ctx.translate(cx, cy);
        ctx.rotate(angleFrom * Math.PI / 180);
        ctx.beginPath();
        ctx.moveTo(0, 0);
        ctx.arc(0, 0, radius, 0, (angleTo - angleFrom) * Math.PI / 180, false);
        ctx.lineTo(0, 0);
        ctx.closePath();
        ctx.fill();
        ctx.restore();
        this._setDirty();
    };

    // ==================================================
    // 存档清理
    // ==================================================
    var _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function() {
        if ($gameMap && $gameMap._guardVisionSprites) {
            $gameMap._guardVisionSprites.forEach(function(sprite) {
                if (sprite && sprite.parent) sprite.parent.removeChild(sprite);
            });
            $gameMap.clearGuardVisionSprites();
        }

        if ($gameMap) {
            $gameMap.events().forEach(function(event) {
                if (event) {
                    event._guardParsed = false;
                    if (event._visionSprite) event._visionSprite = null;
                }
            });
        }

        var contents = _DataManager_makeSaveContents.call(this);
        return contents;
    };

})();