/*:
 * @target MV
 * @plugindesc 护盾系统 - 支持元素类型、抵抗/弱点、日志显示元素名称、新护盾替换提示
 * @author 原作者 + 修改
 * 
 * @param Shield State ID
 * @text 护盾状态ID
 * @type number
 * @default 50
 *
 * @param Default Shield Turns
 * @text 默认护盾持续回合
 * @type number
 * @default 5
 *
 * @param Default Shield CD
 * @text 默认护盾技能CD回合
 * @type number
 * @default 3
 *
 * @param Break Shield SE
 * @text 破盾音效
 * @desc 护盾被完全打破时播放的SE文件名（不带路径和.ogg）。留空则不播放。
 * @default Crash
 *
 * @param Absorb Format
 * @text 吸收提示格式
 * @desc %1=名字 %2=吸收值 %3=剩余护盾 %4=元素名（空则无）
 * @default %1 的%4护盾吸收了 %2 点伤害（剩余 %3 点护盾）
 *
 * @param Break Format
 * @text 破盾提示格式
 * @default %1 的%4护盾被破坏！（吸收 %2 点伤害）
 *
 * @param End Format
 * @text 护盾自然结束提示
 * @default %1 的护盾结束！
 *
 * @param Start Format
 * @text 上护盾提示（无元素）
 * @default %1 获得 %2 护盾，持续%3回合
 *
 * @param Start Format With Element
 * @text 上护盾提示（有元素）
 * @default %1 获得 %2 护盾（%4），持续%3回合
 *
 * @param Replace Format
 * @text 元素类型替换提示
 * @default %1 的护盾元素被替换为 %4！
 *
 * @help
 * 技能备注（note）写法示例：
 * <shield>
 * <shield_element:2>       ← 指定元素ID（2通常是火，根据你的数据库）
 * <rate:35>                ← 基于最大HP的百分比
 * <turns:4>
 * <shield_cd:5>            ← 这个技能的CD（队伍共享）
 * <Resist:2=30,4=20>       ← 火抵抗30%，风抵抗20%（减免）
 * <Weakness:3=25>          ← 冰增加25%伤害
 * <state:50>
 * v1.3 修改内容：
 * - 护盾支持元素类型
 * - 同元素默认减免20%（可被Resist覆盖）
 * - 日志显示元素名称
 * - 新护盾元素不同时显示“元素被替换”提示
 * - 破盾/吸收日志也带元素名称
 */

SHIELD_MAP={
	125:1,
	127:1,
	128	:1
};

(function() {
    'use strict';

    var parameters = PluginManager.parameters('ShieldSystem');
    var shieldStateId = Number(parameters['Shield State ID'] || 50);
    var defaultTurns  = Number(parameters['Default Shield Turns'] || 5);
    var defaultCD     = Number(parameters['Default Shield CD'] || 3);
    var breakSEName   = parameters['Break Shield SE'] || 'Crash';

    var absorbFormat       = parameters['Absorb Format']       || '%1 的%4护盾吸收了 %2 点伤害（剩余 %3 点护盾）';
    var breakFormat        = parameters['Break Format']        || '%1 的%4护盾被破坏！（吸收 %2 点伤害）';
    var endFormat          = parameters['End Format']          || '%1 的护盾结束！';
    var startFormat        = parameters['Start Format']        || '%1 获得 %2 护盾，持续%3回合';
    var startFormatElement = parameters['Start Format With Element'] || '%1 获得 %2 护盾（%4），持续%3回合';
    var replaceFormat      = parameters['Replace Format']      || '%1 的护盾元素被替换为 %4！';

    window.ShieldSystem = {
        stateId: shieldStateId,
        defaultTurns: defaultTurns,
        defaultCD: defaultCD,
        breakSEName: breakSEName,
        absorbFormat: gl_translate ? gl_translate(absorbFormat) : absorbFormat,
        breakFormat: gl_translate ? gl_translate(breakFormat) : breakFormat,
        endFormat: gl_translate ? gl_translate(endFormat) : endFormat,
        startFormat: gl_translate ? gl_translate(startFormat) : startFormat,
        startFormatElement: gl_translate ? gl_translate(startFormatElement) : startFormatElement,
        replaceFormat: gl_translate ? gl_translate(replaceFormat) : replaceFormat
    };

    // 获取元素名称（RPG Maker MV 默认从 $dataSystem.elements 取）
    function getElementName(elementId) {
        if (!elementId || elementId <= 0) return "";
        var name = $dataSystem.elements[elementId];
        return name || "元素" + elementId;
    }

    function parseElementMap(str) {
        var map = {};
        if (str) {
            str.split(',').forEach(function(pair) {
                var parts = pair.split('=');
                if (parts.length === 2) {
                    var key = Number(parts[0].trim());
                    var val = Number(parts[1].trim());
                    if (!isNaN(key) && !isNaN(val)) {
                        map[key] = val;
                    }
                }
            });
        }
        return map;
    }

    // 初始化
    var _Game_BattlerBase_initMembers = Game_BattlerBase.prototype.initMembers;
    Game_BattlerBase.prototype.initMembers = function() {
        _Game_BattlerBase_initMembers.call(this);
        this._shieldValue = 0;
        this._shieldTurns = 0;
        this._shieldElement = null;
        this._shieldResists = {};
        this._shieldWeaknesses = {};
        this._skillCooldowns = {};
    };

    // 技能冷却相关（不变）
    Game_BattlerBase.prototype.getSkillCooldown = function(skillId) {
        return this._skillCooldowns[skillId] || 0;
    };

    Game_BattlerBase.prototype.setSkillCooldown = function(skillId, turns) {
        this._skillCooldowns[skillId] = turns;
    };

    Game_BattlerBase.prototype.clearAllSkillCooldowns = function() {
        this._skillCooldowns = {};
    };

    // 回合结束
    var _Game_Battler_onTurnEnd = Game_Battler.prototype.onTurnEnd;
    Game_Battler.prototype.onTurnEnd = function() {
        _Game_Battler_onTurnEnd.call(this);

        if (this._shieldTurns > 0) {
            this._shieldTurns--;
            if (this._shieldTurns <= 0 && this._shieldValue > 0) {
                this._shieldValue = 0;
                if (BattleManager._logWindow) {
                    var text = ShieldSystem.endFormat.replace(/%1/g, this.name());
                    BattleManager._logWindow.addText(text);
                }
                //this.removeState(this._state);
				this.removeState(12);
				this.removeState(22);
				this.removeState(23);	
				this.removeState(24);	
				this.removeState(25);
				this.removeState(26);	
				this.removeState(27);	
                this._shieldElement = null;
                this._shieldResists = {};
                this._shieldWeaknesses = {};
            }
        }

        for (var id in this._skillCooldowns) {
            if (this._skillCooldowns[id] > 0) {
                this._skillCooldowns[id]--;
                if (this._skillCooldowns[id] <= 0) {
                    delete this._skillCooldowns[id];
                }
            }
        }
    };

    // 技能列表显示冷却
    var _Window_SkillList_drawSkillCost = Window_SkillList.prototype.drawSkillCost;
    Window_SkillList.prototype.drawSkillCost = function(skill, x, y, width) {
        if (skill && skill.meta.shield) {
            var cd = this._actor ? this._actor.getSkillCooldown(skill.id) : 0;
            if (cd > 0) {
                this.changeTextColor(this.textColor(8));
                this.drawText('{冷却回合}：' + cd, x, y, width, 'right');
                this.resetTextColor();
                return;
            }
        }
        _Window_SkillList_drawSkillCost.call(this, skill, x, y, width);
    };

    // 上护盾核心逻辑
    var _Game_Action_apply = Game_Action.prototype.apply;
    Game_Action.prototype.apply = function(target) {
        _Game_Action_apply.call(this, target);

        var item = this.item();
        if (item.meta && item.meta.shield) {
            var user = this.subject();
            var rate = Number(item.meta.rate || 30) / 100;
            var turns = Number(item.meta.turns || ShieldSystem.defaultTurns);
            var shieldValue = Math.floor(user.mhp * rate);
			//var isBaojii = (Math.random() < this.itemCri(target));
			var result = target.result();
			if(result.critical){
				shieldValue = shieldValue * 2;
			}
            var oldShield = target._shieldValue || 0;
            var oldElement = target._shieldElement;
            var newElement = Number(item.meta.shield_element) || null;
            var elementName = getElementName(newElement);
			var state = Number(item.meta.state || ShieldSystem.stateId);
            // 护盾值叠加，持续时间取最大
            target._shieldValue =  shieldValue;
            target._shieldTurns =  turns;
			
            var isReplace = false;
            if (newElement !== null) {
                if (oldElement !== newElement && oldShield > 0) {
                    isReplace = true;
                }
                target._shieldElement = newElement;
                target._shieldResists = parseElementMap(item.meta.Resist);
                target._shieldWeaknesses = parseElementMap(item.meta.Weakness);
            }else{
				target._shieldElement = target._shieldElement;
                target._shieldResists = parseElementMap(item.meta.Resist);
                target._shieldWeaknesses = parseElementMap(item.meta.Weakness);
			}
            // 新护盾无元素 → 保留旧元素（根据需求可改成清空）

            target.addState(state);

            if (BattleManager._logWindow) {
                var log = BattleManager._logWindow;

                // 显示替换提示
                if (isReplace) {
                    var repText = ShieldSystem.replaceFormat
                        .replace(/%1/g, target.name())
                        .replace(/%4/g, elementName);
                    log.addText(repText);
                }

                // 显示获得护盾
                var fmt = newElement ? ShieldSystem.startFormatElement : ShieldSystem.startFormat;
                var startText = fmt
                    .replace(/%1/g, target.name())
                    .replace(/%2/g, shieldValue)
                    .replace(/%3/g, target._shieldTurns)
                    .replace(/%4/g, elementName);
                log.addText(startText);
            }

            // CD（队伍共享）
            var cdTurns = Number(item.meta.shield_cd || ShieldSystem.defaultCD);
            if (cdTurns > 0) {
                var unit = user.friendsUnit();
                var members = (unit === $gameParty) ? $gameParty.battleMembers() : $gameTroop.members();
                members.forEach(function(b) { b.setSkillCooldown(item.id, cdTurns); });
            }
        }
    };

    // 记录本次攻击元素
    var _Game_Action_executeDamage = Game_Action.prototype.executeDamage;
    Game_Action.prototype.executeDamage = function(target, value) {
        target._currentDamageElement = this.item().damage.elementId;
        _Game_Action_executeDamage.call(this, target, value);
        delete target._currentDamageElement;
    };

    // 护盾吸收核心
    var _Game_Battler_gainHp = Game_Battler.prototype.gainHp;
    Game_Battler.prototype.gainHp = function(value) {
        var preShield = this._shieldValue || 0;

        if (this._isRealDamageThisTurn && value < 0 && preShield > 0) {
            var absDamage = -value;
            var elementId = this._currentDamageElement || -1;

            var resist = this._shieldResists?(this._shieldResists[elementId] || 0):0	;
            var weak   = this._shieldWeaknesses?(this._shieldWeaknesses[elementId] || 0):0;

            if (resist === 0 && weak === 0 && this._shieldElement && elementId === this._shieldElement) {
                resist = 20; // 同类型默认20%减免
            }
			console.log("[伤害吸收]攻击元素类型 ="+elementId+",抵抗="+resist);
            var rate = 1.0;
            if (resist > 0) rate = 1 - resist / 100;
            else if (weak > 0) rate = 1 + weak / 100;

            var maxBlock = preShield / rate;
            var blocked = Math.min(absDamage, maxBlock);
            var deduct = Math.floor(blocked * rate);
            var newShield = preShield - deduct;

            this._shieldValue = newShield;

            this._result.hpDamage = blocked;
            this._result.hpAffected = true;
            this.performDamage();

            var eleName = getElementName(this._shieldElement);

            if (BattleManager._logWindow) {
                var fmt = (newShield > 0) ? ShieldSystem.absorbFormat : ShieldSystem.breakFormat;
                var text = fmt
                    .replace(/%1/g, this.name())
                    .replace(/%2/g, Math.round(deduct))
                    .replace(/%3/g, newShield)
                    .replace(/%4/g, eleName);
                BattleManager._logWindow.addText(text);

                if (newShield <= 0 && ShieldSystem.breakSEName) {
                    AudioManager.playSe({
                        name: ShieldSystem.breakSEName,
                        volume: 90,
                        pitch: 100,
                        pan: 0
                    });
                }
				}

            value = 0;

            if (newShield <= 0) {
				value = 0;	
                this._shieldValue = 0;
                this._shieldElement = null;
                this._shieldResists = {};
                this._shieldWeaknesses = {};
                //this.removeState(this._state);
				this.removeState(12);
				this.removeState(22);
				this.removeState(23);	
				this.removeState(24);	
				this.removeState(25);
				this.removeState(26);	
				this.removeState(27);					
            }
        }
		
        _Game_Battler_gainHp.call(this, value);
    };

    // 伤害标记
    var _Game_Action_makeDamageValue = Game_Action.prototype.makeDamageValue;
    Game_Action.prototype.makeDamageValue = function(target, critical) {
        var damage = _Game_Action_makeDamageValue.call(this, target, critical);
        delete target._isRealDamageThisTurn;
        if (damage > 0) target._isRealDamageThisTurn = true;
        return damage;
    };

    // 战斗结束清CD
    var _BattleManager_processVictory = BattleManager.processVictory;
    BattleManager.processVictory = function() {
        BattleManager.clearAllShieldCooldowns();
        _BattleManager_processVictory.call(this);
    };

    var _BattleManager_processEscape = BattleManager.processEscape;
    BattleManager.processEscape = function() {
        BattleManager.clearAllShieldCooldowns();
        _BattleManager_processEscape.call(this);
    };

    var _BattleManager_processDefeat = BattleManager.processDefeat;
    BattleManager.processDefeat = function() {
        BattleManager.clearAllShieldCooldowns();
        _BattleManager_processDefeat.call(this);
    };

    BattleManager.clearAllShieldCooldowns = function() {
        $gameParty.battleMembers().forEach(a => a.clearAllSkillCooldowns());
        $gameTroop.members().forEach(e => e.clearAllSkillCooldowns());
    };

	var _Window_SkillList_isEnabled = Window_SkillList.prototype.isEnabled;

	Window_SkillList.prototype.isEnabled = function(skill) {
		if (skill && skill.meta.shield) {
			var cd = this._actor ? this._actor.getSkillCooldown(skill.id) : 0;
			if (cd > 0) {
				return false;           // ← 关键！禁止使用
			}
		}
		return _Window_SkillList_isEnabled.call(this, skill);
	};
	// 新增：清空所有人的护盾（包括我方和敌方，如果需要）
	BattleManager.clearAllShieldStates = function() {
		// 清我方
		$gameParty.battleMembers().forEach(function(actor) {
			actor._shieldValue = 0;
			actor._shieldTurns = 0;
			actor._shieldElement = null;
			actor._shieldResists = {};
			actor._shieldWeaknesses = {};
			// 如果你用了特定状态ID，也可以在这里移除（视情况）
			actor.removeState(ShieldSystem.stateId);
			// 你代码里之前硬编码的那些状态也可以清
			[12,22,23,24,25,26,27].forEach(function(id) {
				actor.removeState(id);
			});
		});

		// 如果你也想清敌方（通常不需要，但保险起见）
		$gameTroop.members().forEach(function(enemy) {
			enemy._shieldValue = 0;
			enemy._shieldTurns = 0;
			enemy._shieldElement = null;
			enemy._shieldResists = {};
			enemy._shieldWeaknesses = {};
			enemy.removeState(ShieldSystem.stateId);
			[12,22,23,24,25,26,27].forEach(function(id) {
				enemy.removeState(id);
			});
		});
	};
	// 新增可拦截函数，默认返回 true（允许正常AI）
    Game_Enemy.prototype.canProceedWithNormalAI = function() {
		if(this._skillCooldowns	 != undefined && this._skillCooldowns != null)
		{
			//return false;	
			if(this._actions != undefined && this._actions != null && this._actions.length > 0)
			{
				for(var	 i=0;i < this._actions.length ;++i){
					var _action = this._actions[i];
					try{
						if(_action._item._dataClass=='skill')
						{
							if(_action._item._itemId in SHIELD_MAP)
							{
								if(0 < this._shieldValue)
								{
									return false;
								}
								for(let k in SHIELD_MAP)
								{
									if(k in this._skillCooldowns && this._skillCooldowns[k] > 0)
									{
										return false;
									}
								}
							}
						}
						
					}catch(e){
						
					}
				}
			}
		}
		
        return true;
    };
})();