/*:
 * @plugindesc 清理污渍小游戏插件 v2.3 - clear 指令真正强制终止小游戏
 *
 * @param StainRegionId
 * @text 污渍生成Region ID
 * @desc 污渍随机出现的Region ID
 * @type number
 * @min 1
 * @max 255
 * @default 1
 *
 * @param StainCount
 * @text 默认污渍总数
 * @desc 如果没有push事件ID，使用这个数量作为总计
 * @type number
 * @min 1
 * @max 20
 * @default 4
 *
 * @help
 * 【插件指令】
 * StainGame push 5 6 7 8     → 添加污渍事件ID
 * StainGame start           → 开始游戏（随机放置 + 显示进度条）
 * StainGame clear           → 强制终止小游戏（隐藏进度条 + 清空所有状态，可随时重新开始）
 *
 * 【脚本调用】
 * $gameSystem.isLastStain()         → 是否只剩 ≤1 个污渍
 * $gameSystem.clearStainHud()         → 仅隐藏进度条（不终止游戏状态）
 * $gameSystem.forceEndStainGame()    → 强制终止小游戏（等同于插件指令 clear）
 * $gameSystem.isStainGameActive()    → 返回是否正在进行小游戏（true/false）
 */

(function() {
    const pluginName = "StainCleaningGame";
    const parameters = PluginManager.parameters(pluginName);
    const stainRegionId = Number(parameters['StainRegionId'] || 1);
    const totalStainsParam = Number(parameters['StainCount'] || 4);

    let stainEventIds = [];      // 当前这轮的污渍事件ID列表
    let totalStains = totalStainsParam;

    // 插件指令
    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command.toLowerCase() === "staingame") {
            const subCommand = args[0] && args[0].toLowerCase();
            if (subCommand === "push") {
                for (let i = 1; i < args.length; i++) {
                    const id = Number(args[i]);
                    if (id > 0 && stainEventIds.indexOf(id) === -1) {
                        stainEventIds.push(id);
                    }
                }
            } else if (subCommand === "start") {
                startStainGame();
            } else if (subCommand === "clear") {
                forceEndStainGame();  // 强制终止
            }
        }
    };

    function startStainGame() {
        if (stainEventIds.length === 0) return;

        totalStains = stainEventIds.length;

        const mapId = $gameMap.mapId();
        const keyBase = [mapId];

        // 重置所有污渍事件的独立开关 A B C D
        stainEventIds.forEach(function(eventId) {
            ['A','B','C','D'].forEach(function(sw) {
                $gameSelfSwitches.setValue(keyBase.concat(eventId, sw), false);
            });
        });

        // 收集可用位置并随机放置（同之前）
        const candidates = [];
        for (let x = 0; x < $gameMap.width(); x++) {
            for (let y = 0; y < $gameMap.height(); y++) {
                if ($gameMap.regionId(x, y) === stainRegionId && $gameMap.eventsXy(x, y).length === 0) {
                    candidates.push({x: x, y: y});
                }
            }
        }

        if (candidates.length < totalStains) {
            console.warn("污渍位置不足！");
            return;
        }

        // 随机打乱
        for (let i = candidates.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [candidates[i], candidates[j]] = [candidates[j], candidates[i]];
        }

        // 放置事件
        stainEventIds.forEach(function(eventId, i) {
            const event = $gameMap.event(eventId);
            if (event) event.setPosition(candidates[i].x, candidates[i].y);
        });

        if (SceneManager._scene instanceof Scene_Map) {
            SceneManager._scene.createStainHud();
        }
    }

    // 强制终止小游戏（clear 指令调用这个）
    function forceEndStainGame() {
        // 1. 移除进度条
        if (SceneManager._scene instanceof Scene_Map) {
            SceneManager._scene.removeStainHud();
        }

        // 2. 清空状态，彻底终止这轮游戏
        stainEventIds = [];
        totalStains = totalStainsParam;

        console.log("污渍小游戏已强制终止，可重新 push + start");
    }

    // 仅隐藏进度条（不清除状态，适合只是想暂时不显示）
    function clearStainHudOnly() {
        if (SceneManager._scene instanceof Scene_Map) {
            SceneManager._scene.removeStainHud();
        }
    }

    // 公开方法
    Game_System.prototype.isLastStain = function() {
        if (stainEventIds.length === 0) return false;
        const mapId = $gameMap.mapId();
        const keyBase = [mapId];
        let cleaned = 0;
        stainEventIds.forEach(function(eventId) {
            if ($gameSelfSwitches.value(keyBase.concat(eventId, 'A'))) cleaned++;
        });
        return (totalStains - cleaned) <= 1;
    };

    Game_System.prototype.clearStainHud = function() {  // 仅隐藏
        clearStainHudOnly();
    };

    Game_System.prototype.forceEndStainGame = function() {  // 强制终止
        forceEndStainGame();
    };

    Game_System.prototype.isStainGameActive = function() {
        return stainEventIds.length > 0;
    };

    // HUD Sprite（保持不变，增加了全部清理完自动隐藏）
    function Sprite_StainHud() {
        this.initialize.apply(this, arguments);
    }
    Sprite_StainHud.prototype = Object.create(Sprite.prototype);
    Sprite_StainHud.prototype.constructor = Sprite_StainHud;

    Sprite_StainHud.prototype.initialize = function() {
        Sprite.prototype.initialize.call(this);
        this.bitmap = new Bitmap(280, 80);
        this.x = Graphics.width - 300;
        this.y = 10;
        this.opacity = 230;
        this.refresh();
    };

    Sprite_StainHud.prototype.refresh = function() {
        if (stainEventIds.length === 0) {
            this.visible = false;
            return;
        }

        const mapId = $gameMap.mapId();
        const keyBase = [mapId];
        let cleaned = 0;
        stainEventIds.forEach(function(eventId) {
            if ($gameSelfSwitches.value(keyBase.concat(eventId, 'A'))) cleaned++;
        });

        const bitmap = this.bitmap;
        bitmap.clear();

        bitmap.fontSize = 28;
        bitmap.outlineWidth = 4;
        bitmap.outlineColor = 'rgba(0,0,0,0.8)';
        bitmap.drawText(gl_translate('{清理进度}')	, 0, 0, 280, 32, 'center');

        const barX = 30, barY = 40, barW = 220, barH = 20;
        bitmap.fillRect(barX, barY, barW, barH, '#333333');

        const rate = cleaned / totalStains;
        const fillW = Math.floor(barW * rate);
        if (fillW > 0) {
            bitmap.gradientFillRect(barX, barY, fillW, barH, '#00aaff', '#0088ff');
        }

        bitmap.context.strokeStyle = '#ffffff';
        bitmap.context.lineWidth = 2;
        bitmap.context.strokeRect(barX + 1, barY + 1, barW - 2, barH - 2);
        bitmap._baseTexture.update();

        bitmap.fontSize = 24;
        bitmap.drawText(`${cleaned} / ${totalStains}`, 0, 35, 280, 32, 'center');

        // 全部清理完自动隐藏
        if (cleaned >= totalStains) {
            this.visible = false;
        } else {
            this.visible = true;
        }
    };

    Sprite_StainHud.prototype.update = function() {
        Sprite.prototype.update.call(this);
        this.refresh();
    };

    // HUD 创建与移除
    Scene_Map.prototype.createStainHud = function() {
        if (stainEventIds.length > 0 && !this._stainHudSprite) {
            this._stainHudSprite = new Sprite_StainHud();
            this._spriteset.addChild(this._stainHudSprite);
        }
    };

    Scene_Map.prototype.removeStainHud = function() {
        if (this._stainHudSprite) {
            this._spriteset.removeChild(this._stainHudSprite);
            this._stainHudSprite = null;
        }
    };

    const _Scene_Map_createSpriteset = Scene_Map.prototype.createSpriteset;
    Scene_Map.prototype.createSpriteset = function() {
        _Scene_Map_createSpriteset.call(this);
        this.createStainHud();
    };

    const _Scene_Map_terminate = Scene_Map.prototype.terminate;
    Scene_Map.prototype.terminate = function() {
        this.removeStainHud();
        _Scene_Map_terminate.call(this);
    };

})();