/*:
 * @plugindesc 推箱子谜题完成检查器（支持传参变量 + 多语言地图标签 + 公开校验方法 + 调试日志）

 * @param keyTag
 * @text 箱子事件标签
 * @desc 箱子（可推动的事件）在事件备注中使用的标签，格式：<key>
 * @default key

 * @param lockTag
 * @text 目标位置标签
 * @desc 目标位置（固定事件）在事件备注中使用的标签，格式：<lock>
 * @default lock

 * @help
 * 【使用方法】
 * 1. 在地图备注（Note）中写入以下任意一个标签来启用推箱子检查：
 *    - <推箱子>（中文）
 *    - <sokoban>（英文，推荐防止乱码）
 *    - <倉庫番>（日文）
 *    包含任意一个即可启用。
 * 2. 箱子事件：在事件备注（Note）中写入 <key>（所有页共享）
 * 3. 目标位置事件：在事件备注（Note）中写入 <lock>（所有页共享）
 * 4. 设置变量方式（推荐用于过关逻辑）：
 *    创建平行进程事件，插件命令：checkSokoban 变量ID 完成值
 *    示例：checkSokoban 10 1
 * 5. 公开校验方法（用于条件判断、开关等）：
 *    在事件“脚本”命令中直接调用：
 *    $gameMap.isSokobanCompleted()
 *    返回 true（已完成）或 false（未完成）
 *    示例：在条件分支的“脚本”中写入：
 *    $gameMap.isSokobanCompleted()
 *    即可判断谜题是否完成。
 *
 * 【调试功能】
 * - 每次检查（插件命令或公开方法）都会在浏览器控制台（F8）输出详细日志：
 *   - 当前箱子数量、目标数量
 *   - 箱子当前位置列表
 *   - 目标位置列表
 *   - 是否完成
 * - 方便调试箱子位置是否对齐、标签是否识别等。
 *
 * 【说明】
 * - 检查逻辑：箱子当前位置必须与目标位置完全一一对应（数量相等、位置相同）
 * - 插件命令只在完成时设置变量（未完成不修改）
 * - 公开方法随时可用，返回布尔值，不修改任何变量
 * - 目标事件建议设为“穿过”以免阻挡
 */

(function() {
    const parameters = PluginManager.parameters('SokobanChecker');
    const keyTag = parameters['keyTag'] || 'key';
    const lockTag = parameters['lockTag'] || 'lock';

    // 判断地图是否启用推箱子检查
    Game_Map.prototype.isSokobanEnabled = function() {
        if (!$dataMap) return false;
        const note = $dataMap.note;
        return note.includes('<推箱子>') || 
               note.includes('<sokoban>') || 
               note.includes('<倉庫番>');
    };

    // 公开方法：返回谜题是否已完成（true/false）
    Game_Map.prototype.isSokobanCompleted = function() {
        if (!this.isSokobanEnabled()) {
            console.log('[Sokoban] 地图未启用推箱子检查（无对应标签）');
            return false;
        }

        // 获取箱子事件（运行时实例）
        const keyEvents = this.events().filter(function(event) {
            if (!event || !event.event()) return false;
            const meta = event.event().meta;
            return meta && meta[keyTag] !== undefined;
        });

        // 获取目标位置（原始事件数据中的固定位置）
        const lockEvents = $dataMap.events.slice(1).filter(function(ev) {
            return ev && ev.meta && ev.meta[lockTag] !== undefined;
        });

        const keyCount = keyEvents.length;
        const lockCount = lockEvents.length;

        console.log(`[Sokoban] 检查开始 - 箱子数量: ${keyCount}, 目标数量: ${lockCount}`);

        if (keyCount !== lockCount || keyCount === 0) {
            console.log('[Sokoban] 数量不匹配或为0，无法完成');
            return false;
        }

        // 位置数组（排序后比较）
        const keyPositions = keyEvents.map(e => `(${e.x},${e.y})`).sort();
        const lockPositions = lockEvents.map(e => `(${e.x},${e.y})`).sort();

        console.log('[Sokoban] 箱子当前位置:', keyPositions);
        console.log('[Sokoban] 目标位置:', lockPositions);

        // 逐一比较
        let completed = true;
        for (let i = 0; i < keyPositions.length; i++) {
            if (keyPositions[i] !== lockPositions[i]) {
                completed = false;
                break;
            }
        }

        console.log('[Sokoban] 检查结果:', completed ? '已完成！' : '未完成');

        return completed;
    };

    // 插件命令：checkSokoban 变量ID 完成值
    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);

        if (command.toLowerCase() === 'checksokoban') {
            if (args.length < 2) {
                console.warn('checkSokoban 需要两个参数：变量ID 完成值');
                return;
            }

            const variableId = Number(args[0]);
            const completedValue = Number(args[1]);

            if (isNaN(variableId) || isNaN(completedValue)) {
                console.warn('checkSokoban 参数必须为数字');
                return;
            }

            // 调用公开方法进行检查（自动输出日志）
            if ($gameMap.isSokobanCompleted()) {
                $gameVariables.setValue(variableId, completedValue);
                console.log(`[Sokoban] 已完成 → 变量 ${variableId} 设置为 ${completedValue}`);
            } else {
                console.log('[Sokoban] 未完成，变量不修改');
            }
        }
    };
})();