/*:
 * @target MZ
 * @plugindesc 文章表示時の名前を一括変換するプラグイン
 * @author Onmoremind
 *
 * @help NameReplacer.js
 *
 * 文章表示コマンドの「名前」部分を一括で変換
 *
 * @param replacementRules
 * @text 変換ルール
 * @desc 名前の変換ルールを設定します。
 * @type struct<ReplacementRule>[]
 * @default ["{\"originalName\":\"主人公\",\"replacedName\":\"\\\\N[1]\"}"]
 *
 */

/*~struct~ReplacementRule:
 * @param originalName
 * @text 変換元の名前
 * @desc 
 * @type string
 * @default 主人公
 *
 * @param replacedName
 * @text 変換後の名前
 * @desc 制御文字使用可能
 * @type string
 * @default \N[1]
 *
 */

(() => {
    'use strict';

    const pluginName = 'NameReplacer';
    const parameters = PluginManager.parameters(pluginName);
    
    // 変換ルールを解析
    const replacementRules = [];
    try {
        const rulesJson = JSON.parse(parameters['replacementRules'] || '[]');
        for (const ruleJson of rulesJson) {
            const rule = JSON.parse(ruleJson);
            replacementRules.push({
                originalName: rule.originalName || '',
                replacedName: rule.replacedName || ''
            });
        }
    } catch (e) {
        console.error('NameReplacer: 変換ルールの解析に失敗しました。', e);
    }

    /**
     * @param {string} text - 変換するテキスト
     * @returns {string} - 変換後のテキスト
     */
    function convertEscapeCharacters(text) {
        if (!text) return text;
        
        text = text.replace(/\\N\[(\d+)\]/gi, (_, p1) => {
            const actor = $gameActors.actor(parseInt(p1));
            return actor ? actor.name() : '';
        });
        
        text = text.replace(/\\P\[(\d+)\]/gi, (_, p1) => {
            const actor = $gameParty.members()[parseInt(p1) - 1];
            return actor ? actor.name() : '';
        });
        
        text = text.replace(/\\V\[(\d+)\]/gi, (_, p1) => {
            return $gameVariables.value(parseInt(p1));
        });
        
        return text;
    }

    /**
     * 名前を変換する
     * @param {string} name - 元の名前
     * @returns {string} - 変換後の名前
     */
    function replaceName(name) {
        if (!name) return name;
        
        for (const rule of replacementRules) {
            if (name === rule.originalName) {
                return convertEscapeCharacters(rule.replacedName);
            }
        }
        return name;
    }

    const _Window_Message_setSpeakerName = Window_Message.prototype.setSpeakerName;
    Window_Message.prototype.setSpeakerName = function(name) {
        const replacedName = replaceName(name);
        _Window_Message_setSpeakerName.call(this, replacedName);
    };

    const _Game_Message_setSpeakerName = Game_Message.prototype.setSpeakerName;
    Game_Message.prototype.setSpeakerName = function(speakerName) {
        const replacedName = replaceName(speakerName);
        _Game_Message_setSpeakerName.call(this, replacedName);
    };

})();
