/*:
 * @target MZ
 * @plugindesc 円形ゲージプラグイン
 * @author Onmoremind
 *
 * @help 
 *
 *
 * @param currentVariableId
 * @text 現在値の変数ID
 * @type variable
 * @default 1
 *
 * @param maxVariableId
 * @text 最大値の変数ID
 * @type variable
 * @default 2
 *
 * @param visibleSwitchId
 * @text 可視切替スイッチID
 * @type switch
 * @default 0
 * @desc 0=未使用。指定時、スイッチONで表示/ OFFで非表示
 *
 * @param framePicture
 * @text フレーム画像名
 * @type file
 * @dir img/pictures
 * @default IntimacyGauge_Frame
 *
 * @param barPicture
 * @text バー画像名
 * @type file
 * @dir img/pictures
 * @default IntimacyGauge_Bar
 *
 * @param x
 * @text 表示X
 * @type number
 * @default 24
 *
 * @param y
 * @text 表示Y
 * @type number
 * @default 24
 *
 * @param scale
 * @text スケール
 * @type number
 * @decimals 2
 * @min 0.1
 * @default 1.00
 *
 * @param startAngleDeg
 * @text 開始角(度)
 * @type number
 * @min -360
 * @max 360
 * @default -210
 * @desc 0=右, 90=下, -90=上。例: -210 から 30 までの扇形で240度分
 *
 * @param endAngleDeg
 * @text 終了角(度)
 * @type number
 * @min -360
 * @max 360
 * @default 30
 *
 * @param clockwise
 * @text 時計回りに加算
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 *
 *
 * @param showOnMap
 * @text マップで表示
 * @type boolean
 * @default true
 *
 * @param showOnBattle
 * @text 戦闘で表示
 * @type boolean
 * @default true
 *
 * @param picturePriority
 * @text ピクチャプライオリティID
 * @type number
 * @min 1
 * @max 100
 * @default 50
 * @desc ゲージの表示優先度をピクチャIDと同等に設定します。50ならピクチャID49より上、51より下に表示されます。
 *
 * @param fadeFrames
 * @text フェードフレーム数
 * @type number
 * @min 0
 * @default 0
 * @desc スイッチ切替時のフェードフレーム数。0なら即時切替、1以上ならそのフレーム数でフェードイン/アウト。
 *
 */

(() => {
  'use strict';

  const PLUGIN_NAME = 'OncircleGAGE';
  const params = PluginManager.parameters(PLUGIN_NAME);

  const p = {
    curVar: Number(params.currentVariableId || 1),
    maxVar: Number(params.maxVariableId || 2),
    visSw: Number(params.visibleSwitchId || 0),
    framePic: String(params.framePicture || ''),
    barPic: String(params.barPicture || ''),
    x: Number(params.x || 0),
    y: Number(params.y || 0),
    scale: Number(params.scale || 1),
    startDeg: Number(params.startAngleDeg || 0),
    endDeg: Number(params.endAngleDeg || 0),
    clockwise: params.clockwise === 'true',
    showOnMap: params.showOnMap === 'true',
    showOnBattle: params.showOnBattle === 'true',
    picturePriority: Number(params.picturePriority || 50),
    fadeFrames: Number(params.fadeFrames || 0),
  };

  const deg2rad = (d) => (d * Math.PI) / 180;

  class Sprite_VariableArcGauge extends Sprite {
    initialize() {
      super.initialize();
      this.x = p.x;
      this.y = p.y;
      this.scale.set(p.scale, p.scale);
      this.z = p.picturePriority * 10;

      this._rate = -1; 
      this._barSprite = null;
      this._barContainer = null; 
      this._frameSprite = null;
      this._mask = new PIXI.Graphics();
      this._center = { x: 0, y: 0 };
      this._outerRadius = 0;
      this._lastStartDeg = null;
      this._lastEndDeg = null;
      this._lastClockwise = null;
      const initialVisible = p.visSw > 0 ? $gameSwitches.value(p.visSw) : true;
      this._targetOpacity = initialVisible ? 255 : 0;
      this.opacity = this._targetOpacity;
      this.visible = initialVisible;
      this._fadeSpeed = 0;

      if (p.barPic) {
        this._barSprite = new Sprite(ImageManager.loadPicture(p.barPic));
        this._barSprite.anchor.set(0, 0);
        this.addChild(this._barSprite);
        this._barSprite.bitmap.addLoadListener(() => {
          this._center.x = this._barSprite.width / 2;
          this._center.y = this._barSprite.height / 2;
          this._outerRadius = Math.max(this._barSprite.width, this._barSprite.height) / 2;
          this.addChild(this._mask);
          this._mask.renderable = false;
          this._barSprite.mask = this._mask;

          this._applyMask(this.rate()); 
        });

      }

      if (p.framePic) {
        this._frameSprite = new Sprite(ImageManager.loadPicture(p.framePic));
        this._frameSprite.anchor.set(0, 0);
        this._frameSprite.x = this.x;
        this._frameSprite.y = this.y;
        this._frameSprite.scale.set(p.scale, p.scale);
        this._frameSprite.z = this.z + 1;
      }


      this.update();
    }

    rate() {
      const rawCur = $gameVariables.value(p.curVar);
      const rawMax = $gameVariables.value(p.maxVar);
      const cur = Math.max(0, Number(rawCur || 0));
      const maxRaw = Number(rawMax || 0);
      const max = Math.max(1, maxRaw);
      const r = cur / max;
      const result = Math.max(0, Math.min(1, r));
      
      // デバッグログ（前回と値が変わった時のみ出力）
      if (this._lastDebugCur !== cur || this._lastDebugMax !== max) {
        console.log(`[OncircleGAGE] rawCur=${rawCur} (type:${typeof rawCur}), rawMax=${rawMax} (type:${typeof rawMax})`);
        console.log(`[OncircleGAGE] cur=${cur}, max=${max}, r=${r}, result=${result}`);
        this._lastDebugCur = cur;
        this._lastDebugMax = max;
      }
      
      return result;
    }

    update() {
      super.update();
      this._updateFade();

      if (this._frameSprite) {
        this._frameSprite.opacity = this.opacity;
      }

      const r = this.rate();
      const needAngleUpdate = (
        this._lastStartDeg !== p.startDeg ||
        this._lastEndDeg !== p.endDeg ||
        this._lastClockwise !== p.clockwise
      );
      if (this._barSprite && this._outerRadius > 0 && (r !== this._rate || needAngleUpdate)) {
        this._applyMask(r);
        this._rate = r;
        this._lastStartDeg = p.startDeg;
        this._lastEndDeg = p.endDeg;
        this._lastClockwise = p.clockwise;
      }


      if (this._frameSprite && this._frameSprite.parent) {
        this._frameSprite.x = this.x;
        this._frameSprite.y = this.y;
        this._frameSprite.z = this.z + 1;
      }
    }

    _updateFade() {
      let targetVisible = true;
      if (p.visSw > 0) {
        targetVisible = $gameSwitches.value(p.visSw);
      }
      const newTargetOpacity = targetVisible ? 255 : 0;

      if (this._targetOpacity !== newTargetOpacity) {
        this._targetOpacity = newTargetOpacity;
        if (p.fadeFrames > 0) {
          this._fadeSpeed = 255 / p.fadeFrames;
        } else {
          this._fadeSpeed = 0;
          this.opacity = newTargetOpacity;
        }
      }

      if (p.fadeFrames > 0 && this.opacity !== this._targetOpacity) {
        if (this._targetOpacity > this.opacity) {
          this.opacity = Math.min(this.opacity + this._fadeSpeed, 255);
        } else {
          this.opacity = Math.max(this.opacity - this._fadeSpeed, 0);
        }
      }

      this.visible = this.opacity > 0 || this._targetOpacity > 0;
      if (this._frameSprite) {
        this._frameSprite.visible = this.visible;
      }
    }

    _applyMask(rate) {
      const g = this._mask;
      const cx = this._center.x;
      const cy = this._center.y;
      const r = this._outerRadius;
      const start = deg2rad(p.startDeg);
      const end = deg2rad(p.endDeg);
      const sweep = p.clockwise ? (end - start) : (start - end);
      const cur = p.clockwise ? (start + sweep * rate) : (start - sweep * rate);

      g.clear();
      g.beginFill(0xffffff, 1.0);

      g.moveTo(cx, cy);
      if (p.clockwise) {
        g.arc(cx, cy, r, start, cur, false);
      } else {
        g.arc(cx, cy, r, start, cur, true);
      }
      g.lineTo(cx, cy);
      g.endFill();
    }
  }

  function addGaugeToScene(scene) {
    if (!scene || scene._variableArcGauge) return;
    const hud = new Sprite_VariableArcGauge();
    scene._variableArcGauge = hud;

    if (scene._spriteset && scene._spriteset._pictureContainer) {
      const picContainer = scene._spriteset._pictureContainer;
      const insertIndex = Math.max(0, Math.min(p.picturePriority - 1, picContainer.children.length));

      if (hud._frameSprite) {
        picContainer.addChildAt(hud._frameSprite, insertIndex);
        picContainer.addChildAt(hud, insertIndex + 1);
      } else {
        picContainer.addChildAt(hud, insertIndex);
      }
    } else if (scene._spriteset) {
      if (hud._frameSprite) {
        scene._spriteset.addChild(hud._frameSprite);
      }
      scene._spriteset.addChild(hud);
    } else {
      if (hud._frameSprite) {
        scene.addChild(hud._frameSprite);
      }
      scene.addChild(hud);
    }
  }

  if (p.showOnMap) {
    const _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
    Scene_Map.prototype.createDisplayObjects = function() {
      _Scene_Map_createDisplayObjects.call(this);
      addGaugeToScene(this);
    };
  }

  if (p.showOnBattle) {
    const _Scene_Battle_createDisplayObjects = Scene_Battle.prototype.createDisplayObjects;
    Scene_Battle.prototype.createDisplayObjects = function() {
      _Scene_Battle_createDisplayObjects.call(this);
      addGaugeToScene(this);
    };
  }
})();
