/*:
 * @command UI作成
 * @text UIを作成
 * @desc UIを設定し表示する
 *
 * @arg Name
 * @text UI名
 * @type string
 * @desc UI名前を設定
 *
 * @arg UIbaseSettings
 * @text UIbase設定
 * @type struct<UIbaseSettings>
 * @desc UIのドラッグbaseになる画像の設定
 *
 * @arg ActiveSwitchIds
 * @text 有効化スイッチ
 * @type switch[]
 * @desc このUIが表示されている間ONになるスイッチID
 *
 * @arg X
 * @text 全体X座標
 * @type number
 * @default 0
 * @desc UI全体のX座標
 *
 * @arg Y
 * @text 全体Y座標
 * @type number
 * @default 0
 * @desc UI全体のY座標
 *
 * @arg PictureId
 * @text 使用ピクチャID
 * @type number
 * @default 50
 * @desc 使用するピクチャID
 *
 * @arg ImageSprites
 * @text 画像スプライト一覧
 * @type struct<UIImageSprite>[]
 * @desc 静的または単一画像として表示するUIスプライト
 *
 * @arg ChoiceSprites
 * @text 選択肢スプライト一覧
 * @type struct<UIChoiceSprite>[]
 * @desc コモンイベントなどを呼び出す画像選択ボタン一覧
 *
 * @arg GaugeSprites
 * @text ゲージスプライト一覧
 * @type struct<UIGaugeSprite>[]
 * @desc ゲージに使用する画像
 *
 * @arg NumberSprites
 * @text 数値スプライト一覧
 * @type struct<UINumberSprite>[]
 * @desc 変数に応じて数字をスプライトで表示する項目
 *
 * @arg TextSprites
 * @text テキストスプライト一覧
 * @type struct<UITextSprite>[]
 * @desc 変数の値をテキストとして表示するスプライト一覧
 *
 * @command UI消去
 * @text UIを消去
 * @desc 指定したUIを消去
 *
 * @arg name
 * @text UI名
 * @desc 消去するUIの名前
 * @type string
 *
 * @command UI全消去
 * @text すべてのUIを消去
 * @desc すべてのUIを消去
 *
 * @command UI非表示
 * @text UIを非表示
 * @desc 指定したUIをフェードで非表示にする
 *
 * @arg name
 * @text UI名
 * @desc 非表示にするUIの名前
 * @type string
 *
 * @arg fadeFrames
 * @text フェード時間（フレーム）
 * @desc 非表示になるまでのフレーム数（0で即時非表示）
 * @type number
 * @default 30
 * @min 0
 *
 * @command UI表示
 * @text UIを表示
 * @desc 非表示にしたUIをフェードで再表示する
 *
 * @arg name
 * @text UI名
 * @desc 表示するUIの名前
 * @type string
 *
 * @arg fadeFrames
 * @text フェード時間（フレーム）
 * @desc 表示されるまでのフレーム数（0で即時表示）
 * @type number
 * @default 30
 * @min 0
 *
 * @command UI全非表示
 * @text すべてのUIを非表示
 * @desc すべてのUIを一時的に非表示にする
 *
 * @arg fadeFrames
 * @text フェード時間（フレーム）
 * @desc 非表示になるまでのフレーム数（0で即時非表示）
 * @type number
 * @default 0
 * @min 0
 *
 * @command UI再表示
 * @text すべてのUIを再表示
 * @desc 非表示にしたUIをすべて再表示する
 *
 * @arg fadeFrames
 * @text フェード時間（フレーム）
 * @desc 表示されるまでのフレーム数（0で即時表示）
 * @type number
 * @default 0
 * @min 0
 */

/*~struct~UIConfig:
 * @param Name
 * @text UI名
 * @type string
 *
 * @param UIbaseSettings
 * @text UIbase設定
 * @type struct<UIbaseSettings>
 * @desc UIのドラッグbaseになる画像の設定
 *
 * @param ActiveSwitchIds
 * @text 有効化スイッチ
 * @type switch[]
 * @desc このUIが表示されている間ONになるスイッチID
 *
 * @param X
 * @text 全体X座標
 * @type number
 * @default 0
 *
 * @param Y
 * @text 全体Y座標
 * @type number
 * @default 0
 *
 * @param PictureId
 * @text 使用ピクチャID
 * @type number
 * @default 50
 *
 * @param ImageSprites
 * @text 画像スプライト一覧
 * @type struct<UIImageSprite>[]
 * @desc 静的または単一画像として表示するUIスプライト
 *
 * @param ChoiceSprites
 * @text 選択肢スプライト一覧
 * @type struct<UIChoiceSprite>[]
 * @desc コモンイベントなどを呼び出す画像選択ボタン一覧
 *
 * @param GaugeSprites
 * @text ゲージスプライト一覧
 * @type struct<UIGaugeSprite>[]
 * @desc ゲージに使用する画像
 *
 * @param NumberSprites
 * @text 数値スプライト一覧
 * @type struct<UINumberSprite>[]
 * @desc 変数に応じて数字をスプライトで表示する項目
 *
 * @param TextSprites
 * @text テキストスプライト一覧
 * @type struct<UITextSprite>[]
 * @desc 変数の値をテキストとして表示するスプライト一覧
 */

/*~struct~UIbaseSettings:
 *
 * @param EnableSwitchId
 * @text 有効化スイッチID
 * @type switch
 * @desc このスイッチがONのときUI全体をドラッグで移動可能にする
 *
 * @param Baseimage
 * @text base画像
 * @type file
 * @dir img/UI/
 * @desc UIのbaseになるファイル名
 */

/*~struct~UIImageSprite:
 * @param Image
 * @text 表示画像
 * @type file
 * @dir img/UI
 *
 * @param OffsetX
 * @text Xオフセット
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param OffsetY
 * @text Yオフセット
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param ShowSwitch
 * @text 表示スイッチ
 * @type switch
 * @default 0
 * @desc 0の場合は常に表示
 *
 * @param ShowVariable
 * @text 表示変数ID
 * @type variable
 * @default 0
 * @desc 0の場合条件を無視
 *
 * @param ShowOperator
 * @text 比較方法
 * @type select
 * @option 等しい（==）
 * @option 以上（>=）
 * @option 以下（<=）
 * @option より大きい（>）
 * @option より小さい（<）
 * @default 等しい（==）
 *
 * @param ShowValue
 * @text 比較値
 * @type number
 * @default 0
 * @desc 変数と比較する値

 * @param ProhibitSwitch
 * @text 禁止スイッチ
 * @type switch
 * @default 0
 * @desc ONのとき表示を禁止

 * @param ProhibitVariable
 * @text 禁止変数ID
 * @type variable
 * @default 0
 * @desc 0の場合は無効

 * @param ProhibitOperator
 * @text 禁止比較方法
 * @type select
 * @option 等しい（==）
 * @option 以上（>=）
 * @option 以下（<=）
 * @option より大きい（>）
 * @option より小さい（<）
 * @default 等しい（==）

 * @param ProhibitValue
 * @text 禁止比較値
 * @type number
 * @default 0
 * @desc 禁止条件で比較する値
 *
 * @param FadeFrames
 * @text フェード時間
 * @type number
 * @min 0
 * @default 0
 * @desc 0: フェード機能、0ならフェードなし
 *
 * @param Priority
 * @text 表示優先度
 * @type number
 * @min 0
 * @default 0
 * @desc 数値が大きいほど手前に描画
 */

/*~struct~UIChoiceSprite:
 * @param Image
 * @text 表示画像
 * @type file
 * @dir img/UI
 * @desc ボタンとして表示される画像
 *
 * @param CommonEventId
 * @type common_event
 * @desc クリックしたときに呼び出されるコモンイベント
 *
 * @param X
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param Y
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param HoverEffect
 * @text ホバー効果
 * @type struct<UIHoverEffectConfig>
 * @desc ホバー効果
 *
 * @param ClickEffect
 * @text クリック効果
 * @type struct<UIClickEffectConfig>
 * @desc クリック時の演出効果
 *
 * @param ShowCondition
 * @text 表示条件
 * @type struct<ConditionConfig>
 * @default {"SwitchId":"0","VariableId":"0","Operator":"等しい（==）","Value":"0"}
 *
 * @param EnableCondition
 * @text 有効条件
 * @type struct<ConditionConfig>
 * @default {"SwitchId":"0","VariableId":"0","Operator":"等しい（==）","Value":"0"}
 *
 * @param DisabledImage
 * @text 無効時の画像
 * @type file
 * @dir img/UI
 *
 * @param FadeEffect
 * @text フェード効果
 * @type struct<UIFadeEffect>
 * @desc ボタンの表示・非表示時にフェード効果を使用する
 *
 * @param StopEvent
 * @text イベント停止
 * @type boolean
 * @default false
 * @desc この選択肢が表示されたときにイベントを停止するかどうか
 *
 * @param ResumeOnClick
 * @text クリック時にイベント再開
 * @type boolean
 * @default true
 * @desc イベント停止の影響を受けるかどうか
 */

/*~struct~UIHoverEffectConfig:
 *
 * @param UseScale
 * @text 拡大を有効化
 * @type boolean
 * @default false
 *
 * @param Scale
 * @text 拡大率（%）
 * @type number
 * @default 120
 *
 * @param UseOffset
 * @text 位置補正を有効化
 * @type boolean
 * @default false
 *
 * @param OffsetX
 * @text X補正
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param OffsetY
 * @text Y補正
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param UseFrame
 * @text フレームを表示
 * @type boolean
 * @default false
 *
 * @param FrameImage
 * @text フレーム画像
 * @type file
 * @dir img/UI/
 * @desc UI選択肢用のフレーム画像
 *
 * @param SpriteSheet
 * @text スプライトシート設定
 * @type struct<UISpriteSheetConfig>
 * @default
 *
 * @param ExtraImageSetting
 * @text ホバー中表示画像
 * @type struct<UIHoverExtraImageConfig>
 * @default
 *
 * @param HoverSE
 * @text ホバー時SE
 * @type file
 * @dir audio/se/
 * @desc ホバー時に再生するSEファイル
 * @default
 *
 * @param HoverScript
 * @text ホバー時実行スクリプト
 * @type multiline_string
 * @desc ホバー時に実行するJavaScriptコード
 * @default
 *
 * @param EntryEasing
 * @text 拡大開始イージング
 * @type select
 * @option Linear（一定） @value linear
 * @option QuadIn（加速開始） @value easeInQuad
 * @option QuadOut（減速終了） @value easeOutQuad
 * @option QuadInOut（加減速） @value easeInOutQuad
 * @option CubicIn @value easeInCubic
 * @option CubicOut @value easeOutCubic
 * @option CubicInOut @value easeInOutCubic
 * @option QuartIn @value easeInQuart
 * @option QuartOut @value easeOutQuart
 * @option QuartInOut @value easeInOutQuart
 * @option QuintIn @value easeInQuint
 * @option QuintOut @value easeOutQuint
 * @option QuintInOut @value easeInOutQuint
 * @option SineIn @value easeInSine
 * @option SineOut @value easeOutSine
 * @option SineInOut @value easeInOutSine
 * @option ExpoIn @value easeInExpo
 * @option ExpoOut @value easeOutExpo
 * @option ExpoInOut @value easeInOutExpo
 * @option CircIn @value easeInCirc
 * @option CircOut @value easeOutCirc
 * @option CircInOut @value easeInOutCirc
 * @option BackIn @value easeInBack
 * @option BackOut @value easeOutBack
 * @option BackInOut @value easeInOutBack
 * @option ElasticIn @value easeInElastic
 * @option ElasticOut @value easeOutElastic
 * @option ElasticInOut @value easeInOutElastic
 * @option BounceIn @value easeInBounce
 * @option BounceOut @value easeOutBounce
 * @option BounceInOut @value easeInOutBounce
 * @default easeOutBack
 * @desc 拡大/移動アニメ開始時のイージング
 *
 * @param ExitEasing
 * @text 戻りイージング
 * @type select
 * @option Linear（一定） @value linear
 * @option QuadIn（加速開始） @value easeInQuad
 * @option QuadOut（減速終了） @value easeOutQuad
 * @option QuadInOut（加減速） @value easeInOutQuad
 * @option CubicIn @value easeInCubic
 * @option CubicOut @value easeOutCubic
 * @option CubicInOut @value easeInOutCubic
 * @option QuartIn @value easeInQuart
 * @option QuartOut @value easeOutQuart
 * @option QuartInOut @value easeInOutQuart
 * @option QuintIn @value easeInQuint
 * @option QuintOut @value easeOutQuint
 * @option QuintInOut @value easeInOutQuint
 * @option SineIn @value easeInSine
 * @option SineOut @value easeOutSine
 * @option SineInOut @value easeInOutSine
 * @option ExpoIn @value easeInExpo
 * @option ExpoOut @value easeOutExpo
 * @option ExpoInOut @value easeInOutExpo
 * @option CircIn @value easeInCirc
 * @option CircOut @value easeOutCirc
 * @option CircInOut @value easeInOutCirc
 * @option BackIn @value easeInBack
 * @option BackOut @value easeOutBack
 * @option BackInOut @value easeInOutBack
 * @option ElasticIn @value easeInElastic
 * @option ElasticOut @value easeOutElastic
 * @option ElasticInOut @value easeInOutElastic
 * @option BounceIn @value easeInBounce
 * @option BounceOut @value easeOutBounce
 * @option BounceInOut @value easeInOutBounce
 * @default easeOutExpo
 * @desc ホバー解除時の戻り挙動のイージング
 */

/*~struct~UISpriteSheetConfig:

 * @param UseSpriteSheet
 * @text スプライトシートを使用
 * @type boolean
 * @default true
 *
 * @param FrameIndex
 * @text 初期フレーム番号
 * @type number
 * @default 0
 *
 * @param FrameCols
 * @text 列数
 * @type number
 * @default 3 
 *
 * @param FrameRows
 * @text 行数
 * @type number
 * @default 1
 * 
 * @param UseAnimation
 * @type boolean
 * @default true
 *
 * @param Loop
 * @text ループ再生
 * @type boolean
 * @default true 
 *
 * @param StartIndex
 * @text 再生開始フレーム
 * @type number
 * @default 0
 *
 * @param EndIndex
 * @text 再生終了フレーム
 * @type number
 * @default 2
 *
 * @param Interval
 * @text 切替間隔(フレーム)
 * @type number
 * @default 6 
 * 
 * @param OffsetX
 * @text X オフセット
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param OffsetY
 * @text Y オフセット
 * @type number
 * @min -9999
 * @max 9999
 * @default 0 
 */

/*~struct~UIHoverExtraImageConfig:
 *
 * @param Image
 * @text 表示画像ファイル
 * @type file
 * @dir img/UI/
 * @desc ホバー中に表示する追加画像
 *
 * @param X
 * @text オフセットX
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param Y
 * @text オフセットY
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param SwitchId
 * @text 表示条件スイッチID
 * @type switch
 * @default 0
 *
 * @param VariableId
 * @text 表示条件変数ID
 * @type variable
 * @default 0
 *
 * @param CompareValue
 * @text 変数の比較値
 * @type number
 * @default 0
 *
 * @param Operator
 * @text 比較演算子
 * @type select
 * @option ==
 * @option >=
 * @option <=
 * @option >
 * @option <
 * @default ==
 */

/*~struct~ConditionConfig:
*
* @param SwitchId
* @text スイッチID
* @type switch
* @default 0

* @param SwitchValue
* @text スイッチの期待値
* @type boolean
* @on ON
* @off OFF
* @default true
* @desc スイッチがこの値と一致する時に条件を満たす

* @param OrSwitchIds
* @text OR条件スイッチ群
* @type switch[]
* @default []
* @desc 複数指定可。いずれかがONなら条件を満たす

* @param VariableId
* @text 変数ID
* @type variable
* @default 0

* @param Operator
* @text 比較方法
* @type select
* @option 等しい（==）
* @option 以上（>=）
* @option 以下（<=）
* @option より大きい（>）
* @option より小さい（<）
* @default 等しい（==）

* @param Value
* @text 比較値
* @type number
* @default 0
* @desc 変数と比較する対象値
*/

/*~struct~UIFadeEffect:
 *
 * @param UseFade
 * @text フェードを使う
 * @type boolean
 * @default false
 *
 * @param FadeFrames
 * @text フレーム数
 * @type number
 * @default 5
 * @min 1
 */

/*~struct~UIClickEffectConfig:
 *
 * @param UseScale
 * @text 拡大を有効化
 * @type boolean
 * @default false
 *
 * @param Scale
 * @text 拡大率（%）
 * @type number
 * @default 110
 *
 * @param UseOffset
 * @text 位置補正を有効化
 * @type boolean
 * @default false
 *
 * @param OffsetX
 * @text X補正
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param OffsetY
 * @text Y補正
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param UseFrame
 * @text フレームを表示
 * @type boolean
 * @default false
 *
 * @param FrameImage
 * @text フレーム画像
 * @type file
 * @dir img/UI/
 * @desc クリック時に表示するフレーム画像
 *
 * @param SpriteSheet
 * @text スプライトシート設定
 * @type struct<UISpriteSheetConfig>
 * @default
 *
 * @param ClickSE
 * @text クリック時SE
 * @type file
 * @dir audio/se/
 * @desc クリック時に再生するSEファイル
 * @default
 *
 * @param Duration
 * @text 効果時間（フレーム）
 * @type number
 * @default 15
 * @min 1
 * @desc クリック効果の継続フレーム数
 *
 * @param Easing
 * @text イージング
 * @type select
 * @option Linear（一定） @value linear
 * @option QuadIn（加速開始） @value easeInQuad
 * @option QuadOut（減速終了） @value easeOutQuad
 * @option QuadInOut（加減速） @value easeInOutQuad
 * @option CubicIn @value easeInCubic
 * @option CubicOut @value easeOutCubic
 * @option CubicInOut @value easeInOutCubic
 * @option QuartIn @value easeInQuart
 * @option QuartOut @value easeOutQuart
 * @option QuartInOut @value easeInOutQuart
 * @option QuintIn @value easeInQuint
 * @option QuintOut @value easeOutQuint
 * @option QuintInOut @value easeInOutQuint
 * @option SineIn @value easeInSine
 * @option SineOut @value easeOutSine
 * @option SineInOut @value easeInOutSine
 * @option ExpoIn @value easeInExpo
 * @option ExpoOut @value easeOutExpo
 * @option ExpoInOut @value easeInOutExpo
 * @option CircIn @value easeInCirc
 * @option CircOut @value easeOutCirc
 * @option CircInOut @value easeInOutCirc
 * @option BackIn @value easeInBack
 * @option BackOut @value easeOutBack
 * @option BackInOut @value easeInOutBack
 * @option ElasticIn @value easeInElastic
 * @option ElasticOut @value easeOutElastic
 * @option ElasticInOut @value easeInOutElastic
 * @option BounceIn @value easeInBounce
 * @option BounceOut @value easeOutBounce
 * @option BounceInOut @value easeInOutBounce
 * @default easeOutQuad
 * @desc クリックアニメのイージング
 */

/*~struct~UIGaugeSprite:
 * @param Type
 * @text ゲージタイプ
 * @type select
 * @option 伸縮         @value elasticity
 * @option 移動         @value slide
 * @default elasticity
 * @desc ゲージの動作方式（伸縮 or 移動）
 *
 * @param Align
 * @text ゲージの進行方向
 * @type select
 * @option left
 * @option right
 * @option up
 * @option down
 * @default right
 * @desc ゲージ方向
 *
 * @param CurrentVarId
 * @text 現在値の変数ID
 * @type variable
 * @desc ゲージの現在値変数ID
 *
 * @param MaxVarId
 * @text 最大値の変数ID
 * @type variable
 * @desc ゲージの最大値変数ID
 *
 * @param BarImage
 * @text ゲージ画像
 * @type file
 * @dir img/UI
 * @desc ゲージとして使う画像
 *
 * @param MaskImage
 * @text マスク画像
 * @type file
 * @dir img/UI
 * @desc マスクに使用する画像
 *
 * @param X
 * @text 表示X座標
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 *
 * @param Y
 * @text 表示Y座標
 * @type number
 * @default 0
 *
 * @param Decoration
 * @text 装飾設定
 * @type struct<UIGaugeDecoration>
 * @desc ゲージに重ねる装飾画像
 *
 * @param SlideDistance
 * @text ゲージの移動距離
 * @type number
 * @min 0
 * @desc ゲージが移動する距離(移動タイプのみ使用)
 *
 * @param Condition
 * @text 表示条件
 * @type struct<ConditionConfig>
 * @desc 表示条件（空欄なら常に表示）
 *
 * @param UseEasing
 * @text イージングを使用する
 * @type boolean
 * @default true
 * @desc 値が変化したときの移動表現を使用するかどうか
 *
 * @param EasingType
 * @text イージングの種類
 * @type select
 * @option linear
 * @option easeInQuad
 * @option easeOutQuad
 * @option easeInOutQuad
 * @option easeInCubic
 * @option easeOutCubic
 * @option easeInOutCubic
 * @option easeInQuart
 * @option easeOutQuart
 * @option easeInOutQuart
 * @option easeInQuint
 * @option easeOutQuint
 * @option easeInOutQuint
 * @option easeInSine
 * @option easeOutSine
 * @option easeInOutSine
 * @option easeInExpo
 * @option easeOutExpo
 * @option easeInOutExpo
 * @option easeInCirc
 * @option easeOutCirc
 * @option easeInOutCirc
 * @option easeInBack
 * @option easeOutBack
 * @option easeInOutBack
 * @option easeInElastic
 * @option easeOutElastic
 * @option easeInOutElastic
 * @option easeInBounce
 * @option easeOutBounce
 * @option easeInOutBounce
 * @default easeInOutQuad
 * @desc イーシング種類を指定
 *
 * @param EasingSpeed
 * @text イージング速度
 * @type number
 * @decimals 2
 * @min 0.01
 * @max 1.0
 * @default 0.15
 * @desc 値に近づく速さ
 *
 * @param FadeFrames
 * @text フェード時間
 * @type number
 * @min 0
 * @default 0
 * @desc 出現・消失時のフェードフレーム数（0で無効）
 */

/*~struct~UIGaugeDecoration:
 *
 * @param BackgroundImage
 * @text 背景画像
 * @type file
 * @dir img/UI
 * @desc ゲージの後ろに表示される背景画像
 *
 * @param BackgroundOffsetX
 * @text 背景画像X補正
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @desc 背景画像のX座標補正
 *
 * @param BackgroundOffsetY
 * @text 背景画像Y補正
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @desc 背景画像のY座標補正
 *
 * @param FrameImage
 * @text フレーム画像
 * @type file
 * @dir img/UI
 * @desc ゲージの上に重ねる装飾枠画像
 *
 * @param FrameOffsetX
 * @text フレーム画像X補正
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @desc フレーム画像のX座標補正
 *
 * @param FrameOffsetY
 * @text フレーム画像Y補正
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @desc フレーム画像のY座標補正
 */

/*~struct~UINumberSprite:
 * @param VariableId
 * @text 表示変数ID
 * @type variable
 * @desc この変数の値をスプライトで表示
 *
 * @param X
 * @text 表示X座標
 * @type number
 * @default 0
 * @desc 数字の表示位置X
 *
 * @param Y
 * @text 表示Y座標
 * @type number
 * @default 0
 * @desc 数字の表示位置Y
 *
 * @param Bitmap
 * @text 数字画像
 * @type file
 * @dir img/system
 * @default Damage
 * @desc 数字表示に使用する画像
 *
 * @param Digit
 * @text 桁数
 * @type number
 * @default 4
 * @desc 表示する最大桁数
 *
 * @param Padding
 * @text 桁間スペース
 * @type number
 * @min -99
 * @max 99
 * @default 0
 * @desc 各桁の間隔
 *
 * @param Align
 * @text 整列方向
 * @type select
 * @option 左揃え @value left
 * @option 中央揃え @value center
 * @option 右揃え @value right
 * @default right
 * @desc 数字の表示位置を設定（左・中央・右）
 *
 * @param PadZero
 * @text ゼロ埋め
 * @type boolean
 * @default false
 * @desc 桁が足りないときに0で埋めるか
 *
 * @param MaxValue
 * @text 最大表示値
 * @type number
 * @min 0
 * @default 0
 * @desc 表示上の最大値（0で無制限）。実数がこれを超えても表示はこの値で止まる
 *
 * @param SwitchId
 * @text 表示スイッチ
 * @type switch
 * @default 0
 * @desc 0で常に表示
 *
 * @param UseSeparator
 * @text 3桁区切りを使用
 * @type boolean
 * @default false
 * @desc true の場合、3桁ごとに区切りカンマを挿入
 *
 * @param SeparatorBitmap
 * @text 区切り画像
 * @type file
 * @dir img/system
 * @desc カンマ用画像（UseSeparator=true時必須）
 *
 * @param FadeFrames
 * @text フェード時間
 * @type number
 * @min 0
 * @default 0
 * @desc 出現・消失時のフェードフレーム数（0で無効）
 *
 */

/*~struct~UITextSprite:
 * @param VariableId
 * @text 表示する変数ID
 * @type variable
 * @desc 表示したい値が格納されている変数番号
 *
 * @param OffsetX
 * @text Xオフセット
 * @type number
 * @default 0
 * @desc UIConfig.X に対する相対位置
 *
 * @param OffsetY
 * @text Yオフセット
 * @type number
 * @default 0
 * @desc UIConfig.Y に対する相対位置
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @min 8
 * @max 72
 * @default 28
 * @desc テキストのフォントサイズ
 *
 * @param FontFace
 * @text フォント名
 * @type string
 * @default
 * @desc 空欄だとデフォルトフォント
 *
 * @param Color
 * @text テキスト色
 * @type string
 * @default #ffffff
 * @desc テキストの色（#ffffff形式またはCSS色名）
 *
 * @param Align
 * @text 文字配置
 * @type select
 * @option 左揃え @value left
 * @option 中央揃え @value center
 * @option 右揃え @value right
 * @default left
 * @desc テキストの配置位置
 *
 * @param OutlineColor
 * @text アウトラインカラー
 * @type string
 * @default #000000
 * @desc テキストアウトラインの色（#ffffff形式またはCSS色名）
 *
 * @param OutlineWidth
 * @text アウトライン幅
 * @type number
 * @min 0
 * @max 10
 * @default 4
 * @desc アウトラインの太さ（0で無効）
 *
 * @param DisplayCondition
 * @text 表示条件
 * @type struct<ConditionConfig>
 * @desc テキスト表示の条件（スイッチ・変数ベース）
 *
 * @param FadeFrames
 * @text フェード時間
 * @type number
 * @min 0
 * @default 0
 * @desc 出現・消失時のフェードフレーム数（0で無効）
 *
 */

(() => {
  "use strict";

  const pluginName = "OnevUImod";
  const parameters = PluginManager.parameters(pluginName);

  if (window.OnevASSISTANT) {
    if (!OnevASSISTANT._uiList) {
      OnevASSISTANT._uiList = new Map();
    }

    if (!OnevASSISTANT._activeSprites) {
      OnevASSISTANT._activeSprites = {};
    }
    if (!OnevASSISTANT._activeUINames) {
      OnevASSISTANT._activeUINames = new Set();
    }

    if (!OnevASSISTANT._dynamicUIConfigs) {
      OnevASSISTANT._dynamicUIConfigs = new Map();
    }

    if (!OnevASSISTANT._previousStates) {
      OnevASSISTANT._previousStates = new Map();
    }

    
    OnevASSISTANT.checkCondition = function(condition) {
      if (!condition) {
        return true;
      }
      
      try {
        if (condition.SwitchId && Number(condition.SwitchId) > 0) {
          const switchId = Number(condition.SwitchId);
          const expectedValue = condition.SwitchValue === "true" || condition.SwitchValue === true;
          const actualValue = $gameSwitches.value(switchId);
          if (actualValue !== expectedValue) return false;
        }
        
        if (condition.OrSwitchIds) {
          let orSwitchIds = condition.OrSwitchIds;
          if (typeof orSwitchIds === "string") {
            try {
              orSwitchIds = JSON.parse(orSwitchIds);
            } catch (e) {
              orSwitchIds = [];
            }
          }
          if (Array.isArray(orSwitchIds) && orSwitchIds.length > 0) {
            const validIds = orSwitchIds.map(id => Number(id)).filter(id => id > 0);
            if (validIds.length > 0) {
              const anyOn = validIds.some(id => $gameSwitches.value(id));
              if (!anyOn) return false;
            }
          }
        }
        
        if (condition.VariableId && Number(condition.VariableId) > 0) {
          const variableId = Number(condition.VariableId);
          const actualValue = $gameVariables.value(variableId);
          const compareValue = Number(condition.Value || 0);
          const operator = condition.Operator || "等しい（==）";
          
          switch (operator) {
            case "等しい（==）":
              if (actualValue !== compareValue) return false;
              break;
            case "以上（>=）":
              if (actualValue < compareValue) return false;
              break;
            case "以下（<=）":
              if (actualValue > compareValue) return false;
              break;
            case "より大きい（>）":
              if (actualValue <= compareValue) return false;
              break;
            case "より小さい（<）":
              if (actualValue >= compareValue) return false;
              break;
          }
        }
        
        return true; 
      } catch (error) {
        console.warn("ConditionConfig check error:", error);
        return true; 
      }
    };
    
    OnevASSISTANT.updateOnevUImodStates = function() {
      if (!this._monitoredUIs) {
        return;
      }

      const ensureStateEntry = (uiName) => {
        if (!this._previousStates) {
          this._previousStates = new Map();
        }
        let state = this._previousStates.get(uiName);
        if (!state) {
          state = {
            baseVisible: undefined,
            imageVisible: {},
            choiceVisible: {},
            choiceAlpha: {},
            choiceEnabled: {}
          };
          this._previousStates.set(uiName, state);
        }
        return state;
      };

      const applyVisibilityIfChanged = (stateBucket, index, sprite, isVisible) => {
        if (!sprite) {
          return;
        }
        const prev = stateBucket[index];
        if (prev === isVisible) {
          return;
        }
        stateBucket[index] = isVisible;
        
        if (sprite._choiceMeta && sprite._fadeConfig) {
          sprite._targetVisible = isVisible;
          sprite._nextVisible = isVisible;
          
          if (String(sprite._fadeConfig.UseFade) === "true") {
          } else {
            sprite.visible = isVisible;
            sprite.alpha = isVisible ? 1 : 0;
          }
        } else {
          sprite.visible = isVisible;
        }
      };

      const applyAlphaIfChanged = (stateBucket, index, sprite, alpha) => {
        if (!sprite) {
          return;
        }
        const prev = stateBucket[index];
        if (prev !== undefined && Math.abs(prev - alpha) < 0.001) {
          return;
        }
        stateBucket[index] = alpha;
        
        sprite.alpha = alpha;
      };

      const applyTouchEnabledIfChanged = (stateBucket, index, sprite, enabled) => {
        if (!sprite) {
          return;
        }
        const prev = stateBucket[index];
        if (prev === enabled) {
          return;
        }
        stateBucket[index] = enabled;
        
        if (sprite._choiceMeta) {
          sprite.interactive = enabled;
          sprite.buttonMode = enabled;
          if (sprite._content) {
            sprite._content.interactive = enabled;
            sprite._content.buttonMode = enabled;
          }
        } else if (sprite._touchEnabled !== undefined) {
          sprite._touchEnabled = enabled;
        }
      };

      const normalizeOp = (op) => {
        switch (op) {
          case "==":
          case "等しい（==）":
            return "==";
          case "!=":
          case "等しくない（!=）":
            return "!=";
          case ">=":
          case "以上（>=）":
            return ">=";
          case "<=":
          case "以下（<=）":
            return "<=";
          case ">":
          case "より大きい（>）":
            return ">";
          case "<":
          case "より小さい（<）":
            return "<";
          default:
            return "==";
        }
      };

      const compareByOperator = (value, operator, compareValue) => {
        const op = normalizeOp(operator);
        switch (op) {
          case "==":
            return value == compareValue;
          case "!=":
            return value != compareValue;
          case ">=":
            return value >= compareValue;
          case "<=":
            return value <= compareValue;
          case ">":
            return value > compareValue;
          case "<":
            return value < compareValue;
          default:
            return true;
        }
      };

      const isImageProhibited = (imageConfig) => {
        if (!imageConfig) {
          return false;
        }

        const prohibitSwitch = Number(imageConfig.ProhibitSwitch || 0);
        if (prohibitSwitch > 0 && $gameSwitches.value(prohibitSwitch)) {
          return true;
        }

        const prohibitVariable = Number(imageConfig.ProhibitVariable || 0);
        if (prohibitVariable > 0) {
          const op = imageConfig.ProhibitOperator || "等しい（==）";
          const cmp = Number(imageConfig.ProhibitValue || 0);
          const variableValue = $gameVariables.value(prohibitVariable);
          if (compareByOperator(variableValue, op, cmp)) {
            return true;
          }
        }

        return false;
      };
      
      for (const [name, uiData] of this._monitoredUIs.entries()) {
        if (!uiData.sprite) {
          continue;
        }
        
        if (!uiData.sprite.parent) {
          continue;
        }
        
        const config = uiData.config;
        const sprite = uiData.sprite;
        const stateCache = ensureStateEntry(name);
        
        let processedImageSprites = config.ImageSprites;
        let processedChoiceSprites = config.ChoiceSprites;
        
        if (typeof config.ImageSprites === 'string') {
          try {
            processedImageSprites = JSON.parse(config.ImageSprites);
          } catch (error) {
            processedImageSprites = [];
          }
        }
        
        if (typeof config.ChoiceSprites === 'string') {
          try {
            processedChoiceSprites = JSON.parse(config.ChoiceSprites);
          } catch (error) {
            processedChoiceSprites = [];
          }
        }
        
        const base = typeof config.UIbaseSettings === "string" 
          ? this.safeJsonParse(config.UIbaseSettings)
          : config.UIbaseSettings;
        
        if (base && base.DisplayCondition) {
          const shouldShow = this.checkCondition(base.DisplayCondition);
          if (stateCache.baseVisible !== shouldShow) {
            sprite.visible = shouldShow;
            stateCache.baseVisible = shouldShow;
          }
        } else {
          stateCache.baseVisible = sprite.visible;
        }
        
        if (sprite._onevImageChildren && processedImageSprites) {
          sprite._onevImageChildren.forEach((imageSprite, index) => {
            let imageConfig = processedImageSprites[index];
            
            if (typeof imageConfig === 'string') {
              try {
                imageConfig = JSON.parse(imageConfig);
              } catch (error) {
                imageConfig = {};
              }
            }
            
            let visibilityHandled = false;
            const hasProhibitSetting =
              (imageConfig && Number(imageConfig.ProhibitSwitch || 0) > 0) ||
              (imageConfig && Number(imageConfig.ProhibitVariable || 0) > 0);

            if (imageConfig && imageConfig.DisplayCondition) {
              const shouldShow = this.checkCondition(imageConfig.DisplayCondition);
              const finalShow = shouldShow && !isImageProhibited(imageConfig);
              applyVisibilityIfChanged(stateCache.imageVisible, index, imageSprite, finalShow);
              visibilityHandled = true;
            }
            
            if (imageConfig) {
              const showVarNum = Number(imageConfig.ShowVariable);
              const showSwitchNum = Number(imageConfig.ShowSwitch);
              
              if (showVarNum > 0 || showSwitchNum > 0) {
                const condition = {};
                
                if (showSwitchNum > 0) {
                  condition.SwitchId = showSwitchNum;
                  condition.SwitchValue = true;
                }
                
                if (showVarNum > 0) {
                  condition.VariableId = showVarNum;
                  
                  const operator = imageConfig.ShowOperator || "等しい（==）";
                  const value = Number(imageConfig.ShowValue || 0);
                  
                  if (operator === "等しい（==）") {
                    condition.VariableValue = value;
                  } else if (operator === "以上（>=）") {
                    condition.VariableMin = value;
                  } else if (operator === "以下（<=）") {
                    condition.VariableMax = value;
                  } else if (operator === "より大きい（>）") {
                    condition.VariableMin = value + 1;
                  } else if (operator === "より小さい（<）") {
                    condition.VariableMax = value - 1;
                  }
                }
                
                const shouldShow = this.checkCondition(condition);
                const finalShow = shouldShow && !isImageProhibited(imageConfig);
                applyVisibilityIfChanged(stateCache.imageVisible, index, imageSprite, finalShow);
                visibilityHandled = true;
              } else if (!visibilityHandled && hasProhibitSetting) {
                const finalShow = !isImageProhibited(imageConfig);
                applyVisibilityIfChanged(stateCache.imageVisible, index, imageSprite, finalShow);
                visibilityHandled = true;
              }
            }
          });
        }
        
        if (sprite._onevChoiceChildren && processedChoiceSprites) {
          
          sprite._onevChoiceChildren.forEach((choiceSprite, index) => {
            let choiceConfig = null;
            
            if (choiceSprite._choiceMeta && choiceSprite._choiceMeta.Image) {
              const spriteImageName = choiceSprite._choiceMeta.Image;
              
              choiceConfig = processedChoiceSprites.find(config => {
                if (typeof config === 'string') {
                  try {
                    const parsedConfig = JSON.parse(config);
                    return parsedConfig.Image === spriteImageName;
                  } catch (error) {
                    return false;
                  }
                }
                return config.Image === spriteImageName;
              });
              
              if (!choiceConfig) {
                choiceConfig = processedChoiceSprites[index];
              }
              
            } else {
              choiceConfig = processedChoiceSprites[index];
            }
            
            if (typeof choiceConfig === 'string') {
              try {
                choiceConfig = JSON.parse(choiceConfig);
              } catch (error) {
                choiceConfig = {};
              }
            }
            
            if (choiceConfig) {
              let showCondition = null;
              
              if (choiceConfig.DisplayCondition) {
                showCondition = choiceConfig.DisplayCondition;
              }
              else if (choiceConfig.ShowCondition) {
                try {
                  const parsedShowCondition = JSON.parse(choiceConfig.ShowCondition);
                  
                  const hasValidCondition = Number(parsedShowCondition.SwitchId) > 0 || Number(parsedShowCondition.VariableId) > 0;
                  
                  if (hasValidCondition) {
                    showCondition = {};
                    
                    if (Number(parsedShowCondition.SwitchId) > 0) {
                      showCondition.SwitchId = Number(parsedShowCondition.SwitchId);
                      showCondition.SwitchValue = parsedShowCondition.SwitchValue !== "false" && parsedShowCondition.SwitchValue !== false;
                    }
                    if (Number(parsedShowCondition.VariableId) > 0) {
                      showCondition.VariableId = Number(parsedShowCondition.VariableId);
                      
                      const operator = parsedShowCondition.Operator || "等しい（==）";
                      const value = Number(parsedShowCondition.Value || 0);
                      
                      if (operator === "等しい（==）") {
                        showCondition.VariableValue = value;
                      } else if (operator === "以上（>=）") {
                        showCondition.VariableMin = value;
                      } else if (operator === "以下（<=）") {
                        showCondition.VariableMax = value;
                      } else if (operator === "より大きい（>）") {
                        showCondition.VariableMin = value + 1;
                      } else if (operator === "より小さい（<）") {
                        showCondition.VariableMax = value - 1;
                      }
                    }
                  }
                } catch (error) {
                }
              }
              
              if (showCondition) {
                const shouldShow = this.checkCondition(showCondition);
                applyVisibilityIfChanged(stateCache.choiceVisible, index, choiceSprite, shouldShow);
              } else {
                applyVisibilityIfChanged(stateCache.choiceVisible, index, choiceSprite, true);
              }
              
              let enableCondition = null;
              
              if (choiceConfig.EnableCondition) {
                if (typeof choiceConfig.EnableCondition === 'string') {
                  try {
                    const parsedEnableCondition = JSON.parse(choiceConfig.EnableCondition);
                    
                    const hasValidCondition = Number(parsedEnableCondition.SwitchId) > 0 || Number(parsedEnableCondition.VariableId) > 0;
                    
                    if (hasValidCondition) {
                      enableCondition = {};
                      
                      if (Number(parsedEnableCondition.SwitchId) > 0) {
                        enableCondition.SwitchId = Number(parsedEnableCondition.SwitchId);
                        enableCondition.SwitchValue = parsedEnableCondition.SwitchValue !== "false" && parsedEnableCondition.SwitchValue !== false;
                      }
                      if (Number(parsedEnableCondition.VariableId) > 0) {
                        enableCondition.VariableId = Number(parsedEnableCondition.VariableId);
                        
                        const operator = parsedEnableCondition.Operator || "等しい（==）";
                        const value = Number(parsedEnableCondition.Value || 0);
                        
                        if (operator === "等しい（==）") {
                          enableCondition.VariableValue = value;
                        } else if (operator === "以上（>=）") {
                          enableCondition.VariableMin = value;
                        } else if (operator === "以下（<=）") {
                          enableCondition.VariableMax = value;
                        }
                      }
                    }
                  } catch (error) {
                  }
                } else {
                  enableCondition = choiceConfig.EnableCondition;
                }
              }
              
              if (enableCondition) {
                const isEnabled = this.checkCondition(enableCondition);
                applyAlphaIfChanged(stateCache.choiceAlpha, index, choiceSprite, isEnabled ? 1.0 : 0.5);
                applyTouchEnabledIfChanged(stateCache.choiceEnabled, index, choiceSprite, isEnabled);
              }
            }
          });
        }
      }
    };
    
    OnevASSISTANT.forceUpdateUIStates = function(modeOrOptions) {
      var mode = "full";

      if (typeof modeOrOptions === "string") {
        mode = modeOrOptions;
      } else if (modeOrOptions && typeof modeOrOptions === "object" && modeOrOptions.mode) {
        mode = modeOrOptions.mode;
      }

      if (mode === "light") {
        if (this.updateOnevUImodStates) {
          this.updateOnevUImodStates();
        }
        return;
      }

      this.updateOnevUImodStates();

      if (this.updateAllUIConditions) {
        this.updateAllUIConditions();
      } else if (this.checkAndUpdateUIVisibility) {
        this.checkAndUpdateUIVisibility();
      } else if (this.refreshUIStates) {
        this.refreshUIStates();
      }
    };



    const originalCreateUI = OnevASSISTANT.createUI;

    OnevASSISTANT.createUI = function (name) {
      if (OnevASSISTANT._dynamicUIConfigs.has(name)) {
        const dynamicConfig = OnevASSISTANT._dynamicUIConfigs.get(name);

        if (OnevASSISTANT._activeUINames.has(name)) {
          const pc = SceneManager._scene?._spriteset?._pictureContainer;
          const old = OnevASSISTANT._activeSprites[name];

          if (!old || old.parent !== pc) {
            if (old && old.parent) old.parent.removeChild(old);
            delete OnevASSISTANT._activeSprites[name];
            OnevASSISTANT._activeUINames.delete(name);
          } else {
            return;
          }
        }

        const target = dynamicConfig;

        OnevASSISTANT._activeUINames.add(name);
        $gameSystem.showCustomUI(name);
        


        const base = typeof target.UIbaseSettings === "string" 
          ? OnevASSISTANT.safeJsonParse(target.UIbaseSettings)
          : target.UIbaseSettings;

        if (!base) {
          return;
        }

        const sprite = base.Baseimage 
          ? new Sprite(ImageManager.loadBitmap("img/UI/", base.Baseimage))
          : new Sprite();
        sprite.sortableChildren = true;
        const pos = $gameSystem.getUIPosition(name);
        sprite.x = Number(pos?.x || target.X || 0);
        sprite.y = Number(pos?.y || target.Y || 0);
        sprite.z = Number(target.PictureId || 50);
        sprite._onevUIName = name;
        sprite._createdByUImod = true; 
        sprite._uiConfig = target; 

        const imageSprites = OnevASSISTANT.createAndAttachImageSprites(
          sprite,
          JSON.stringify(target.ImageSprites || []),
          []
        );
        sprite._onevImageChildren = imageSprites;
        
        if (!OnevASSISTANT._monitoredUIs) {
          OnevASSISTANT._monitoredUIs = new Map();
        }
        
        const monitorData = {
          sprite: sprite,
          config: target,
          type: 'dynamic'
        };
        
        OnevASSISTANT._monitoredUIs.set(name, monitorData);

        const switchId = Number(base.EnableSwitchId || 0);
        if (switchId > 0) {
          if (!OnevASSISTANT._dragInfo) {
            OnevASSISTANT._dragInfo = {};
          }
          OnevASSISTANT._dragInfo[name] = {
            sprite: sprite,
            switchId: switchId,
            dragging: false,
            offsetX: 0,
            offsetY: 0,
          };
        }

        try {
          const pc = SceneManager._scene._spriteset._pictureContainer;
          if (pc) {
            const idx = Math.max(0, Number(target.PictureId || 50) - 1);
            const safeIdx = Math.min(idx, pc.children.length);
            pc.addChildAt(sprite, safeIdx);
          } else {
            if (SceneManager._scene._spriteset) {
              SceneManager._scene._spriteset.addChild(sprite);
            }
          }
        } catch (error) {}

        sprite._onevGaugeChildren = sprite._onevGaugeChildren || [];
        sprite._onevNumberChildren = sprite._onevNumberChildren || [];
        sprite._onevImageChildren = sprite._onevImageChildren || [];
        sprite._onevChoiceChildren = sprite._onevChoiceChildren || [];
        sprite._onevTextChildren = sprite._onevTextChildren || [];

        OnevASSISTANT._activeSprites[name] = sprite;

        try {
          const choiceSpriteData = target.ChoiceSprites || [];
          if (OnevASSISTANT.createChoiceSprites) {
            OnevASSISTANT.createChoiceSprites(sprite, choiceSpriteData);
            sprite._onevChoiceChildren = sprite._onevChoiceChildren || [];
          }
        } catch (error) {
          sprite._onevChoiceChildren = [];
        }

        try {
          let gaugeDefs = target.GaugeSprites || [];

          if (gaugeDefs.length > 0 && typeof gaugeDefs[0] === "string") {
            gaugeDefs = gaugeDefs
              .map((def) => {
                try {
                  return JSON.parse(def);
                } catch (error) {
                  return null;
                }
              })
              .filter((def) => def !== null);
          }

          if (OnevASSISTANT.createGaugeSprites) {
            const gaugeSprites = OnevASSISTANT.createGaugeSprites(
              sprite,
              gaugeDefs
            );
            sprite._onevGaugeChildren = gaugeSprites || [];
          }
          sprite._onevGaugeChildren = sprite._onevGaugeChildren || [];
        } catch (error) {
          sprite._onevGaugeChildren = [];
        }

        try {
          let numberDefs = target.NumberSprites || [];

          if (numberDefs.length > 0 && typeof numberDefs[0] === "string") {
            numberDefs = numberDefs
              .map((def) => {
                try {
                  return JSON.parse(def);
                } catch (error) {
                  return null;
                }
              })
              .filter((def) => def !== null);
          }

          if (OnevASSISTANT.createNumberSprites) {
            const numberSprites = OnevASSISTANT.createNumberSprites(
              sprite,
              numberDefs
            );
            sprite._onevNumberChildren = numberSprites || [];
          }
          sprite._onevNumberChildren = sprite._onevNumberChildren || [];
        } catch (error) {
          sprite._onevNumberChildren = [];
        }

        try {
          let textDefs = target.TextSprites || [];

          if (textDefs.length > 0 && typeof textDefs[0] === "string") {
            textDefs = textDefs
              .map((def) => {
                try {
                  return JSON.parse(def);
                } catch (error) {
                  return null;
                }
              })
              .filter((def) => def !== null);
          }

          if (OnevASSISTANT.createTextSprites) {
            const textSprites = OnevASSISTANT.createTextSprites(
              sprite,
              textDefs
            );
            sprite._onevTextChildren = textSprites || [];
          }
          sprite._onevTextChildren = sprite._onevTextChildren || [];
        } catch (error) {
          sprite._onevTextChildren = [];
        }

        const activeIds = target.ActiveSwitchIds || [];
        activeIds.forEach((id) => {
          if (+id > 0) $gameSwitches.setValue(+id, true);
        });

        if (OnevASSISTANT.registerUIForMonitoring) {
          OnevASSISTANT.registerUIForMonitoring(name, sprite, target);
        }
        
        if (OnevASSISTANT._uiConfigs) {
          OnevASSISTANT._uiConfigs[name] = target;
        }
        if (OnevASSISTANT._uiSprites) {
          OnevASSISTANT._uiSprites[name] = sprite;
        }
        if (OnevASSISTANT._registeredUIs) {
          OnevASSISTANT._registeredUIs.set(name, { sprite, config: target });
        }
        
        if (OnevASSISTANT._monitorTargets) {
          if (!OnevASSISTANT._monitorTargets[name]) {
            OnevASSISTANT._monitorTargets[name] = [];
          }
          
          if (sprite._onevImageChildren) {
            sprite._onevImageChildren.forEach((imgSprite, index) => {
              const imgConfig = target.ImageSprites && target.ImageSprites[index];
              const hasShowCondition =
                Number((imgConfig && imgConfig.ShowVariable) || 0) > 0 ||
                Number((imgConfig && imgConfig.ShowSwitch) || 0) > 0;
              const hasProhibitCondition =
                Number((imgConfig && imgConfig.ProhibitVariable) || 0) > 0 ||
                Number((imgConfig && imgConfig.ProhibitSwitch) || 0) > 0;
              if (imgConfig && (hasShowCondition || hasProhibitCondition)) {
                OnevASSISTANT._monitorTargets[name].push({
                  type: 'image',
                  sprite: imgSprite,
                  config: imgConfig,
                  index: index
                });
              }
            });
          }
          
          if (sprite._onevChoiceChildren) {
            sprite._onevChoiceChildren.forEach((choiceSprite, index) => {
              const choiceConfig = target.ChoiceSprites && target.ChoiceSprites[index];
              if (choiceConfig && (choiceConfig.ShowCondition || choiceConfig.EnableCondition)) {
                OnevASSISTANT._monitorTargets[name].push({
                  type: 'choice',
                  sprite: choiceSprite,
                  config: choiceConfig,
                  index: index
                });
              }
            });
          }
          
        }
        
        if (OnevASSISTANT.updateOnevUImodStates) {
          OnevASSISTANT.updateOnevUImodStates();
        }
        
        setTimeout(() => {
          if (OnevASSISTANT.updateOnevUImodStates) {
            OnevASSISTANT.updateOnevUImodStates();
          }
          
          if (OnevASSISTANT.forceUpdateUIStates) {
            OnevASSISTANT.forceUpdateUIStates();
          }
          if (OnevASSISTANT.updateUI) {
            OnevASSISTANT.updateUI(name);
          }
          if (OnevASSISTANT.refreshUI) {
            OnevASSISTANT.refreshUI(name);
          }
          if (OnevASSISTANT.checkUIConditions) {
            OnevASSISTANT.checkUIConditions();
          }
        }, 100);

        return target;
      } else {
        return originalCreateUI.call(this, name);
      }
    };

    const _Scene_Map_update_UImod = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
      _Scene_Map_update_UImod.call(this);
      
      if (!this._onevUImodUpdateCounter) this._onevUImodUpdateCounter = 0;
      this._onevUImodUpdateCounter++;
      
      if (this._onevUImodUpdateCounter >= 300) {
        this._onevUImodUpdateCounter = 0;
        if (window.OnevASSISTANT && OnevASSISTANT.updateOnevUImodStates && !$gameVariables._onevUImodUpdating) {
          console.log("[OnevUImod] 定期更新を実行");
          OnevASSISTANT.updateOnevUImodStates();
        }
      }
    };

    const _Scene_Battle_update_UImod = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function() {
      _Scene_Battle_update_UImod.call(this);
      
      if (!this._onevUImodUpdateCounter) this._onevUImodUpdateCounter = 0;
      this._onevUImodUpdateCounter++;
      
      if (this._onevUImodUpdateCounter >= 300) {
        this._onevUImodUpdateCounter = 0;
        if (window.OnevASSISTANT && OnevASSISTANT.updateOnevUImodStates && !$gameVariables._onevUImodUpdating) {
          console.log("[OnevUImod] 定期更新を実行");
          OnevASSISTANT.updateOnevUImodStates();
        }
      }
    };
  }

  PluginManager.registerCommand(pluginName, "UI作成", function (args) {
    if (!window.OnevASSISTANT) {
      console.error("OnevASSISTANT.js発見できず");
      return;
    }

    const baseSettings = args.UIbaseSettings
      ? JSON.parse(args.UIbaseSettings)
      : {};
    const activeSwitchIds = args.ActiveSwitchIds
      ? JSON.parse(args.ActiveSwitchIds)
      : [];
    const imageSprites = args.ImageSprites ? JSON.parse(args.ImageSprites) : [];
    const choiceSprites = args.ChoiceSprites
      ? JSON.parse(args.ChoiceSprites)
      : [];
    const gaugeSprites = args.GaugeSprites ? JSON.parse(args.GaugeSprites) : [];
    const numberSprites = args.NumberSprites
      ? JSON.parse(args.NumberSprites)
      : [];
    const textSprites = args.TextSprites ? JSON.parse(args.TextSprites) : [];
    const name = args.Name;
    const configData = {
      Name: name,
      UIbaseSettings: baseSettings,
      X: Number(args.X || 0),
      Y: Number(args.Y || 0),
      PictureId: Number(args.PictureId || 50),
      ImageSprites: imageSprites,
      ChoiceSprites: choiceSprites,
      GaugeSprites: gaugeSprites,
      NumberSprites: numberSprites,
      TextSprites: textSprites,
      ActiveSwitchIds: activeSwitchIds,
    };

    if (!window.OnevASSISTANT._uiConfigs) {
      window.OnevASSISTANT._uiConfigs = {};
    }
    window.OnevASSISTANT._uiConfigs[name] = configData;

    const options = {
      enableSwitchId: Number(baseSettings.EnableSwitchId || 0),
      baseImage: baseSettings.Baseimage || "",
      x: Number(args.X || 0),
      y: Number(args.Y || 0),
      pictureId: Number(args.PictureId || 50),
      imageSprites: imageSprites,
      choiceSprites: choiceSprites,
      gaugeSprites: gaugeSprites,
      numberSprites: numberSprites,
      textSprites: textSprites,
      activeSwitchIds: activeSwitchIds.map((id) => Number(id)),
    };

    if (OnevASSISTANT && OnevASSISTANT.createUI) {
      try {
        OnevASSISTANT._dynamicUIConfigs.set(name, configData);

        const newUI = OnevASSISTANT.createUI(name);
        
        if (OnevASSISTANT._stopEventDuringChoice && choiceSprites.length > 0) {
          
          setTimeout(() => {
            if (OnevASSISTANT.updateUIVisibility) {
              OnevASSISTANT.updateUIVisibility();
            }
          }, 100); 
        }
      } catch (error) {
        console.error('[OnevUImod] UI作成エラー:', error);
      }
    }
  });

  PluginManager.registerCommand(pluginName, "UI消去", function (args) {
    if (!window.OnevASSISTANT) {
      console.error("OnevASSISTANT.js発見できず");
      return;
    }

    const name = args.name;
    if (!name) {
      console.error("UI名指定なし");
      return;
    }

    try {
      if (OnevASSISTANT.removeUI) {
        OnevASSISTANT.removeUI(name);
      }
    } catch (error) {}
  });

  PluginManager.registerCommand(pluginName, "UI全消去", function (args) {
    if (!window.OnevASSISTANT) {
      console.error("OnevASSISTANT.js発見できず");
      return;
    }

    try {
      if (OnevASSISTANT.removeAllUI) {
        OnevASSISTANT.removeAllUI();
      }
    } catch (error) {}
  });

  PluginManager.registerCommand(pluginName, "UI非表示", function (args) {
    if (!window.OnevASSISTANT) {
      console.error("OnevASSISTANT.js発見できず");
      return;
    }

    const name = args.name;
    if (!name) {
      console.error("UI名指定なし");
      return;
    }

    const fadeFrames = Number(args.fadeFrames || 0);

    try {
      if (OnevASSISTANT.hideUIWithFade) {
        OnevASSISTANT.hideUIWithFade(name, fadeFrames);
      } else if (OnevASSISTANT.hideUI) {
        OnevASSISTANT.hideUI(name);
      }
    } catch (error) {
      console.error('[OnevUImod] UI非表示エラー:', error);
    }
  });

  PluginManager.registerCommand(pluginName, "UI表示", function (args) {
    if (!window.OnevASSISTANT) {
      console.error("OnevASSISTANT.js発見できず");
      return;
    }

    const name = args.name;
    if (!name) {
      console.error("UI名指定なし");
      return;
    }

    const fadeFrames = Number(args.fadeFrames || 0);

    try {
      if (OnevASSISTANT.showUIWithFade) {
        OnevASSISTANT.showUIWithFade(name, fadeFrames);
      } else if (OnevASSISTANT.showUI) {
        OnevASSISTANT.showUI(name);
      }
    } catch (error) {
      console.error('[OnevUImod] UI表示エラー:', error);
    }
  });

  PluginManager.registerCommand(pluginName, "UI全非表示", function (args) {
    if (!window.OnevASSISTANT) {
      console.error("OnevASSISTANT.js発見できず");
      return;
    }

    const fadeFrames = Number(args.fadeFrames || 0);

    try {
      if (fadeFrames > 0 && OnevASSISTANT.hideAllUIWithFade) {
        OnevASSISTANT.hideAllUIWithFade(fadeFrames);
      } else if (OnevASSISTANT.hideAllUI) {
        OnevASSISTANT.hideAllUI();
      }
    } catch (error) {
      console.error('[OnevUImod] UI全非表示エラー:', error);
    }
  });

  PluginManager.registerCommand(pluginName, "UI再表示", function (args) {
    if (!window.OnevASSISTANT) {
      console.error("OnevASSISTANT.js発見できず");
      return;
    }

    const fadeFrames = Number(args.fadeFrames || 0);

    try {
      if (fadeFrames > 0 && OnevASSISTANT.showAllUIWithFade) {
        OnevASSISTANT.showAllUIWithFade(fadeFrames);
      } else if (OnevASSISTANT.showAllUI) {
        OnevASSISTANT.showAllUI();
      }
    } catch (error) {
      console.error('[OnevUImod] UI再表示エラー:', error);
    }
  });

  if (window.Game_Variables && window.Game_Switches) {
    let _onevUImodUpdating = false;
    let _onevUImodUpdateScheduled = false;
    
    const scheduleUIUpdate = function() {
      if (_onevUImodUpdateScheduled || _onevUImodUpdating) {
        return;
      }
      _onevUImodUpdateScheduled = true;
      
      setTimeout(() => {
        _onevUImodUpdateScheduled = false;
        if (_onevUImodUpdating) {
          return;
        }
        _onevUImodUpdating = true;
        try {
          if (window.OnevASSISTANT) {
            if (OnevASSISTANT.forceUpdateUIStates) {
              OnevASSISTANT.forceUpdateUIStates("light");
            } else if (OnevASSISTANT.updateOnevUImodStates) {
              OnevASSISTANT.updateOnevUImodStates();
            } else if (OnevASSISTANT.checkUIConditions) {
              OnevASSISTANT.checkUIConditions();
            }
            
            if (OnevASSISTANT.updateGaugeSprites) {
              OnevASSISTANT.updateGaugeSprites();
            }
            
            if (OnevASSISTANT.updateNumberSprites) {
              OnevASSISTANT.updateNumberSprites();
            }
            
            if (OnevASSISTANT.updateTextSprites) {
              OnevASSISTANT.updateTextSprites();
            }
          }
        } finally {
          _onevUImodUpdating = false;
        }
      }, 0);
    };

    const _Game_Variables_setValue = Game_Variables.prototype.setValue;
    Game_Variables.prototype.setValue = function(variableId, value) {
      _Game_Variables_setValue.call(this, variableId, value);
      scheduleUIUpdate();
    };

    const _Game_Switches_setValue = Game_Switches.prototype.setValue;
    Game_Switches.prototype.setValue = function(switchId, value) {
      _Game_Switches_setValue.call(this, switchId, value);
      scheduleUIUpdate();
    };
  }
})();
