/*:
 * @target MZ
 * @plugindesc Spinespine補助プラグイン
 * @author Onmoremind
 * 
 * @param resetAnimationName
 * @text リセット用アニメ名
 * @type string
 * @default idle
 * @desc トラックリセット時に使用するアニメ名
 *
 * @param pictureIdVariable
 * @text ピクチャID変数
 * @type variable
 * @default 0
 * @desc ピクチャIDを格納する変数
 *
 * @param xVariable
 * @text X座標変数
 * @type variable
 * @default 0
 * @desc X座標を格納する変数
 *
 * @param yVariable
 * @text Y座標変数
 * @type variable
 * @default 0
 * @desc Y座標を格納する変数
 *
 * @param scaleVariable
 * @text 拡大率変数
 * @type variable
 * @default 0
 * @desc 拡大率を格納する変数
 *
 * @param skeletonNameVariable
 * @text スケルトン名変数
 * @type variable
 * @default 0
 * @desc スケルトン名を格納する変数
 *
 * @param trackVariable
 * @text トラック番号変数
 * @type variable
 * @default 0
 * @desc トラック番号を格納する変数
 *
 * @param animationNameVariable
 * @text アニメーション名変数
 * @type variable
 * @default 0
 * @desc アニメーション名を格納する変数
 *
 * @command CallSpineFull
 * @text Spine呼出
 * @desc 指定した設定でSpine呼出
 *
 * @arg pictureId
 * @type number
 * @text ピクチャID
 * @desc 使用するピクチャ番号（例：5）
 *
 * @arg x
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @text X座標
 *
 * @arg y
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @text Y座標
 *
 * @arg pictureScale
 * @type number
 * @min 1
 * @max 500
 * @default 100
 * @text ピクチャ拡大率（％）
 *
 * @arg skeletonName
 * @type string
 * @text スケルトン名
 * 
 * @arg trackSettings
 * @type struct<SpineTrack>[]
 * @text アニメーション設定

 * @arg layerSkinSettings
 * @type struct<SkinLayerEntry>[]
 * @text レイヤースキン設定
 * @desc レイヤー名ごとに適用するスキン名を指定（例: レイヤー=face, スキン=normal → face/normal を適用）
 *
 * @arg mixValue
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.3
 * @text ミックス時間（秒）
 * 
 * @arg mosaicSettings
 * @type struct<MosaicData>[]
 * @text モザイク設定リスト
 *
 * @arg useFade
 * @type boolean
 * @default false
 * @text フェード表示
 *
 * @arg fadeDuration
 * @type number
 * @default 30
 * @text フェード時間（フレーム）
 * 
 * @command CallSpineVar
 * @text Spinevar呼出
 * @desc 変数参照でSpine呼出（pictureID、X、Y座標に加え拡大率・スケルトン名も変数から取得可能）
 *
 * @arg pictureIdVar
 * @type variable
 * @text ピクチャID変数
 * @desc ピクチャIDが格納されている変数
 *
 * @arg xVar
 * @type variable
 * @text X座標変数
 * @desc X座標が格納されている変数
 *
 * @arg yVar
 * @type variable
 * @text Y座標変数
 * @desc Y座標が格納されている変数
 *
 * @arg pictureScaleVar
 * @type variable
 * @text 拡大率変数
 * @desc 拡大率に使用している変数
 *
 * @arg skeletonNameVar
 * @type variable
 * @text スケルトン名変数
 * @desc スケルトン名に使用している変数
 * 
 * @arg trackSettings
 * @type struct<SpineTrack>[]
 * @text アニメーション設定

 * @arg layerSkinSettings
 * @type struct<SkinLayerEntry>[]
 * @text レイヤースキン設定
 * @desc レイヤー名ごとに適用するスキン名を指定
 *
 * @arg mixValue
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.3
 * @text ミックス時間（秒）
 * 
 * @arg mosaicSettings
 * @type struct<MosaicData>[]
 * @text モザイク設定リスト
 *
 * @arg useFade
 * @type boolean
 * @default false
 * @text フェード表示
 *
 * @arg fadeDuration
 * @type number
 * @default 30
 * @text フェード時間（フレーム）
 * 
 * @command updateSpine
 * @text Spine再設定
 * @desc 表示済みSpineにアニメ・ミックス・再生速度を再適用
 *
 * @arg pictureId
 * @type number
 * @default 10
 * @text ピクチャID
 * 
 * @arg trackSettings
 * @type struct<SpineTrack>[]
 * @default []
 * @text アニメ設定リスト
 *
 * @arg mixValue
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.3
 * @text setmix値
 *
 * @arg timeScale
 * @type number
 * @decimals 2
 * @min 0.01
 * @default 1.0
 * @text アニメ再生速度
 * 
 * @arg resetOtherTracks
 * @type boolean
 * @default false
 * @text 他トラックをリセット
 * @desc 指定されていないトラックはリセットアニメに置き換える

 *
 * @command changeSkin
 * @text Spineスキン変更
 * @desc 表示済みSpineのスキンを変更
 *
 * @arg pictureId
 * @type number
 * @default 10
 * @text ピクチャID
 * 
 * @arg layerSkinSettings
 * @type struct<LayerSkinSetting>[]
 * @default []
 * @text レイヤー別スキン設定
 * 
 * @command playRandomAnimation
 * @text Spineランダム再生
 * @desc 指定トラックにランダム再生（重み付き）でアニメーションを設定
 *
 * @arg pictureId
 * @type number
 * @default 10
 * @text ピクチャID
 *
 * @arg trackId
 * @type number
 * @default 0
 * @text トラック番号
 *
 * @arg randomEntries
 * @type struct<RandomAnim>[]
 * @default []
 * @text ランダム再生設定
 * @desc アニメーション名と回数の設定
 * 
 * @command setSpineTimeScale
 * @text Spine再生速度変更
 * @desc 指定したSpineのアニメ再生速度だけを変更します
 *
 * @arg pictureId
 * @type number
 * @default 99
 * @text ピクチャID
 *
 * @arg timeScale
 * @type number
 * @decimals 2
 * @min 0.01
 * @default 1.0
 * @text アニメ再生速度
 *
 * @command setSpineAlpha
 * @text Spineアルファ設定
 *
 * @arg pictureId
 * @type number
 * @default 99
 * @text ピクチャID
 *
 * @arg alphaTracks
 * @type struct<AlphaTrack>[]
 * @text アルファ設定リスト
 * 
 * @command showTestSpine
 * @text 座標確認
 * @desc 座標確認用コマンド
 *
 * @arg pictureName
 * @type file
 * @dir img/pictures/
 * @text ドラック用画像
 *
 * @arg skeletonName
 * @type string
 * @default UI
 * @text スケルトン名
 *
 * @arg animationName
 * @type string
 * @default 1
 * @text アニメーション名
 * 
 * @arg skinNames
 * @type string[]
 * @default []
 * @text スキン名リスト
 * @desc 複数スキンを順に適用
 * 
 * @arg x
 * @type number
 * @default 400
 * @text X座標
 *
 * @arg y
 * @type number
 * @default 300
 * @text Y座標
 *
 * @arg scalePercent
 * @type number
 * @default 100
 * @text 拡大率（％）
 * 
 * @command fadeOutAndErase
 * @text 消去
 * @desc 指定したSpineをフェードアウトしてから消去
 *
 * @arg pictureId
 * @type number
 * @text ピクチャID
 * @desc 消去するピクチャ番号
 *
 * @arg fadeDuration
 * @type number
 * @default 30
 * @text フェード時間（フレーム）
 * @desc フェードアウトにかける時間
 * 
 * @command VarSpinecall
 * @text VarSpinecall
 * @desc 設定した変数の値を使用してSpineを呼出
 * 
 * @command VaranimCall
 * @text VaranimCall
 * @desc 設定した変数の値を使用してSpineアニメーションを変更
 *
 * @arg option
 * @type struct<AnimationOption>
 * @text オプション設定
 * @desc アニメーション再生のオプション設定
*/

/*~struct~SpineTrack:
 * @param trackId
 * @type number
 * @text トラック番号
 *
 * @param animations
 * @type string[]
 * @text アニメーション名リスト
 *
 * @param order
 * @type select
 * @option sequential
 * @option random
 * @option shuffle
 * @default sequential
 * @text 再生順序
 *
 * @param continuance
 * @type select
 * @option continue
 * @option reset
 * @option none
 * @default continue
 * @text 継続オプション
 *
 * @param interrupt
 * @type boolean
 * @default false
 * @text 割り込み
 */

/*~struct~MosaicData:
 * @param image
 * @type string
 * @text 対象パーツ名
 *
 * @param size
 * @type number
 * @min 1
 * @text モザイクサイズ（px）
 */

/*~struct~AlphaTrack:
 * @param trackId
 * @type number
 * @text トラック番号
 *
 * @param alpha
 * @type number
 * @decimals 2
 * @min 0
 * @max 1
 * @default 1.0
 * @text アルファ値（0～1.0）
 *
 * @param overwrite
 * @type boolean
 * @default false
 * @text 上書き（overwrite）
 */

/*~struct~AnimationOption:
 * @param order
 * @type select
 * @option sequential
 * @option random
 * @option shuffle
 * @default sequential
 * @text 順序オプション
 * @desc アニメーションの再生順序
 *
 * @param continuance
 * @type select
 * @option continue
 * @option reset
 * @option none
 * @default continue
 * @text 継続オプション
 * @desc アニメーション終了後の動作
 *
 * @param interrupt
 * @type boolean
 * @default false
 * @text 割り込みフラグ
 * @desc 現在のアニメーションを中断するか
 */

/*~struct~RandomAnim:
 * @param name
 * @type string
 * @text アニメーション名
 *
 * @param times
 * @type number
 * @min 1
 * @default 1
 * @text 回数（重み）
 */

/*~struct~SkinLayerEntry:
 * @param layerName
 * @type string
 * @text レイヤー名
 * @desc スキンのレイヤー（カテゴリ）名
 *
 * @param skinName
 * @type string
 * @text スキン名
 * @desc 指定レイヤーに適用するスキン名
 */

/*~struct~LayerSkinSetting:
 * @param layerName
 * @type string
 * @text レイヤー名
 * @desc スキンのレイヤー（カテゴリ）名
 *
 * @param skinName
 * @type string
 * @text スキン名
 * @desc 指定レイヤーに適用するスキン名
 */

(() => {
    const pluginName = "OnspineCALL";
    const params = PluginManager.parameters(pluginName);
    const resetAnimationName = params.resetAnimationName || "idle";
    const trackState = window._spineTrackStates = window._spineTrackStates || {};
    const skinState = window._spineSkinStates = window._spineSkinStates || {};
    const testSpineState = {
        active: false,
        pictureId: 99,
        waitFrames: 0,
        maxWait: 0,
        skeletonApplied: false,
        sprite: null,
        infoWindow: null,
        config: null,
        spineRef: null
    };

    

    function detachTestSpineSprite() {
        const sprite = testSpineState.sprite;
        if (!sprite) return;
        if (sprite._wheelListenerBound) {
            window.removeEventListener("wheel", sprite._wheelListenerBound);
            sprite._wheelListenerBound = null;
        }
        if (sprite._draggableOriginalUpdate) {
            sprite.update = sprite._draggableOriginalUpdate;
            sprite._draggableOriginalUpdate = null;
        }
        if (sprite._draggableOriginalHitTest) {
            sprite.hitTest = sprite._draggableOriginalHitTest;
            sprite._draggableOriginalHitTest = null;
        }
        if (sprite._draggableInfoWindow && sprite._draggableInfoWindow.parent) {
            sprite._draggableInfoWindow.parent.removeChild(sprite._draggableInfoWindow);
        }
        sprite._draggableInfoWindow = null;
        sprite._draggableUpdateInstalled = false;
        sprite._draggableSetupDone = false;
        testSpineState.sprite = null;
        testSpineState.infoWindow = null;
    }

    function cleanupTestSpineState({ erasePicture = false } = {}) {
        detachTestSpineSprite();
        if (testSpineState.infoWindow && testSpineState.infoWindow.parent) {
            testSpineState.infoWindow.parent.removeChild(testSpineState.infoWindow);
        }
        if (erasePicture && $gameScreen.picture(testSpineState.pictureId)) {
            $gameScreen.erasePicture(testSpineState.pictureId);
        }
        testSpineState.active = false;
        testSpineState.waitFrames = 0;
        testSpineState.maxWait = 0;
        testSpineState.skeletonApplied = false;
        testSpineState.infoWindow = null;
        testSpineState.config = null;
            testSpineState.spineRef = null;
    }

    function getSpine(pictureId) {
        return $gameScreen.spine(pictureId);
    }

    function applySkins(spine, skinNames, replace) {
        if (skinNames.length === 0) return;
        spine.setSkin(...skinNames);
    }

    function updateLayerSkins(pictureId, layerSkinSettings) {
        const spine = getSpine(pictureId);
        if (!spine) return;

        skinState[pictureId] = skinState[pictureId] || {};
        
        for (const setting of layerSkinSettings) {
            if (setting.layerName && setting.skinName) {
                skinState[pictureId][setting.layerName] = setting.skinName;
            }
        }
        
        const allSkinNames = Object.values(skinState[pictureId]).filter(name => name);
        
        if (allSkinNames.length > 0) {
            applySkins(spine, allSkinNames, true);
        }
    }

    function initializeSkinState(pictureId, layerSkinSettings) {
        skinState[pictureId] = {};
        for (const setting of layerSkinSettings) {
            if (setting.layerName && setting.skinName) {
                skinState[pictureId][setting.layerName] = setting.skinName;
            }
        }
    }

    function applyMix(spine, mixValue) {
        if (mixValue >= 0) spine.setMix("/default", "", mixValue);
    }

    function applyTracks(spine, pictureId, tracks, resetUnused) {
        const activeTracks = new Set();
        trackState[pictureId] = trackState[pictureId] || {};
        const stateMap = trackState[pictureId];

        for (const track of tracks) {
            spine.setAnimation(track.trackId, track.animations, track.order, track.continuance, track.interrupt);
            stateMap[track.trackId] = track.animations[0] || "";
            activeTracks.add(track.trackId);
        }

        if (resetUnused) {
            for (const trackId in stateMap) {
                if (!activeTracks.has(Number(trackId))) {
                    spine.setAnimation(Number(trackId), resetAnimationName, true);
                    stateMap[trackId] = resetAnimationName;
                }
            }
        }
    }

    PluginManager.registerCommand(pluginName, "CallSpineFull", args => {
        const pictureId = Number(args.pictureId);
        const x = Number(args.x);
        const y = Number(args.y);
        const scale = Number(args.pictureScale);
        const skeletonName = args.skeletonName;
        const skinNames = JSON.parse(args.skinNames || "[]");
        const layerSkinSettings = JSON.parse(args.layerSkinSettings || "[]").map(raw => {
            const obj = JSON.parse(raw);
            return { layerName: String(obj.layerName || '').trim(), skinName: String(obj.skinName || '').trim() };
        }).filter(e => e.layerName && e.skinName);
        const mixValue = Number(args.mixValue);
        const useFade = args.useFade === "true";
        const fadeDuration = Number(args.fadeDuration);

        const trackSettings = JSON.parse(args.trackSettings || "[]").map(raw => {
            const obj = JSON.parse(raw);
            return {
                trackId: Number(obj.trackId),
                animations: JSON.parse(obj.animations || "[]"),
                order: obj.order || "sequential",
                continuance: obj.continuance || "continue",
                interrupt: obj.interrupt === "true"
            };
        });

        const mosaicSettings = JSON.parse(args.mosaicSettings || "[]").map(raw => {
            const obj = JSON.parse(raw);
            return {
                image: obj.image.trim(),
                size: Number(obj.size)
            };
        }).filter(m => m.image && m.size > 0);

        if ($gameScreen.picture(pictureId)) {
            $gameScreen.erasePicture(pictureId);
        }
        $gameScreen.showPicture(pictureId, "", 0, x, y, scale, scale, 255, 0);

        const spine = getSpine(pictureId);
        if (!spine) return;

        spine.setSkeleton(skeletonName);
        spine.setScale(1.0, 1.0);

    initializeSkinState(pictureId, layerSkinSettings);

    const layerCombined = layerSkinSettings.map(e => e.skinName);
    const combinedSkinNames = [...skinNames, ...layerCombined].filter((v, i, a) => a.indexOf(v) === i);
    applySkins(spine, combinedSkinNames, true);  
        applyMix(spine, mixValue);
        applyTracks(spine, pictureId, trackSettings, false);

        for (const mosaic of mosaicSettings) {
            spine.setMosaic(mosaic.image, mosaic.size);
        }

        if (useFade) {
            spine.setColor(1, 1, 1, 0);
            const step = 1 / fadeDuration;
            let alpha = 0.0;
            const updateAlpha = () => {
                alpha += step;
                if (alpha >= 1.0) {
                    alpha = 1.0;
                } else {
                    requestAnimationFrame(updateAlpha);
                }
                spine.setColor(1, 1, 1, alpha);
            };
            updateAlpha();
        }
    });

    PluginManager.registerCommand(pluginName, "CallSpineVar", args => {
        const pictureIdVar = Number(args.pictureIdVar);
        const xVar = Number(args.xVar);
        const yVar = Number(args.yVar);
        const pictureId = $gameVariables.value(pictureIdVar);
        const x = $gameVariables.value(xVar);
        const y = $gameVariables.value(yVar);
        const scaleVarId = Number(args.pictureScaleVar || 0);
        let scale = 100;
        if (scaleVarId > 0) {
            const v = Number($gameVariables.value(scaleVarId));
            if (isFinite(v)) {
                scale = v;
            }
        }

        const skeletonNameVarId = Number(args.skeletonNameVar || 0);
        let skeletonName = "";
        if (skeletonNameVarId > 0) {
            const v = $gameVariables.value(skeletonNameVarId);
            skeletonName = String(v ?? "");
        }
        if (!skeletonName) return;
        const skinNames = JSON.parse(args.skinNames || "[]");
        const layerSkinSettings = JSON.parse(args.layerSkinSettings || "[]").map(raw => {
            const obj = JSON.parse(raw);
            return { layerName: String(obj.layerName || '').trim(), skinName: String(obj.skinName || '').trim() };
        }).filter(e => e.layerName && e.skinName);
        const mixValue = Number(args.mixValue);
        const useFade = args.useFade === "true";
        const fadeDuration = Number(args.fadeDuration);

        const trackSettings = JSON.parse(args.trackSettings || "[]").map(raw => {
            const obj = JSON.parse(raw);
            return {
                trackId: Number(obj.trackId),
                animations: JSON.parse(obj.animations || "[]"),
                order: obj.order || "sequential",
                continuance: obj.continuance || "continue",
                interrupt: obj.interrupt === "true"
            };
        });

        const mosaicSettings = JSON.parse(args.mosaicSettings || "[]").map(raw => {
            const obj = JSON.parse(raw);
            return {
                image: obj.image.trim(),
                size: Number(obj.size)
            };
        }).filter(m => m.image && m.size > 0);

        if ($gameScreen.picture(pictureId)) {
            $gameScreen.erasePicture(pictureId);
        }
        $gameScreen.showPicture(pictureId, "", 0, x, y, scale, scale, 255, 0);

        const spine = getSpine(pictureId);
        if (!spine) return;

        spine.setSkeleton(skeletonName);
        spine.setScale(1.0, 1.0);

    initializeSkinState(pictureId, layerSkinSettings);

    const layerCombined = layerSkinSettings.map(e => e.skinName);
    const combinedSkinNames = [...skinNames, ...layerCombined].filter((v, i, a) => a.indexOf(v) === i);
    applySkins(spine, combinedSkinNames, true);  
        applyMix(spine, mixValue);
        applyTracks(spine, pictureId, trackSettings, false);

        for (const mosaic of mosaicSettings) {
            spine.setMosaic(mosaic.image, mosaic.size);
        }

        if (useFade) {
            spine.setColor(1, 1, 1, 0);
            const step = 1 / fadeDuration;
            let alpha = 0.0;
            const updateAlpha = () => {
                alpha += step;
                if (alpha >= 1.0) {
                    alpha = 1.0;
                } else {
                    requestAnimationFrame(updateAlpha);
                }
                spine.setColor(1, 1, 1, alpha);
            };
            updateAlpha();
        }
    });

    PluginManager.registerCommand(pluginName, "updateSpine", args => {
        const pictureId = Number(args.pictureId || 99);
        const mixValue = Number(args.mixValue || 0.3);
        const timeScale = Number(args.timeScale || 1.0);
        const resetOtherTracks = args.resetOtherTracks === "true";

        const trackSettings = JSON.parse(args.trackSettings || "[]").map(raw => {
            const obj = JSON.parse(raw);
            return {
                trackId: Number(obj.trackId),
                animations: JSON.parse(obj.animations || "[]"),
                order: obj.order || "sequential",
                continuance: obj.continuance || "continue",
                interrupt: obj.interrupt === "true"
            };
        });

        const spine = getSpine(pictureId);
        if (!spine) {
            return;
        }

        applyMix(spine, mixValue);
        if (timeScale >= 0) spine.setTimeScale(timeScale);
        applyTracks(spine, pictureId, trackSettings, resetOtherTracks);
    });

    PluginManager.registerCommand(pluginName, "changeSkin", args => {
        const pictureId = Number(args.pictureId || 10);
        const layerSkinSettings = JSON.parse(args.layerSkinSettings || "[]").map(raw => {
            return JSON.parse(raw);
        });

        const spine = getSpine(pictureId);
        if (!spine) {
            console.warn(`Spine not found for picture ID: ${pictureId}`);
            return;
        }

        if (layerSkinSettings.length > 0) {
            console.log(`Changing layer skins for picture ${pictureId}:`, layerSkinSettings);
            updateLayerSkins(pictureId, layerSkinSettings);
        } else {
            console.warn(`No valid layer skin settings provided for picture ID: ${pictureId}`);
        }
    });

    PluginManager.registerCommand(pluginName, "playRandomAnimation", args => {
        const pictureId = Number(args.pictureId || 0);
        const trackId = Number(args.trackId || 0);
        const entriesRaw = JSON.parse(args.randomEntries || "[]");

        if (!pictureId) {
            return;
        }

        const spine = getSpine(pictureId);
        if (!spine) {
            return;
        }

        let continuance = "reset";
        let interrupt = false;

        const animations = [];
        for (const raw of entriesRaw) {
            try {
                const obj = JSON.parse(raw);
                const base = String(obj.name || "").trim();
                let times = Number(obj.times || 1);
                if (!base) continue;
                if (!isFinite(times) || times <= 0) times = 1;
                const anim = `${base}/times=${times}`;
                animations.push(anim);
            } catch (_) {  }
        }

        if (animations.length === 0) {
            return;
        }

        try {
            spine.setAnimation(trackId, animations, 'random', continuance, interrupt);
        } catch (e) {
            return;
        }

        trackState[pictureId] = trackState[pictureId] || {};
        trackState[pictureId][trackId] = animations.join(",");
    });

    PluginManager.registerCommand(pluginName, "setSpineTimeScale", args => {
        const pictureId = Number(args.pictureId || 0);
        const timeScale = Number(args.timeScale || 1.0);
        if (!pictureId || !(timeScale >= 0)) return;
        const spine = getSpine(pictureId);
        if (!spine) return;
        spine.setTimeScale(timeScale);
    });

    PluginManager.registerCommand(pluginName, "showTestSpine", args => {
        const pictureId = 99;
        const pictureName = args.pictureName || "test";
        const skeletonName = args.skeletonName || "test";
        const animationName = args.animationName || "test";
        const x = Number(args.x || 400);
        const y = Number(args.y || 300);
        const scale = Math.max(0.1, Number(args.scalePercent || 100) / 100);
        const skinNames = JSON.parse(args.skinNames || "[]");

        cleanupTestSpineState();

        testSpineState.pictureId = pictureId;
        testSpineState.config = {
            skeletonName,
            animationName,
            skinNames,
            scale
        };
        testSpineState.waitFrames = 0;
        testSpineState.maxWait = 600;
        testSpineState.skeletonApplied = false;
        testSpineState.active = true;

        $gameScreen.showPicture(pictureId, pictureName, 0, x, y, scale * 100, scale * 100, 255, 0);

        if (!Input.keyMapper[67]) Input.keyMapper[67] = "copy";
    });

    function showToast(text) {
        const toast = new Window_Base(new Rectangle(0, 0, 350, 60));
        toast.opacity = 200;
        toast.contents.clear();
        toast.drawText(text, 0, 0, 320, 36, "center");
        const scene = SceneManager._scene;
        if (!scene) return;
        scene.addChild(toast);
        toast.x = (Graphics.width - toast.width) / 2;
        toast.y = (Graphics.height - toast.height) / 2;
        setTimeout(() => scene.removeChild(toast), 1500);
    }

    function setupDraggableTestSprite(sprite) {
        const scene = SceneManager._scene;
        if (!scene) return;

        if (sprite._draggableInfoWindow && sprite._draggableInfoWindow.parent) {
            sprite._draggableInfoWindow.parent.removeChild(sprite._draggableInfoWindow);
        }

        const infoWindow = new Window_Base(new Rectangle(10, 10, 450, 80));
        infoWindow.opacity = 180;
        scene.addChild(infoWindow);

        sprite._dragging = false;
        sprite._copyMode = "x";
        sprite._lastCopyPressed = false;

        sprite._draggableInfoWindow = infoWindow;
        testSpineState.infoWindow = infoWindow;
        testSpineState.sprite = sprite;

        if (!sprite._draggableOriginalHitTest) {
            sprite._draggableOriginalHitTest = sprite.hitTest;
        }
        sprite.hitTest = function (x, y) {
            const left = this.x;
            const top = this.y;
            const right = left + this.width * this.scale.x;
            const bottom = top + this.height * this.scale.y;
            return x >= left && x < right && y >= top && y < bottom;
        };

        sprite._onWheel = function (event) {
            const delta = Math.sign(event.deltaY);
            const step = 0.01;
            let newScale = this.scale.x - delta * step;
            newScale = Math.max(0.1, Math.min(3.0, newScale));
            this.scale.set(newScale, newScale);
            const pic = $gameScreen.picture(testSpineState.pictureId);
            if (pic) {
                pic._scaleX = pic._scaleY = newScale * 100;
            }
        };

        if (!sprite._wheelListenerBound) {
            sprite._wheelListenerBound = event => {
                if (!testSpineState.active || testSpineState.sprite !== sprite) return;
                sprite._onWheel(event);
            };
            window.addEventListener("wheel", sprite._wheelListenerBound);
        }

        if (!sprite._draggableUpdateInstalled) {
            const originalUpdate = sprite.update;
            sprite._draggableOriginalUpdate = originalUpdate;
            sprite.update = function () {
                originalUpdate.call(this);
                if (!testSpineState.active || testSpineState.sprite !== this) return;
                if (!this.visible || !this.bitmap || !this.bitmap.isReady()) return;

                if (!this._dragging && TouchInput.isTriggered() && this.hitTest(TouchInput.x, TouchInput.y)) {
                    this._dragging = true;
                    this._offsetX = TouchInput.x - this.x;
                    this._offsetY = TouchInput.y - this.y;
                } else if (this._dragging && TouchInput.isPressed()) {
                    this.x = TouchInput.x - this._offsetX;
                    this.y = TouchInput.y - this._offsetY;
                    const pic = $gameScreen.picture(testSpineState.pictureId);
                    if (pic) {
                        pic._x = this.x;
                        pic._y = this.y;
                    }
                } else if (this._dragging && !TouchInput.isPressed()) {
                    this._dragging = false;
                }

                if (Input.isPressed("control") && Input.isPressed("copy") && !this._lastCopyPressed) {
                    const value = this._copyMode === "x" ? Math.round(this.x) : Math.round(this.y);
                    const label = this._copyMode.toUpperCase();
                    const text = `${value}`;
                    navigator.clipboard?.writeText(text).then(() => {
                        showToast(`${label}: ${value} をコピーしました`);
                    }).catch(() => {
                        showToast(`${label}: ${value} をコピーしました`);
                    });
                    this._copyMode = this._copyMode === "x" ? "y" : "x";
                    this._lastCopyPressed = true;
                } else if (!Input.isPressed("control") || !Input.isPressed("copy")) {
                    this._lastCopyPressed = false;
                }

                const infoWindow = this._draggableInfoWindow;
                if (infoWindow && infoWindow.contents) {
                    const zoom = Math.round(this.scale.x * 100);
                    infoWindow.contents.clear();
                    infoWindow.drawText(`X: ${Math.round(this.x)}`, 0, 0, 160);
                    infoWindow.drawText(`Y: ${Math.round(this.y)}`, 0, 24, 160);
                    infoWindow.drawText(`倍率: ${zoom}%`, 160, 0, 160);
                }
            };
            sprite._draggableUpdateInstalled = true;
        }

        sprite._draggableSetupDone = true;
    }

    function updateTestSpineState() {
        if (!testSpineState.active) return;
        const state = testSpineState;

        const picture = $gameScreen.picture(state.pictureId);
        if (!picture) {
            cleanupTestSpineState();
            return;
        }

        const spine = getSpine(state.pictureId);
        if (!spine) {
            state.waitFrames++;
            if (state.maxWait && state.waitFrames > state.maxWait) {
                
                cleanupTestSpineState();
            }
            return;
        }

        if (state.spineRef !== spine) {
            state.spineRef = spine;
            state.skeletonApplied = false;
        }
        state.waitFrames = 0;

        if (!state.skeletonApplied && state.config) {
            try {
                spine.setSkeleton(state.config.skeletonName);
                if (state.config.skinNames.length > 0) spine.setSkin(...state.config.skinNames);
                spine.setAnimation(0, state.config.animationName, true);
                state.skeletonApplied = true;
            } catch (error) {
                cleanupTestSpineState();
                return;
            }
        }

        const scene = SceneManager._scene;
        if (!scene || !scene._spriteset || !scene._spriteset._pictureContainer) return;

        const sprite = scene._spriteset._pictureContainer.children.find(s => s && s._pictureId === state.pictureId);
        if (!sprite) return;

        if (state.sprite && state.sprite !== sprite) {
            detachTestSpineSprite();
        }

        if (!sprite._draggableSetupDone) {
            setupDraggableTestSprite(sprite);
        }
    }

    if (!SceneManager.updateMain._OnspineCALL_showTestSpine) {
        const _OnspineCALL_SceneManager_updateMain = SceneManager.updateMain;
        SceneManager.updateMain = function () {
            _OnspineCALL_SceneManager_updateMain.call(this);
            updateTestSpineState();
        };
        SceneManager.updateMain._OnspineCALL_showTestSpine = true;
    }

        PluginManager.registerCommand(pluginName, "setSpineAlpha", args => {
        const pictureId = Number(args.pictureId);
        const spine = getSpine(pictureId);
        if (!spine) {
            return;
        }

        const alphaTracks = JSON.parse(args.alphaTracks || "[]").map(raw => {
            const obj = JSON.parse(raw);
            return {
                trackId: Number(obj.trackId),
                alpha: Number(obj.alpha),
                overwrite: obj.overwrite === "true"
            };
        });

        for (const { trackId, alpha, overwrite } of alphaTracks) {
            if (!isFinite(trackId) || !isFinite(alpha)) continue;
            try {
                spine.setAlpha(trackId, alpha, overwrite);
            } catch (e) {
            }
        }
    });

    PluginManager.registerCommand(pluginName, "fadeOutAndErase", args => {
        const pictureId = Number(args.pictureId);
        const fadeDuration = Number(args.fadeDuration) || 30;
        
        const spine = getSpine(pictureId);
        if (!spine) {
            $gameScreen.erasePicture(pictureId);
            return;
        }

        const step = 1 / fadeDuration;
        let alpha = 1.0;
        
        const updateAlpha = () => {
            alpha -= step;
            if (alpha <= 0.0) {
                alpha = 0.0;
                spine.setColor(1, 1, 1, alpha);
                $gameScreen.erasePicture(pictureId);
                if (trackState[pictureId]) {
                    delete trackState[pictureId];
                }
            } else {
                spine.setColor(1, 1, 1, alpha);
                requestAnimationFrame(updateAlpha);
            }
        };
        updateAlpha();
    });

    PluginManager.registerCommand(pluginName, "VarSpinecall", args => {
        const pictureIdVar = Number(params.pictureIdVariable);
        const xVar = Number(params.xVariable);
        const yVar = Number(params.yVariable);
        const scaleVar = Number(params.scaleVariable);
        const skeletonNameVar = Number(params.skeletonNameVariable);
        const trackVar = Number(params.trackVariable);
        const animationNameVar = Number(params.animationNameVariable);
        const pictureId = pictureIdVar > 0 ? $gameVariables.value(pictureIdVar) || 0 : 0;
        const x = xVar > 0 ? $gameVariables.value(xVar) || 0 : 0;
        const y = yVar > 0 ? $gameVariables.value(yVar) || 0 : 0;
        const scale = scaleVar > 0 ? $gameVariables.value(scaleVar) || 100 : 100;
        const skeletonName = skeletonNameVar > 0 ? $gameVariables.value(skeletonNameVar) || "" : "";
        const trackNum = trackVar > 0 ? $gameVariables.value(trackVar) || 0 : 0;
        const animationName = animationNameVar > 0 ? $gameVariables.value(animationNameVar) || "" : "";

        if (!pictureId || !skeletonName || !animationName) {
            return;
        }

        const animationList = String(animationName)
            .split(",")
            .map(n => n.trim())
            .filter(n => n.length > 0);
        
        const trackSettings = [{
            trackId: trackNum,
            animations: animationList.length > 0 ? animationList : [String(animationName)],
            order: "sequential",
            continuance: "continue",
            interrupt: false
        }];

        const mosaicSettings = [];
        const useFade = false;
        const skinNames = [];
        const mixValue = 0.3;

        if ($gameScreen.picture(pictureId)) {
            $gameScreen.erasePicture(pictureId);
        }
        $gameScreen.showPicture(pictureId, "", 0, x, y, scale, scale, 255, 0);

        const spine = getSpine(pictureId);
        if (!spine) return;

        spine.setSkeleton(skeletonName);
        spine.setScale(1.0, 1.0);

        applySkins(spine, skinNames, true);  
        applyMix(spine, mixValue);
        applyTracks(spine, pictureId, trackSettings, false);

        for (const mosaic of mosaicSettings) {
            spine.setMosaic(mosaic.image, mosaic.size);
        }
    });

    PluginManager.registerCommand(pluginName, "VaranimCall", args => {
        const pictureIdVar = Number(params.pictureIdVariable);
        const trackVar = Number(params.trackVariable);
        const animationNameVar = Number(params.animationNameVariable);
        const pictureId = pictureIdVar > 0 ? $gameVariables.value(pictureIdVar) || 0 : 0;
        const trackNum = trackVar > 0 ? $gameVariables.value(trackVar) || 0 : 0;
        const animationName = animationNameVar > 0 ? $gameVariables.value(animationNameVar) || "" : "";

        let order = "sequential";
        let continuance = "continue";
        let interrupt = false;
        
        if (args.option) {
            try {
                const optionData = JSON.parse(args.option);
                order = optionData.order || "sequential";
                continuance = optionData.continuance || "continue";
                interrupt = optionData.interrupt === "true";
            } catch (e) {
            }
        }

        if (!pictureId || animationName === "") {
            return;
        }

        const spine = getSpine(pictureId);
        if (!spine) {
            return;
        }

        const animationList = animationName.split(',').map(name => name.trim()).filter(name => name);
        
        if (animationList.length === 0) {
            return;
        }

        if (animationList.length === 1) {
            spine.setAnimation(trackNum, animationList[0], continuance, interrupt);
        } else {
            spine.setAnimation(trackNum, animationList, order, continuance, interrupt);
        }
        
        if (!trackState[pictureId]) {
            trackState[pictureId] = {};
        }
        trackState[pictureId][trackNum] = animationName;
    });

})();

(() => {
    if (typeof Sprite_Spine === 'undefined' || !Sprite_Spine.prototype.onEvent) return;
    const _OnspineCALL_Sprite_Spine_onEvent = Sprite_Spine.prototype.onEvent;
    Sprite_Spine.prototype.onEvent = function(entry, event) {
        _OnspineCALL_Sprite_Spine_onEvent.call(this, entry, event);
        if (this._isRestore) return;
        if (!event) return;
        if (!event._commonEventsParsed) {
            event._commonEventsParsed = true;
            const list = [];
            const raw = (event.stringValue || event.data?.stringValue || '').replace(/ +/g, '');
            if (raw) {
                for (const part of raw.split(/,/)) {
                    if (part.match(/^CE:(\d+)$/i)) {
                        const id = Number(RegExp.$1);
                        if (id > 0 && !$dataCommonEvents[id]) {
                        }
                        if (id > 0) list.push(id);
                    }
                }
            }
            event._commonEventIds = list;
        }
        if (Array.isArray(event._commonEventIds)) {
            for (const id of event._commonEventIds) {
                if (id > 0 && $dataCommonEvents[id]) {
                    const interpreter = new Game_Interpreter();
                    interpreter.setup($dataCommonEvents[id].list, 0);
                    
                    if (!$gameMap._parallelCommonEventInterpreters) {
                        $gameMap._parallelCommonEventInterpreters = [];
                    }
                    $gameMap._parallelCommonEventInterpreters.push(interpreter);
                }
            }
        }
    };
    
    const _Game_Map_update = Game_Map.prototype.update;
    Game_Map.prototype.update = function(sceneActive) {
        _Game_Map_update.call(this, sceneActive);
        
        if (this._parallelCommonEventInterpreters) {
            for (let i = this._parallelCommonEventInterpreters.length - 1; i >= 0; i--) {
                const interpreter = this._parallelCommonEventInterpreters[i];
                interpreter.update();
                
                if (!interpreter.isRunning()) {
                    this._parallelCommonEventInterpreters.splice(i, 1);
                }
            }
        }
    };
})();