/*:
 * @target MZ
 * @plugindesc 文章デフォルト設定
 * @author
 *
 * @param TextColor
 * @text 既定の文字色
 * @type string
 * @default
 *
 * @param OutlineColor
 * @text 既定のアウトライン色
 * @type string
 * @default
 *
 * @param OutlineWidth
 * @text 既定のアウトライン太さ
 * @type number
 * @min 0
 * @desc -1 で既定のまま
 * @default -1
 *
 * @param AllowOutlineTranslucent
 * @text アウトライン半透明？
 * @type boolean
 * @on 許可する
 * @off 許可しない
 * @default true
 *
 */
(() => {
    'use strict';

    const PLUGIN_NAME = document.currentScript.src.match(/([^/]+)\.js$/)[1];
    const params = PluginManager.parameters(PLUGIN_NAME);

    function numOrDefault(str, def) {
        if (str === undefined || str === null || str === '') return def;
        const n = Number(str);
        return Number.isFinite(n) ? n : def;
    }

    const P = {
        textColor: String(params.TextColor || ''),
        outlineColor: String(params.OutlineColor || ''),
        outlineWidth: numOrDefault(params.OutlineWidth, -1),
        allowOutlineTranslucent: params.AllowOutlineTranslucent === 'true',
        fontFace: String(params.FontFace || ''),
        fontSize: numOrDefault(params.FontSize, -1)
    };

    function toOpaqueColor(color) {
        if (!color) return color;
        const mRgba = color.match(/^rgba\s*\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*([0-9.]+)\s*\)\s*$/i);
        if (mRgba) return `rgb(${mRgba[1]},${mRgba[2]},${mRgba[3]})`;
        const mHsla = color.match(/^hsla\s*\(\s*([^,]+)\s*,\s*([^,]+)\s*,\s*([^,]+)\s*,\s*([0-9.]+)\s*\)\s*$/i);
        if (mHsla) return `hsl(${mHsla[1]},${mHsla[2]},${mHsla[3]})`;
        const mHex8 = color.match(/^#([0-9a-f]{8})$/i);
        if (mHex8) return `#${mHex8[1].slice(0, 6)}`;
        const mHex4 = color.match(/^#([0-9a-f]{4})$/i);
        if (mHex4) return `#${mHex4[1].slice(0, 3)}`;
        return color;
    }

    function applyDefaultsToBitmap(bitmap) {
        if (!bitmap) return;
        if (P.outlineColor) {
            bitmap.outlineColor = P.allowOutlineTranslucent ? P.outlineColor : toOpaqueColor(P.outlineColor);
        } else if (!P.allowOutlineTranslucent) {
            bitmap.outlineColor = '#000000';
        }
        if (P.outlineWidth >= 0) {
            bitmap.outlineWidth = P.outlineWidth;
        }
        if (P.fontFace) bitmap.fontFace = P.fontFace;
        if (P.fontSize > 0) bitmap.fontSize = P.fontSize;
    }

    const _Window_Base_resetTextColor = Window_Base.prototype.resetTextColor;
    Window_Base.prototype.resetTextColor = function() {
        _Window_Base_resetTextColor.call(this);
        if (P.textColor) this.changeTextColor(P.textColor);
    };

    const _Window_Base_resetFontSettings = Window_Base.prototype.resetFontSettings;
    Window_Base.prototype.resetFontSettings = function() {
        _Window_Base_resetFontSettings.call(this);
        applyDefaultsToBitmap(this.contents);
    };

    const _Window_createContents = Window.prototype.createContents;
    Window.prototype.createContents = function() {
        _Window_createContents.call(this);
        applyDefaultsToBitmap(this.contents);
    };


    const _Window_resetFontSettings = Window.prototype.resetFontSettings;
    if (_Window_resetFontSettings) {
        Window.prototype.resetFontSettings = function() {
            _Window_resetFontSettings.call(this);
            applyDefaultsToBitmap(this.contents);
        };
    }
})();
