// AWY_ItemTrigger.js Ver.1.0.2
// MIT License (C) 2026 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ
* @plugindesc Trigger events with items and skills.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/519624636.html
* @help Ver.1.0.2
* Using items or skills in the menu will trigger adjacent events.
*
* [Comment]
* Place a comment on the first line of the event contents
* and enter it in the following format to be reflected.
* <itemTrigger:1> //Trigger with Item 1
* <itemTrigger:1,3> //Trigger with Item 1 or 3
* <itemTrigger:0,3> //Trigger with Action Button or Item 3 
* <skillTrigger:1> //Trigger with Skill 1
* <skillTrigger:0,1,3> //Trigger with Action Button or Item 1 or 3 
*
* [Note (Item/Skill)]
* <triggerAction> //On the menu screen, make it trigger-only.
* *Its effects disappear in the menu.
*  Consumed only when an event is triggered.
*  If the trigger fails, a buzzer will sound.
*  Its effects in battle remain unchanged.
*
* <triggerAction:1> //If not triggered, start Common Event 1
* <triggerAction:1,true> //As above, but always consumes
*
* [Scripts]
* It can be used to branch after an event is triggered.
* $gameTemp.triggerItemId() //Get the item ID used in the trigger
* $gameTemp.triggerSkillId() //Get the skill ID used in the trigger
*
* [Specifications]
* Autorun or parallel events cannot be triggered.
*
* @param seParams
* @text SE Settings
* @desc Set the sound effect when triggered.
* System SE: Blank
* @default 
* @type struct<seParams>
*
*/

/*~struct~seParams:
*
* @param item
* @desc System SE: Blank
* @default 
* @type struct<se>
*
* @param skill
* @desc System SE: Blank
* @default 
* @type struct<se>
*/

/*~struct~se:
*
* @param name
* @default Item1
* @type file
* @dir audio/se
*
* @param volume
* @default 90
* @type number
*
* @param pitch
* @default 100
* @type number
*
* @param pan
* @default 0
* @min -100
* @max 100
* @type number
*/

/*:ja
* @target MZ
* @plugindesc アイテムやスキルでイベントを起動します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/519624636.html
* @help メニューでアイテムやスキルを使用すると目の前のイベントを起動できます。
*
* [注釈]
* イベントの実行内容一行目に注釈を配置し、以下の形式で入力すると反映されます。
* <itemTrigger:1> //アイテム１で起動
* <itemTrigger:1,3> //アイテム１、または３で起動
* <itemTrigger:0,3> //普通に話しかけるまたはアイテム３で起動
* <skillTrigger:1> //スキル１で起動
* <skillTrigger:0,1,3> //普通に話しかけるまたはスキル１またはスキル３で起動
*
* [メモ（アイテム/スキル）]
* <triggerAction> //メニュー画面ではトリガー専用にする
* ※メニューでの使用効果が無くなる。イベント起動成功時にコスト消費。
*   イベント起動に失敗するとブザーが鳴る。
*   戦闘時の使用効果は変わらない。
*
* <triggerAction:1> //イベントが起動しないならコモンイベント１を起動
* <triggerAction:1,true> //上記と同様だが失敗時もコスト消費する
*
* [スクリプト]
* イベント起動後に条件分岐するのに使えます。
* $gameTemp.triggerItemId() //起動に使用したアイテムIDを取得
* $gameTemp.triggerSkillId() //起動に使用したスキルIDを取得
*
* [仕様]
* トリガーが自動実行や並列処理のイベントは起動できません。
*
* [更新履歴]
* 2026/01/03：Ver.1.0.0　公開。
* 2026/01/04：Ver.1.0.1　<triggerAction>で使用可能時[メニュー画面]のアイテムが戦闘で使えてしまう問題を修正。
* 2026/01/12：Ver.1.0.2　競合対策。
*
* @param seParams
* @text 効果音設定
* @desc イベント起動時の効果音を設定します。
* システム効果音：空欄
* @default 
* @type struct<seParams>
*
*/

/*~struct~seParams:ja
*
* @param item
* @text アイテム
* @desc システム効果音：空欄
* @default 
* @type struct<se>
*
* @param skill
* @text スキル
* @desc システム効果音：空欄
* @default 
* @type struct<se>
*/

/*~struct~se:ja
*
* @param name
* @text 名前
* @desc 選択したファイルを再生します。
* @default Item1
* @type file
* @dir audio/se
*
* @param volume
* @text 音量
* @desc オーディオ再生の音量です。
* @default 90
* @type number
*
* @param pitch
* @text ピッチ
* @desc オーディオ再生のピッチです。
* @default 100
* @type number
*
* @param pan
* @text 位相
* @desc オーディオ再生の位相です。
* @default 0
* @min -100
* @max 100
* @type number
*/

var AWY = AWY || {};
'use strict';

{
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const parameters = PluginManager.parameters(pluginName);
	const extendMovement = PluginManager.parameters("SimpleCharacterCollision").extendMovement === "true";
	function audioParams(str) {
		if (!str) return null;
		const params = JSON.parse(str);
		params.name = params.name || "";
		params.volume = Number(params.volume || 90);
		params.pitch = Number(params.pitch || 100);
		params.pan = Number(params.pan || 0);
		return params;
	}
	const seParams = JSON.parse(parameters.seParams || "{}");
	const itemSe = audioParams(seParams.item);
	const skillSe = audioParams(seParams.skill);

	//-----------------------------------------------------------------------------
	// Game_Temp

	Game_Temp.prototype.triggerItemAction = function() {
		return this._triggerItemAction;
	};

	Game_Temp.prototype.clearTriggerItem = function() {
		this._triggerItemAction = false;
	};

	Game_Temp.prototype.setTriggerItem = function(item) {
		if (DataManager.isItem(item)) {
			this._triggerItemAction = true;
			this._triggerItemId = item.id;
		} else if (DataManager.isSkill(item)) {
			this._triggerItemAction = true;
			this._triggerItemId = -item.id;
		} else {
			this.clearTriggerItem();
			this._triggerItemId = 0;
		}
	};

	Game_Temp.prototype.triggerItemId = function() {
		return this._triggerItemId > 0 ? this._triggerItemId : 0;
	};

	Game_Temp.prototype.triggerSkillId = function() {
		return this._triggerItemId < 0 ? -this._triggerItemId : 0;
	};

	//-----------------------------------------------------------------------------
	// Game_BattlerBase

	const _Game_BattlerBase_isOccasionOk = Game_BattlerBase.prototype.isOccasionOk;
	Game_BattlerBase.prototype.isOccasionOk = function(item) {
		return _Game_BattlerBase_isOccasionOk.call(this, item) || (item.occasion === 1 && item.meta && item.meta.triggerAction);
	};

	//-----------------------------------------------------------------------------
	// Game_Player

	Game_Player.prototype.triggerItemAction = function(item) {
		$gameTemp.setTriggerItem(item);
		if ($gameTemp.triggerItemAction() && this.canMove()) {
			this.checkEventTriggerHere([0, 1, 2]);
			$gameMap.refreshIfNeeded();
			if ($gameMap.setupStartingMapEvent()) {
				$gameTemp.clearTriggerItem()
				return true;
			}
			if (extendMovement && this.isCollisionEnabled()) {
				const x = this._x;
				const y = this._y;
				switch (this.direction()) {
				case 2:
					this._y += this._collision[0];
					break;
				case 4:
					this._x -= this._collision[1];
					break;
				case 6:
					this._x += this._collision[2];
					break;
				case 8:
					this._y -= this._collision[3];
					break;
				}
				[this._x, this._y] = [$gameMap.roundX(this.x), $gameMap.roundY(this.y)];
				this.checkEventTriggerThere([0, 1, 2]);
				this._x = x;
				this._y = y;
			} else {
				this.checkEventTriggerThere([0, 1, 2]);
			}
			$gameMap.refreshIfNeeded();
			if ($gameMap.setupStartingMapEvent()) {
				$gameTemp.clearTriggerItem()
				return true;
			}
		}
		$gameTemp.clearTriggerItem()
		return false;
	};

	//-----------------------------------------------------------------------------
	// Game_Event

	const _Game_Event_start = Game_Event.prototype.start;
	Game_Event.prototype.start = function() {
		if (this.isItemTriggerIn()) _Game_Event_start.call(this);
	};

	Game_Event.prototype.isItemTriggerIn = function() {
		const itemTriggers = this.page().itemTrigger;
		if ($gameTemp.triggerItemAction()) {
			if (itemTriggers && itemTriggers.contains($gameTemp._triggerItemId)) {
				return true;
			}
		} else {
			if (!itemTriggers || itemTriggers.contains(0)) {
				$gameTemp._triggerItemId = 0;
				return true;
			}
		}
		return false;
	};

	//-----------------------------------------------------------------------------
	// Scene_ItemBase

	const _Scene_ItemBase_determineItem = Scene_ItemBase.prototype.determineItem;
	Scene_ItemBase.prototype.determineItem = function() {
		if (!this.triggerItemAction()) _Scene_ItemBase_determineItem.call(this);
	};

	Scene_ItemBase.prototype.triggerItemAction = function() {
		const item = this.item();
		if ($gamePlayer.triggerItemAction(item)) {
			//トリガー成功
			useItem.call(this, item);
			SceneManager.goto(Scene_Map);
			this.checkGameover();
			this._actorWindow.refresh();
			this._statusWindow && this._statusWindow.refresh();
			this._itemWindow.refresh();
			return true;
		} else if (item.meta && item.meta.triggerAction) {
			//トリガー失敗
			const params = getTriggerParams(item.meta.triggerAction);
			if (params) {
				const commonEventId = params[0];
				const payCost = params[1];
				if (payCost) {
					useItem.call(this, item);
				}
				if (commonEventId > 0) $gameTemp.reserveCommonEvent(commonEventId);
				SceneManager.goto(Scene_Map);
				this.checkGameover();
				if (payCost) {
					this._actorWindow.refresh();
					this._statusWindow && this._statusWindow.refresh();
					this._itemWindow.refresh();
				}
			} else {
				SoundManager.playBuzzer();
				this.activateItemWindow();
			}
			return true;
		}
		return false;
	};

	function getTriggerParams(str) {
		if (str === true) return null;
		const params = str.split(",");
		params[0] = Number(params[0] || 0);
		params[1] = params[1] === "true";
		return params;
	}

	function playSe(item) {
		let param;
		if (DataManager.isItem(item)) {
			param = itemSe;
		} else if (DataManager.isSkill(item)) {
			param = skillSe;
		}
		if (param) {
			AudioManager.playStaticSe(param);
		}
		return !!param;
	}

	function useItem(item) {
		if (!playSe(item)) {
			this.playSeForItem();
		}
		const action = new Game_Action(this.user());
		this.user().useItem(item);
		action.setItemObject(item);
		action.updateLastUsed();
		action.updateLastSubject();
	}

	//-----------------------------------------------------------------------------
	// DataManager
	//マップデータロード時の処理に追加

	if (!AWY.addPageSettings) {
		AWY._pageSettingHandlers = [];
		const _DataManager_onLoad = DataManager.onLoad;
		DataManager.onLoad = function(object) {
			_DataManager_onLoad.apply(this, arguments);
			AWY.addPageSettings(object);
		};

		AWY.addPageSettings = function(object) {
			if (DataManager.isMapObject(object) && Array.isArray(object.events)) {
				for (const event of object.events) {
					if (event && event.pages){
						extractMetadata(event);
					}
				}
			}
		};

		function extractMetadata(data) {
			for (const page of data.pages) {
				const comment = findComment(page);
				const commentData = {"note": comment};
				DataManager.extractMetadata(commentData);
				AWY._pageSettingHandlers.forEach(handler => handler(page, commentData.meta));
			}
		}

		function findComment(page) {
			const list = page.list;
			if (!list[0] && list[0].code !== 108) {
				return "";
			}
			let comment = list[0].parameters[0];
			for (let i = 1; list[i] && list[i].code === 408; i++) {
				comment += list[i].parameters[0];
			}
			return comment;
		}
	}

	AWY._pageSettingHandlers.push(
		function(page, meta) {
			if (meta.itemTrigger) {
				page.itemTrigger = meta.itemTrigger.split(",").map(Number);
			}
			if (meta.skillTrigger) {
				const skillTrigger = meta.skillTrigger.split(",").map(n => -n);
				page.itemTrigger = page.itemTrigger ? page.itemTrigger.concat(skillTrigger) : skillTrigger;
			}
		}
	);
}