// DeathConditionalState.js Ver.1.2.0
// MIT License (C) 2022 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ MV
* @plugindesc Create states that result in instant death or defeat.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/488855304.html
* @help Ver.1.2.0
* 【Parameters】
* ・Instant Death Message
* Displays the message "%1 has fallen!" and "%1 is slain!" after instant death is added.
* If set to false, no death message will be displayed.
*
* ・Avoid Overlooking Log
* Normally status and disable messages are displayed separately, but enabling
* this parameter will display them together when instant death conditions
* are the direct cause of death.
*
* ・Instant Death State
* Battler dies when this state is added.
*
* ・Defeat State
* If all members are in these states or die, you lose.
*
* @param deathStateMessage
* @text Instant Death Message
* @desc Display a death message when receiving an instant death state.
* @type boolean
* @default false
*
* @param overlookLog
* @text Avoid Overlooking Log
* @desc Improved log display for instant death.
* This parameter is only valid if "Instant Death Message" is enabled.
* @type boolean
* @default true
*
* @param battler
* @text Battler's Instant Death State
* @desc These states kill both actors and enemies instantly.
* @type state[]
*
* @param actor
* @text Actor's Instant Death State
* @desc These states kill actors instantly.
* @type state[]
*
* @param enemy
* @text Enemy's Instant Death State
* @desc These states kill enemies instantly.
* @type state[]
*
* @param unit
* @text Unit's Defeat State
* @desc If all members of a party or enemy group are affected by these states, they are treated as annihilated.
* @type state[]
*
* @param party
* @text Party's Defeat State
* @desc If all members of a party are affected by these states, they are treated as annihilated.
* @type state[]
*
* @param troop
* @text Troop's Defeat State
* @desc If all members of a enemy group are affected by these states, they are treated as annihilated.
* @type state[]
*
*/

/*:ja
* @target MZ MV
* @plugindesc 即死ステートまたは全滅条件になるステートを作成します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/488855304.html
* @help 【パラメータ】
* ・戦闘不能メッセージ表示
* 即死ステート付与後に戦闘不能メッセージ「%1は倒れた！」「%1を倒した！」を表示します。
* falseにすると戦闘不能メッセージを表示しません。
*
* ・ログ見逃し予防
* 通常、ステート付与メッセージと戦闘不能メッセージは別々に表示されますが、
* このパラメータを有効にした場合、即死ステートが直接の死因になった時に、
* これらのメッセージを一度に表示します。
*
* ・即死ステート
* 即死ステート付与に成功した時に戦闘不能を付与します。
*
* ・全滅ステート
* メンバー全員が設定したステート、または戦闘不能になった場合、
* 敗北扱いとなります。
* 行動不能による詰みやバトルが冗長になってしまう事への対策、
* 特殊条件下のバトルなどに。
*
* [更新履歴]
* 2022/06/13：Ver.1.0.0　公開
* 2022/06/20：Ver.1.1.0　戦闘不能メッセージ表示の有無を設定可能にしました。
* 2023/08/29：Ver.1.2.0　挙動修正。
*
* @param deathStateMessage
* @text 戦闘不能メッセージ表示
* @desc 即死ステートで戦闘不能になった場合に戦闘不能メッセージを表示します。
* @type boolean
* @default false
*
* @param overlookLog
* @text ログ見逃し予防
* @desc 即死ステートで戦闘不能になった場合のログ表示を改善します。
* 「戦闘不能メッセージ表示」有効時のみ有効なパラメータです。
* @type boolean
* @default true
*
* @param battler
* @text バトラー即死ステート
* @desc アクター、エネミー両方とも即死するステートです。
* @type state[]
*
* @param actor
* @text アクター即死ステート
* @desc アクターが即死するステートです。
* @type state[]
*
* @param enemy
* @text エネミー即死ステート
* @desc エネミーが即死するステートです。
* @type state[]
*
* @param unit
* @text ユニット全滅ステート
* @desc パーティまたは敵グループのメンバー全員がこれらのステートにかかった場合、全滅扱いになります。
* @type state[]
*
* @param party
* @text パーティ全滅ステート
* @desc パーティメンバー全員がこれらのステートにかかった場合、全滅扱いになります。
* @type state[]
*
* @param troop
* @text 敵グループ全滅ステート
* @desc 敵グループのメンバー全員がこれらのステートにかかった場合、全滅扱いになります。
* @type state[]
*
*/

'use strict';

{

	const useMZ = Utils.RPGMAKER_NAME === "MZ";
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const parameter = PluginManager.parameters(pluginName);

	const deathStateMessage = parameter["deathStateMessage"] === "true";
	const overlookLog = deathStateMessage && parameter["overlookLog"] === "true";
	const paramBattler = new Set(JSON.parse(parameter["battler"] || "[]").map(Number));
	const paramActor = new Set(JSON.parse(parameter["actor"] || "[]").map(Number).concat(...paramBattler));
	const paramEnemy = new Set(JSON.parse(parameter["enemy"] || "[]").map(Number).concat(...paramBattler));
	const paramUnit = new Set(JSON.parse(parameter["unit"] || "[]").map(Number));
	const paramParty = new Set(JSON.parse(parameter["party"] || "[]").map(Number).concat(...paramUnit));
	const paramTroop = new Set(JSON.parse(parameter["troop"] || "[]").map(Number).concat(...paramUnit));

	//-----------------------------------------------------------------------------
	// BattleManager

	const _BattleManager_processVictory = BattleManager.processVictory;
	BattleManager.processVictory = function() {
		$gameTroop.isAnnihilateMembers().forEach(member => {
			member.addState(member.deathStateId());
			member.performCollapse();
		});
		_BattleManager_processVictory.call(this);
	};

	//-----------------------------------------------------------------------------
	// Game_Unit

	const _Game_Unit_isAllDead = Game_Unit.prototype.isAllDead;
	Game_Unit.prototype.isAllDead = function() {
		return _Game_Unit_isAllDead.call(this) || this.isAnnihilate();
	};

	Game_Unit.prototype.isAnnihilate = function() {
		return this.aliveMembers().every(member => member.isAnnihilateStateAffected());
	};

	Game_Unit.prototype.isAnnihilateMembers = function() {
		return this.aliveMembers().filter(member => member.isAnnihilateStateAffected());
	};

	//-----------------------------------------------------------------------------
	// Game_BattlerBase

	Game_BattlerBase.prototype.includesDeathStateId = function(stateId) {
		return paramBattler.has(stateId);
	};

	Game_BattlerBase.prototype.isAnnihilateStateAffected = function() {
		return this._states.find(state => paramUnit.has(state));
	};

	//-----------------------------------------------------------------------------
	// Game_Battler

	const _Game_Battler_addState = Game_Battler.prototype.addState;
	Game_Battler.prototype.addState = function(stateId) {
		const addable = this.isStateAddable(stateId);
		_Game_Battler_addState.call(this, stateId);
		if (addable && this.includesDeathStateId(stateId)) {
			this.addState(this.deathStateId());
		}
	};

	//-----------------------------------------------------------------------------
	// Game_Actor

	Game_Actor.prototype.includesDeathStateId = function(stateId) {
		return paramActor.has(stateId);
	};

	Game_Actor.prototype.isAnnihilateStateAffected = function() {
		return this._states.find(state => paramParty.has(state));
	};

	//-----------------------------------------------------------------------------
	// Game_Enemy

	Game_Enemy.prototype.includesDeathStateId = function(stateId) {
		return paramEnemy.has(stateId);
	};

	Game_Enemy.prototype.isAnnihilateStateAffected = function() {
		return this._states.find(state => paramTroop.has(state));
	};

	//-----------------------------------------------------------------------------
	// Window_BattleLog

	if (overlookLog || !deathStateMessage) {
		Window_BattleLog.prototype.findMarkedForDeath = function(target) {
			const result = target.result();
			const states = result.addedStateObjects();
			if (states.length === 0 || states[0].id === target.deathStateId()) {
				return;
			}
			return states.find(state => target.includesDeathStateId(state.id));
		};

		const _Window_BattleLog_displayAddedStates = Window_BattleLog.prototype.displayAddedStates;
		Window_BattleLog.prototype.displayAddedStates = function(target) {
			const result = target.result();
			const message = target.isActor() ? "message1" : "message2";
			const mfd = this.findMarkedForDeath(target);
			const death = $dataStates[target.deathStateId()];
			const mfdMessage = mfd ? mfd[message] : "";
			const deathMessage = death[message];
			if (mfd) {
				death[message] = "";
			}
			_Window_BattleLog_displayAddedStates.call(this, target);
			death[message] = deathMessage;
		};

		if (overlookLog) {
			Window_BattleLog.prototype.displayAddedStates = function(target) {
				const result = target.result();
				const message = target.isActor() ? "message1" : "message2";
				const mfd = this.findMarkedForDeath(target);
				const death = $dataStates[target.deathStateId()];
				const mfdMessage = mfd ? mfd[message] : "";
				const deathMessage = death[message];
				if (mfdMessage && deathMessage) {
					death[message] = `${mfdMessage}\n${useMZ ? deathMessage : target.name() + deathMessage}`;
				}
				_Window_BattleLog_displayAddedStates.call(this, target);
				death[message] = deathMessage;
			};

			const _Window_BattleLog_addText = Window_BattleLog.prototype.addText;
			Window_BattleLog.prototype.addText = function(text) {
				const texts = text.split("\n");
				texts.forEach(text => _Window_BattleLog_addText.call(this, text));
			};
		}

	}

}