//=============================================================================
// MPP_SmoothBattleLog_Op2.js
//=============================================================================
// Copyright (c) 2021 - 2024 Mokusei Penguin
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc Display the log only with the item name or skill name.
 * @author Mokusei Penguin
 * @url 
 * 
 * @base MPP_SmoothBattleLog
 * @orderAfter MPP_SmoothBattleLog
 *
 * @help [version 2.1.1]
 * - This plugin is for RPG Maker MZ.
 * 
 * ▼ Supplement
 *  - When using an item or skill, the log display is only the name,
 *    but the battle log confirmation command displays the normal log.
 *  - When displaying the state log, the window settings on the 
 *    MPP_SmoothBattleLog side are applied to the position and size of the
 *    log window.
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠ is half-width)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 * 
 *  @param Not Display Skills
 *      @desc 
 *      @type skill[]
 *      @default ["1","2"]
 * 
 *  @param Show Icon?
 *      @desc 
 *      @type boolean
 *      @default false
 * 
 *  @param Background Type
 *      @desc 
 *      @type select
 *          @option 0:Window
 *              @value 0
 *          @option 1:Dim
 *              @value 1
 *          @option 2:Transparent
 *              @value 2
 *      @default 0
 * 
 *  @param View State Log?
 *      @desc 
 *      @type boolean
 *      @default false
 * 
 */

/*:ja
 * @target MZ
 * @plugindesc ログ表示をアイテム名またはスキル名のみにします。
 * @author 木星ペンギン
 * @url 
 * 
 * @base MPP_SmoothBattleLog
 * @orderAfter MPP_SmoothBattleLog
 *
 * @help [version 2.1.1]
 * - このプラグインはRPGツクールMZ用です。
 * 
 * ▼ 補足
 *  - アイテムやスキルを使用した際のログ表示は名前のみとなりますが、
 *    戦闘ログの確認コマンドでは通常のログが表示されます。
 *  - ステートログを表示する場合、ログウィンドウの位置とサイズは
 *    MPP_SmoothBattleLog側のウィンドウ設定が適用されます。
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠は半角)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 * 
 *  @param Not Display Skills
 *      @text 表示しないスキルID
 *      @desc 
 *      @type skill[]
 *      @default ["1","2"]
 * 
 *  @param Font Size
 *      @text 文字の大きさ
 *      @desc 
 *      @type number
 *          @min 6
 *          @max 99
 *      @default 26
 * 
 *  @param Show Icon?
 *      @text アイコン表示
 *      @desc 
 *      @type boolean
 *      @default false
 * 
 *  @param Background Type
 *      @text ウィンドウ背景
 *      @desc 
 *      @type select
 *          @option 0:ウィンドウ
 *              @value 0
 *          @option 1:暗くする
 *              @value 1
 *          @option 2:透明
 *              @value 2
 *      @default 0
 * 
 *  @param View State Log?
 *      @text ステートログの表示
 *      @desc 
 *      @type boolean
 *      @default false
 * 
 */

(() => {
    'use strict';
    
    const pluginName = 'MPP_SmoothBattleLog_Op2';
    
    // Plugin Parameters
    const parameters = PluginManager.parameters(pluginName);
    const convertToSet = (param) => {
        return JSON.parse(param).reduce((r, item) => r.add(+item), new Set());
    };
    const param_NotDisplaySkills = convertToSet(parameters['Not Display Skills'] || '[]');
    const param_ShowIcon = parameters['Show Icon?'] === 'true';
    const param_FontSize = Number(parameters['Font Size'] || 26);
    const param_BackgroundType = Number(parameters['Background Type'] || 0);
    const param_ViewStateLog = parameters['View State Log?'] === 'true';

    //-------------------------------------------------------------------------
    // Window_BattleLog

    const _Window_BattleLog_initialize = Window_BattleLog.prototype.initialize;
    Window_BattleLog.prototype.initialize = function(rect) {
        _Window_BattleLog_initialize.apply(this, arguments);
        this._sendStateLog = false;
    };
    
    Window_BattleLog.prototype.setItemNameWindow = function(itemNameWindow) {
        this._itemNameWindow = itemNameWindow;
    };

    const _Window_BattleLog_clear = Window_BattleLog.prototype.clear;
    Window_BattleLog.prototype.clear = function() {
        _Window_BattleLog_clear.apply(this, arguments);
        if (this._itemNameWindow) {
            this._itemNameWindow.clear();
        }
    };
    
    const _Window_BattleLog_addText = Window_BattleLog.prototype.addText;
    Window_BattleLog.prototype.addText = function(text) {
        _Window_BattleLog_addText.apply(this, arguments);
        if (this._sendStateLog) {
            this.popupText(text);
        }
    };

    Window_BattleLog.prototype.isPopupText = function() {
        return false;
    };

    const _Window_BattleLog_displayAction = Window_BattleLog.prototype.displayAction;
    Window_BattleLog.prototype.displayAction = function(subject, item) {
        if (DataManager.isItem(item) || !param_NotDisplaySkills.has(item.id)) {
            this.push('addItemText', item);
        }
        _Window_BattleLog_displayAction.apply(this, arguments);
    };

    Window_BattleLog.prototype.addItemText = function(item) {
        if (this._itemNameWindow) {
            this._itemNameWindow.setItem(item);
        }
    };

    Window_BattleLog.prototype.sendStateLogOn = function() {
        this._sendStateLog = param_ViewStateLog;
    };

    Window_BattleLog.prototype.sendStateLogOff = function() {
        this._sendStateLog = false;
    };

    const _Window_BattleLog_displayAddedStates = Window_BattleLog.prototype.displayAddedStates;
    Window_BattleLog.prototype.displayAddedStates = function(target) {
        this.push('sendStateLogOn');
        _Window_BattleLog_displayAddedStates.apply(this, arguments);
        this.push('sendStateLogOff');
    };
    
    const _Window_BattleLog_displayRemovedStates = Window_BattleLog.prototype.displayRemovedStates;
    Window_BattleLog.prototype.displayRemovedStates = function(target) {
        this.push('sendStateLogOn');
        _Window_BattleLog_displayRemovedStates.apply(this, arguments);
        this.push('sendStateLogOff');
    };
    
    const _Window_BattleLog_displayBuffs = Window_BattleLog.prototype.displayBuffs;
    Window_BattleLog.prototype.displayBuffs = function(target, buffs, fmt) {
        this.push('sendStateLogOn');
        _Window_BattleLog_displayBuffs.apply(this, arguments);
        this.push('sendStateLogOff');
    };

    //-------------------------------------------------------------------------
    // Window_ItemName

    class Window_ItemName extends Window_Base {
        constructor(rect) {
            super(rect);
            this._item = null;
            this._text = '';
            this.setBackgroundType(param_BackgroundType);
        }

        updatePadding() {
            this.padding = 8;
        }
        
        fontSize() {
            return param_FontSize;
        }
    
        lineHeight() {
            const lineHeight = super.lineHeight();
            return lineHeight - $gameSystem.mainFontSize() + this.fontSize();
        }

        calcTextHeight() {
            return this.lineHeight();
        }

        resetFontSettings() {
            super.resetFontSettings();
            this.contents.fontSize = this.fontSize();
        }
    
        setItem(item) {
            if (this._item !== item) {
                this._item = item;
                this._text = item ? this.centerName(item) : '';
                this.refresh();
            }
        }
    
        clear() {
            this.setItem(null);
        }
    
        refresh() {
            this.updatePlacement();
            this.createContents();
            this.refreshDimmerBitmap();
            if (this._text) {
                this.drawTextEx(this._text, 8, 0, this.innerWidth);
            }
        }
        
        updatePlacement() {
            if (this._item) {
                this.width = this.windowWidth();
                this.height = this.itemHeight() + this.padding * 2;
            } else {
                this.width = 0;
                this.height = 0;
            }
            this.x = (Graphics.boxWidth - this.width) / 2;
        }
        
        windowWidth() {
            const textWidth = this.textSizeEx(this._text).width;
            const width = textWidth + 16 + this.padding * 2;
            return Math.min(width, Graphics.boxWidth);
        }
        
        centerName(item) {
            const iconText = param_ShowIcon ? `\\i[${item.iconIndex}]` : '';
            return '  ' + iconText + item.name + '  ';
        }

    }

    //-------------------------------------------------------------------------
    // Scene_Battle

    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function() {
        _Scene_Battle_createAllWindows.apply(this, arguments);
        this.createItemNameWindow();
    };

    Scene_Battle.prototype.createItemNameWindow = function() {
        const rect = this.itemNameWindowRect();
        this._itemNameWindow = new Window_ItemName(rect);
        this._logWindow.setItemNameWindow(this._itemNameWindow);
        this.addWindow(this._itemNameWindow);
    };
    
    Scene_Battle.prototype.itemNameWindowRect = function() {
        const ww = 0;
        const wh = 0;
        const wx = (Graphics.boxWidth - ww) / 2;
        const wy = 0;
        return new Rectangle(wx, wy, ww, wh);
    };
    
    Scene_Battle.prototype.updateLogWindowVisibility = function() {
        const helpVisible = this._helpWindow.visible;
        this._logWindow.visible = !helpVisible && param_ViewStateLog;
        this._itemNameWindow.visible = !helpVisible;
    };

})();
