// TilesetEx.js Ver.4.2.0

/*:
* @target MZ
* @plugindesc Make it possible to use two or more types of tilesets at the same time.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/492557242.html
*
* @noteParam bgName
* @noteDir img/parallaxes/
* @noteType file
* @noteData maps
*
* @help Ver.4.2.0
* 【How to use】
* (1) First, prepare a parent map (base map) and a child map (overlay map).
* (2) Group the two maps so that the name of the child map contains "tilesetEx".
* (3) Set separate tilesets for parent and child maps.
* (4) Place tiles on each map.
* (5) When you start the game and go to the parent map, it will be composited
* with the child map.
* Of course, the two tilesets coexist.
* (6) The third and subsequent ones can be added in the same way.
*
* Caution：
* ・The width and height of the two maps must be the same.
* ・Specify the ID of the parent map when transferring the player.
* ・Tiles, shadows, and regions are prioritized from the basemap.
* ・Parallax Background are not merged.
* ・Events created in the child map will be generated with the changed event id.
* Of course, the event ID specified in the event command will be different,
* so it is recommended not to create events other than map decoration.
*
* 【How to make a beautiful map】
* (1) Right-click the parent map and do [Save as Image].
* (2) Set the output location to img\parallaxes.
* (3) Specifies the image of the parent map as the Parallax Background of the child map.
* (4) Parent maps are displayed in child maps for intuitive map editing.
* Since the parallax background of the child map is not reflected in the game,
* you can test play immediately.
*
* Caution：
* ・Since the Parallax Background is used, the image remains
* even if [Exclude unused files] checked during deployment.
* ・If the width or height exceeds 63, the image will be reduced and saved,
* so please enlarge it with paint software etc. as appropriate.
* 
* 【Note (map properties)】
* Used in parent map notes.
*
* <bgName:Filename>　//Image specification of parallax background
* In the actual game, do not use the parallax background image specified
* in the editor,
* but use the image specified in this note.
*
* <noBg>　//No parallax background
* The parallax background is not displayed in the actual game.
* 
* 【Plugin Command】
* You can change the tileset of the child map by executing
* the event command [Change Tileset] after this plugin command.
*
* @param numTilesets
* @text Number of Tilesets
* @desc The specified number of tilesets will be available.
* @type number
* @default 2
* @min 2
*
* @param tilesetExName
* @text Child Map Name
* @desc Synthesize a map with a map name containing this string.
* @default tilesetEx
*
* @param changeBaseMap
* @text Change Base Map
* @desc Treat child map as base map.
* @type boolean
* @default false
*
* @command changeTilesetEx
* @text Change Tileset Ex
* @desc Changes the tileset used by the current child map temporarily.
*
* @arg index
* @text Child Map Index
* @desc Input the index of the child map to change tileset.
* The first one from the top is 0, the second one is 1.
* @type number
* @default 0
* 
*/

/*:ja
* @target MZ
* @plugindesc タイルセットを２種類以上同時に使えるようにします。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/492557242.html
*
* @noteParam bgName
* @noteDir img/parallaxes/
* @noteType file
* @noteData maps
*
* @help 【使い方】
* (1) まず、親マップ（ベースマップ）と子マップ（重ねるマップ）
* を用意します。
* (2) ２つのマップをグループ化し、子マップの名前が"tilesetEx"を
* 含むようにして下さい。
* (3) 親マップと子マップに別々のタイルセットを設定します。
* (4) それぞれのマップにタイルを設置します。
* (5) ゲームを起動して親マップに移動すると子マップと合成されています。
* もちろんタイルセットが２つ共存しています。
* (6) ３つ目以降も同様に追加可能です。
*
* 注意：
* ・２つのマップは幅・高さを同じにする必要があります。
* ・マップ移動時は、親マップのIDを指定してください。
* ・タイルや影、リージョンはベースマップのものが優先されます。
* ・遠景は合成されません。
* ・子マップで作成したイベントはイベントIDが変更された状態で生成されます。
* 当然イベントコマンドで指定したイベントIDとのズレが発生するため、
* マップ装飾用のイベント以外は作らないことを推奨します。
*
* 【綺麗にマップを作る方法】
* (1) 親マップを右クリックし［画像として保存］を行います。
* (2) 出力場所をimg\parallaxesにします。
* (3) 子マップの遠景として親マップの画像を指定します。
* (4) 子マップに親マップが表示され、直感的にマップを編集可能になります。
* 子マップの遠景はゲーム上に反映されないのでテストプレイもすぐにできます。
*
* 注意：
* ・遠景を利用しているため、デプロイメント時に［未使用ファイルを除外する］に
* チェックをいれても画像が残ります。
* ・幅または高さが63を超えた場合、画像が縮小されて保存されるため、
* 適宜ペイントソフトなどで拡大して使用して下さい。
* 
* 【メモ（マップの設定）】
* 親マップのメモ欄に使用します。
* 本プラグインではエディタの遠景表示機能を利用したマップ編集を要求する為、
* それにより発生する遠景画像の設定の煩わしさを解消する機能です。
* これらのメモタグを活用する事でスムーズにマップ編集からテストプレイに
* 移行できます。
*
* <bgName:ファイル名>　//遠景の画像指定
* 実際のゲームではエディタで指定した遠景画像を使わず、このメモで指定した画像を
* 使うようにします。
*
* <noBg>　//遠景なし
* 実際のゲームでは遠景を使いません。
* 
* 【プラグインコマンド】
* プラグインコマンドの直後にイベントコマンド［タイルセットの変更］を行うことで、
* 子マップのタイルセットを変更できます。
*
* [更新履歴]
* 2022/10/17：Ver.1.0.0　公開。
* 2022/10/17：Ver.1.0.1　処理を最適化。
* 2022/10/20：Ver.1.0.2　合成したタイルセットの格納先を$dataTilesetExに変更。
* 2022/10/23：Ver.1.1.0　ベースマップを入れ替える機能を追加。MV版のプラグインコマンドに対応。
* 2022/11/15：Ver.1.1.1　イベントテスト時にエラーが起こる問題の修正。
* 2022/12/07：Ver.2.0.0　競合対策。バグ修正。プラグインコマンドの仕様を変更。
* 2022/12/08：Ver.2.0.1　$gameMap.hasMapEx()の判定基準をより厳密に。
* 2022/12/08：Ver.2.0.2　競合対策。
* 2022/12/09：Ver.2.0.3　競合対策。
* 2023/02/01：Ver.3.0.0　タイルセット数を無制限にしました。
* 2023/02/12：Ver.4.0.0　余分なデータを破棄する仕様に変更。
* 2023/03/13：Ver.4.0.1　カウンター属性のタイルのグラフィックを修正。
* 2024/01/09：Ver.4.0.2　オートタイル属性を修正。キャラクターグラフィックの処理を削減。
* 2024/01/24：Ver.4.1.0　影の合成に対応しました。
* 2024/02/25：Ver.4.1.1　特定状況下で通行判定が正しく反映されない現象を修正。
* 2024/11/29：Ver.4.2.0　特定状況下でタイルセットが正しく反映されない不具合を修正。
*
* @param numTilesets
* @text タイルセットの数
* @desc 指定した数だけタイルセットが使用可能になります。
* @type number
* @default 3
* @min 2
*
* @param tilesetExName
* @text 重ねるマップの名前
* @desc マップ名にこの文字列が含まれている時、マップを重ねます。
* 日本語使用可。
* @default tilesetEx
*
* @param changeBaseMap
* @text ベースマップ変更
* @desc 子マップをベースマップ、親マップを重ねるマップとして扱います。
* @type boolean
* @default false
*
* @command changeTilesetEx
* @text タイルセットEXの変更
* @desc 現在のマップで使用されている子マップのタイルセットを一時的に変更します。
*
* @arg index
* @text 子マップのインデックス
* @desc 変更したい子マップのインデックスを指定します。
* 上から一つ目なら0、二つ目なら1。
* @type number
* @default 0
* 
*/

'use strict';

{
	//プラグイン名取得。
	const script = document.currentScript;
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const parameters = PluginManager.parameters(pluginName);
	const tilesetExName = parameters["tilesetExName"];
	const changeBaseMap = parameters["changeBaseMap"] === "true";
	const numTilesets = Number(parameters["numTilesets"]);
	const numTilesetsEx = numTilesets - 1;

	window.$dataMapEx = [];
	for (let i = 0; i < numTilesetsEx; i++) {
		window["$tempMapEx%1".format(i)] = null;
	}
	window.$dataTilesetEx = null;

	//-----------------------------------------------------------------------------
	// PluginManager

	let changeTilesetEx = false;
	let tilesetIndex = 0;
	PluginManager.registerCommand(pluginName, "changeTilesetEx", function(args) {
		changeTilesetEx = true;
		tilesetIndex = Number(args.index || 0);
	});

	//-----------------------------------------------------------------------------
	// Game_Interpreter（MV版から移行する時用）

	const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
	Game_Interpreter.prototype.pluginCommand = function(command, args) {
		_Game_Interpreter_pluginCommand.apply(this, arguments);
		if (command === "changeTilesetEx") {
			changeTilesetEx = true;
			tilesetIndex = Number(args || 0);
		}
	};

	//-----------------------------------------------------------------------------
	// Sprite_Character
	//イベントが追加したタイルセットの画像を表示可能に

	const _Sprite_Character_updateBitmap = Sprite_Character.prototype.updateBitmap;
	Sprite_Character.prototype.updateBitmap = function() {
		const tilesetId = $gameMap._tilesetId;
		if (Tilemap.isTileEx(this._tileId)) {
			const index = Math.floor(this._tileId / Tilemap.TILE_ID_MAX) - 1;
			$gameMap._tilesetId = $gameMap.tilesetExId(index);
		}
		_Sprite_Character_updateBitmap.call(this);
		$gameMap._tilesetId = tilesetId;
	};

	const _Sprite_Character_tilesetBitmap = Sprite_Character.prototype.tilesetBitmap;
	Sprite_Character.prototype.tilesetBitmap = function(tileId) {
		if (Tilemap.isTileEx(tileId)) {
			const offsetNumber = Tilemap.offsetNumber(tileId);
			tileId = Tilemap.originalTileId(tileId);
			tileId += offsetNumber * 256;//setNumberを合わせるための補正値（9*256）
		}
		return _Sprite_Character_tilesetBitmap.call(this, tileId);
	};

	const _Sprite_Character_updateTileFrame = Sprite_Character.prototype.updateTileFrame;
	Sprite_Character.prototype.updateTileFrame = function() {
		const tileId = this._tileId;
		this._tileId = Tilemap.originalTileId(tileId);
		_Sprite_Character_updateTileFrame.call(this);
		this._tileId = tileId;
	};

	//-----------------------------------------------------------------------------
	// Tile type checkers
	//追加したタイルを通常のタイルと同じように判別可能にする。
	Tilemap.isTileEx = function(tileId) {
		return tileId >= this.TILE_ID_MAX;
	};

	Tilemap.originalTileId = function(tileId) {
		return tileId % this.TILE_ID_MAX;
	};

	Tilemap.offsetNumber = function(tileId) {
		return 9 * Math.floor(tileId / this.TILE_ID_MAX);
	};

	const _Tilemap_isVisibleTile = Tilemap.isVisibleTile;
	Tilemap.isVisibleTile = function(tileId) {
		return _Tilemap_isVisibleTile.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_isAutotile = Tilemap.isAutotile;
	Tilemap.isAutotile = function(tileId) {
		return _Tilemap_isAutotile.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_getAutotileKind = Tilemap.getAutotileKind;
	Tilemap.getAutotileKind = function(tileId) {
		return _Tilemap_getAutotileKind.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_getAutotileShape = Tilemap.getAutotileShape;
	Tilemap.getAutotileShape = function(tileId) {
		return _Tilemap_getAutotileShape.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_isTileA1 = Tilemap.isTileA1;
	Tilemap.isTileA1 = function(tileId) {
		return _Tilemap_isTileA1.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_isTileA2 = Tilemap.isTileA2;
	Tilemap.isTileA2 = function(tileId) {
		return _Tilemap_isTileA2.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_isTileA3 = Tilemap.isTileA3;
	Tilemap.isTileA3 = function(tileId) {
		return _Tilemap_isTileA3.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_isTileA4 = Tilemap.isTileA4;
	Tilemap.isTileA4 = function(tileId) {
		return _Tilemap_isTileA4.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_isTileA5 = Tilemap.isTileA5;
	Tilemap.isTileA5 = function(tileId) {
		return _Tilemap_isTileA5.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_isWaterTile = Tilemap.isWaterTile;
	Tilemap.isWaterTile = function(tileId) {
		return _Tilemap_isWaterTile.call(this, this.originalTileId(tileId));
	};

	const _Tilemap_isWaterfallTile = Tilemap.isWaterfallTile;
	Tilemap.isWaterfallTile = function(tileId) {
		return _Tilemap_isWaterfallTile.call(this, this.originalTileId(tileId));
	};

	//-----------------------------------------------------------------------------
	// Tilemap
	//追加されたタイルセットの画像インデックスを指定。

	let offsetNumber = 0;
	const _Tilemap__addTile = Tilemap.prototype._addTile;
	Tilemap.prototype._addTile = function(layer, tileId, dx, dy) {
		if (Tilemap.isTileEx(tileId)) {
			offsetNumber = Tilemap.offsetNumber(tileId);
			tileId = Tilemap.originalTileId(tileId);
		}
		_Tilemap__addTile.call(this, layer, tileId, dx, dy);
		offsetNumber = 0;
	};

	const _Tilemap__addTableEdge = Tilemap.prototype._addTableEdge;
	Tilemap.prototype._addTableEdge = function(layer, tileId, dx, dy) {
		if (Tilemap.isTileEx(tileId)) {
			offsetNumber = Tilemap.offsetNumber(tileId);
			tileId = Tilemap.originalTileId(tileId);
		}
		_Tilemap__addTableEdge.call(this, layer, tileId, dx, dy);
		offsetNumber = 0;
	};

	const _Tilemap__isTableTile = Tilemap.prototype._isTableTile;
	Tilemap.prototype._isTableTile = function(tileId) {
		if (offsetNumber) {
			tileId += Math.floor(offsetNumber / 9) * Tilemap.TILE_ID_MAX;
		}
		return _Tilemap__isTableTile.call(this, tileId);
	};

	//-----------------------------------------------------------------------------
	// Tilemap.Layer
	//テクスチャ―の上限を増やす。
	Tilemap.Layer.MAX_GL_TEXTURES = Math.ceil(numTilesets * 9 / 4);

	const _Tilemap_Layer_addRect = Tilemap.Layer.prototype.addRect;
	Tilemap.Layer.prototype.addRect = function(setNumber, sx, sy, dx, dy, w, h) {
		_Tilemap_Layer_addRect.call(this, setNumber + offsetNumber, sx, sy, dx, dy, w, h);
	};

	//-----------------------------------------------------------------------------
	// Tilemap.Renderer
	//なんか追加したらちゃんとタイルが表示された。
	Tilemap.Renderer.prototype._createShader = function() {
		const vertexSrc =
			"attribute float aTextureId;" +
			"attribute vec4 aFrame;" +
			"attribute vec2 aSource;" +
			"attribute vec2 aDest;" +
			"uniform mat3 uProjectionMatrix;" +
			"varying vec4 vFrame;" +
			"varying vec2 vTextureCoord;" +
			"varying float vTextureId;" +
			"void main(void) {" +
			"  vec3 position = uProjectionMatrix * vec3(aDest, 1.0);" +
			"  gl_Position = vec4(position, 1.0);" +
			"  vFrame = aFrame;" +
			"  vTextureCoord = aSource;" +
			"  vTextureId = aTextureId;" +
			"}";
		const fragmentSrc =
			"varying vec4 vFrame;" +
			"varying vec2 vTextureCoord;" +
			"varying float vTextureId;" +
			createSampler("uniform sampler2D uSampler%1;") +
			"void main(void) {" +
			"  vec2 textureCoord = clamp(vTextureCoord, vFrame.xy, vFrame.zw);" +
			"  int textureId = int(vTextureId);" +
			"  vec4 color;" +
			"  if (textureId < 0) {" +
			"    color = vec4(0.0, 0.0, 0.0, 0.5);" +
			"  }" + createSampler("else if (textureId == %1) {    color = texture2D(uSampler%1, textureCoord / 2048.0);  }") +
			"  gl_FragColor = color;" +
			"}";

		const arg = {
			uProjectionMatrix: new PIXI.Matrix()
		}
		for (let i = 0; i < Tilemap.Layer.MAX_GL_TEXTURES; i++) {
			arg[`uSampler${i}`] = 0;
		}
		return new PIXI.Shader(PIXI.Program.from(vertexSrc, fragmentSrc), arg);
	};

	function createSampler(arg) {
		let str = "";
		for (let i = 0; i < Tilemap.Layer.MAX_GL_TEXTURES; i++) {
			str += arg.format(i);
		}
		return str;
	}

	//-----------------------------------------------------------------------------
	// DataManager
	//追加したマップのロードも条件に追加。
	const _DataManager_isMapLoaded = DataManager.isMapLoaded;
	DataManager.isMapLoaded = function() {
		const mapLoaded = _DataManager_isMapLoaded.call(this);
		if (sceneMapIsReady) {
			return mapLoaded && this.isMapExLoaded();
		} else {
			return mapLoaded;
		}
	};

	DataManager.isMapExLoaded = function() {
		this.checkError();
		for (let i = 0; i < numTilesetsEx; i++) {
			const name = "$tempMapEx%1".format(i);
			if (!window[name]) {
				return false;
			}
		}
		return true;
	};

	DataManager.loadMapExData = function(parentMapId) {
		const info = $dataMapInfos[parentMapId];
		const maps = $dataMapInfos.filter(info => info && info.parentId === parentMapId && info.name.indexOf(tilesetExName) > -1);
		maps.sort((a, b) => a.order - b.order);
		for (let i = 0; i < numTilesetsEx; i++) {
			const name = "$tempMapEx%1".format(i);
			const mapId = maps[i] ? maps[i].id : 0;
			if (mapId > 0) {
				const filename = "Map%1.json".format(mapId.padZero(3));
				this.loadDataFile(name, filename);
			} else {
				this.makeEmptyMapEx(name);
			}
		}
	};

	DataManager.makeEmptyMapEx = function(name) {
		window[name] = {};
	};

	DataManager.onMapExLoaded = function() {
		this.pushMapEx();
		this.synthesizeMapData();
		this.clearDataMapEx();
	};

	DataManager.pushMapEx = function() {
		for (let i = 0; i < numTilesetsEx; i++) {
			const name = "$tempMapEx%1".format(i);
			const dataMap = window[name];
			if (dataMap.data && dataMap.width === $dataMap.width) {
				$dataMapEx.push(dataMap);
			} else {
				this.makeEmptyMapEx(name);
			}
		}
	};

	DataManager.synthesizeMapData = function() {
		for (let i = 0; i < $dataMapEx.length; i++) {
			this.synthesizeEvents(i);
			this.synthesizeMap(i);
		}
		this.synthesizeTileset();
	};
	
	//マップを合体
	DataManager.synthesizeMap = function(index) {
		this.synthesizeTile(index);
		this.synthesizeShadow(index);
		this.synthesizeRegion(index);
	};

	DataManager.synthesizeTile = function(index) {
		const mapEx = $dataMapEx[index];
		const shadowStart = (4 * $dataMap.height) * $dataMap.width;
		for (let i = 0; i < shadowStart; i++) {
			const tileId = $dataMap.data[i];
			const tileExId = mapEx.data[i];
			if (tileExId !== 0 && (!changeBaseMap || tileId === 0)) {
				$dataMap.data[i] = tileExId + Tilemap.TILE_ID_MAX * (index + 1);
			}
		}
	};

	DataManager.synthesizeShadow = function(index) {
		const mapEx = $dataMapEx[index];
		const shadowStart = (4 * $dataMap.height) * $dataMap.width;
		const regionStart = (5 * $dataMap.height) * $dataMap.width;
		for (let i = shadowStart; i < regionStart; i++) {
			const shadowBits = $dataMap.data[i];
			const shadowExBits = mapEx.data[i];
			$dataMap.data[i] = shadowExBits | shadowBits;
		}
	};

	DataManager.synthesizeRegion = function(index) {
		const mapEx = $dataMapEx[index];
		const regionStart = (5 * $dataMap.height) * $dataMap.width;
		for (let i = regionStart; i < $dataMap.data.length; i++) {
			const regionId = $dataMap.data[i];
			const regionExId = mapEx.data[i];
			if (regionExId !== 0 && (!changeBaseMap || regionId === 0)) {
				$dataMap.data[i] = regionExId;
			}
		}
	};
	//イベントを合体
	DataManager.synthesizeEvents = function(index) {
		const mapEx = $dataMapEx[index];
		mapEx.events.forEach((event, id) => {
			if (!event) return;
			event.id += $dataMap.events.length;
			event.pages.forEach(page => {
				const tileId = page.image.tileId;
				if (tileId > 0) {
					page.image.tileId += Tilemap.TILE_ID_MAX*(index+1);
				}
			});
		});
		$dataMap.events = $dataMap.events.concat(mapEx.events);
	};

	DataManager.synthesizeTileset = function() {
		$dataMap.tilesetExIds = [];
		for (let i = 0; i < $dataMapEx.length; i++) {
			$dataMap.tilesetExIds[i] = $dataMapEx[i].tilesetId;
		}
	};

	DataManager.clearDataMapEx = function() {
		$dataMapEx = [];
		for (let i = 0; i < numTilesetsEx; i++) {
			const name = "$tempMapEx%1".format(i);
			this.makeEmptyMapEx(name);
		}
	};

	//-----------------------------------------------------------------------------
	// Scene_Map

	const _Scene_Map_create = Scene_Map.prototype.create;
	Scene_Map.prototype.create = function() {
		$gameMap.initTilesetEx();
		_Scene_Map_create.call(this);
		if (this._transfer) {
			DataManager.loadMapExData($gamePlayer.newMapId());
		} else {
			DataManager.loadMapExData($gameMap.mapId());
		}
	};

	let sceneMapIsReady = false;
	const _Scene_Map_isReady = Scene_Map.prototype.isReady;
	Scene_Map.prototype.isReady = function() {
		sceneMapIsReady = true;
		const result = _Scene_Map_isReady.call(this);
		sceneMapIsReady = false;
		return result;
	};

	const _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
	Scene_Map.prototype.onMapLoaded = function() {
		this.onMapExLoaded();
		_Scene_Map_onMapLoaded.call(this);
	};
	//マップ切り替え時以外は$gameMap.setup()を経由しないのでここでタイルセットを合成しておく。
	Scene_Map.prototype.onMapExLoaded = function() {
		DataManager.onMapExLoaded();
		if (!this._transfer) {
			$gameMap.synthesizeTileset();
		}
	};

	//-----------------------------------------------------------------------------
	// Game_Player

	let setup = false;
	const _Game_Player_performTransfer = Game_Player.prototype.performTransfer;
	Game_Player.prototype.performTransfer = function() {
		setup = false;
		const isTransferring = this.isTransferring();
		_Game_Player_performTransfer.call(this);
		if (isTransferring && !setup) {
			$gameMap.synthesizeTileset();
		}
		setup = false;
	};

	//-----------------------------------------------------------------------------
	// Game_Map

	const _Game_Map_initialize = Game_Map.prototype.initialize;
	Game_Map.prototype.initialize = function() {
		_Game_Map_initialize.call(this);
		this._tilesetExIds = [];
	};
	//この関数は同じIDのマップに移動する時や、ロード時には読み込まれない。
	const _Game_Map_setup = Game_Map.prototype.setup;
	Game_Map.prototype.setup = function(mapId) {
		_Game_Map_setup.call(this, mapId);
		if (!$dataMap.tilesetExIds) {
			$dataMap.tilesetExIds = [];
		}
		this._tilesetExIds = [];
		for (let i = 0; i < $dataMap.tilesetExIds.length; i++) {
			this._tilesetExIds[i] = $dataMap.tilesetExIds[i];
		}
		this.synthesizeTileset();
		setup = true;
	};
	//タイルセットを合体
	Game_Map.prototype.synthesizeTileset = function() {
		this.initTilesetEx();
		if (!$dataMap.tilesetExIds) {
			$dataMap.tilesetExIds = [];
		}
		if ($dataMap.tilesetExIds.length > 0) {
			$dataTilesetEx = JsonEx.makeDeepCopy($dataTilesets[this._tilesetId]);
			$dataTilesetEx.flags.length = Tilemap.TILE_ID_MAX;
		}
		for (let i = 0; i < $dataMap.tilesetExIds.length; i++) {
			const tilesetId = this._tilesetExIds[i];
			const tileset = JsonEx.makeDeepCopy($dataTilesets[tilesetId]);
			tileset.flags.length = Tilemap.TILE_ID_MAX;
			$dataTilesetEx.tilesetNames = $dataTilesetEx.tilesetNames.concat(tileset.tilesetNames);
			$dataTilesetEx.flags = $dataTilesetEx.flags.concat(tileset.flags);
		}
	};

	const _Game_Map_setupParallax = Game_Map.prototype.setupParallax;
	Game_Map.prototype.setupParallax = function() {
		if ($dataMap.meta) {
			if ($dataMap.meta.bgName) {
				$dataMap.parallaxName = $dataMap.meta.bgName || "";
			}
			if ($dataMap.meta.noBg) {
				$dataMap.parallaxName = "";
			}
		}
		_Game_Map_setupParallax.call(this);
	};
	//タイルセット合体時のみ読み込む。
	const _Game_Map_tileset = Game_Map.prototype.tileset;
	Game_Map.prototype.tileset = function() {
		return this.tilesetEx() || _Game_Map_tileset.call(this);
	};
	//タイルセットを切り替えるたびにタイルセットを合体させる。
	const _Game_Map_changeTileset = Game_Map.prototype.changeTileset;
	Game_Map.prototype.changeTileset = function(tilesetId) {
		if (changeTilesetEx) {
			changeTilesetEx = false;
			this.changeTilesetEx(tilesetId, tilesetIndex);
			return;
		}
		_Game_Map_changeTileset.call(this, tilesetId);
		this.synthesizeTileset();
	};

	Game_Map.prototype.changeTilesetEx = function(tilesetId, index) {
		if (index < $dataMap.tilesetExIds.length) {
			this._tilesetExIds[index] = tilesetId;
			this.refresh();
			this.synthesizeTileset();
		}
	};

	Game_Map.prototype.tilesetEx = function() {
		return $dataTilesetEx;
	};

	Game_Map.prototype.tilesetExId = function(index) {
		return this._tilesetExIds[index];
	};

	Game_Map.prototype.initTilesetEx = function() {
		$dataTilesetEx = null;
	};

	Game_Map.prototype.autotileType = function(x, y, z) {
		const tileId = Tilemap.originalTileId(this.tileId(x, y, z));
		return tileId >= 2048 ? Math.floor((tileId - 2048) / 48) : -1;
	};

}
