/*:
 * @param newGameSe
 * @text ニューゲームSE
 * @desc ニューゲーム決定時に再生する効果音です。未設定なら通常の決定SEを再生します。
 * @default
 * @type struct<AudioSe>
 *
 * @param anotherDataList
 * @text アナザーニューゲームリスト
 * @desc アナザーニューゲームのコマンド一覧です。
 * @default []
 * @type struct<COMMAND>[]
 *
 * @param defaultCursorIndex
 * @text カーソル初期位置
 * @desc セーブデータがないときのタイトルコマンドのカーソル初期位置です。0を指定するとデフォルト仕様に準拠します。
 * @default 0
 * @type number
 *
 * @param commandImageList
 * @text タイトル画像コマンド設定
 * @desc タイトルコマンドごとの画像・座標・非選択時のトーン・決定時スクリプト設定。
 * @default []
 * @type struct<CommandImage>[]
 *
 * @param ShutdownText
 * @text ゲーム終了コマンド名
 * @desc タイトル・ゲーム終了画面に追加するゲーム終了コマンドの名称です。
 * @default ゲーム終了
 *
 * @param UseGameEnd
 * @text ゲーム終了画面に追加
 * @desc ゲーム終了画面(Window_GameEnd)にもゲーム終了(Shutdown)を追加します。
 * @type boolean
 * @default true
 *
 * @param AnimationMode
 * @text アニメーションタイプ（未使用）
 * @desc 互換性のために残しているだけで、実際にはアニメーションは行いません。
 * @type select
 * @option なし
 * @value 0
 * @option 拡大縮小
 * @value 1
 * @option シェイク
 * @value 2
 * @default 0
 *
 * @param LeftRightInput
 * @text 左右入力で選択
 * @desc 左右キーでコマンドを移動するかどうか。
 * @type boolean
 * @default true
 *
 * @param SlideX
 * @text 位置オフセットX
 * @desc 全コマンド共通のXオフセット（アニメではなく固定オフセット）
 * @type number
 * @default 0
 *
 * @param SlideY
 * @text 位置オフセットY
 * @desc 全コマンド共通のYオフセット（アニメではなく固定オフセット）
 * @type number
 * @default 0
 *
 * @command SETTING
 * @text コマンド制御
 * @desc アナザーニューゲームコマンドの表示と利用可否を制御します。
 *
 * @arg type
 * @text 操作種別
 * @desc 操作種別を選択します。
 * @default VISIBLE
 * @type select
 * @option コマンド表示
 * @value VISIBLE
 * @option コマンド非表示
 * @value HIDDEN
 * @option コマンド利用許可
 * @value ENABLE
 * @option コマンド利用禁止
 * @value DISABLE
 *
 * @arg index
 * @text コマンドインデックス
 * @desc 操作対象のコマンドインデックスです。(開始位置は0)
 * @type number
 * @default 0
 *
 * @command NEW_GAME_SETTING
 * @text ニューゲーム制御
 * @desc ニューゲームコマンドの表示を制御します。設定次第でゲームが開始できなくなるので注意してください。
 *
 * @arg type
 * @text 操作種別
 * @desc 操作種別を選択します。
 * @default VISIBLE
 * @type select
 * @option コマンド表示
 * @value VISIBLE
 * @option コマンド非表示
 * @value HIDDEN
 *
 * @command SET_DEFAULT_CURSOR
 * @text カーソル初期位置設定
 * @desc セーブデータがないときのタイトルコマンドのカーソル初期位置を設定します。
 *
 * @arg index
 * @text カーソル初期位置
 * @desc カーソル初期位置です。(先頭は0)
 * @type number
 * @default 0
 *
 */

/*~struct~AudioSe:
 * @param name
 * @text ファイル名
 * @desc 再生するSEファイル名です。(audio/se フォルダ)
 * @default
 * @require 1
 * @dir audio/se/
 * @type file
 *
 * @param volume
 * @text 音量
 * @desc 再生音量(0〜100)です。
 * @default 90
 * @type number
 * @min 0
 * @max 100
 *
 * @param pitch
 * @text ピッチ
 * @desc 再生ピッチ(50〜150)です。
 * @default 100
 * @type number
 * @min 50
 * @max 150
 *
 * @param pan
 * @text 位相
 * @desc 左右バランス(-100〜100)です。
 * @default 0
 * @type number
 * @min -100
 * @max 100
 */

/*~struct~COMMAND:
 * @param name
 * @text コマンド名称
 * @desc タイトル画面に表示されるコマンド名です。
 * @default Another New Game
 *
 * @param mapId
 * @text マップID
 * @desc 移動先のマップIDです。0を指定した場合、場所移動しません。
 * @default 1
 * @type number
 *
 * @param mapX
 * @text X座標
 * @desc 移動先のX座標です。
 * @default 1
 * @type number
 *
 * @param mapY
 * @text Y座標
 * @desc 移動先のY座標です。
 * @default 1
 * @type number
 *
 * @param hidden
 * @text デフォルト非表示
 * @desc デフォルトで選択肢を非表示にします。プラグインコマンドで有効化できます。
 * @default false
 * @type boolean
 *
 * @param disable
 * @text デフォルト使用禁止
 * @desc デフォルトで選択肢を選択禁止にします。プラグインコマンドで有効化できます。
 * @default false
 * @type boolean
 *
 * @param addPosition
 * @text 追加位置
 * @desc アナザーニューゲームのコマンド追加位置です。(1:ニューゲームの上、2:コンティニューの上、3:オプションの上)
 * @default 0
 * @type select
 * @option オプションの下
 * @value 0
 * @option ニューゲームの上
 * @value 1
 * @option コンティニューの上
 * @value 2
 * @option オプションの上
 * @value 3
 *
 * @param demoWaitFrame
 * @text デモ待機フレーム数
 * @desc 指定したフレーム数ぶん無操作状態が続いたときに開始します。指定するとコマンドには出現しなくなります。
 * @default 0
 * @type number
 *
 * @param switchId
 * @text 開始時有効スイッチ
 * @desc アナザーニューゲーム開始時に自動でONになるスイッチを指定できます。
 * @default 0
 * @type switch
 *
 * @param fileLoad
 * @text ファイルロード
 * @desc アナザーニューゲーム選択時に、ロード画面に遷移して既存セーブデータをロードします。
 * @default false
 * @type boolean
 *
 * @param noFadeout
 * @text フェードアウト無効
 * @desc アナザーニューゲーム選択時に、オーディオや画面がフェードアウトしなくなります。
 * @default false
 * @type boolean
 */

/*~struct~CommandImage:
 * @param symbol
 * @text コマンド識別子
 * @desc newGame / continue / options / nameGame2_0 / nameGame2_1 / ... / shutdown など。
 * @default
 *
 * @param filename
 * @text 画像ファイル名
 * @desc img/titles2 内のファイル名（拡張子なし）
 * @default
 *
 * @param posX
 * @text 表示X座標
 * @desc このコマンド画像の中心X座標（ピクセル）。未設定ならデフォルト配置。
 * @type number
 * @default 0
 *
 * @param posY
 * @text 表示Y座標
 * @desc このコマンド画像の中心Y座標（ピクセル）。未設定ならデフォルト配置。
 * @type number
 * @default 0
 *
 * @param toneR
 * @text 非選択時トーンR
 * @type number
 * @min -255
 * @max 255
 * @default 0
 *
 * @param toneG
 * @text 非選択時トーンG
 * @type number
 * @min -255
 * @max 255
 * @default 0
 *
 * @param toneB
 * @text 非選択時トーンB
 * @type number
 * @min -255
 * @max 255
 * @default 0
 *
 * @param toneGray
 * @text 非選択時トーングレー
 * @type number
 * @min 0
 * @max 255
 * @default 0
 *
 * @param okScript
 * @text 決定時スクリプト
 * @desc このコマンドが決定されたときに eval で実行されるスクリプト。空なら何もしない。
 * @type note
 * @default ""
 */

(function() {
    const script     = document.currentScript;
    const parameters = PluginManagerEx.createParameter(script);

    const CommandImageSettings = {};
    if (parameters.commandImageList) {
        parameters.commandImageList.forEach(cfg => {
            if (cfg && cfg.symbol) {
                CommandImageSettings[String(cfg.symbol)] = cfg;
            }
        });
    }
    const NewGameSeSetting = parameters.newGameSe || null;
    const shutdownText = parameters.ShutdownText || "ゲーム終了";
    const useGameEnd   = (parameters.UseGameEnd !== undefined) ? parameters.UseGameEnd : true;


    let localExtraStageIndex = -1;

    PluginManagerEx.registerCommand(script, 'SETTING', args => {
        const index = args.index;
        switch (args.type) {
            case 'VISIBLE':
                ANGSettingManager.setVisible(index, true);
                break;
            case 'ENABLE':
                ANGSettingManager.setEnable(index, true);
                break;
            case 'HIDDEN':
                ANGSettingManager.setVisible(index, false);
                break;
            case 'DISABLE':
                ANGSettingManager.setEnable(index, false);
                break;
        }
        ANGSettingManager.save();
    });

    PluginManagerEx.registerCommand(script, 'NEW_GAME_SETTING', args => {
        ANGSettingManager.newGameHidden = args.type === 'HIDDEN';
        ANGSettingManager.save();
    });

    PluginManagerEx.registerCommand(script, 'SET_DEFAULT_CURSOR', args => {
        ANGSettingManager.setDefaultCursorIndex(args.index);
        ANGSettingManager.save();
    });

    Game_Map.prototype.abortInterpreter = function() {
        if (this.isEventRunning()) {
            this._interpreter.command115();
        }
    };

    const _Scene_Boot_onDatabaseLoaded = Scene_Boot.prototype.onDatabaseLoaded;
    Scene_Boot.prototype.onDatabaseLoaded = function() {
        _Scene_Boot_onDatabaseLoaded.apply(this, arguments);
        ANGSettingManager.loadData();
    };

    const _Scene_Boot_isReady    = Scene_Boot.prototype.isReady;
    Scene_Boot.prototype.isReady = function() {
        return _Scene_Boot_isReady.apply(this, arguments) && ANGSettingManager.isLoaded();
    };

    const _Scene_Title_commandContinue    = Scene_Title.prototype.commandContinue;
    Scene_Title.prototype.commandContinue = function() {
        _Scene_Title_commandContinue.call(this);
        localExtraStageIndex = -1;
    };

    const _Window_TitleCommand_selectLast = Window_TitleCommand.prototype.selectLast;
    Window_TitleCommand.prototype.selectLast = function() {
        const index = ANGSettingManager._defaultCursorIndex;
        if (index > 0) {
            this.select(index);
        }
        _Window_TitleCommand_selectLast.apply(this, arguments);
    };

    const _Scene_Title_update = Scene_Title.prototype.update;
    Scene_Title.prototype.update = function() {
        _Scene_Title_update.apply(this, arguments);
        if (!this.isBusy() && !this._callAnotherNewGame) {
            this.updateWait();
        }
    };

    Scene_Title.prototype.updateWait = function() {
        if (this._commandWindowIndex !== this._commandWindow.index()) {
            this._commandWindowIndex = this._commandWindow.index();
            this._waitFrame = 0;
        } else {
            this._waitFrame++;
        }
        ANGSettingManager.findList(true).forEach(command => {
            if (this._waitFrame >= command.demoWaitFrame) {
                this.callAnotherNewGame(command);
            }
        });
        if (this._callAnotherNewGame) {
            this._commandWindow.open();
            this._commandWindow.deactivate();
        }
    };

    const _Scene_Title_commandNewGameSecond    = Scene_Title.prototype.commandNewGameSecond;
    Scene_Title.prototype.commandNewGameSecond = function(index) {
        if (_Scene_Title_commandNewGameSecond) {
            _Scene_Title_commandNewGameSecond.apply(this, arguments);
        }
        const command = ANGSettingManager.findList(false)[index];
        this.callAnotherNewGame(command);
    };

    Scene_Title.prototype.callAnotherNewGame = function(command) {
        if (command.noFadeout) {
            this._noFadeout = true;
        }
        if (!command.fileLoad) {
            const preMapId  = $dataSystem.startMapId;
            const preStartX = $dataSystem.startX;
            const preStartY = $dataSystem.startY;
            const newMapId  = command.mapId;
            if (newMapId > 0) {
                $dataSystem.startMapId = newMapId;
                $dataSystem.startX     = command.mapX || 1;
                $dataSystem.startY     = command.mapY || 1;
            }
            this.commandNewGame();
            $dataSystem.startMapId = preMapId;
            $dataSystem.startX     = preStartX;
            $dataSystem.startY     = preStartY;
            const switchId         = command.switchId;
            if (switchId > 0) {
                $gameSwitches.setValue(switchId, true);
            }
        } else {
            this.commandContinue();
            localExtraStageIndex = ANGSettingManager.findList(false).indexOf(command);
        }
        this._callAnotherNewGame = true;
    };

    const _Scene_Title_createCommandWindow    = Scene_Title.prototype.createCommandWindow;
    Scene_Title.prototype.createCommandWindow = function() {
        _Scene_Title_createCommandWindow.call(this);

        ANGSettingManager.findList(false).forEach((command, index) => {
            this._commandWindow.setHandler('nameGame2_' + index,
                this.commandNewGameSecond.bind(this, index));
        });

        this._commandWindow.setHandler('shutdown', this.commandShutdown.bind(this));
    };

    Scene_Title.prototype.commandShutdown = function() {
        this._commandWindow.close();
        this.fadeOutAll();
        SceneManager.goto(Scene_Terminate);
    };

    Scene_Title.prototype.fadeOutAll = function() {
        if (!this._noFadeout) {
            Scene_Base.prototype.fadeOutAll.apply(this, arguments);
        }
    };

    const _Scene_Load_onLoadSuccess    = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
        _Scene_Load_onLoadSuccess.call(this);
        if (localExtraStageIndex >= 0) {
            const command = parameters.anotherDataList[localExtraStageIndex];
            const mapId   = command.mapId;
            if (mapId > 0) {
                const x = command.mapX || 1;
                const y = command.mapY || 1;
                $gamePlayer.reserveTransfer(mapId, x, y);
            }
            $gameMap.abortInterpreter();
            DataManager.selectSavefileForNewGame();
            const switchId = command.switchId;
            if (switchId > 0) {
                $gameSwitches.setValue(switchId, true);
            }
        }
    };

   const _Window_TitleCommand_makeCommandList    = Window_TitleCommand.prototype.makeCommandList;
    Window_TitleCommand.prototype.makeCommandList = function() {
        _Window_TitleCommand_makeCommandList.call(this);

        ANGSettingManager.findList(false).forEach((command, index) => {
            this.makeAnotherNewGameCommand(command, index);
        });

        if (ANGSettingManager.newGameHidden) {
            this.eraseCommandNewGame();
        }

        const hasShutdown = this._list.some(c => c.symbol === 'shutdown');
        if (!hasShutdown) {
            this.addCommand(shutdownText, 'shutdown');
        }

        this.height = this.fittingHeight(this._list.length);
        this.createContents();
        this.updatePlacement();
    };

    Window_TitleCommand.prototype.makeAnotherNewGameCommand = function(command, index) {
        this.addCommand(command.name, 'nameGame2_' + index, ANGSettingManager.isEnable(index));
        const addPosition = command.addPosition;
        if (addPosition > 0) {
            const anotherCommand = this._list.pop();
            this._list.splice(addPosition - 1, 0, anotherCommand);
        }
    };

    Window_TitleCommand.prototype.eraseCommandNewGame = function() {
        this._list = this._list.filter(function(command) {
            return command.symbol !== 'newGame';
        });
    };

    Window_TitleCommand.prototype.updatePlacement = function() {
        const addSize = this._list.length - 3;
        if (addSize > 0) {
            this.y -= addSize * this.itemHeight() / 2;
        }
    };

    const _Window_TitleCommand_processOk = Window_TitleCommand.prototype.processOk;
    Window_TitleCommand.prototype.processOk = function() {
        const symbol = this.currentSymbol();
        const cfg = CommandImageSettings[symbol];

        if (cfg && cfg.okScript !== undefined && cfg.okScript !== null) {
            try {
                let scriptText = String(cfg.okScript);
                scriptText = scriptText.trim();
                if (scriptText !== "" && scriptText !== '""' && scriptText !== "''") {
                    eval(scriptText);
                }
            } catch (e) {
                console.error("TitleCommand okScript error:", e);
            }
        }

        _Window_TitleCommand_processOk.call(this);
    };

    const _Window_TitleCommand_playOkSound = Window_TitleCommand.prototype.playOkSound;
    Window_TitleCommand.prototype.playOkSound = function() {
        if (NewGameSeSetting && NewGameSeSetting.name && this.currentSymbol() === 'newGame') {
            const se = {
                name:   String(NewGameSeSetting.name),
                volume: Number(NewGameSeSetting.volume ?? 90),
                pitch:  Number(NewGameSeSetting.pitch  ?? 100),
                pan:    Number(NewGameSeSetting.pan    ?? 0)
            };
            AudioManager.playStaticSe(se);
        } else {
            _Window_TitleCommand_playOkSound.apply(this, arguments);
        }
    };

    function ANGSettingManager() {
        throw new Error('This is a static class');
    }

    ANGSettingManager._fileName = 'anotherNewGame';

    ANGSettingManager._visibleList  = [];
    ANGSettingManager._enableList   = [];
    ANGSettingManager.newGameHidden = false;
    ANGSettingManager._defaultCursorIndex = parameters.defaultCursorIndex;

    ANGSettingManager.make = function() {
        const info         = {};
        info.visibleList   = this._visibleList;
        info.enableList    = this._enableList;
        info.newGameHidden = this.newGameHidden;
        info.cursorIndex   = this._defaultCursorIndex;
        return info;
    };

    ANGSettingManager.isVisible = function(index) {
        if (this._visibleList[index] !== undefined && this._visibleList[index] !== null) {
            return this._visibleList[index];
        } else {
            return !parameters.anotherDataList[index].hidden;
        }
    };

    ANGSettingManager.setVisible = function(index, value) {
        this._visibleList[index] = value;
    };

    ANGSettingManager.isEnable = function(index) {
        if (this._enableList[index] !== undefined && this._enableList[index] !== null) {
            return this._enableList[index];
        } else {
            return !parameters.anotherDataList[index].disable;
        }
    };

    ANGSettingManager.findList = function (waitFlag) {
        return parameters.anotherDataList.filter((command, index) => {
            return this.isVisible(index) && (command.demoWaitFrame > 0) === waitFlag;
        });
    };

    ANGSettingManager.setEnable = function(index, value) {
        this._enableList[index] = value;
    };

    ANGSettingManager.setDefaultCursorIndex = function(index) {
        this._defaultCursorIndex = index;
    };

    ANGSettingManager.loadData = function() {
        StorageManager.loadObject(this._fileName).then(info => {
            this._visibleList  = info.visibleList || [];
            this._enableList   = info.enableList || [];
            this.newGameHidden = !!info.newGameHidden;
            this._defaultCursorIndex = info.cursorIndex || parameters.defaultCursorIndex;
            this._loaded       = true;
        }).catch(() => {
            this._loaded = true;
        });
    };

    ANGSettingManager.isLoaded = function() {
        return this._loaded;
    };

    ANGSettingManager.save = function() {
        StorageManager.saveObject(this._fileName, ANGSettingManager.make());
    };

    var Imported = Imported || {};
    Imported.MOG_TitleCommands = true;
    var Moghunter = Moghunter || {};

    Moghunter.title_sideInput     = String(parameters.LeftRightInput || "true");
    Moghunter.title_slideXaxis    = Number(parameters.SlideX || 0);
    Moghunter.title_slideYaxis    = Number(parameters.SlideY || 0);

    Moghunter.title_com_pos = [
        "180,440",
        "350,440",
        "520,440",
        "690,440",
        "345,498",
        "345,530",
        "0,192",
        "0,224",
        "0,256",
        "0,288"
    ];

    var _mog_titleCom_create = Scene_Title.prototype.create;
    Scene_Title.prototype.create = function() {
        _mog_titleCom_create.call(this);
        if (this._titleField3) {
            this._titleField3.children.sort((a, b) => a.z - b.z);
        }
    };

    var _mog_titleCom_createCommandWindow = Scene_Title.prototype.createCommandWindow;
    Scene_Title.prototype.createCommandWindow = function() {
        _mog_titleCom_createCommandWindow.call(this);
        if (!this._titleField3) {
            this.createTitleField3();
        }
        this.createTitlePictureCommands();
        this._sideInput = String(Moghunter.title_sideInput) === "true";
    };

    var _mog_titleCom_update = Scene_Title.prototype.update;
    Scene_Title.prototype.update = function() {
        _mog_titleCom_update.call(this);
        this.updatePicCommands();
    };

    Scene_Title.prototype.createTitleField3 = function() {
        this._titleField3 = new Sprite();
        this._titleField3.z = 200;
        this.addChild(this._titleField3);
    };

    Scene_Title.prototype.createTitlePictureCommands = function() {
        this._picComE = false;
        this._TpictureCom = [];
        this._tComTouch = [TouchInput.x, TouchInput.y];
        this._picComIndex = this._commandWindow._index;
        for (let i = 0; i < this._commandWindow._list.length; i++) {
            this._TpictureCom[i] = new TpictureCom(this._commandWindow, i);
            this._TpictureCom[i].z = 300;
            this._titleField3.addChild(this._TpictureCom[i]);
        }
        this._commandWindow.x = -(Graphics.width * 2);
    };

    Scene_Title.prototype.checkTPicCom = function() {
        for (let i = 0; i < this._TpictureCom.length; i++) {
            if (this._TpictureCom[i].isOnPicCom()) {
                this._commandWindow._index = i;
                if (this._picComIndex === this._commandWindow._index) {
                    this._commandWindow.processOk();
                } else {
                    this._commandWindow.playCursorSound();
                }
                this._picComIndex = this._commandWindow._index;
            }
        }
    };

    Scene_Title.prototype.picComNeedCheckTouch = function() {
        if (this._tComTouch[0] !== TouchInput.x) return true;
        if (this._tComTouch[1] !== TouchInput.y) return true;
        return false;
    };

    Scene_Title.prototype.updateTitleTouchInputCom = function() {
        if (TouchInput.isTriggered()) {
            this.checkTPicCom();
        }
        if (this.picComNeedCheckTouch()) {
            this.updateTComMouseIsOnPic();
        }
        this._tComTouch = [TouchInput.x, TouchInput.y];
    };

    Scene_Title.prototype.updateComSideInput = function() {
        if (Input.isRepeated('right')) {
            this.addTitleComIndex(1);
        } else if (Input.isRepeated('left')) {
            this.addTitleComIndex(-1);
        }
    };

    Scene_Title.prototype.updateTComMouseIsOnPic = function() {
        for (let i = 0; i < this._TpictureCom.length; i++) {
            if (this._TpictureCom[i].isOnPicCom()) {
                this._commandWindow._index = i;
                if (this._picComIndex !== this._commandWindow._index) {
                    this._commandWindow.playCursorSound();
                }
                this._picComIndex = this._commandWindow._index;
            }
        }
    };

    Scene_Title.prototype.addTitleComIndex = function(value) {
        SoundManager.playCursor();
        const maxIndex = this._commandWindow._list.length - 1;
        this._commandWindow._index += value;
        if (this._commandWindow._index < 0) {
            this._commandWindow._index = maxIndex;
        } else if (this._commandWindow._index > maxIndex) {
            this._commandWindow._index = 0;
        }
    };

    Scene_Title.prototype.updatePicCommands = function() {
        if (!this._picComE) {
            this.updateTitleTouchInputCom();
            if (this._sideInput) {
                this.updateComSideInput();
            }
        }
        if (!this._picComE && this._commandWindow.isClosing()) {
            this._picComE = true;
        }
    };

    function TpictureCom() {
        this.initialize.apply(this, arguments);
    }

    TpictureCom.prototype = Object.create(Sprite.prototype);
    TpictureCom.prototype.constructor = TpictureCom;

    TpictureCom.prototype.initialize = function(data, index) {
        Sprite.prototype.initialize.call(this);
        this._index = index;     
        this._data  = data;       
        this._index2 = this._data._index; 
        this._wait = 5 * index;   
        this.opacity = 0;
        this._orgXY = this.set_tcp(Moghunter.title_com_pos[index]);
        this._imageSetting = null;
        this.scale.x = 1.0;
        this.scale.y = 1.0;
        this.prepareBitmap();
    };

    TpictureCom.prototype.isEnabled = function() {
        if (this._data && typeof this._data.isCommandEnabled === "function") {
            return this._data.isCommandEnabled(this._index);
        }
        return true;
    };

    TpictureCom.prototype.prepareBitmap = function() {
        const cmd    = this._data._list[this._index];
        const symbol = cmd ? String(cmd.symbol || "") : "";
        const cfg    = CommandImageSettings[symbol];
        this._imageSetting = cfg || null;

        let name = "";
        if (this._imageSetting && this._imageSetting.filename) {
            name = String(this._imageSetting.filename);
        } else {
            name = "Command_" + String(this._index);
        }
        this.bitmap = ImageManager.loadTitle2(name);
    };

    TpictureCom.prototype.set_tcp = function(value) {
        if (!value) return [0, 0];
        const s = value.split(',');
        if (!s[0] || !s[1]) return [0, 0];
        return [Number(s[0]), Number(s[1])];
    };

    TpictureCom.prototype.getData = function() {
        this.anchor.x = 0.5;
        this.anchor.y = 0.5;
        this._cw = this.bitmap.width;
        this._ch = this.bitmap.height / 2; 

        const hasCustomPos =
            this._imageSetting &&
            this._imageSetting.posX !== undefined &&
            this._imageSetting.posY !== undefined;

        if (hasCustomPos) {
            this._orgXY[0] = Number(this._imageSetting.posX || 0);
            this._orgXY[1] = Number(this._imageSetting.posY || 0);
        } else {
            const fx = (Graphics.width  - 816) / 2;
            const fy = (Graphics.height - 624) / 2;
            this._orgXY[0] += (this._cw / 2) + fx;
            this._orgXY[1] += fy;
        }

        this._orgXY[0] += Moghunter.title_slideXaxis;
        this._orgXY[1] += Moghunter.title_slideYaxis;

        this.x = this._orgXY[0];
        this.y = this._orgXY[1];

        this._pw1 = this.x - (this._cw / 2);
        this._pw2 = this.x + (this._cw / 2);
        this._ph1 = this.y - (this._ch / 2);
        this._ph2 = this.y + (this._ch / 2);

        this.refreshFrameAndTone();
    };

    TpictureCom.prototype.refreshFrameAndTone = function() {
        const enabled  = this.isEnabled();
        const selected = (this._index === this._data._index);

        const rectY = selected ? 0 : this._ch;
        this.setFrame(0, rectY, this._cw, this._ch);

        if (!this._imageSetting) {
            this.setColorTone([0, 0, 0, 0]);
            return;
        }

        if (enabled) {
            this.setColorTone([0, 0, 0, 0]);
        } else {
            const r    = Number(this._imageSetting.toneR    || 0);
            const g    = Number(this._imageSetting.toneG    || 0);
            const b    = Number(this._imageSetting.toneB    || 0);
            const gray = Number(this._imageSetting.toneGray || 0);
            this.setColorTone([r, g, b, gray]);
        }
    };

    TpictureCom.prototype.isOnPicCom = function() {
        if (TouchInput.x < this._pw1) return false;
        if (TouchInput.x > this._pw2) return false;
        if (TouchInput.y < this._ph1) return false;
        if (TouchInput.y > this._ph2) return false;
        return true;
    };

    TpictureCom.prototype.updatePicCommand = function() {
        if (this._wait > 0) {
            this._wait--;
            return;
        }

        this.refreshFrameAndTone();

        if (this.opacity < 255) {
            this.opacity += 15;
        }
    };

    TpictureCom.prototype.update = function() {
        Sprite.prototype.update.call(this);
        if (!this._cw) {
            if (this.bitmap && this.bitmap.isReady()) {
                this.getData();
            }
        } else {
            this.updatePicCommand();
        }
    };

    const _Scene_GameEnd_createCommandWindow = Scene_GameEnd.prototype.createCommandWindow;
    Scene_GameEnd.prototype.createCommandWindow = function() {
        _Scene_GameEnd_createCommandWindow.call(this);
        if (useGameEnd) {
            this._commandWindow.setHandler('shutdown', this.commandShutdown.bind(this));
        }
    };

    Scene_GameEnd.prototype.commandShutdown = function() {
        this._commandWindow.close();
        this.fadeOutAll();
        SceneManager.goto(Scene_Terminate);
    };

    const _Scene_GameEnd_commandWindowRect = Scene_GameEnd.prototype.commandWindowRect;
    Scene_GameEnd.prototype.commandWindowRect = function() {
        const rect = _Scene_GameEnd_commandWindowRect.call(this);
        if (useGameEnd) {
            rect.height = this.calcWindowHeight(3, true);
        }
        return rect;
    };

    const _Window_GameEnd_makeCommandList = Window_GameEnd.prototype.makeCommandList;
    Window_GameEnd.prototype.makeCommandList = function() {
        _Window_GameEnd_makeCommandList.call(this);
        if (useGameEnd) {
            this.addCommand(shutdownText, 'shutdown');
        }
    };

    function Scene_Terminate() {
        this.initialize.apply(this, arguments);
    }
    Scene_Terminate.prototype = Object.create(Scene_Base.prototype);
    Scene_Terminate.prototype.constructor = Scene_Terminate;
    Scene_Terminate.prototype.initialize = function() {
        Scene_Base.prototype.initialize.call(this);
    };
    Scene_Terminate.prototype.start = function() {
        Scene_Base.prototype.start.call(this);
        SceneManager.terminate();
    };

})();
