/*:
 * 
 * @param FolderBlockRules
 * @text フォルダ再生禁止ルール
 * @type struct<FolderBlockRule>[]
 * @default []
 * @desc 条件スイッチがONの間、ここで指定した \m[フォルダ] の再生をスキップします。
 *
 */
/*~struct~FolderBlockRule:
 * @param switchId
 * @text スイッチID
 * @type switch
 * @default 0
 *
 * @param folders
 * @text 禁止フォルダ一覧（\m[xxx]）
 * @type string[]
 * @default []
 * @desc 例: 001, 002, voice/001 など。完全一致で判定します。
*/
(() => {
"use strict";

const pluginName = "AudioVoice_MZ";
const AudioVoiceParams = PluginManager.parameters(pluginName);
function parseBlockRules() {
    const raw = AudioVoiceParams["FolderBlockRules"] || "[]";
    let list = [];
    try { list = JSON.parse(raw); } catch (e) { list = []; }
    const rules = [];
    for (const it of list) {
        try {
            const obj = typeof it === "string" ? JSON.parse(it) : it;
            const switchId = Number(obj.switchId || 0);
            let folders = obj.folders;
            if (typeof folders === "string") {
                try { folders = JSON.parse(folders); } catch(_) { folders = [folders]; }
            }
            folders = Array.isArray(folders) ? folders.map(s => String(s || "").trim()) : [];
            rules.push({ switchId, folders });
        } catch(_) {}
    }
    return rules;
}
AudioVoice._blockRules = parseBlockRules();

AudioVoice.isFolderBlocked = function(folder) {
    const f = String(folder || "");
    for (const r of this._blockRules) {
        if (r.switchId > 0 && $gameSwitches && $gameSwitches.value(r.switchId)) {
            if (r.folders && r.folders.includes(f)) {
                if (window.KZ_DEBUG_VOICE) console.warn("[AudioVoice] blocked by switch", r.switchId, "folder:", f);
                return true;
            }
        }
    }
    return false;
};


function AudioVoice() {
    throw new Error('This is a static class');
}

AudioVoice._vcVolume       = 100;
AudioVoice._vcBuffers      = [];
AudioVoice._staticBuffers  = [];
AudioVoice._charaBuffers   = [];
AudioVoice._charaObj       = null;
AudioVoice._path           = 'audio/';
AudioVoice._Plugpath       = 'audio/voice/';
AudioVoice._blobUrl        = null; 


if (typeof ConfigManager !== 'undefined') {
    if (!Object.prototype.hasOwnProperty.call(ConfigManager, 'Voice')) ConfigManager.Voice = true;
    if (!Object.prototype.hasOwnProperty.call(ConfigManager, 'battleVoice')) ConfigManager.battleVoice = true;
}

Object.defineProperty(AudioVoice, 'vcVolume', {
    get: function() {
        return this._vcVolume;
    },
    set: function(value) {
        this._vcVolume = value;
    },
    configurable: true
});

AudioVoice.playVc = function(vc, folder) {
    if (typeof ConfigManager !== 'undefined' && ConfigManager.hasOwnProperty('Voice') && !ConfigManager.Voice) return;
    if (folder && AudioVoice.isFolderBlocked(folder)) return;
    if (vc && vc.name) {
        this._vcBuffers = this._vcBuffers.filter(a => a && a.isPlaying());
        const buffer = this.createBuffer(folder, vc.name);
        this.updateVcParameters(buffer, vc);
        buffer.play(false);
        this._vcBuffers.push(buffer);
    }
};

AudioVoice.playbattleVc = function(vc, chara) {
    if (typeof ConfigManager !== 'undefined' && ConfigManager.hasOwnProperty('battleVoice') && !ConfigManager.battleVoice) return;
    this._charaObj = chara;
    if (vc && vc.name) {
        this._vcBuffers = this._vcBuffers.filter(a => a && a.isPlaying());
        let buffer;
        if (chara && chara.isActor && chara.isActor()) {
            const actorid = chara.actorId();
            buffer = this.createBuffer('voice/actor0' + String(actorid), vc.name);
        } else if (chara && chara.enemyId) {
            const enemyid = chara.enemyId();
            buffer = this.createBuffer('voice/enemy0' + String(enemyid), vc.name);
        }
        if (buffer) {
            this.updateVcParameters(buffer, vc);
            buffer.play(false);
            this._vcBuffers.push(buffer);
        }
    }
};

AudioVoice.explaybattleVc = function(vc, chara, addstr) {
    if (typeof ConfigManager !== 'undefined' && ConfigManager.hasOwnProperty('battleVoice') && !ConfigManager.battleVoice) return;
    this._charaObj = chara;
    if (vc && vc.name) {
        this._vcBuffers = this._vcBuffers.filter(a => a && a.isPlaying());
        let buffer;
        if (chara && chara.isActor && chara.isActor()) {
            const actorid = chara.actorId();
            buffer = this.createBuffer('voice/actor0' + String(actorid), vc.name + addstr);
        } else if (chara && chara.enemyId) {
            const enemyid = chara.enemyId();
            buffer = this.createBuffer('voice/enemy0' + String(enemyid), vc.name + addstr);
        }
        if (buffer) {
            this.updateVcParameters(buffer, vc);
            buffer.play(false);
            this._vcBuffers.push(buffer);
        }
    }
};

AudioVoice.updateVcParameters = function(buffer, vc) {
    let cfg = this._vcVolume || 100;
    if (typeof AudioManager !== "undefined" &&
        typeof AudioManager.getConfigVolume_Vsc === "function") {
        cfg = Math.round(cfg * AudioManager.getConfigVolume_Vsc() / 100);
    } else if (typeof AudioManager !== "undefined" &&
               typeof AudioManager.masterVolume === "number") {
        cfg = Math.round(cfg * AudioManager.masterVolume / 100);
    }

    this.updateBufferParameters(buffer, cfg, vc);
};


AudioVoice.stopVc = function() {
    this._vcBuffers.forEach(buffer => buffer && buffer.stop());
    this._vcBuffers = [];
};

AudioVoice.playStaticVc = function(vc) {
    if (vc && vc.name) {
        this.loadStaticVc(vc);
        for (let i = 0; i < this._staticBuffers.length; i++) {
            const buffer = this._staticBuffers[i];
            if (buffer._reservedVcName === vc.name) {
                buffer.stop();
                this.updateVcParameters(buffer, vc);
                buffer.play(false);
                break;
            }
        }
    }
};

AudioVoice.loadStaticVc = function(vc) {
    if (vc.name && !this.isStaticVc(vc)) {
        const buffer = this.createBuffer('voice', vc.name);
        buffer._reservedVcName = vc.name;
        this._staticBuffers.push(buffer);
    }
};

AudioVoice.isStaticVc = function(vc) {
    return this._staticBuffers.some(buffer => buffer._reservedVcName === vc.name);
};

AudioVoice.makeEmptyAudioObject = function() {
    return { name: '', volume: 0, pitch: 0, pan: 0 };
};

AudioVoice.createBuffer = function(folder, name) {
    const lc = (name || "").toLowerCase();
    const known = lc.endsWith(".ogg") || lc.endsWith(".m4a") || lc.endsWith(".mp3") || lc.endsWith(".wav");
    const ext = known ? "" : this.audioFileExt();
    const url = this._path + folder + '/' + encodeURIComponent(name) + ext;
    if (window.KZ_DEBUG_VOICE) console.log("[AudioVoice] load:", url);
    return new WebAudio(url);
};

AudioVoice.updateBufferParameters = function(buffer, configVolume, audio) {
    if (buffer && audio !== undefined) {
        const vol = (audio.volume == null ? 100 : audio.volume);
        const pitch = (audio.pitch == null ? 100 : audio.pitch);
        const pan = (audio.pan == null ? 0 : audio.pan);
        const cfg = (configVolume == null ? 100 : configVolume);
        buffer.volume = (cfg * vol) / 10000; // 0.0 - 1.0
        const gv = (typeof ConfigManager !== 'undefined' && typeof ConfigManager.voiceVolume === 'number') ? ConfigManager.voiceVolume : 100;
        buffer.volume *= Math.max(0, Math.min(1, gv / 100));
        buffer.pitch  = pitch / 100;
        buffer.pan    = pan / 100;
    }
};

AudioVoice.audioFileExt = function() {
    return AudioManager.audioFileExt();
};

AudioVoice.checkErrors = function() {
    this._vcBuffers.forEach(buffer => this.checkWebAudioError(buffer));
    this._staticBuffers.forEach(buffer => this.checkWebAudioError(buffer));
};

AudioVoice.checkWebAudioError = function(webAudio) {
    if (webAudio && webAudio.isError && webAudio.isError()) {
        throw new Error('Failed to load: ' + (webAudio._url || ""));
    }
};

window.AudioVoice = AudioVoice;
})();