/*:
 * @param EnableReplay
 * @text もう一度再生を有効にする
 * @type boolean
 * @default true
 *
 * @param TriggerKey
 * @text トリガーキー
 * @type select
 * @option OK(決定) @value ok
 * @option キャンセル @value cancel
 * @option Shift @value shift
 * @option Control @value control
 * @option Tab @value tab
 * @option PageUp @value pageup
 * @option PageDown @value pagedown
 * @default ok
 *
 * @param UseIcon
 * @text アイコン表示（クリック無効）
 * @type boolean
 * @default true
 *
 * @param IconIndex
 * @text アイコン番号
 * @type number
 * @min 0
 * @default 86
 *
 * @param PictureName
 * @text ピクチャ表示（クリック無効）
 * @type file
 * @dir img/pictures
 * @default
 *
 * @param PositionMode
 * @text 座標基準
 * @type select
 * @option メッセージウィンドウ基準 @value window
 * @option 画面基準 @value screen
 * @option 名前の後ろ @value afterName
 * @default window
 *
 * @param OffsetX
 * @text 表示X
 * @type number
 * @default 0
 *
 * @param OffsetY
 * @text 表示Y
 * @type number
 * @default -48
 *
 * @param NoNameX
 * @text 「名前なし」時X
 * @desc PositionMode=afterName のとき、namebox に名前がない場合のX座標（画面基準）
 * @type number
 * @default 0
 *
 * @param NoNameY
 * @text 「名前なし」時Y
 * @desc PositionMode=afterName のとき、namebox に名前がない場合のY座標（画面基準）
 * @type number
 * @default -48
 *
 * @param ShowOnlyWhenVoice
 * @text 直近ボイスがある時だけ表示
 * @type boolean
 * @default true
 *
 * @param AlwaysOnTop
 * @text 最前面に表示
 * @type boolean
 * @default true
 *
 */

(() => {
"use strict";

const PN = "BasicMapVoiceReplayKeyOnly_MZ";
const P = PluginManager.parameters(PN);

const EnableReplay       = P["EnableReplay"] === "true";
const TriggerKey         = String(P["TriggerKey"] || "ok");
const UseIcon            = P["UseIcon"] === "true";
const IconIndex          = Number(P["IconIndex"] || 86);
const PictureName        = String(P["PictureName"] || "");
const PositionMode       = String(P["PositionMode"] || "window");
const OffsetX            = Number(P["OffsetX"] || 0);
const OffsetY            = Number(P["OffsetY"] || -48);
// 「名前なし」時の座標（未設定なら OffsetX/OffsetY を流用）
const NoNameX            = (P["NoNameX"] !== undefined && P["NoNameX"] !== "")
                         ? Number(P["NoNameX"])
                         : OffsetX;
const NoNameY            = (P["NoNameY"] !== undefined && P["NoNameY"] !== "")
                         ? Number(P["NoNameY"])
                         : OffsetY;
const ShowOnlyWhenVoice  = P["ShowOnlyWhenVoice"] === "true";
const AlwaysOnTop        = P["AlwaysOnTop"] === "true";

if (window.AudioVoice && !AudioVoice._kzReplayKeyPatched) {
    AudioVoice._kzReplayKeyPatched = true;

    const _AudioVoice_playVc = AudioVoice.playVc;
    AudioVoice._lastPlayed = null;

    AudioVoice.playVc = function(vc, folder) {
        if (this._lastPlayed && typeof this.stopVc === "function") {
            try {
                this.stopVc(this._lastPlayed.vc, this._lastPlayed.folder);
            } catch (e) {
                if (window.KZ_DEBUG_VOICE) {
                    console.warn("[AudioVoice] stopVc error", e);
                }
            }
        }
        _AudioVoice_playVc.call(this, vc, folder);
        this._lastPlayed = { vc, folder };
        if (window.KZ_DEBUG_VOICE) {
            console.log("[AudioVoice] lastPlayed =", this._lastPlayed);
        }
    };
}

class Sprite_VoiceReplayKeyOnly extends Sprite {
    initialize() {
        super.initialize();
        this._bitmapReady = false;
        this._isIcon = !!UseIcon;
        this.bitmap = new Bitmap(32, 32);
        this.createGfx();
        this.visible = false;
    }

    createGfx() {
        if (this._isIcon) {
            const sheet = ImageManager.loadSystem("IconSet");
            sheet.addLoadListener(() => {
                const w = ImageManager.iconWidth;
                const h = ImageManager.iconHeight;
                const sx = (IconIndex % 16) * w;
                const sy = Math.floor(IconIndex / 16) * h;
                this.bitmap = new Bitmap(w, h);
                this.bitmap.blt(sheet, sx, sy, w, h, 0, 0);
                this._bitmapReady = true;
            });
        } else if (PictureName) {
            const pic = ImageManager.loadPicture(PictureName);
            pic.addLoadListener(() => {
                this.bitmap = pic.clone();
                this._bitmapReady = true;
            });
        } else {
            this.bitmap = new Bitmap(32, 32);
            this.bitmap.drawCircle(16, 16, 14, "#ffffff");
            this.bitmap.drawCircle(16, 16, 10, "#00aaff");
            this._bitmapReady = true;
        }
    }

    canReplay() {
        if (!window.AudioVoice) return false;
        const lp = AudioVoice._lastPlayed;
        if (!lp || !lp.vc || !lp.folder) return false;
        return true;
    }

    replay() {
        const lp = AudioVoice._lastPlayed;
        if (!lp) return;
        if (window.KZ_DEBUG_VOICE) console.log("[Replay] (key only)", lp.folder, lp.vc);
        AudioVoice.playVc(lp.vc, lp.folder);
    }

    _shouldBeVisible() {
        if (!EnableReplay || !this._bitmapReady) return false;

        const scene = SceneManager._scene;
        const w = scene && scene._messageWindow;
        if (!w) return false;

        const overlayHidden   = !!window.KZ_MSGOVERLAY_HIDDEN;
        const windowHidden    = !w.visible || w.contentsOpacity === 0;
        const globalHidden    = overlayHidden || windowHidden;

        const inMessage =
            w.isOpen() &&
            !w.isClosing() &&
            w.visible &&
            w.openness > 0 &&
            w.contentsOpacity > 0;

        const haveVoice = this.canReplay();
        const hasZ      = !!w._kzHasVoiceTag;

        if (globalHidden) return false;
        if (!inMessage) return false;
        if (!hasZ) return false;
        if (ShowOnlyWhenVoice && !haveVoice) return false;

        return true;
    }

    update() {
        super.update();

        const wantVisible = this._shouldBeVisible();
        if (wantVisible !== this.visible) {
            this.visible = wantVisible;
        }
        if (!this.visible) return;

        const scene = SceneManager._scene;
        const w = scene && scene._messageWindow;

        if (PositionMode === "window" && w) {
            this.x = (w.x || 0) + OffsetX;
            this.y = (w.y || 0) + OffsetY;

        } else if (PositionMode === "afterName") {
            const nb   = w && w._nameBoxWindow;
            const name = nb && typeof nb._name === "string" ? nb._name : "";

            const hasNameBox =
                nb &&
                nb.visible &&
                nb.openness > 0 &&
                name.length > 0;

            if (hasNameBox) {
                const bw = this.bitmap ? this.bitmap.width : this.width || 0;
                const bh = this.bitmap ? this.bitmap.height : this.height || 0;
                let nx = nb.x + nb.width + OffsetX;
                let ny = nb.y + Math.max(0, (nb.height - bh) / 2) + OffsetY;
                const maxX = (Graphics && Graphics.width ? Graphics.width : 816) - bw;
                const maxY = (Graphics && Graphics.height ? Graphics.height : 624) - bh;
                this.x = Math.min(Math.max(0, nx), maxX);
                this.y = Math.min(Math.max(0, ny), maxY);
            } else {
                this.x = NoNameX;
                this.y = NoNameY;
            }

        } else {
            this.x = OffsetX;
            this.y = OffsetY;
        }
    }
}

const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
Scene_Map.prototype.createAllWindows = function() {
    _Scene_Map_createAllWindows.call(this);
    this._voiceReplaySprite = new Sprite_VoiceReplayKeyOnly();
    if (AlwaysOnTop) {
        this.addChild(this._voiceReplaySprite);
    } else {
        this._windowLayer.addChild(this._voiceReplaySprite);
    }
};

const _Scene_Map_update = Scene_Map.prototype.update;
Scene_Map.prototype.update = function() {
    _Scene_Map_update.call(this);
    const s = this._voiceReplaySprite;
    if (!s || !EnableReplay) return;

    if (Input.isTriggered(TriggerKey)) {
        if (s.visible && s.canReplay()) {
            s.replay();
        }
    }
};

const _Window_Message_processEscapeCharacter =
    Window_Message.prototype.processEscapeCharacter;
Window_Message.prototype.processEscapeCharacter = function(code, textState) {
    _Window_Message_processEscapeCharacter.call(this, code, textState);
    if (code === "Z") { 
        this._kzHasVoiceTag = true;
    }
};

const _Window_Message_startMessage = Window_Message.prototype.startMessage;
Window_Message.prototype.startMessage = function() {
    this._kzHasVoiceTag = false;
    _Window_Message_startMessage.call(this);
};

})();
