/*:
 * @param FolderPrefix
 * @text フォルダ接頭辞
 * @type string
 * @default voice/
 * @desc 再生時に前置するフォルダ。標準配置なら "voice/"。直下なら空文字。
 *
 * @param AutoVolumeset
 * @text 自動BGM音量調整
 * @type boolean
 * @default true
 *
 * @param Setbgmvolume
 * @text BGM一時音量
 * @type number
 * @min 0
 * @max 100
 * @default 20
 *
 * @param AutoVolumesetather
 * @text \zが無い行は音量を戻す
 * @type boolean
 * @default true
 *
 * @command SetFolder
 * @text SetFolder（フォルダ記憶）
 * @desc 以後の再生で使うフォルダ番号を記憶（3桁化）
 * @arg FolderNumber
 * @text フォルダ番号
 * @type number
 * @min 0
 * @default 0
 *
 * @command PlayByIndex
 * @text ボイス単発再生！
 * @desc MapVoice.json の voices[index] を再生（制御文字 \z[i] と同じルーチン）
 * @arg Index
 * @text index
 * @type number
 * @min 0
 * @default 0
 * @arg FolderNumber
 * @text フォルダ番号（任意）
 * @type number
 * @min 0
 * @default
 * @arg AdjustBgm
 * @text BGM一時的減衰
 * @type select
 * @option auto
 * @option true
 * @option false
 * @default auto
 *
 * @command StopVoice
 * @text StopVoice（ボイス停止）
 *
 * @command RestoreBgm
 * @text RestoreBgm（BGM復帰）
 *
 */

(() => {
"use strict";

const pluginName = "BasicMapVoice_MZ";
const params = PluginManager.parameters(pluginName);
const FolderPrefix       = String(params["FolderPrefix"] || "voice/");
const AutoVolumeset      = params["AutoVolumeset"] === "true";
const Setbgmvolume       = Number(params["Setbgmvolume"] || 20);
const AutoVolumesetather = params["AutoVolumesetather"] === "true";

const Volume_Back = { Volume: 0 };
const BMV_State   = { folder: "000" }; 

function to3(n){ return String(Number(n||0)).padZero(3); }
function fullFolder(folder3){
  let pre = FolderPrefix || "";
  if (pre && !pre.endsWith("/")) pre += "/";
  return pre + folder3;
}
function voices(){ return (window.$datamapvoice && $datamapvoice.voices) || []; }

function stopAllVoices(){
  if (window.AudioVoice && AudioVoice.stopVc) {
    try { AudioVoice.stopVc(); } catch(_){}
  }
}

function lowerBgmIfNeeded(){
  if (!AutoVolumeset) return;
  if (Volume_Back.Volume !== 0) return;
  Volume_Back.Volume = AudioManager.bgmVolume;
  AudioManager.bgmVolume = Setbgmvolume;
  AudioManager.updateBgmParameters(AudioManager._currentBgm);
}
function restoreBgmIfNeeded(force=false){
  if (Volume_Back.Volume === 0) return;
  if (force || true) {
    AudioManager.bgmVolume = Volume_Back.Volume;
    AudioManager.updateBgmParameters(AudioManager._currentBgm);
    Volume_Back.Volume = 0;
  }
}

function playByIndex(index, folder3){
  const list = voices();
  const obj  = list[Number(index)||0];
  if (!obj) return;
  if (!(window.AudioVoice && AudioVoice.playVc)) return;
  stopAllVoices();
  AudioVoice.playVc(obj, fullFolder(folder3));

  if (typeof BgvManager !== "undefined" && BgvManager.autoPauseOnVoice) {
    const buffers = AudioVoice._vcBuffers || [];
    const lastBuffer = buffers[buffers.length - 1];
    if (lastBuffer && typeof lastBuffer.addStopListener === "function") {
      BgvManager.autoPauseOnVoice();
      lastBuffer.addStopListener(function() {
        BgvManager.resume();
      });
    }
  }
}


const _Window_Message_processEscapeCharacter =
  Window_Message.prototype.processEscapeCharacter;

Window_Message.prototype.processEscapeCharacter = function(code, textState){
  switch (code) {
    case "M": { 
      stopAllVoices();
      const n = this.obtainEscapeParam(textState);
      const f3 = to3(n);
      this._vcFolder   = f3;
      BMV_State.folder = f3;
      return;
    }
    case "Z": { 
      const i = this.obtainEscapeParam(textState);
      lowerBgmIfNeeded();
      const f3 = this._vcFolder || BMV_State.folder || "000";
      playByIndex(i, f3);
      return;
    }
    default:
      _Window_Message_processEscapeCharacter.call(this, code, textState);
  }
};

const _Window_Base_convertEscapeCharacters =
  Window_Base.prototype.convertEscapeCharacters;
Window_Base.prototype.convertEscapeCharacters = function(text){
  const hasZ = /\\z\[(\d+)\]/i.test(text);
  const result = _Window_Base_convertEscapeCharacters.call(this, text);
  if (!hasZ && AutoVolumeset && AutoVolumesetather && Volume_Back.Volume !== 0) {
    restoreBgmIfNeeded(true);
  }
  return result;
};

const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
Window_Message.prototype.terminateMessage = function(){
  if (!$gameParty.inBattle()) stopAllVoices();
  if (AutoVolumeset && Volume_Back.Volume !== 0) restoreBgmIfNeeded(true);
  _Window_Message_terminateMessage.call(this);
};

PluginManager.registerCommand(pluginName, "SetFolder", args => {
  const n  = Number(args.FolderNumber || 0);
  const f3 = to3(n);
  stopAllVoices();
  BMV_State.folder = f3;
});

PluginManager.registerCommand(pluginName, "PlayByIndex", args => {
  const i  = Number(args.Index || 0);
  const fn = args.FolderNumber;
  const f3 = (fn === undefined || fn === "" || fn === null) ? (BMV_State.folder || "000") : to3(fn);
  const adjust = String(args.AdjustBgm || "auto").toLowerCase();
  const shouldLower = (adjust === "true") || (adjust === "auto" && AutoVolumeset);
  if (shouldLower) lowerBgmIfNeeded();
  playByIndex(i, f3);
});

PluginManager.registerCommand(pluginName, "StopVoice", _args => {
  stopAllVoices();
});

PluginManager.registerCommand(pluginName, "RestoreBgm", _args => {
  if (Volume_Back.Volume !== 0) restoreBgmIfNeeded(true);
});

if (typeof Scene_Title !== "undefined") {
  const _BMV_Scene_Title_start = Scene_Title.prototype.start;
  Scene_Title.prototype.start = function() {
    stopAllVoices();
    if (AutoVolumeset && Volume_Back.Volume !== 0) {
      restoreBgmIfNeeded(true);
    }
    _BMV_Scene_Title_start.call(this);
  };
}

if (typeof Scene_Options !== "undefined" && Scene_Options.prototype.commandToTitle) {
  const _BMV_Scene_Options_commandToTitle = Scene_Options.prototype.commandToTitle;
  Scene_Options.prototype.commandToTitle = function() {
    stopAllVoices();
    if (AutoVolumeset && Volume_Back.Volume !== 0) {
      restoreBgmIfNeeded(true);
    }
    _BMV_Scene_Options_commandToTitle.call(this);
  };
}

})();
