/*:
 * @plugindesc スクリーンショットギャラリー
 * @author DarkPlasma
 * @license MIT
 * @target MZ
 * @url https://github.com/elleonard/DarkPlasma-MZ-Plugins/tree/release
 *
 * @param fileNameVariableId
 * @text ファイル名を保存する変数ID
 * @type variable
 * @default 0
 *
 * @param imageSettings
 * @text 画像設定リスト（変数39で小画像）
 * @type struct<ImageCondition>[]
 * @default []
 *
 * @param commonEventId
 * @text 撮影前コモンイベント
 * @type common_event
 * @default 0
 *
 * @param postScreenshotCommonEventId
 * @text 撮影後コモンイベント
 * @type common_event
 * @default 0
 *
 * @param switchId
 * @text 撮影許可スイッチID
 * @type switch
 * @default 0
 *
 * @param key
 * @text 撮影キー
 * @type select
 * @option control
 * @option tab
 * @default control
 *
 * @param tweetKey
 * @text ツイートキー（任意）
 * @type select
 * @option pageup
 * @option pagedown
 * @option shift
 * @option control
 * @option tab
 * @option
 * @default shift
 *
 * @param scenes
 * @text 撮影可能シーン
 * @type string[]
 * @default ["Scene_Base"]
 *
 * @param format
 * @text フォーマット
 * @type select
 * @option png
 * @option jpg
 * @default png
 *
 * @param se
 * @text 効果音
 * @type struct<Se>
 * @default {"name":"Switch2", "volume":"90", "pitch":"100", "pan":"0"}
 *
 * @param flash
 * @text フラッシュ
 * @type struct<Flash>
 * @default {"red":"255", "green":"255", "blue":"255", "power":"170", "duration":"30"}
 *
 * @param directory
 * @text 保存先フォルダ名
 * @type string
 * @default screenshot
 *
 * @param maxView
 * @text 表示最大数
 * @type number
 * @default 30
 *
 * @param preview
 * @text プレビュー設定
 * @type struct<Preview>
 * @default {"show":"true", "frameWidth":"4", "duration":"60", "rect":"{\"x\":\"16\", \"y\":\"16\", \"width\":\"102\", \"height\":\"78\"}"}
 *
 * @param watermarkSettings
 * @text ウォーターマーク設定
 * @type struct<Watermark>[]
 * @default []
 *
 * @param parallaxFileName
 * @text 背景パララックス
 * @desc ギャラリー背景に使う画像（img/parallaxes）。未指定ならSystem/Background。
 * @type file
 * @dir img/parallaxes
 * @default
 *
 * @param foregroundPicture
 * @text 前景ピクチャ
 * @type struct<Foreground>
 * @default {"fileName":"", "x":"0", "y":"0"}
 *
 * @command sceneScreenshot
 * @text スクショギャラリーを開く
 *
 * @help
 * version: 1.1.2+fg
 * ギャラリー画面にパララックス背景と前景ピクチャを重ねて表示します。
 * プレビュー中は前景を自動で非表示にし、終了時に復帰します。
 * ブラウザプレイ非対応。Scene_ScreenshotGallery で直接呼び出し可。
 */

/*~struct~Se:
 * @param name
 * @text SEファイル
 * @type file
 * @dir audio/se
 * @param volume
 * @text 音量
 * @type number
 * @default 90
 * @max 100
 * @param pitch
 * @text ピッチ
 * @type number
 * @default 100
 * @min 50
 * @max 150
 * @param pan
 * @text 位相
 * @type number
 * @default 0
 * @min -100
 * @max 100
 */

/*~struct~Flash:
 * @param red
 * @text 赤
 * @type number
 * @default 255
 * @max 255
 * @param green
 * @text 緑
 * @type number
 * @default 255
 * @max 255
 * @param blue
 * @text 青
 * @type number
 * @default 255
 * @max 255
 * @param power
 * @text 強さ
 * @type number
 * @default 170
 * @max 255
 * @param duration
 * @text 時間(フレーム)
 * @type number
 * @default 30
 * @min 1
 */

/*~struct~Preview:
 * @param show
 * @text プレビューを表示する
 * @type boolean
 * @default true
 * @param frameWidth
 * @text フレーム幅
 * @type number
 * @default 4
 * @param duration
 * @text 表示時間(フレーム)
 * @type number
 * @default 60
 * @param rect
 * @text 位置とサイズ
 * @type struct<Rectangle>
 * @default {"x":"16", "y":"16", "width":"102", "height":"78"}
 */

/*~struct~Rectangle:
 * @param x
 * @text X座標
 * @type number
 * @default 16
 * @param y
 * @text Y座標
 * @type number
 * @default 16
 * @param width
 * @text 幅
 * @type number
 * @default 102
 * @param height
 * @text 高さ
 * @type number
 * @default 78
 */

/*~struct~ImageCondition:
 * @param variableXValue
 * @text 変数39の値
 * @type string
 * @default ""
 * @param fileName
 * @text 画像ファイル名（pictures）
 * @type file
 * @dir img/pictures
 * @param x
 * @text X座標
 * @type number
 * @default 0
 * @param y
 * @text Y座標
 * @type number
 * @default 0
 */

/*~struct~Watermark:
 * @param fileName
 * @text ウォーターマーク（pictures）
 * @type file
 * @dir img/pictures
 * @param x
 * @text X座標
 * @type number
 * @default 0
 * @param y
 * @text Y座標
 * @type number
 * @default 0
 */

/*~struct~Foreground:
 * @param fileName
 * @text 前景ピクチャ（pictures）
 * @type file
 * @dir img/pictures
 * @default
 * @param x
 * @text X座標
 * @type number
 * @default 0
 * @param y
 * @text Y座標
 * @type number
 * @default 0
 */

(() => {
  'use strict';

  const pluginName = 'DarkPlasma_ScreenshotGallery';
  const pluginParameters = PluginManager.parameters(pluginName);

  const settings = {
    key: String(pluginParameters.key || `control`),
    tweetKey: String(pluginParameters.tweetKey || `shift`),
    scenes: JSON.parse(pluginParameters.scenes || '["Scene_Base"]').map((e) => String(e || ``)),
    format: String(pluginParameters.format || `png`),
    se: ((p) => { const o = JSON.parse(p);
      return { name: String(o.name||``), volume: Number(o.volume||90), pitch: Number(o.pitch||100), pan: Number(o.pan||0) };
    })(pluginParameters.se || '{"name":"Switch2","volume":"90","pitch":"100","pan":"0"}'),
    flash: ((p) => { const o = JSON.parse(p);
      return { red:Number(o.red||255), green:Number(o.green||255), blue:Number(o.blue||255), power:Number(o.power||170), duration:Number(o.duration||30)};
    })(pluginParameters.flash || '{"red":"255","green":"255","blue":"255","power":"170","duration":"30"}'),
    directory: String(pluginParameters.directory || `screenshot`),
    maxView: Number(pluginParameters.maxView || 30),
    preview: ((p)=>{ const o=JSON.parse(p);
      return { show: String(o.show||true)==='true', frameWidth:Number(o.frameWidth||4), duration:Number(o.duration||60),
        rect: ((q)=>{ const r=JSON.parse(q);
          return { x:Number(r.x||16), y:Number(r.y||16), width:Number(r.width||102), height:Number(r.height||78) };
        })(o.rect || '{"x":"16","y":"16","width":"102","height":"78"}')
      };
    })(pluginParameters.preview || '{"show":"true","frameWidth":"4","duration":"60","rect":"{\\"x\\":\\"16\\",\\"y\\":\\"16\\",\\"width\\":\\"102\\",\\"height\\":\\"78\\"}"}'),
    imageSettings: JSON.parse(pluginParameters.imageSettings || '[]').map(s=>{ const o=JSON.parse(s);
      return { variableXValue:String(o.variableXValue||''), fileName:String(o.fileName||''), x:Number(o.x||0), y:Number(o.y||0) };
    }),
    watermarkSettings: JSON.parse(pluginParameters.watermarkSettings || '[]').map(w=>{ const o=JSON.parse(w);
      return { fileName:String(o.fileName||''), x:Number(o.x||0), y:Number(o.y||0) };
    }),
    commonEventId: Number(pluginParameters.commonEventId || 0),
    switchId: Number(pluginParameters.switchId || 0),
    fileNameVariableId: Number(pluginParameters.fileNameVariableId || 0),
    postScreenshotCommonEventId: Number(pluginParameters.postScreenshotCommonEventId || 0),

    parallaxFileName: String(pluginParameters.parallaxFileName || ''),
    foreground: ((p)=>{
      try { const o = JSON.parse(p || '{}');
        return { fileName: String(o.fileName||''), x: Number(o.x||0), y: Number(o.y||0) };
      } catch(_) { return { fileName:'', x:0, y:0 }; }
    })(pluginParameters.foregroundPicture)
  };

  const command_sceneScreenshot = 'sceneScreenshot';

  // ===== SceneManager extensions =====
  function SceneManager_ScreenshotGalleryMixIn(sceneManager) {
    sceneManager.saveScreenshot = function (format, callback) {
      const dataURLFormat = format === 'jpg' ? 'image/jpeg' : `image/${format}`;
      const now = new Date();
      const name = `${now.getFullYear()}-${(now.getMonth()+1).toString().padStart(2,'0')}-${now.getDate().toString().padStart(2,'0')}-${now.getHours().toString().padStart(2,'0')}${now.getMinutes().toString().padStart(2,'0')}${now.getSeconds().toString().padStart(2,'0')}${now.getMilliseconds().toString().padStart(4,'0')}`;

      if (settings.fileNameVariableId > 0) $gameVariables.setValue(settings.fileNameVariableId, name);
      ImageManager.setLatestScreenshotName(name);

      const screenshotCanvas = this.snap().canvas;
      const context = screenshotCanvas.getContext('2d');

      const watermarkPromises = settings.watermarkSettings.map(wm => new Promise((resolve) => {
        if (wm.fileName) {
          const bmp = ImageManager.loadBitmap('img/pictures/', wm.fileName);
          if (bmp.isReady()) { context.drawImage(bmp.canvas, wm.x, wm.y); resolve(); }
          else bmp.addLoadListener(()=>{ context.drawImage(bmp.canvas, wm.x, wm.y); resolve(); });
        } else resolve();
      }));

      Promise.all(watermarkPromises).then(()=>{
        const data = screenshotCanvas.toDataURL(dataURLFormat, 1).replace(/^.*,/, '');
        this.saveImage(name, format, data);
        if (settings.postScreenshotCommonEventId > 0) $gameTemp.reserveCommonEvent(settings.postScreenshotCommonEventId);
        if (typeof callback === 'function') callback();
      }).catch(()=>{
        const data = screenshotCanvas.toDataURL(dataURLFormat, 1).replace(/^.*,/, '');
        this.saveImage(name, format, data);
        if (settings.postScreenshotCommonEventId > 0) $gameTemp.reserveCommonEvent(settings.postScreenshotCommonEventId);
        if (typeof callback === 'function') callback();
      });
    };

    sceneManager.saveImage = function (filename, format, base64Image) {
      const fs = require('fs');
      const path = require('path');
      const dirpath = StorageManager.screenshotDirPath();
      if (!fs.existsSync(dirpath)) fs.mkdirSync(dirpath, { recursive:true });
      fs.writeFileSync(path.join(dirpath, `${filename}.${format}`), Buffer.from(base64Image, 'base64'));
    };
  }
  SceneManager_ScreenshotGalleryMixIn(SceneManager);

  // ===== ImageManager extensions =====
  function ImageManager_ScreenshotGalleryMixIn(imageManager) {
    imageManager.setLatestScreenshotName = function (name) { this._latestScreenshotName = name; };
    imageManager.loadLatestScreenshot = function () { return this.loadScreenshot(this._latestScreenshotName); };
    imageManager.loadScreenshot = function (filename) { return this.loadBitmap(`${StorageManager.screenshotDirPath()}`, filename); };
    imageManager.loadAllScreenshot = function () {
      const fs = require('fs'); const path = require('path');
      const dirpath = StorageManager.screenshotDirPath();
      if (!fs.existsSync(dirpath)) fs.mkdirSync(dirpath, { recursive:true });
      const filenames = fs.readdirSync(dirpath, { withFileTypes:true })
        .filter(e=>e.isFile()).map(e=>e.name.replace(/\..+$/,'')).slice(0, settings.maxView);
      return filenames.map(fn=>this.loadScreenshot(fn)).reverse();
    };
    imageManager.validScreenshotCount = function () { return this.loadAllScreenshot().length; };
  }
  ImageManager_ScreenshotGalleryMixIn(ImageManager);

  // ===== StorageManager extensions =====
  function StorageManager_ScreenshotGalleryMixIn(storageManager) {
    const path = require('path');
    storageManager.screenshotDirPath = function () {
      return path.join(path.resolve(''), settings.directory) + path.sep;
    };
  }
  StorageManager_ScreenshotGalleryMixIn(StorageManager);

  PluginManager.registerCommand(pluginName, command_sceneScreenshot, function () {
    SceneManager.push(Scene_ScreenshotGallery);
  });

  // ===== Bitmap helper =====
  function Bitmap_ScreenshotGalleryMixIn(bitmap) {
    const _startLoading = bitmap._startLoading;
    bitmap._startLoading = function () {
      if (this._url && this._url.startsWith(`${StorageManager.screenshotDirPath()}`)) {
        this._startDecrypting = () => {
          this._image.src = this._url;
          if (this._image.width > 0) { this._image.onload = null; this._onLoad(); }
        };
      }
      _startLoading.call(this);
    };
    bitmap.drawFrame = function (x,y,w,h,thick,color) {
      this._context.strokeStyle = color;
      this._context.lineWidth = thick;
      this._context.strokeRect(x,y,w,h);
      this._context.restore();
      this._baseTexture.update();
    };
  }
  Bitmap_ScreenshotGalleryMixIn(Bitmap.prototype);

  // ===== inject capture hotkey to listed scenes =====
  function Scene_ScreenshotGalleryHotkeyMixIn(sceneClass) {
    const _start = sceneClass.start;
    sceneClass.start = function () {
      _start.call(this);
      this._isScreenshotPending = false;
      if (settings.preview.show) this.createPreviewContainer();
    };
    sceneClass.createPreviewContainer = function () {
      this._previewContainer = new Sprite();
      this._previewContainer.bitmap = new Bitmap(
        settings.preview.rect.width + settings.preview.frameWidth*2,
        settings.preview.rect.height + settings.preview.frameWidth*2
      );
      this._previewContainer.x = settings.preview.rect.x - settings.preview.frameWidth;
      this._previewContainer.y = settings.preview.rect.y - settings.preview.frameWidth;
      this._previewContainer.bitmap.drawFrame(
        0, 0,
        settings.preview.rect.width + settings.preview.frameWidth*2,
        settings.preview.rect.height + settings.preview.frameWidth*2,
        settings.preview.frameWidth, ColorManager.textColor(0)
      );
      this.addChild(this._previewContainer);
      this._previewSprite = new Sprite();
      this._previewSprite.x = settings.preview.frameWidth;
      this._previewSprite.y = settings.preview.frameWidth;
      this._previewSprite.scale.x = settings.preview.rect.width / Graphics.width;
      this._previewSprite.scale.y = settings.preview.rect.height / Graphics.height;
      this._previewContainer.addChild(this._previewSprite);
      this._previewContainer.hide();
    };
    sceneClass.startPreview = function () {
      this._previewDuration = settings.preview.duration;
      this._previewSprite.bitmap = ImageManager.loadLatestScreenshot();
      this._previewContainer.show();
    };
    sceneClass.hidePreview = function () { if (this._previewContainer.visible) this._previewContainer.hide(); };
    sceneClass.startFlash = function () { this._flashDuration = settings.flash.duration; this._flashOpacity = settings.flash.power; this.updateFlash(); };
    sceneClass.clearFlash  = function () { if (this._flashDuration>0){ this._flashDuration=0; this.updateColorFilter(); } };
    const _update = sceneClass.update;
    sceneClass.update = function () {
      _update.call(this);
      if (this._isScreenshotPending && !$gameMap.isEventRunning()) {
        this._isScreenshotPending = false;
        this.clearFlash(); this.hidePreview();
        SceneManager.saveScreenshot(settings.format, () => {
          if (settings.se.name) { AudioManager.playSe(settings.se); this.startFlash(); }
          if (settings.preview.show) this.startPreview();
        });
      }
      if (Input.isTriggered(settings.key)) {
        if (settings.switchId > 0 && !$gameSwitches.value(settings.switchId)) return;
        if (settings.commonEventId > 0) {
          $gameTemp.reserveCommonEvent(settings.commonEventId);
          this._isScreenshotPending = true;
        } else {
          this.clearFlash(); this.hidePreview();
          SceneManager.saveScreenshot(settings.format, () => {
            if (settings.se.name) { AudioManager.playSe(settings.se); this.startFlash(); }
            if (settings.preview.show) this.startPreview();
          });
        }
      }
      this.updateFlash();
      this.updatePreview();
    };
    sceneClass.updateColorFilter = function () { this._colorFilter.setBlendColor(this.blendColor()); };
    sceneClass.updateFlash = function () {
      if (this._flashDuration > 0) { this._flashOpacity *= (this._flashDuration-1)/this._flashDuration; this._flashDuration--; }
    };
    sceneClass.updatePreview = function () { if (this._previewDuration>0){ this._previewDuration--; if (this._previewDuration<=0) this._previewContainer.hide(); } };
    sceneClass.blendColor = function () { if (this._fadeDuration===0 && this._flashDuration>0) return [settings.flash.red,settings.flash.green,settings.flash.blue,this._flashOpacity];
      const c = this._fadeWhite ? 255 : 0; return [c,c,c,this._fadeOpacity]; };
  }
  settings.scenes.filter((scene)=>scene in globalThis).forEach((scene)=>{ Scene_ScreenshotGalleryHotkeyMixIn(globalThis[scene].prototype); });

  // ===== Gallery Scene =====
  class Scene_ScreenshotGallery extends Scene_Base {
    create() {
      Scene_Base.prototype.create.call(this);
      this.createBackground();        // 背景（パララックス対応）
      this.createConditionalImage();  // 背景の上に小画像（変数39）
      this.createWindowLayer();
      this.createGalleryWindow();     // サムネイル一覧
      this.createSprite();            // 拡大表示
      this.createForegroundPicture(); // 最前面
      this._galleryWindow.select(0);
    }

    // 変数39の値に対応した小画像
    createConditionalImage() {
      const variableX = String($gameVariables.value(39));
      const imageSetting = settings.imageSettings.find(s => s.variableXValue === variableX);
      if (imageSetting && imageSetting.fileName) {
        this._smallImage = new Sprite(ImageManager.loadBitmap("img/pictures/", imageSetting.fileName));
        this._smallImage.x = imageSetting.x;
        this._smallImage.y = imageSetting.y;
        this.addChild(this._smallImage);
      }
    }

    // 背景：パララックス指定があれば使用。未指定ならSystem/Background
    createBackground() {
      this._backgroundSprite = new Sprite();
      if (settings.parallaxFileName) {
        const bmp = ImageManager.loadParallax(settings.parallaxFileName);
        this._backgroundSprite.bitmap = bmp;
        this._backgroundSprite.anchor.x = 0;
        this._backgroundSprite.anchor.y = 0;
        const fit = () => {
          const sx = Graphics.width  / bmp.width;
          const sy = Graphics.height / bmp.height;
          this._backgroundSprite.scale.set(sx, sy);
        };
        if (bmp.isReady()) fit(); else bmp.addLoadListener(fit);
      } else {
        this._backgroundSprite.bitmap = ImageManager.loadSystem('Background');
      }
      this.addChild(this._backgroundSprite);
    }

    // 前景：最前面
    createForegroundPicture() {
      const fp = settings.foreground;
      if (fp && fp.fileName) {
        this._foregroundSprite = new Sprite(ImageManager.loadBitmap("img/pictures/", fp.fileName));
        this._foregroundSprite.x = fp.x || 0;
        this._foregroundSprite.y = fp.y || 0;
        this.addChild(this._foregroundSprite);
      }
    }

    createGalleryWindow() {
      this._galleryWindow = new Window_ScreenshotGallery(this.galleryWindowRect());
      this._galleryWindow.setHandler('ok', () => this.openLargeImage());
      this._galleryWindow.setHandler('cancel', () => this.popScene());
      this.addWindow(this._galleryWindow);
      this._galleryWindow.activate();
      this._galleryWindow.refresh();
    }

    galleryWindowRect() {
      return new Rectangle(240, 110, Graphics.boxWidth - 480, Graphics.boxHeight - 260);
    }

    createSprite() {
      this._sprite = new Sprite_Screenshot();
      this._sprite.hide();
      this._sprite.setOkHandler(() => this.closeLargeImage());
      this.addChild(this._sprite);
    }

    openLargeImage() {
      if (this._galleryWindow.index() < 0) {
        this._galleryWindow.activate();
      } else {
        this._sprite.bitmap = this._galleryWindow.currentItem();
        this._sprite.show();
        if (this._smallImage) this._smallImage.visible = false;
        this._galleryWindow.deactivate();
      }
    }

    closeLargeImage() {
      SoundManager.playCancel();
      this._sprite.hide();
      if (this._smallImage) this._smallImage.visible = true;
      this._galleryWindow.activate();
    }

    // ===== 前景のプレビュー連動（ここが追加／修正点） =====
    startPreview() {
      if (this._foregroundSprite) this._foregroundSprite.visible = false;
      if (Scene_Base.prototype.startPreview) {
        Scene_Base.prototype.startPreview.call(this);
      }
    }

    hidePreview() {
      if (Scene_Base.prototype.hidePreview) {
        Scene_Base.prototype.hidePreview.call(this);
      }
      if (this._foregroundSprite) this._foregroundSprite.visible = true;
    }

    updatePreview() {
      if (Scene_Base.prototype.updatePreview) {
        Scene_Base.prototype.updatePreview.call(this);
      }
      if (this._previewDuration <= 0 && this._previewContainer && !this._previewContainer.visible) {
        if (this._foregroundSprite) this._foregroundSprite.visible = true;
      }
    }
  }

  class Window_ScreenshotGallery extends Window_Selectable {
    constructor() { super(...arguments); this._images = []; }
    initialize(rect) { super.initialize(rect); }
    maxCols() { return 3; }
    maxItems() { return ImageManager.validScreenshotCount(); }
    itemHeight() { return Math.floor(Graphics.height / (Graphics.width / this.itemWidth())); }
    itemRectWithPadding(index) {
      const rect = super.itemRectWithPadding(index);
      const padding = this.itemPadding();
      rect.y += padding; rect.height -= padding * 2;
      return rect;
    }
    drawItem(index) {
      if (this._images[index]) {
        const rect = this.itemRectWithPadding(index);
        const bitmap = this._images[index];
        this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, rect.x, rect.y, rect.width, rect.height);
      }
    }
    refresh() { this.makeItemList(); super.refresh(); }
    makeItemList() {
      this._images = ImageManager.loadAllScreenshot();
      this._images.find((image)=>!image.isReady())?.addLoadListener(()=>this.refresh());
    }
    currentItem() { return this._images[this.index()]; }
  }

  class Sprite_Screenshot extends Sprite_Clickable {
    initialize() { super.initialize(); }
    update() { super.update(); this.processHandling(); }
    setOkHandler(handler) { this._okHandler = handler; }
    processHandling() {
      if (this.isClickEnabled() && this._okHandler && this.bitmap) {
        if (Input.isTriggered('ok') || Input.isTriggered('cancel')) {
          this._okHandler();
        } else if (settings.tweetKey && Input.isTriggered(settings.tweetKey)) {
          SceneManager.tweetImage?.(this.bitmap);
        } else {
          this.processTouch();
        }
      }
    }
    onClick() { if (this._okHandler) this._okHandler(); }
  }

  globalThis.Scene_ScreenshotGallery = Scene_ScreenshotGallery;
})();
