/*:
 * @plugindesc イベントテキストのログ保持と表示（拡張一体化版）
 * @author DarkPlasma
 * @license MIT
 * @target MZ
 *
 * @param disableLoggingSwitch @text ログ記録無効スイッチ @type switch @default 0
 * @param openLogKeys @text ログ開閉ボタン @type select[] @option shift @option control @option tab @option pageup @option pagedown @default ["tab"]
 * @param disableLogWindowSwitch @text ログウィンドウ無効スイッチ @type switch @default 0
 * @param lineSpacing @text ログの行間 @type number @default 0
 * @param messageSpacing @text メッセージ間隔 @type number @default 0
 * @param logSplitter @text ログ区切り線 @type string @default -------------------------------------------------------
 * @param autoSplit @text 自動区切り線 @type boolean @default true
 * @param choiceFormat @text 選択肢フォーマット @type string @default 選択肢:{choice}
 * @param choiceColor @text 選択肢色 @type number @default 17
 * @param choiceCancelText @text キャンセルログ @type string @default キャンセル
 * @param smoothBackFromLog @text テキスト再表示なし @type boolean @default true
 * @param backgroundImage @text 背景画像 @type file @dir img
 * @param showLogWindowFrame @text ウィンドウ枠表示 @type boolean @default true
 * @param escapeCharacterCodes @text 無視する制御文字 @type string[] @default []
 * @param scrollSpeed @text スクロール速さ @type number @default 1 @min 1
 * @param scrollSpeedHigh @text 高速スクロール速さ @type number @default 10 @min 1
 * @param maxLogMessages @text ログメッセージ保持数 @type number @default 200
 * @param logX @text ログX @type number @default 0
 * @param logY @text ログY @type number @default 0
 * @param logWidth @text ログ幅 @type number @default 0
 * @param logHeight @text ログ高さ @type number @default 0
 * @param voiceIconIndex @text ボイス行アイコン（IconSet） @type number @default 314
 * @param voiceIconOffsetX @text アイコンXオフセット @type number @default 6
 * @param voiceIconOffsetY @text アイコンYオフセット @type number @default 2
 *
 * @command showTextLog @text ログウィンドウを開く
 * @command insertLogSplitter @text ログに区切り線を追加する
 * @command insertTextLog @text ログに指定したテキストを追加する
 * @arg text @text テキスト @type string
 */
(() => {
  'use strict';

  const pluginName = 'DarkPlasma_TextLog';
  const params = PluginManager.parameters(pluginName);
  const parseJSON = (s, d) => { try { return JSON.parse(s); } catch (_) { return d; } };

  const settings = {
    disableLoggingSwitch: Number(params.disableLoggingSwitch || 0),
    openLogKeys: parseJSON(params.openLogKeys || '["tab"]', ["tab"]).map(String),
    disableLogWindowSwitch: Number(params.disableLogWindowSwitch || 0),
    lineSpacing: Number(params.lineSpacing || 0),
    messageSpacing: Number(params.messageSpacing || 0),
    logSplitter: String(params.logSplitter || `-------------------------------------------------------`),
    autoSplit: String(params.autoSplit || true) === 'true',
    choiceFormat: String(params.choiceFormat || `選択肢:{choice}`),
    choiceColor: Number(params.choiceColor || 17),
    choiceCancelText: String(params.choiceCancelText || `キャンセル`),
    smoothBackFromLog: String(params.smoothBackFromLog || true) === 'true',
    backgroundImage: String(params.backgroundImage || ``),
    showLogWindowFrame: String(params.showLogWindowFrame || true) === 'true',
    escapeCharacterCodes: parseJSON(params.escapeCharacterCodes || '[]', []).map(s => String(s || '').trim()).filter(Boolean),
    scrollSpeed: Number(params.scrollSpeed || 1),
    scrollSpeedHigh: Number(params.scrollSpeedHigh || 10),
    maxLogMessages: Number(params.maxLogMessages || 200),
    logX: Number(params.logX || 0),
    logY: Number(params.logY || 0),
    logWidth: Number(params.logWidth || 0),
    logHeight: Number(params.logHeight || 0),
    voiceIconIndex: Number(params.voiceIconIndex || 314),
    voiceIconOffsetX: Number(params.voiceIconOffsetX || 6),
    voiceIconOffsetY: Number(params.voiceIconOffsetY || 2),
  };

  const stripIgnoredEscapes = (text) => {
    if (!text) return '';
    let out = String(text);
    settings.escapeCharacterCodes.forEach(code => {
      if (!code) return;
      const c = code.replace(/[-/\\^$*+?.()|[\]{}]/g, '\\$&');
      const reBracket = new RegExp(`\\\\${c}\\[[^\\]]*\\]`, 'gi');
      const reBare    = new RegExp(`\\\\${c}(?![A-Za-z])`, 'gi');
      out = out.replace(reBracket, '');
      out = out.replace(reBare, '');
    });
    return out;
  };

  function Window_ObtainEscapeParamTextMixIn(windowClass) {
    windowClass.obtainEscapeParamText = function (textState) {
      const m = /\[(.+?)\]/.exec(textState.text.slice(textState.index));
      if (m) { textState.index += m[0].length; return m[1].split(','); }
      return [];
    };
  }

  PluginManager.registerCommand(pluginName, 'showTextLog', function () {
    if ($gameMessage && $gameMessage.isBusy()) {
      SceneManager.push(Scene_TextLog);
    }
  });
  PluginManager.registerCommand(pluginName, 'insertTextLog', function (args) {
    const text = String(args.text || ``);
    $gameTemp.eventTextLog().pushLog('', text, null);
  });
  PluginManager.registerCommand(pluginName, 'insertLogSplitter', function () {
    $gameTemp.eventTextLog().pushSplitter();
  });

  class EvacuatedMessageAndSubWindows {
    constructor(messageWindow, goldWindow, nameBoxWindow, choiceListWindow, numberInputWindow) {
      this._messageWindow = messageWindow;
      this._goldWindow = goldWindow;
      this._nameBoxWindow = nameBoxWindow;
      this._choiceListWindow = choiceListWindow;
      this._numberInputWindow = numberInputWindow;
    }
    get messageWindow() { return this._messageWindow; }
    get goldWindow() { return this._goldWindow; }
    get nameBoxWindow() { return this._nameBoxWindow; }
    get choiceListWindow() { return this._choiceListWindow; }
    get numberInputWindow() { return this._numberInputWindow; }
  }
  globalThis.EvacuatedMessageAndSubWindows = EvacuatedMessageAndSubWindows;

  (function mix_Game_Temp(proto) {
    const _initialize = proto.initialize;
    proto.initialize = function () {
      _initialize.call(this);
      this._evacuatedMessageAndSubWindows = null;
      this._eventTextLog = new Game_EventTextLog();
      this._callTextLogOnMap = false;
      this._lastMapVoice = null;
    };
    proto.evacuatedMessageAndSubWindows = function () {
      return this._evacuatedMessageAndSubWindows;
    };
    proto.setEvacuatedMessageAndSubWindows = function (w) {
      this._evacuatedMessageAndSubWindows = w;
    };
    proto.clearEvacuatedMessageAndSubWindows = function () {
      this._evacuatedMessageAndSubWindows = null;
    };
    proto.eventTextLog = function () {
      if (!this._eventTextLog) this._eventTextLog = new Game_EventTextLog();
      return this._eventTextLog;
    };
    proto.requestCallTextLogOnMap = function () {
      this._callTextLogOnMap = true;
    };
    proto.clearCallTextLogOnMapRequest = function () {
      this._callTextLogOnMap = false;
    };
    proto.isCallTextLogOnMapRequested = function () {
      return this._callTextLogOnMap;
    };
  })(Game_Temp.prototype);

  class Game_EventTextLog {
    constructor() {
      this._messages = [];
    }
    get messages() { return this._messages; }
    pushLog(speakerName, text, voiceMeta) {
      this._messages.push(new Game_LogMessage(speakerName, text, voiceMeta));
      if (settings.maxLogMessages < this._messages.length) {
        this._messages.splice(0, this._messages.length - settings.maxLogMessages);
      }
    }
    pushSplitter() {
      this.pushLog('', settings.logSplitter, null);
    }
    latestMessageIsLogSplitter() {
      return this._messages.length > 0 && this._messages[this._messages.length - 1].isLogSplitter();
    }
    findLastVoiceIndex() {
      for (let i = this._messages.length - 1; i >= 0; i--) {
        if (this._messages[i].hasVoice()) return i;
      }
      return -1;
    }
  }
  globalThis.Game_EventTextLog = Game_EventTextLog;

  class Game_LogMessage {
    constructor(speakerName, message, voiceMeta) {
      this._speakerName = speakerName;
      this._message = message;
      this._voiceMeta = voiceMeta;
    }
    get speakerName() { return this._speakerName; }
    get message() { return this._message; }
    get voiceMeta() { return this._voiceMeta; }
    hasVoice() {
      return !!(this._voiceMeta && (this._voiceMeta.index !== undefined));
    }
    text() {
      // 区切り線の場合は従来通りそのまま返す
      if (this.isLogSplitter()) {
        return this._message;
      }

      // 名前行は必ず1行目として確保する（空欄でも空行として残す）
      const name = this._speakerName != null ? String(this._speakerName) : '';
      const msg  = this._message != null ? String(this._message) : '';
      return name + '\n' + msg;
    }
    isLogSplitter() {
      return this.message === settings.logSplitter;
    }
  }
  globalThis.Game_LogMessage = Game_LogMessage;

  (function mix_Game_Message(proto) {
    const _clear = proto.clear;
    proto.clear = function () {
      _clear.call(this);
      this._chosenIndex = null;
    };
    const _onChoice = proto.onChoice;
    proto.onChoice = function (n) {
      this._chosenIndex = n;
      _onChoice.call(this, n);
    };
    proto.chosenIndex = function () {
      return this._chosenIndex || 0;
    };
    proto.chosenText = function () {
      if (this.choiceCancelType() < 0 && this.chosenIndex() < 0) {
        return settings.choiceCancelText;
      }
      return this.choices()[this.chosenIndex()];
    };
  })(Game_Message.prototype);

  (function mix_Game_Interpreter(proto) {
    const _terminate = proto.terminate;
    proto.terminate = function () {
      if (this.mustSplitLogOnTeminate()) {
        $gameTemp.eventTextLog().pushSplitter();
      }
      _terminate.call(this);
    };
    proto.mustSplitLogOnTeminate = function () {
      return (
        settings.autoSplit &&
        this._depth === 0 &&
        this._eventId > 0 &&
        !this.isOnParallelEvent() &&
        $gameTemp.eventTextLog().messages.length > 0 &&
        !$gameTemp.eventTextLog().latestMessageIsLogSplitter()
      );
    };
    proto.isOnParallelEvent = function () {
      return $gameMap.event(this._eventId)?.isTriggerIn([4]) && this.isOnCurrentMap();
    };
  })(Game_Interpreter.prototype);

  class Scene_TextLog extends Scene_Base {
    create() {
      super.create();
      this.createBackground();
      this.createWindowLayer();
      this.createTextLogWindow();
      this.createButtons();
    }
    createBackground() {
      this._backgroundSprite = new Sprite();
      this._backgroundSprite.bitmap = this.backgroundImage();
      this.addChild(this._backgroundSprite);
    }
    backgroundImage() {
      return settings.backgroundImage ? ImageManager.loadBitmap('img/', settings.backgroundImage) : SceneManager.backgroundBitmap();
    }
    createTextLogWindow() {
      this._textLogWindow = new Window_TextLog(this.textLogWindowRect());
      this._textLogWindow.setHandler('cancel', this.popScene.bind(this));
      if (!settings.showLogWindowFrame) this._textLogWindow.setBackgroundType(2);
      this.addWindow(this._textLogWindow);
    }
    textLogWindowRect() {
      const yBase = ConfigManager.touchUI ? this.buttonAreaBottom() : 0;
      const x = Math.max(0, settings.logX);
      const y = Math.max(yBase, settings.logY + yBase);
      const w = settings.logWidth > 0 ? settings.logWidth : Graphics.boxWidth;
      const hFull = Graphics.boxHeight - yBase;
      const h = settings.logHeight > 0 ? settings.logHeight : (hFull - (y - yBase));
      return new Rectangle(x, y, Math.max(1, w), Math.max(1, h));
    }
    createButtons() {
      if (ConfigManager.touchUI) {
        this._cancelButton = new Sprite_Button('cancel');
        this._cancelButton.x = Graphics.boxWidth - this._cancelButton.width - 4;
        this._cancelButton.y = this.buttonY();
        this.addChild(this._cancelButton);
      }
    }
  }
  globalThis.Scene_TextLog = Scene_TextLog;

  (function mix_Scene_Map(proto) {
    const _start = proto.start;
    proto.start = function () {
      _start.call(this);
      this.textLogCalling = false;
    };
    const _createAllWindows = proto.createAllWindows;
    proto.createAllWindows = function () {
      _createAllWindows.call(this);
      $gameTemp.clearEvacuatedMessageAndSubWindows();
    };
    const _createMessageWindow = proto.createMessageWindow;
    proto.createMessageWindow = function () {
      if (settings.smoothBackFromLog && $gameTemp.evacuatedMessageAndSubWindows()) {
        this._messageWindow = $gameTemp.evacuatedMessageAndSubWindows().messageWindow;
        this.addWindow(this._messageWindow);
      } else {
        _createMessageWindow.call(this);
      }
      if (settings.smoothBackFromLog) this._messageWindow.destroy = () => {};
    };
    const _createGoldWindow = proto.createGoldWindow;
    proto.createGoldWindow = function () {
      if (settings.smoothBackFromLog && $gameTemp.evacuatedMessageAndSubWindows()) {
        this._goldWindow = $gameTemp.evacuatedMessageAndSubWindows().goldWindow;
        this.addWindow(this._goldWindow);
      } else {
        _createGoldWindow.call(this);
      }
      if (settings.smoothBackFromLog) this._goldWindow.destroy = () => {};
    };
    const _createNameBoxWindow = proto.createNameBoxWindow;
    proto.createNameBoxWindow = function () {
      if (settings.smoothBackFromLog && $gameTemp.evacuatedMessageAndSubWindows()) {
        this._nameBoxWindow = $gameTemp.evacuatedMessageAndSubWindows().nameBoxWindow;
        this.addWindow(this._nameBoxWindow);
      } else {
        _createNameBoxWindow.call(this);
      }
      if (settings.smoothBackFromLog) this._nameBoxWindow.destroy = () => {};
    };
    const _createChoiceListWindow = proto.createChoiceListWindow;
    proto.createChoiceListWindow = function () {
      if (settings.smoothBackFromLog && $gameTemp.evacuatedMessageAndSubWindows()) {
        this._choiceListWindow = $gameTemp.evacuatedMessageAndSubWindows().choiceListWindow;
        this.addWindow(this._choiceListWindow);
      } else {
        _createChoiceListWindow.call(this);
      }
      if (settings.smoothBackFromLog) this._choiceListWindow.destroy = () => {};
    };
    const _createNumberInputWindow = proto.createNumberInputWindow;
    proto.createNumberInputWindow = function () {
      if (settings.smoothBackFromLog && $gameTemp.evacuatedMessageAndSubWindows()) {
        this._numberInputWindow = $gameTemp.evacuatedMessageAndSubWindows().numberInputWindow;
        this.addWindow(this._numberInputWindow);
      } else {
        _createNumberInputWindow.call(this);
      }
      if (settings.smoothBackFromLog) this._numberInputWindow.destroy = () => {};
    };
    const _update = proto.update;
    proto.update = function () {
      _update.call(this);
      if (!SceneManager.isSceneChanging()) this.updateCallTextLog();
    };
    proto.updateCallTextLog = function () {
      if (this.isTextLogEnabled()) {
        if (this.isTextLogCalled()) {
          this.textLogCalling = true;
          $gameTemp.clearCallTextLogOnMapRequest();
        }
        if (this.textLogCalling && !$gamePlayer.isMoving()) this.callTextLog();
      } else {
        this.textLogCalling = false;
      }
    };
    proto.isTextLogEnabled = function () {
      const enabledBySwitch =
        !settings.disableLogWindowSwitch || !$gameSwitches.value(settings.disableLogWindowSwitch);
      const duringMessage = $gameMessage && $gameMessage.isBusy();
      return enabledBySwitch && duringMessage;
    };
    proto.isTextLogCalled = function () {
      return settings.openLogKeys.some((key) => Input.isTriggered(key)) || $gameTemp.isCallTextLogOnMapRequested();
    };
    proto.callTextLog = function () {
      if (settings.smoothBackFromLog) {
        $gameTemp.setEvacuatedMessageAndSubWindows(
          new EvacuatedMessageAndSubWindows(this._messageWindow, this._goldWindow, this._nameBoxWindow, this._choiceListWindow, this._numberInputWindow)
        );
      }
      SoundManager.playCursor();
      SceneManager.push(Scene_TextLog);
      $gameTemp.clearDestination();
      this._mapNameWindow?.hide();
      this._waitCount = 2;
    };
  })(Scene_Map.prototype);

  class LogMessageForView {
    constructor(message, height, heightFromBottom) {
      this._message = message;
      this._height = height;
      this._heightFromBottom = heightFromBottom;
    }
    get height() { return this._height; }
    get heightFromBottom() { return this._heightFromBottom; }
    text() { return this._message.text(); }
    raw() { return this._message; }
  }

  class Window_TextLog extends Window_Selectable {
    initialize(rect) {
      this._messages = [];
      this._index = -1;
      this._lineTops = [];
      this._lineHeights = [];
      super.initialize(rect);
      this.setupLogMessages();
      this.rebuildLayout();
      this.jumpToBottom();
      this.refresh();
      this.activate();
    }

    itemRect(index) {
      const top = this._lineTops[index] ?? 0;
      const h   = this._lineHeights[index] ?? this.lineHeight();
      return new Rectangle(0, top - this.scrollBaseY(), this.innerWidth, h);
    }

    maxItems() {
      return Array.isArray(this._messages) ? this._messages.length : 0;
    }
    itemHeight() {
      return this.lineHeight();
    }
    lineHeight() {
      return super.lineHeight() + settings.lineSpacing;
    }

    setupLogMessages() {
      let fromBottom = 0;
      const src = ($gameTemp.eventTextLog() && $gameTemp.eventTextLog().messages) || [];
      this._messages = Array.from(src)
        .reverse()
        .map((message) => {
          const height = this.calcMessageHeight(message);
          fromBottom += height;
          return new LogMessageForView(message, height, fromBottom);
        })
        .reverse();
    }

    rebuildLayout() {
      this._lineTops = [];
      this._lineHeights = [];
      let y = 0;
      for (let i = 0; i < this._messages.length; i++) {
        const h = this._messages[i].height;
        this._lineTops.push(y);
        this._lineHeights.push(h);
        y += h;
      }
    }

    overallHeight() {
      return this._messages.length > 0 ? this._messages[0].heightFromBottom : this.innerHeight;
    }
    isCursorMovable() {
      return true;
    }

    select(index) {
      const n = this.maxItems();
      if (n <= 0) {
        this._index = -1;
        this.setCursorRect(0, 0, 0, 0);
        return;
      }
      this._index = Math.max(0, Math.min(index, n - 1));
      this.updateCursorForIndex();
    }

    cursorUp() {
      if (this._index > 0) this.select(this._index - 1);
      this.smoothScrollToKeepCursor();
    }
    cursorDown() {
      if (this._index < this.maxItems() - 1) this.select(this._index + 1);
      this.smoothScrollToKeepCursor();
    }
    cursorPageup() {
      const targetTop = Math.max(0, this.scrollBaseY() - this.innerHeight);
      const newIndex = this.indexAtY(targetTop + 1);
      this.select(newIndex);
      this.smoothScrollTo(0, targetTop);
    }
    cursorPagedown() {
      const targetTop = Math.min(this.maxScrollY(), this.scrollBaseY() + this.innerHeight);
      const newIndex = this.indexAtY(targetTop + this.innerHeight - 1);
      this.select(newIndex);
      this.smoothScrollTo(0, targetTop);
    }

    processOk() {
      this.playVoiceOfCurrent();
    }

    localTouchPoint() {
      const px = TouchInput.x;
      const py = TouchInput.y;
      if (this.worldTransform && this.worldTransform.applyInverse && window.PIXI) {
        const pIn = new PIXI.Point(px, py);
        const pOut = new PIXI.Point();
        this.worldTransform.applyInverse(pIn, pOut);
        return pOut;
      }
      return new Point(px - this.x, py - this.y);
    }

    onTouchOk() {
      if (this.isTouchedInsideFrame()) {
        const p = this.localTouchPoint();
        const y = p.y + this.scrollBaseY();
        const i = this.indexAtY(y);
        this.select(i);
        this.playVoiceOfCurrent();
      }
    }
    onTouchSelect() {
      if (this.isTouchedInsideFrame()) {
        const p = this.localTouchPoint();
        const y = p.y + this.scrollBaseY();
        const i = this.indexAtY(y);
        this.select(i);
      }
    }

    updateCursorForIndex() {
      if (this._index < 0) {
        this.setCursorRect(0, 0, 0, 0);
        return;
      }
      const top = this._lineTops[this._index] ?? 0;
      const h   = this._lineHeights[this._index] ?? this.lineHeight();
      const cy = top - this.scrollBaseY();
      this.setCursorRect(0, cy, this.innerWidth, h);
      if (cy < 0) this.smoothScrollTo(0, top);
      const bottom = cy + h;
      if (bottom > this.innerHeight) this.smoothScrollTo(0, top - (this.innerHeight - h));
    }

    smoothScrollToKeepCursor() {
      if (this._index < 0) return;
      const top = this._lineTops[this._index] ?? 0;
      const h   = this._lineHeights[this._index] ?? this.lineHeight();
      const cy = top - this.scrollBaseY();
      if (cy < 0) this.smoothScrollTo(0, top);
      const bottom = cy + h;
      if (bottom > this.innerHeight) this.smoothScrollTo(0, top - (this.innerHeight - h));
    }

    indexAtY(yAbs) {
      if (!this._lineTops || !this._lineHeights || !this._lineTops.length) return 0;
      for (let i = 0; i < this._lineTops.length; i++) {
        const top = this._lineTops[i];
        const h = this._lineHeights[i];
        if (yAbs >= top && yAbs < top + h) return i;
      }
      return this._lineTops.length - 1;
    }

    update() {
      super.update();
      this.updateArrows();
      if (this.isOpenAndActive() && Input.isTriggered("ok")) this.processOk();
    }

    paint() {
      if (this.contents) {
        this.contents.clear();
        this.drawTextLog();
      }
    }

    drawTextLog() {
      let height = 0;
      const x = 4;
      this._messages.forEach((mView) => {
        this.drawTextEx(mView.text(), x, height + Math.floor(settings.lineSpacing / 2) - this.scrollBaseY());
        const raw = mView.raw();
        if (raw.hasVoice() && settings.voiceIconIndex >= 0) {
          const name = String(raw.speakerName || "");
          const nameW = name ? this.textSizeEx(name).width : 0;
          const iconX = x + (name ? nameW : 0) + settings.voiceIconOffsetX;
          const iconY = height + settings.voiceIconOffsetY - this.scrollBaseY();
          this.drawIcon(settings.voiceIconIndex, iconX, iconY);
        }
        height += mView.height;
      });
      this.updateCursorForIndex();
    }

    textSizeEx(text) {
      this.resetFontSettings();
      const safe = String(text ?? "");
      const ts = this.createTextState(safe, 4, 0, this.innerWidth);
      ts.drawing = false;
      this.processAllText(ts);
      return { width: ts.outputWidth, height: ts.outputHeight };
    }

    calcMessageHeight(message) {
      return this.textSizeEx(message.text()).height + settings.messageSpacing;
    }

    processEscapeCharacter(code, textState) {
      if (settings.escapeCharacterCodes.includes(code)) {
        this.obtainEscapeParamText(textState);
        return;
      }
      super.processEscapeCharacter(code, textState);
    }

    playVoiceOfCurrent() {
      if (this._index < 0 || this._index >= this._messages.length) return;
      const gm = this._messages[this._index].raw();
      const meta = gm.voiceMeta;
      if (!meta) return;
      const list = (window.$datamapvoice && $datamapvoice.voices) || [];
      const vc = list[Number(meta.index) || 0];
      if (!vc) return;
      let prefix = "voice/";
      try {
        const p = PluginManager.parameters("BasicMapVoice_MZ") || {};
        prefix = String(p["FolderPrefix"] || "voice/");
      } catch (_) {}
      if (prefix && !prefix.endsWith("/")) prefix += "/";
      const folder = prefix + String(meta.folder3 || "000");
      if (window.AudioVoice && AudioVoice.playVc) {
        if (AudioVoice.stopVc) {
          try {
            AudioVoice.stopVc();
          } catch (_) {}
        }
        AudioVoice.playVc(vc, folder);
      }
    }
  }

  Window_ObtainEscapeParamTextMixIn(Window_TextLog.prototype);
  globalThis.Window_TextLog = Window_TextLog;

  (function mix_Window_Message(proto) {
    const _processEscapeCharacter = proto.processEscapeCharacter;
    proto.processEscapeCharacter = function (code, textState) {
      if (code === "Z") {
        const m = /^\[(\d+)\]/.exec(textState.text.slice(textState.index));
        const i = m ? Number(m[1]) : 0;
        let f3 = "000";
        try {
          f3 = this._vcFolder || (window.BMV_State && BMV_State.folder) || "000";
        } catch (_) {}
        if (window.$gameTemp) $gameTemp._lastMapVoice = { index: i, folder3: String(f3) };
      }
      _processEscapeCharacter.call(this, code, textState);
    };

    const _terminateMessage = proto.terminateMessage;
    proto.terminateMessage = function () {
      if (!settings.disableLoggingSwitch || !$gameSwitches.value(settings.disableLoggingSwitch)) {
        const raw = $gameMessage.allText();
        if (raw) {
          // namebox に入力された名前だけをそのまま使う
          // 名前欄が空なら、ログの話者名も空文字のまま
          const speaker = this.convertEscapeCharacters($gameMessage.speakerName());
          const converted = this.convertEscapeCharacters(raw);
          const body = stripIgnoredEscapes(converted);
          $gameTemp.eventTextLog().pushLog(
            speaker,
            body,
            $gameTemp._lastMapVoice ? { ...$gameTemp._lastMapVoice } : null
          );
        }
        if ($gameMessage.isChoice()) {
          const line = settings.choiceFormat.replace(
            /\{choice\}/gi,
            `\x1bC[${settings.choiceColor}]${$gameMessage.chosenText()}\x1bC[0]`
          );
          $gameTemp.eventTextLog().pushLog('', stripIgnoredEscapes(line), null);
        }
      }
      delete $gameTemp._lastMapVoice;
      _terminateMessage.call(this);
    };
  })(Window_Message.prototype);

  Window_TextLog.prototype.jumpToBottom = function () {
    const n = this.maxItems();
    if (n <= 0) {
      this._index = -1;
      this._scrollY = 0;
      this.setCursorRect(0, 0, 0, 0);
      return;
    }
    this.select(n - 1);
    this._scrollY = this.maxScrollY();
    this.updateCursorForIndex();
  };



  // ============================================================
// LL_GalgeChoiceWindow 表示中はテキストログを無効化
// ============================================================
(() => {
  const _Scene_Map_isTextLogEnabled = Scene_Map.prototype.isTextLogEnabled;

  Scene_Map.prototype.isTextLogEnabled = function () {
    // 通常条件が false なら即終了
    if (!_Scene_Map_isTextLogEnabled.call(this)) return false;

    // LL_GalgeChoiceWindow の選択肢が開いているかチェック
    const scene = SceneManager._scene;
    const w = scene && scene._galgeChoiceListWindow;

    if (w && w.isOpen() && w.active) {
      return false; // ログ禁止
    }

    return true;
  };
})();

})();
