/*:
 * @param scenes
 * @text シーン設定
 * @type struct<SceneConfig>[]
 * @desc 複数のシーンごとに背景画像とコモンイベントを設定します。
 * @default []
 * 
 * @param scale
 * @text 背景画像のスケール
 * @type number
 * @decimals 2
 * @desc 背景画像のスケールを指定します (1.0 で等倍、0.5 で半分の大きさ)。
 * @default 1.0
 * 
 * @param blurRadius
 * @text ブラー効果の半径
 * @type number
 * @decimals 2
 * @desc 背景画像に適用するブラーの半径を指定します。0でブラーなし。
 * @default 0
 */

/*~struct~SceneConfig:
 * @param sceneId
 * @text シーン識別子
 * @desc 背景画像とコモンイベントを設定するシーンの識別子 (例: Scene_Map, Scene_Custom_Menu)
 * 
 * @param variableId
 * @text 画像ファイル名の変数ID
 * @desc 画像ファイル名を格納している変数のID (変数にサブフォルダを含むパスが格納される)
 * @type variable
 * @default 1
 * 
 * @param commonEventId
 * @text コモンイベントID
 * @desc このシーンで発生させるコモンイベントのID
 * @type common_event
 * @default 0
 */

(() => {
    const pluginName = "DynamicBackgroundSetter";
    const parameters = PluginManager.parameters(pluginName);
    const sceneConfigs = JSON.parse(parameters["scenes"] || "[]").map(config => JSON.parse(config));
    const backgroundScale = parseFloat(parameters["scale"] || 1.0);
    const blurRadius = parseFloat(parameters["blurRadius"] || 0);

    const applySceneConfig = (sceneClass, variableId, commonEventId) => {
        if (!sceneClass?.prototype) return;

        const _originalStart = sceneClass.prototype.start;
        sceneClass.prototype.start = function() {
            _originalStart.call(this);
            this.setDynamicBackground(variableId);
            if (commonEventId > 0) {
                triggerCommonEvent(commonEventId);
            }
        };

        sceneClass.prototype.setDynamicBackground = function(variableId) {
            const imagePath = $gameVariables.value(variableId);
            if (!imagePath) return;

            console.log(`背景画像のパス: img/pictures/${imagePath}`);
            const bitmap = ImageManager.loadBitmap("img/pictures/", imagePath);

            if (this._backgroundSprite) {
                this.removeChild(this._backgroundSprite);
            }

            const sprite = new Sprite(bitmap);
            sprite.anchor.set(0.5, 0.5); 
            sprite.scale.set(backgroundScale); 
            sprite.x = Graphics.width / 2; 
            sprite.y = Graphics.height / 2;

            if (blurRadius > 0) {
                const blurFilter = new PIXI.filters.BlurFilter();
                blurFilter.blur = blurRadius;
                sprite.filters = [blurFilter];
            }

            this._backgroundSprite = sprite;
            this._backgroundSprite.opacity = 0;

            this.addChildAt(this._backgroundSprite, 0);

            bitmap.addLoadListener(() => {
                this._backgroundSprite.opacity = 255;
            });
        };
    };

    const triggerCommonEvent = (commonEventId) => {
        if ($dataCommonEvents[commonEventId]) {
            $gameTemp.reserveCommonEvent(commonEventId);
        }
    };

    sceneConfigs.forEach(config => {
        const sceneId = config.sceneId;
        const variableId = Number(config.variableId);
        const commonEventId = Number(config.commonEventId);

        const sceneClass = window[sceneId];

        if (sceneClass) {
            applySceneConfig(sceneClass, variableId, commonEventId);
        } else {
            const checkScene = () => {
                if (SceneManager._scene && SceneManager._scene.constructor.name === sceneId) {
                    applySceneConfig(SceneManager._scene.constructor, variableId, commonEventId);
                    if (commonEventId > 0) {
                        triggerCommonEvent(commonEventId);
                    }
                }
            };

            SceneManager._sceneCreateListeners = SceneManager._sceneCreateListeners || [];
            SceneManager._sceneCreateListeners.push(checkScene);

            checkScene();
        }
    });

    const _SceneManager_onSceneCreate = SceneManager.onSceneCreate;
    SceneManager.onSceneCreate = function() {
        _SceneManager_onSceneCreate.call(this);
        if (SceneManager._sceneCreateListeners) {
            SceneManager._sceneCreateListeners.forEach(listener => listener());
        }
    };

    const _Scene_Title_create = Scene_Title.prototype.create;
    Scene_Title.prototype.create = function() {
        _Scene_Title_create.call(this);

        sceneConfigs.forEach(config => {
            const variableId = Number(config.variableId);
            $gameVariables.setValue(variableId, 0); 
        });

        if (this._backgroundSprite) {
            this.removeChild(this._backgroundSprite);
            this._backgroundSprite = null;
        }
    };
})();
