// FesStyleConfirm.js Ver.1.1.1
// MIT License (C) 2023 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ
* @orderAfter RemoveSaveFile
* @plugindesc Generates a Maker Fes-style confirmation dialog.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/499410179.html
* @help Ver.1.1.1
*
* @command showConfirm
* @text Show Confirm
* @desc Display a confirmation dialog.
*
* @arg text
* @text Text
* @type note
* @default 
*
* @arg ok
* @desc Decision text.
* @text OK
* @type string
* @default Yes
*
* @arg cancel
* @desc Cancellation text.
* @text Cancel
* @type string
* @default No
*
* @arg switchId
* @text Switch ID
* @desc Assigned when the dialog ends.
* ON at decision, OFF at cancellation.
* @type switch
* @default 1
*
* @command showAlert
* @text Show Alert
* @desc Show alert.
*
* @arg text
* @text Text
* @type note
* @default 
*
* @arg ok
* @desc Decision text.
* @text OK
* @type string
* @default OK
*
* @param mzStyle
* @text Make it MZ/MV style
* @desc Make the design MZ/MV.
* @type boolean
* @default false
*
* @param windowSkin
* @text Window Skin
* @desc 
* @default 
* @type file
* @dir img/system
*
* @param windowOpacity
* @text Window Opacity
* @desc -1 is system default.
* @default -1
* @type number
* @min -1
*
* @param windowWidth
* @text Window Width
* @desc 
* @type number
* @default 528
*
* @param numLines
* @text Number of Rows
* @desc The window height is calculated from the number of rows.
* @type number
* @default 8
*
* @param maxItemWidth
* @text Max Item Width
* @desc Limit the width of the selection.
* @type number
* @default 180
*
* @param itemOffsetX
* @text Item Offset X
* @desc Adjust the X coordinate of the selection.
* @type number
* @default 32
*
* @param itemOffsetY
* @text Item Offset Y
* @desc Adjust the Y coordinate of the selection.
* @type number
* @default 32
*
* @param terms
* @text Terms
* @type struct<terms>
* @default {"saveConfirm":"\"Save data.\\nAre you sure?\"","overwriteConfirm":"\"Overwrite data.\\nAre you sure?\"","loadConfirm":"\"Load data.\\nAre you sure?\"","removeConfirm":"\"Remove data.\\nAre you sure?\"","gameEndConfirm":"\"Back to the title?\"","ok":"Yes","cancel":"No"}
*
*/

/*:ja
* @target MZ
* @orderAfter RemoveSaveFile
* @plugindesc ツクールフェス風の確認ダイアログを生成します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/499410179.html
* @help 
*
* FesStyleConfirm[更新履歴]
* 2023/05/20：Ver.1.0.0　公開。
* 2023/07/26：Ver.1.1.0　制御文字や中央揃えが正しく機能しない不具合を修正。
* 2024/01/16：Ver.1.1.1　MZ風にした時のアウトライン幅を修正。
*
* @command showConfirm
* @text 確認ダイアログの表示
* @desc 確認ダイアログを表示します。
*
* @arg text
* @text テキスト
* @type note
* @default 
*
* @arg ok
* @desc 決定した時のテキストです。
* @text 決定
* @type string
* @default はい
*
* @arg cancel
* @desc キャンセルした時のテキストです。
* @text キャンセル
* @type string
* @default いいえ
*
* @arg switchId
* @text スイッチID
* @desc ダイアログ終了時に代入されます。
* 決定の時ON、キャンセルの時OFF。
* @type switch
* @default 1
*
* @command showAlert
* @text アラートの表示
* @desc アラートを表示します。
*
* @arg text
* @text テキスト
* @type note
* @default 
*
* @arg ok
* @desc 決定のテキストです。
* @text 決定
* @type string
* @default はい
*
* @param mzStyle
* @text MZ/MV風にする
* @desc デザインをMZ/MVの物にします。
* @type boolean
* @default false
*
* @param windowSkin
* @text ウィンドウスキン
* @desc ファイル名を指定します。
* @default 
* @type file
* @dir img/system
*
* @param windowOpacity
* @text ウィンドウ不透明度
* @desc ウィンドウの不透明度を指定します。
* -1でシステムの初期値。
* @default -1
* @type number
* @min -1
*
* @param windowWidth
* @text ウィンドウ幅
* @desc 
* @type number
* @default 528
*
* @param numLines
* @text 行数
* @desc 行数からウィンドウ高さが算出されます。
* @type number
* @default 8
*
* @param maxItemWidth
* @text 要素最大幅
* @desc 選択肢の横幅を制限します。
* @type number
* @default 180
*
* @param itemOffsetX
* @text 要素オフセットX
* @desc 選択肢のX座標を調整します。
* @type number
* @default 32
*
* @param itemOffsetY
* @text 要素オフセットY
* @desc 選択肢のY座標を調整します。
* @type number
* @default 32
*
* @param terms
* @text 用語
* @type struct<terms>
* @default {"saveConfirm":"\"データをセーブします。\\nよろしいですか？\"","overwriteConfirm":"\"データを上書きします。\\nよろしいですか？\"","loadConfirm":"\"データをロードします。\\nよろしいですか？\"","removeConfirm":"\"データを削除します。\\nよろしいですか？\"","gameEndConfirm":"\"タイトルに戻りますか？\"","ok":"はい","cancel":"いいえ"}
*
*/

/*~struct~terms:
*
* @param saveConfirm
* @text Save Confirm
* @desc 
* @type note
* @default "Save data.\nAre you sure?"
*
* @param overwriteConfirm
* @text Overwrite Confirm
* @desc 
* @type note
* @default "Overwrite data.\nAre you sure?"
*
* @param loadConfirm
* @text Load Confirm
* @desc 
* @type note
* @default "Load data.\nAre you sure?"
*
* @param removeConfirm
* @text Remove Confirm
* @desc 
* @type note
* @default "Remove data.\nAre you sure?"
*
* @param gameEndConfirm
* @text Game End Confirm
* @desc 
* @type note
* @default "Back to the title?"
*
* @param ok
* @desc Decision text.
* @text OK
* @type string
* @default Yes
*
* @param cancel
* @desc Cancellation text.
* @text Cancel
* @type string
* @default No
*
*/

/*~struct~terms:ja
*
* @param saveConfirm
* @text セーブ確認
* @desc 
* @type note
* @default "データをセーブします。\nよろしいですか？"
*
* @param overwriteConfirm
* @text 上書き確認
* @desc 
* @type note
* @default "データを上書きします。\nよろしいですか？"
*
* @param loadConfirm
* @text ロード確認
* @desc 
* @type note
* @default "データをロードします。\nよろしいですか？"
*
* @param removeConfirm
* @text 削除確認
* @desc 
* @type note
* @default "データを削除します。\nよろしいですか？"
*
* @param gameEndConfirm
* @text ゲーム終了確認
* @desc 
* @type note
* @default "タイトルに戻りますか？"
*
* @param ok
* @desc 決定のテキストです。
* @text 決定
* @type string
* @default はい
*
* @param cancel
* @desc キャンセルのテキストです。
* @text キャンセル
* @type string
* @default いいえ
*
*/

'use strict';

{
	//プラグイン名取得。
	const script = document.currentScript;
	const pluginName = (() => {
	const src = (script && script.src) ? script.src : "";
	// 末尾のファイル名（.js有無、?query、#hash を吸収）
	const m = src.match(/([^\/?#]+?)(?:\.js)?(?:[?#].*)?$/);
	// 取れなければ固定名にフォールバック（ファイル名に合わせる）
	return (m && m[1]) ? m[1] : "FesStyleConfirm";
	})();
	const parameters = PluginManager.parameters(pluginName);

	const mzStyle = parameters["mzStyle"] === "true";
	const windowSkin = parameters["windowSkin"];
	const windowOpacity = Number(parameters["windowOpacity"]);
	const windowWidth = +parameters["windowWidth"];
	const maxItemWidth = +parameters["maxItemWidth"];
	const itemOffsetX = +parameters["itemOffsetX"];
	const itemOffsetY = +parameters["itemOffsetY"];
	const numLines = +parameters["numLines"];
	const terms = JSON.parse(parameters["terms"]||"{}");
	for (const prop in terms) {
		if (prop !== "ok" && prop !== "cancel") {
			terms[prop] = JSON.parse(terms[prop]);
		}
	}

	//-----------------------------------------------------------------------------
	// PluginManager

	PluginManager.registerCommand(pluginName, "showConfirm", function (args) {
		const text = JSON.parse(args.text);
		const choices = [args.cancel, args.ok];
		const switchId = args.switchId;
		SceneManager._scene.createConfirmWindow();
		$gameMessage.setComfirm(text, choices);
		SceneManager._scene._confirmWindow.startConfirm("event", switchId);
		this.setWaitMode("message");
	});

	PluginManager.registerCommand(pluginName, "showAlert", function (args) {
		const text = JSON.parse(args.text);
		const choices = [args.ok];
		SceneManager._scene.createConfirmWindow();
		$gameMessage.setComfirm(text, choices);
		SceneManager._scene._confirmWindow.startAlert("event");
		this.setWaitMode("message");
	});

	//-----------------------------------------------------------------------------
	// Game_Message

	const _Game_Message_clear = Game_Message.prototype.clear;
	Game_Message.prototype.clear = function() {
		_Game_Message_clear.call(this);
		this._confirmChoices = [];
		this._confirmText = "";
	};

	Game_Message.prototype.isConfirm = function() {
		return this._confirmChoices.length > 0;
	};

	Game_Message.prototype.confirmChoices = function() {
		return this._confirmChoices;
	};

	Game_Message.prototype.confirmText = function() {
		return this._confirmText;
	};

	Game_Message.prototype.setComfirm = function(text, choices) {
		this._confirmText = text;
		this._confirmChoices = choices;
	};

	const _Game_Message_isBusy = Game_Message.prototype.isBusy;
	Game_Message.prototype.isBusy = function() {
		return this.isConfirm() || _Game_Message_isBusy.call(this);
	};

	//-----------------------------------------------------------------------------
	// Scene_Base

	Scene_Base.prototype.createConfirmWindow = function() {
		if (this._confirmWindow) {
			this.addWindow(this._confirmWindow);
			return;
		}
		const rect = this.confirmWindowRect();
		this._confirmWindow = new Window_Confirm(rect);
		this.addWindow(this._confirmWindow);
	};

	Scene_Base.prototype.confirmWindowRect = function() {
		const ww = windowWidth;
		const wh = this.calcWindowHeight(numLines, true);
		const wx = (Graphics.boxWidth - ww) / 2;
		const wy = (Graphics.boxHeight - wh) / 2;
		return new Rectangle(wx, wy, ww, wh);
	};

	//-----------------------------------------------------------------------------
	// Scene_File

	Scene_File.prototype.openConfirmWindow = function() {
		this.createConfirmWindow();
		this._confirmWindow.setHandler("ok", this.onConfirmOk.bind(this));
		this._confirmWindow.setHandler("cancel", this.onConfirmCancel.bind(this));
		const savefileId = this.savefileId();
		const isOverwrite = !!DataManager.savefileInfo(savefileId);
		this._confirmWindow.setOverwrite(isOverwrite);
		this._confirmWindow.startConfirm(this.mode());
	};

	//-----------------------------------------------------------------------------
	// Scene_Save

	const _Scene_Save_onSavefileOk = Scene_Save.prototype.onSavefileOk;
	Scene_Save.prototype.onSavefileOk = function() {
		const savefileId = this.savefileId();
		if (this.isSavefileEnabled(savefileId)) {
			Window_HorzCommand.prototype.playOkSound.call(this);
			this.openConfirmWindow();
		} else {
			_Scene_Save_onSavefileOk.call(this);
		}
	};

	Scene_Save.prototype.onConfirmOk = function() {
		_Scene_Save_onSavefileOk.call(this);
		this._confirmWindow.close();
		this._confirmWindow.deactivate();
	};

	Scene_Save.prototype.onConfirmCancel = function() {
		this._confirmWindow.close();
		this._confirmWindow.deactivate();
		this.activateListWindow();
	};

	const _Scene_Save_onSaveFailure = Scene_Save.prototype.onSaveFailure;
	Scene_Save.prototype.onSaveFailure = function() {
		if (this._confirmWindow) {
			this._confirmWindow.close();
			this._confirmWindow.deactivate();
		}
		_Scene_Save_onSaveFailure.call(this);
	};

	//-----------------------------------------------------------------------------
	// Scene_Load

	const _Scene_Load_onSavefileOk = Scene_Load.prototype.onSavefileOk;
	Scene_Load.prototype.onSavefileOk = function() {
		const savefileId = this.savefileId();
		if (this.isSavefileEnabled(savefileId)) {
			Window_HorzCommand.prototype.playOkSound.call(this);
			this.openConfirmWindow();
		} else {
			_Scene_Load_onSavefileOk.call(this);
		}
	};

	Scene_Load.prototype.onConfirmOk = function() {
		_Scene_Load_onSavefileOk.call(this);
		this._confirmWindow.close();
		this._confirmWindow.deactivate();
	};

	Scene_Load.prototype.onConfirmCancel = function() {
		this._confirmWindow.close();
		this._confirmWindow.deactivate();
		this.activateListWindow();
	};

	const _Scene_Load_onLoadFailure = Scene_Load.prototype.onLoadFailure;
	Scene_Load.prototype.onLoadFailure = function() {
		if (this._confirmWindow) {
			this._confirmWindow.close();
			this._confirmWindow.deactivate();
		}
		_Scene_Load_onLoadFailure.call(this);
	};

	//-----------------------------------------------------------------------------
	// Scene_Remove

	if (window.Scene_Remove) {
		const _Scene_Remove_onSavefileOk = Scene_Remove.prototype.onSavefileOk;
		Scene_Remove.prototype.onSavefileOk = function() {
			const savefileId = this.savefileId();
			if (this.isSavefileEnabled(savefileId)) {
				Window_HorzCommand.prototype.playOkSound.call(this);
				this.openConfirmWindow();
			} else {
				_Scene_Remove_onSavefileOk.call(this);
			}
		};

		Scene_Remove.prototype.onConfirmOk = function() {
			_Scene_Remove_onSavefileOk.call(this);
			this._confirmWindow.close();
			this._confirmWindow.deactivate();
		};

		Scene_Remove.prototype.onConfirmCancel = function() {
			this._confirmWindow.close();
			this._confirmWindow.deactivate();
			this.activateListWindow();
		};

		const _Scene_Remove_onRemoveFailure = Scene_Remove.prototype.onRemoveFailure;
		Scene_Remove.prototype.onRemoveFailure = function() {
			if (this._confirmWindow) {
				this._confirmWindow.close();
				this._confirmWindow.deactivate();
			}
			_Scene_Remove_onRemoveFailure.call(this);
		};
	}

	//-----------------------------------------------------------------------------
	// Scene_GameEnd

	Scene_GameEnd.prototype.createCommandWindow = function() {
		this.createConfirmWindow();
		this._commandWindow = this._confirmWindow;
		this._confirmWindow.setHandler("ok", this.commandToTitle.bind(this));
		this._confirmWindow.setHandler("cancel", this.popScene.bind(this));
		this._confirmWindow.startConfirm("gameEnd");
	};

	//-----------------------------------------------------------------------------
	// Window_Confirm

	function Window_Confirm() {
		this.initialize(...arguments);
	}

	window.Window_Confirm = Window_Confirm;

	Window_Confirm.prototype = Object.create(Window_HorzCommand.prototype);
	Window_Confirm.prototype.constructor = Window_Confirm;

	Window_Confirm.prototype.initialize = function(rect) {
		Window_HorzCommand.prototype.initialize.call(this, rect);
		this._isWindow = false;
		this._mode = "";
		this._alert = false;
		this._switchId = 0;
		this.openness = 0;
		this.deactivate();
		this._background = 0;
		this._canRepeat = false;
		this._overwrite = false;
	};

	Window_Confirm.prototype.pushTextWidth = function(textState) {
		textState.textWidths.push(textState.x - textState.startX);
	};

	Window_Confirm.prototype.setAlignCenter = function(textState) {
		const textWidth = textState.textWidths.shift() || 0;
		textState.x = (this.contentsWidth() - textWidth) / 2;
	};

	Window_Confirm.prototype.processAllText = function(textState) {
		if (textState.drawingCenter) {
			this.setAlignCenter(textState);
		}
		Window_HorzCommand.prototype.processAllText.apply(this, arguments);
		if (textState.calcTextWidths) {
			this.pushTextWidth(textState);
		}
	};

	Window_Confirm.prototype.processNewLine = function(textState) {
		if (textState.calcTextWidths) {
			this.pushTextWidth(textState);
		}
		Window_HorzCommand.prototype.processNewLine.call(this, textState);
		if (textState.drawingCenter) {
			this.setAlignCenter(textState);
		}
	};

	Window_Confirm.prototype.calcTextWidths = function(text) {
		this.resetFontSettings();
		const textState = this.createTextState(text, 0, 0, 0);
		textState.drawing = false;
		textState.calcTextWidths = true;
		textState.textWidths = [];
		this.processAllText(textState);
		return textState.textWidths;
	};

	Window_Confirm.prototype.drawTextExCenter = function(text, x, y, textWidths) {
		this.resetFontSettings();
		const textState = this.createTextState(text, x, y, 0);
		textState.drawingCenter = true;
		textState.textWidths = textWidths;
		this.processAllText(textState);
	};

	Window_Confirm.prototype.processCancel = function() {
		if (this.index() > 0) {
			this.select(0);
			this.playCursorSound();
			this.updateInputData();
			return;
		}
		SoundManager.playCancel();
		this.updateInputData();
		this.deactivate();
		this.callCancelHandler();
	};

	if (windowSkin) {
		Window_Confirm.prototype.loadWindowskin = function() {
			this.windowskin = ImageManager.loadSystem(windowSkin);
		};
	}

	if (windowOpacity > -1) {
		Window_Confirm.prototype.updateBackOpacity = function() {
			this.backOpacity = windowOpacity;
		};
	};

	Window_Confirm.prototype._updateCursor = function() {
		Window_HorzCommand.prototype._updateCursor.call(this);
		if (!mzStyle) {
			this._cursorSprite.visible = false;
		}
	};

	if (mzStyle) {
		Window_Confirm.prototype._backCursorCanInvisible = function() {
			return this.fesGradCursorEnabled();
		};
	}

	Window_Confirm.prototype.maxCols = function() {
		return this._alert ? 1 : 2;
	};

	Window_Confirm.prototype.setOverwrite = function(bool) {
		this._overwrite = bool;
	};

	Window_Confirm.prototype.itemTextAlign = function() {
		return "center";
	};

	Window_Confirm.prototype.startConfirm = function(mode, switchId) {
		this._mode = mode;
		this._alert = false;
		this._switchId = switchId || 0;
		this.updatePlacement();
		this.createContents();
		this.refresh();
		this.scrollTo(0, 0);
		this.selectDefault();
		this.open();
		this.activate();
	};

	Window_Confirm.prototype.startAlert = function(mode) {
		this._mode = mode;
		this._alert = true;
		this._switchId = 0;
		this.updatePlacement();
		this.createContents();
		this.refresh();
		this.scrollTo(0, 0);
		this.selectDefault();
		this.open();
		this.activate();
	};

	Window_Confirm.prototype.selectDefault = function() {
		this.select(1);
	};

	Window_Confirm.prototype.updatePlacement = function() {
		const ww = windowWidth;
		const wh = Window_HorzCommand.prototype.fittingHeight(numLines);
		this.width = ww + ww % 2;
		this.height = wh + wh % 2;
		this.x = (Graphics.boxWidth - this.width) / 2;
		this.y = (Graphics.boxHeight - this.height) / 2;
	};

	Window_Confirm.prototype.makeCommandList = function() {
		if (this._mode === "event") {
			const choices = $gameMessage.confirmChoices();
			for (const choice of choices) {
				this.addCommand(choice, "choice");
			}
		} else {
			if (!this._alert) {
				this.addCommand(terms.cancel, "cancel");
			}
			this.addCommand(terms.ok, "ok");
		}
	};

	Window_Confirm.prototype.drawAllItems = function() {
		Window_HorzCommand.prototype.drawAllItems.call(this);
		const rect = this.itemRect(0);
		const allText = this.getMessage();
		const allSize = this.textSizeEx(allText);
		let startY = (rect.y - allSize.height) / 2;
		const texts = allText.split("\n");
		while(texts.length) {
			const text = texts.shift();
			const size = this.textSizeEx(text);
			const startX = (this.innerWidth - size.width) / 2;
			this.drawTextEx(text, startX, startY, size.width);
			startY += size.height;
		}
	};

	Window_Confirm.prototype.drawAllItems = function() {
		Window_HorzCommand.prototype.drawAllItems.call(this);
		const bottomY = this.itemRect(0).y;
		const allText = this.getMessage();
		const allTextHeight = this.textSizeEx(allText).height;
		const textWidths = this.calcTextWidths(allText);
		const y = (bottomY - allTextHeight) / 2;
		this.drawTextExCenter(allText, 0, y, textWidths);
	};

	Window_Confirm.prototype.getMessage = function() {
		let text = "";
		switch (this._mode) {
		case "event":
			text = $gameMessage.confirmText();
			break;
		case "save":
			text = this._overwrite ? terms.overwriteConfirm : terms.saveConfirm;
			break;
		case "load":
			text = terms.loadConfirm;
			break;
		case "remove":
			text = terms.removeConfirm;
			break;
		case "gameEnd":
			text = terms.gameEndConfirm;
			break;
		}
		return text || "";
	};

	Window_Confirm.prototype.resetTextColor = function() {
		Window_HorzCommand.prototype.resetTextColor.call(this);
		if (drawItem) {
			this.changeTextColor("black");
			this.changeOutlineColor("white");
		}
	};

	let drawItem = false;
	Window_Confirm.prototype.drawItem = function(index) {
		drawItem = !mzStyle;
		const outlineWidth = this.contents.outlineWidth;
		if (!mzStyle) {
			this.contents.outlineWidth = 2;
		}
		Window_HorzCommand.prototype.drawItem.call(this, index);
		this.contents.outlineWidth = outlineWidth;
		drawItem = false;
	};

	Window_Confirm.prototype.drawItemBackground = function(index) {
		Window_HorzCommand.prototype.drawItemBackground.call(this, index);
		if (mzStyle) return;
		const rect = this.itemRect(index);
		this.drawConfirmBack(rect, index === this.index());
	};

	Window_Confirm.prototype.drawBackgroundRect = function(rect) {
		if (mzStyle) {
			Window_HorzCommand.prototype.drawBackgroundRect.call(this, rect);
		}
	};
		

	Window_Confirm.prototype.drawConfirmBack = function(rect, selected) {
		const col1 = this.drawFesStyleConfirmColor1(selected);
		const col2 = this.drawFesStyleConfirmColor2(selected);
		const x = rect.x;
		const y = rect.y;
		const w = rect.width;
		const h = rect.height;
		this.contentsBack.drawFesStyleConfirmBack(x, y, w, h, col1);
		this.contentsBack.drawFesStyleConfirmBack(x+2, y+2, w-4, h-4, col2);
	};

	Window_Confirm.prototype.itemHeight = function() {
		return Window_HorzCommand.prototype.itemHeight.call(this) + 16;
	};

	Window_Confirm.prototype.itemWidth = function() {
		const offsetW = getItemRect ? itemOffsetX * 2 : 0;
		return Math.floor((this.innerWidth - offsetW) / this.maxCols());
	};

	Window_Confirm.prototype.clearConfirmBack = function(rect) {
		this.contentsBack.clearRect(rect.x, rect.y, rect.width, rect.height);
	};

	Window_Confirm.prototype.drawFesStyleConfirmColor1 = function(selected) {
		return selected ? "rgb(188, 157, 68)" : "rgb(192, 192, 192)";
	};

	Window_Confirm.prototype.drawFesStyleConfirmColor2 = function(selected) {
		return selected ? "rgb(253, 212, 92)" : "rgb(255, 255, 255)";
	};

	Window_Confirm.prototype.select = function(index) {
		const lastIndex = this._index;
		Window_HorzCommand.prototype.select.call(this, index);
		if (!mzStyle && lastIndex !== index) {
			const rect1 = this.itemRect(lastIndex);
			const rect2 = this.itemRect(index);
			this.clearConfirmBack(rect1);
			this.clearConfirmBack(rect2);
			if (lastIndex < this.maxItems()) {
				this.drawConfirmBack(rect1, false);
			}
			this.drawConfirmBack(rect2, true);
		}
	};

	let getItemRect = false;
	Window_Confirm.prototype.itemRect = function(index) {
		getItemRect = true;
		const rect = Window_HorzCommand.prototype.itemRect.call(this, index);
		rect.y = this.innerHeight - rect.height - rect.y - itemOffsetY;
		if (maxItemWidth) {
			const rectWidth = rect.width;
			rect.width = Math.min(rectWidth, maxItemWidth);
			rect.x += (rectWidth - rect.width)/2
		}
		rect.x += itemOffsetX;
		getItemRect = false;
		return rect;
	};

	Window_Confirm.prototype.isCancelEnabled = function() {
		return this.maxCols() >= 2;
	};

	Window_Confirm.prototype.callOkHandler = function() {
		if (this._mode === "event") {
			this.onEventOk();
		} else {
			Window_HorzCommand.prototype.callOkHandler.call(this);
		}
	};

	Window_Confirm.prototype.callCancelHandler = function() {
		if (this._mode === "event") {
			this.onEventCancel();
		} else {
			Window_HorzCommand.prototype.callCancelHandler.call(this);
		}
	};

	Window_Confirm.prototype.onEventOk = function() {
		const swicthId = this._switchId;
		if (swicthId) {
			$gameSwitches.setValue(swicthId, this.index() === 1);
		}
		$gameMessage.clear();
		this._mode = "";
		this._switchId = 0;
		this.close();
	};

	Window_Confirm.prototype.playOkSound = function() {
		if (this.currentSymbol() === "ok" && ["save", "load", "remove"].includes(this._mode)) {
			return;
		}
		Window_HorzCommand.prototype.playOkSound.call(this);
	};

	Window_Confirm.prototype.onEventCancel = function() {
		this.callOkHandler();
	};

	//-----------------------------------------------------------------------------
	// Bitmap

	Bitmap.prototype.drawFesStyleConfirmBack = function(x, y, width, height, color) {
		const context = this.context;
		context.save();
		context.fillStyle = color;
		const radius = height / 2;
		context.beginPath();
		context.moveTo(x+radius, y);
		context.arcTo(x+width, y, x+width, y+radius, radius);
		context.arcTo(x+width, y+height, x+radius, y+height, radius);
		context.arcTo(x, y+height, x, y+radius, radius);
		context.arcTo(x, y, x+radius, y, radius);
		context.closePath(x+radius, y);
		context.fill();
		context.restore();
		this._baseTexture.update();
	};

}