//-----------------------------------------------------------------------------
//  Galv's Action Indicators MZ
//-----------------------------------------------------------------------------
//  For: RPGMAKER MZ
//  Galv.ActionIndicatorsMZ.js
//-----------------------------------------------------------------------------
//  2021-01-02 - Version 1.0 - release
//-----------------------------------------------------------------------------
// Terms can be found at:
// galvs-scripts.com
//-----------------------------------------------------------------------------

var Imported = Imported || {};
Imported.Galv_ActionIndicators = true;

var Galv = Galv || {};        // Galv's main object
Galv.AI = Galv.AI || {};      // Plugin Object
Galv.AI.pluginName = "Galv_ActionIndicatorsMZ";

//-----------------------------------------------------------------------------
/*:
 * @plugindesc (v.1.0) Display an icon when the player is able to interact with an event. View help for comment tag.
 * @url http://galvs-scripts.com
 * @target MZ
 * @author Galv
 *
 * @param yOffset
 * @name Y Offset
 * @desc Pixel offset for icon's Y position
 * @default -10
 *
 * @param zPosition
 * @name Z Position
 * @desc The Z position (controls if it appears over/under map objects)
 * @default 5
 *
 * @param autoHide
 * @name Auto Hide
 * @desc true or false. If true, icons will disappear when an event is running
 * @type boolean
 * @on YES
 * @off NO
 * @default true
 *
 * @param iconOpacity
 * @name Icon Opacity
 * @desc 0-255. The opacity of the icon
 * @type number
 * @default 200
 *
 * @help
 *   Galv's Action Indicators
 * ----------------------------------------------------------------------------
 * This plugin will enable you to display an icon when the player is facing an
 * event that has the below code in a 'comment' command anywhere in the active
 * event page.
 *
 *
 *   <actionIcon: id,yo>      // The code to use in a COMMENT within an event
 *                            // id = the icon ID to use for the indicator
 *                            // yo = the y offset in pixels for the indicator
 *
 *
 * This plugin only does ONE icon above a single event that the player is
 * facing, the idea being to give the player an indication they can press the
 * action key to action with what they are looking at.
 * It is not designed for multiple icons over events.
 *
 * ----------------------------------------------------------------------------
 *  PLUGIN COMMANDS
 * ----------------------------------------------------------------------------
 *
 *    Galv.AI.status(x);    // x is true or false
 *   
 */

//-----------------------------------------------------------------------------
//  CODE STUFFS
//-----------------------------------------------------------------------------

Galv.AI.params = PluginManager.parameters(Galv.AI.pluginName);
Galv.AI.y = Number(Galv.AI.params["yOffset"]);
Galv.AI.z = Number(Galv.AI.params["zPosition"]);
Galv.AI.opacity = Number(Galv.AI.params["iconOpacity"]);
Galv.AI.autoHide = eval(Galv.AI.params["autoHide"]);
Galv.AI.needRefresh = false;

Galv.AI.status = function(status) {
	$gameSystem.actionIndicatorVisible = status;
};

Galv.AI.checkActionIcon = function() {
	const x2 = $gameMap.roundXWithDirection($gamePlayer._x, $gamePlayer._direction);
    const y2 = $gameMap.roundYWithDirection($gamePlayer._y, $gamePlayer._direction);
	let action = null;
	
	// CHECK EVENT STANDING ON
	$gameMap.eventsXy($gamePlayer._x, $gamePlayer._y).forEach(function(event) {
		action = Galv.AI.checkEventForIcon(event);
	});
	
	// CHECK EVENT IN FRONT
	if (!action) {
		$gameMap.eventsXy(x2, y2).forEach(function(event) {
			if (event.isNormalPriority()) action = Galv.AI.checkEventForIcon(event);
		});
	};
	
	// CHECK COUNTER
	if (!action && $gameMap.isCounter(x2, y2)) {
		const direction = $gamePlayer.direction();
		const x3 = $gameMap.roundXWithDirection(x2, direction);
        const y3 = $gameMap.roundYWithDirection(y2, direction);
		$gameMap.eventsXy(x3, y3).forEach(function(event) {
			if (event.isNormalPriority()) action = Galv.AI.checkEventForIcon(event);
		});
	};
	action = action || {'eventId': 0, 'iconId': 0};
	$gamePlayer.actionIconTarget = action;
};

Galv.AI.checkEventForIcon = function(event) {
    let text = '';
    let offset = { x: 0, y: 0 }; // デフォルトのオフセット値

    if (event.page()) {
        const listCount = event.page().list.length;
        const variableValue = $gameVariables.value(39); // 変数39の値を取得

        for (let i = 0; i < listCount; i++) {
            if (event.page().list[i].code === 108) {
                const line = event.page().list[i].parameters[0];

                // 注釈をパターンで取得
                if (variableValue === "Xbox" && line.match(/<actionText1: (.+)>/i)) {
                    const match = line.match(/<actionText1: (.+)>/i);
                    const params = match[1].split(',');
                    text = params[0].trim();
                    offset.y = params[1] ? Number(params[1].trim()) : 0;
                } else if (variableValue === "Standard" && line.match(/<actionText2: (.+)>/i)) {
                    const match = line.match(/<actionText2: (.+)>/i);
                    const params = match[1].split(',');
                    text = params[0].trim();
                    offset.y = params[1] ? Number(params[1].trim()) : 0;
                } else if (variableValue === "Generic" && line.match(/<actionText3: (.+)>/i)) {
                    const match = line.match(/<actionText3: (.+)>/i);
                    const params = match[1].split(',');
                    text = params[0].trim();
                    offset.y = params[1] ? Number(params[1].trim()) : 0;
                }
            }
        }
    }

    // テキストが存在すればインディケーターを返す
    return text ? { eventId: event._eventId, text: text, xy: offset } : null;
};


// GAME SYSTEM
//-----------------------------------------------------------------------------

Galv.Game_System_initialize = Game_System.prototype.initialize;
Game_System.prototype.initialize = function() {
	Galv.Game_System_initialize.call(this);
	this.actionIndicatorVisible = true;
};


// GAME MAP
//-----------------------------------------------------------------------------

Galv.Game_Map_requestRefresh = Game_Map.prototype.requestRefresh;
Game_Map.prototype.requestRefresh = function(mapId) {
	Galv.Game_Map_requestRefresh.call(this,mapId);
	Galv.AI.needRefresh = true;
};


// GAME PLAYER
//-----------------------------------------------------------------------------

Galv.Game_CharacterBase_moveStraight = Game_CharacterBase.prototype.moveStraight;
Game_CharacterBase.prototype.moveStraight = function(d) {
	Galv.Game_CharacterBase_moveStraight.call(this,d);
	Galv.AI.needRefresh = true;
};


// SPRITESET MAP
//-----------------------------------------------------------------------------

Galv.Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
Spriteset_Map.prototype.createLowerLayer = function() {
	Galv.Spriteset_Map_createLowerLayer.call(this);
	this.createActionIconSprite();
};

Spriteset_Map.prototype.createActionIconSprite = function() {
    this._actionIconSprite = new Sprite_ActionText(); // テキスト用スプライトに変更
    this._tilemap.addChild(this._actionIconSprite);
};

function Sprite_ActionText() {
    this.initialize(...arguments);
}

Sprite_ActionText.prototype = Object.create(Sprite.prototype);
Sprite_ActionText.prototype.constructor = Sprite_ActionText;

Sprite_ActionText.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    $gamePlayer.actionIconTarget = $gamePlayer.actionIconTarget || { eventId: 0, text: '', xy: { x: 0, y: 0 } }; 
    this._text = '';
    this.z = Galv.AI.z;
    this._tileWidth = $gameMap.tileWidth();
    this._tileHeight = $gameMap.tileHeight();
    this._offsetX = 0;
    this._offsetY = Galv.AI.y;
    this.anchor.y = 1;
    this._float = 0.1;
    this.mod = 0.2;
    Galv.AI.needRefresh = true;
    this.bitmap = new Bitmap(Graphics.width, 50); // 制御文字用に幅を十分に確保
};

Sprite_ActionText.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (Galv.AI.needRefresh) Galv.AI.checkActionIcon();

    if ($gamePlayer.actionIconTarget.eventId != this._eventId) {
        this._eventId = $gamePlayer.actionIconTarget.eventId;
        this._text = $gamePlayer.actionIconTarget.text;
        this.refreshText();
    }

    if (!this._text) return;

    const commentX = $gamePlayer.actionIconTarget.xy.x;
    const commentY = $gamePlayer.actionIconTarget.xy.y;

    this.x = $gamePlayer.screenX() + this._offsetX + commentX;
    this.y = $gamePlayer.screenY() + this._offsetY + this._float + commentY;

    this._float += this.mod;
    if (this._float < -0.1) {
        this.mod = Math.min(this.mod + 0.01, 0.2);
    } else if (this._float >= 0.1) {
        this.mod = Math.max(this.mod - 0.01, -0.2);
    }
};

Sprite_ActionText.prototype.refreshText = function() {
    // 前回の描画内容をクリア
    this.bitmap.clear();

    if (this._text) {
        const fontSize = 20; // フォントサイズ
        const iconScale = 1; // アイコンの拡大縮小倍率
        const padding = 10; // テキストの余白
        const resolution = 2; // 解像度スケール（2倍）

        // 仮のWindow_Baseを作成して制御文字を解析
        const tempWindow = new Window_Base(new Rectangle(0, 0, Graphics.width, Graphics.height));
        tempWindow.contents.fontSize = fontSize;

        // 制御文字を処理した後のテキストを取得
        const processedText = tempWindow.convertEscapeCharacters(this._text);

        // テキスト幅と高さを計算
        const textWidth = tempWindow.textWidth(processedText) + padding * 2;
        const lineHeight = tempWindow.lineHeight() + padding * 2;

        // 必要なビットマップサイズを設定（解像度スケールを考慮）
        this.bitmap = new Bitmap(textWidth * resolution, lineHeight * resolution);
        this.bitmap.fontSize = fontSize * resolution;

        // 描画用のWindow_Baseを再作成してフォントサイズを設定
        const drawWindow = new Window_Base(new Rectangle(0, 0, textWidth * resolution, lineHeight * resolution));
        drawWindow.contents.fontSize = fontSize * resolution;

        // アイコンを拡大縮小して描画する
        const originalDrawIcon = Window_Base.prototype.drawIcon;
        Window_Base.prototype.drawIcon = function(iconIndex, x, y) {
            const bitmap = ImageManager.loadSystem("IconSet");
            const pw = ImageManager.iconWidth;
            const ph = ImageManager.iconHeight;
            const sx = (iconIndex % 16) * pw;
            const sy = Math.floor(iconIndex / 16) * ph;
            this.contents.blt(bitmap, sx, sy, pw, ph, x, y, pw * iconScale, ph * iconScale);
        };

        // テキストを描画
        drawWindow.drawTextEx(this._text, padding * resolution, padding * resolution);

        // Window_Baseの内容をビットマップにコピー
        this.bitmap.blt(
            drawWindow.contents,
            0,
            0,
            textWidth * resolution,
            lineHeight * resolution,
            0,
            0
        );

        // 仮Windowを破棄
        tempWindow.destroy();
        drawWindow.destroy();

        // 元のdrawIconを復元
        Window_Base.prototype.drawIcon = originalDrawIcon;

        // 解像度スケールに基づいて表示サイズを調整
        this.scale.set(1.3 / resolution);
    }
};


// SPRITE ACTIONICON
//-----------------------------------------------------------------------------

function Sprite_ActionIcon() {
    this.initialize(...arguments);
}

Sprite_ActionIcon.prototype = Object.create(Sprite.prototype);
Sprite_ActionIcon.prototype.constructor = Sprite_ActionIcon;

Sprite_ActionIcon.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
	$gamePlayer.actionIconTarget = $gamePlayer.actionIconTarget || {eventId: 0, iconId: 0, xy:{x:0,y:0}}; 
	this._iconIndex = 0;
	this.z = Galv.AI.z;
	this.changeBitmap();
	this._tileWidth = $gameMap.tileWidth();
	this._tileHeight = $gameMap.tileHeight();
	this._offsetX = -(ImageManager.iconWidth / 2);
	this._offsetY = -38 + Galv.AI.y;
	this.anchor.y = 1;
	this._float = 0.1;
	this.mod = 0.2;
	Galv.AI.needRefresh = true;
};

Sprite_ActionIcon.prototype.changeBitmap = function() {
    if ($gamePlayer.actionIconTarget.eventId <= 0) {
        this._iconIndex = 0;
    } else {
        this._iconIndex = $gamePlayer.actionIconTarget.iconId;
    }

    const resolution = 2; // 解像度スケール（2倍）
    const pw = ImageManager.iconWidth;
    const ph = ImageManager.iconHeight;
    const sx = this._iconIndex % 16 * pw;
    const sy = Math.floor(this._iconIndex / 16) * ph;

    // 高解像度でビットマップを作成
    this.bitmap = new Bitmap(pw * resolution, ph * resolution);
    if (this._iconIndex <= 0) return;
    const bitmap = ImageManager.loadSystem("IconSet");
    this.bitmap.blt(
        bitmap,
        sx,
        sy,
        pw,
        ph,
        0,
        0,
        pw * resolution,
        ph * resolution
    );

    // スケールを設定して元のサイズに戻す
    this.scale.set(1 / resolution);
};


Sprite_ActionIcon.prototype.initPopVars = function() {
	this.scale.y = 0.1;
	this.opacity = 0;
	this.mod = 0.2;
	this._float = 0.1;
};

if (Galv.AI.autoHide) {
	Sprite_ActionIcon.prototype.updateOpacity = function() {
		if ($gameMap.isEventRunning()) {
			this.opacity -= 40;
		} else {
			this.opacity = $gameSystem.actionIndicatorVisible ? Galv.AI.opacity : 0;
		};
	};
} else {
	Sprite_ActionIcon.prototype.updateOpacity = function() {
		this.opacity = $gameSystem.actionIndicatorVisible ? Galv.AI.opacity : 0;
	};
};

Sprite_ActionText.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (Galv.AI.needRefresh) Galv.AI.checkActionIcon();

    // イベントが実行中であれば非表示にする
    if ($gameMap.isEventRunning()) {
        this.opacity = 0;
        return; // 表示を更新しない
    }

    // ターゲットが変わった場合にリフレッシュ
    if ($gamePlayer.actionIconTarget.eventId !== this._eventId) {
        this._eventId = $gamePlayer.actionIconTarget.eventId;
        this._text = $gamePlayer.actionIconTarget.text;
        this.refreshText();
    }

    if (!this._text) return;

    const commentX = $gamePlayer.actionIconTarget.xy.x;
    const commentY = $gamePlayer.actionIconTarget.xy.y;

    // テキストの位置を設定
    this.x = $gamePlayer.screenX() + this._offsetX + commentX;
    this.y = $gamePlayer.screenY() + this._offsetY + this._float + commentY;

    // 縮小エフェクト
    this._float += this.mod;
    if (this._float < -0.1) {
        this.mod = Math.min(this.mod + 0.01, 0.2);
    } else if (this._float >= 0.1) {
        this.mod = Math.max(this.mod - 0.01, -0.2);
    }

    // 不透明度を更新
    this.updateOpacity();
};

Sprite_ActionText.prototype.updateOpacity = function() {
    // イベントが実行中なら非表示
    if ($gameMap.isEventRunning()) {
        this.opacity = 0;
    } else {
        this.opacity = Galv.AI.opacity;
    }
};

// --- イベントページが更新されたら、強制リフレッシュ処理 ---
Galv.Game_Event_refresh = Game_Event.prototype.refresh;
Game_Event.prototype.refresh = function() {
    Galv.Game_Event_refresh.call(this);
    if (this._erased) return;

    // マップリフレッシュ要求
    $gameMap.requestRefresh();

    // プレイヤーとイベントの関係をチェック
    const player = $gamePlayer;
    const x2 = $gameMap.roundXWithDirection(player._x, player._direction);
    const y2 = $gameMap.roundYWithDirection(player._y, player._direction);

    const frontEvents = $gameMap.eventsXy(x2, y2);
    const standingEvents = $gameMap.eventsXy(player._x, player._y);

    const isFacingOrStanding = standingEvents.includes(this) || frontEvents.includes(this);

    if (isFacingOrStanding) {
        // プレイヤーが見てるイベントなら、新しいactionTextを取得し直す
        const newAction = Galv.AI.checkEventForIcon(this);
        if (newAction) {
            player.actionIconTarget = newAction;
        } else {
            // 新しいページにactionTextが無い場合は非表示
            player.actionIconTarget = { eventId: 0, text: '', xy: { x: 0, y: 0 } };
        }
        Galv.AI.needRefresh = true; // 描画更新要求
    }
};

// --- Sprite_ActionTextを正しくリフレッシュする ---
Sprite_ActionText.prototype.update = function() {
    Sprite.prototype.update.call(this);

    if (Galv.AI.needRefresh) {
        Galv.AI.checkActionIcon();
        Galv.AI.needRefresh = false;
    }

    if ($gameMap.isEventRunning()) {
        this.opacity = 0;
        return;
    }

    const actionTarget = $gamePlayer.actionIconTarget;

    // イベントIDまたはテキスト内容が変わったらリフレッシュ
    if (actionTarget.eventId !== this._eventId || actionTarget.text !== this._text) {
        this._eventId = actionTarget.eventId;
        this._text = actionTarget.text;
        this.refreshText();
    }

    if (!this._text) {
        this.opacity = 0;
        return;
    } else {
        this.opacity = Galv.AI.opacity;
    }

    const commentX = actionTarget.xy.x;
    const commentY = actionTarget.xy.y;

    this.x = $gamePlayer.screenX() + this._offsetX + commentX;
    this.y = $gamePlayer.screenY() + this._offsetY + this._float + commentY;

    this._float += this.mod;
    if (this._float < -0.1) {
        this.mod = Math.min(this.mod + 0.01, 0.2);
    } else if (this._float >= 0.1) {
        this.mod = Math.max(this.mod - 0.01, -0.2);
    }
};
